%feature("docstring") getLibSBMLVersion "
Returns the version number of this copy of libSBML as an integer.

@return the libSBML version as an integer; version 1.2.3 becomes 10203.
";


%feature("docstring") getLibSBMLDottedVersion "
Returns the version number of this copy of libSBML as a string.

@return the libSBML version as a string; version 1.2.3 becomes
\'1.2.3\'.

@see getLibSBMLVersionString()
";


%feature("docstring") getLibSBMLVersionString "
Returns the version number of this copy of libSBML as a string without
periods.

@return the libSBML version as a string: version 1.2.3 becomes \'10203\'.

@see getLibSBMLDottedVersion()
";


%feature("docstring") isLibSBMLCompiledWith "
Returns an indication whether libSBML has been compiled with
against a specific library. 

@param option the library to test against, this can be one of
\'expat\', \'libxml\', \'xerces-c\', \'bzip2\', \'zip\'

@return 0 in case the libSBML has not been compiled against 
that library and nonzero otherwise (for libraries 
that define an integer version number that number will 
be returned).

@see getLibSBMLDependencyVersionOf()
";


%feature("docstring") getLibSBMLDependencyVersionOf "
Returns the version string for the dependency library used. 

@param option the library for which the version
should be retrieved, this can be one of
\'expat\', \'libxml\', \'xerces-c\', \'bzip2\', \'zip\'

@return None in case libSBML has not been compiled against 
that library and a version string otherwise.

@see isLibSBMLCompiledWith()
";


%feature("docstring") OperationReturnValue_toString "
This method takes an SBML operation return value and returns a string representing
the code.

@param returnValue the operation return value to convert to a string.

@return a human readable name for the given
@if clike #OperationReturnValues_t value@else operation return value @endif.

@note The caller does not own the returned string and is therefore not
allowed to modify it.
";


%feature("docstring") IdList "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html Maintains a list of SIds.
@internal
";


%feature("docstring") IdList::append "
@internal

@internal
";


%feature("docstring") IdList::contains "
@internal

@internal
";


%feature("docstring") IdList::empty "
@internal

@internal
";


%feature("docstring") IdList::removeIdsBefore "
@internal

@internal
";


%feature("docstring") IdList::size "
@internal

@internal
";


%feature("docstring") IdentifierTransformer "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html Base class for identifier transformers.
@internal
";


%feature("docstring") ElementFilter "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html Base class for filter functions.

@htmlinclude libsbml-facility-only-warning.html

Some libSBML objects provide the ability to return lists of components.
To provide callers with greater control over exactly what is
returned, these methods take optional arguments in the form of filters.
The ElementFilter class is the parent class for these filters.
";


%feature("docstring") ElementFilter::ElementFilter "
Creates a new ElementFilter object.
";


%feature("docstring") ElementFilter::filter "
Predicate to test elements.

This is the central predicate of the ElementFilter class.  In subclasses
of ElementFilter, callers should implement this method such that it
returns @c True for @p element arguments that are \'desirable\' and
@c False for those that are \'undesirable\' in whatever filtering context the
ElementFilter subclass is designed to be used.

@param element the element to be tested.

@return @c True if the @p element is desirable or should be kept,
@c False otherwise.
";


%feature("docstring") ElementFilter::getUserData "
Returns the user data that has been previously set via setUserData().

Callers can attach private data to ElementFilter objects using
setUserData().  This user data can be used by an application to store
custom information to be accessed by the ElementFilter in its work.  In
case of a deep copy, the data will passed as it is.  The attribute will
never be interpreted by libSBML.

@return the user data of this node, or @c None if no user data has been
set.

@warning This <em>user data</em> is specific to an ElementFilter object
instance, and is not the same as the user data that may be attached to
an SBML object using SBase.setUserData().

@see setUserData()
";


%feature("docstring") ElementFilter::setUserData "
Sets the user data of this element.

Callers can attach private data to ElementFilter objects using this
method, and retrieve them using getUserData().  Such user data can be
used by an application to store information to be accessed by the
ElementFilter in its work.  In case of a deep copy, this data will
passed as it is.  The attribute will never be interpreted by libSBML.

@param userData specifies the new user data.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@warning This <em>user data</em> is specific to an ElementFilter object
instance, and is not the same as the user data that may be attached to
an SBML object using SBase.setUserData().

@see getUserData()
";


%feature("docstring") SBMLReader "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html File and text-string SBML reader.

@htmlinclude not-sbml-warning.html

The SBMLReader class provides the main interface for reading SBML content
from files and strings.  The methods for reading SBML all return an
SBMLDocument object representing the results.  In the case of failures
(such as if the SBML contains errors or a file cannot be read), the errors
will be recorded with the SBMLErrorLog object kept in the SBMLDocument
returned by SBMLReader.  Consequently, immediately after calling a method
on SBMLReader, callers should always check for errors and warnings using
the methods for this purpose provided by SBMLDocument.

For convenience as well as easy access from other languages besides C++,
this file also defines two global functions,
@sbmlglobalfunction{readSBML, String} and
@sbmlglobalfunction{readSBMLFromString, String}.  They are
equivalent to creating an SBMLReader object and then calling the
@if python @link SBMLReader.readSBML() SBMLReader.readSBML()@endlink@endif@if java SBMLReader.readSBML()@endif@if cpp SBMLReader.readSBML()@endif@if csharp SBMLReader.readSBML()@endif@~ and
@if python @link SBMLReader.readSBMLFromString() SBMLReader.readSBMLFromString()@endlink@endif@if java SBMLReader.readSBMLFromString()@endif@if cpp SBMLReader.readSBMLFromString()@endif@if csharp SBMLReader.readSBMLFromString()@endif@~ methods, respectively.

@section compression Support for reading compressed files

LibSBML provides support for reading (as well as writing) compressed
SBML files.  The process is transparent to the calling
application---the application does not need to do anything
deliberate to invoke the functionality.  If a given SBML filename ends
with an extension for the @em gzip, @em zip or @em bzip2 compression
formats (respectively, @c .gz, @c .zip, or @c .bz2), then the methods
@if python @link SBMLReader.readSBML() SBMLReader.readSBML()@endlink@endif@if java @link SBMLReader.readSBML() SBMLReader.readSBML(String)@endlink@endif@if cpp SBMLReader.readSBML()@endif@if csharp SBMLReader.readSBML()@endif@~ and
@if python @link SBMLWriter.writeSBML() SBMLWriter.writeSBML()@endlink@endif@if java @link SBMLWriter.writeSBML() SBMLWriter.writeSBML(String)@endlink@endif@if cpp SBMLWriter.writeSBML()@endif@if csharp SBMLWriter.writeSBML()@endif@~
will automatically decompress and compress the file while reading and
writing it.  If the filename has no such extension, it will be read and
written uncompressed as normal.

The compression feature requires that the @em zlib (for @em gzip and @em
zip formats) and/or @em bzip2 (for @em bzip2 format) be available on the
system running libSBML, and that libSBML was configured with their
support compiled-in.  Please see the libSBML
@if java <a href=\'../../../libsbml-installation.html\'>installation instructions</a> @else <a href=\'libsbml-installation.html\'>installation instructions</a>@endif@~
for more information about this.  The methods
@if java SBMLReader.hasZlib()@else hasZlib()@endif@~ and
@if java SBMLReader.hasBzip2()@else hasBzip2()@endif@~
can be used by an application to query at run-time whether support
for the compression libraries is available in the present copy of
libSBML.

Support for compression is not mandated by the SBML standard, but
applications may find it helpful, particularly when large SBML models
are being communicated across data links of limited bandwidth.
";


%feature("docstring") SBMLReader::SBMLReader "
Creates a new SBMLReader object and returns it.

The libSBML SBMLReader object offers methods for reading SBML in
XML form from files and text strings.
";


%feature("docstring") SBMLReader::readSBML "
Reads an SBML document from the given file.

If the file named @p filename does not exist or its content is not valid
SBML, one or more errors will be logged with the SBMLDocument object
returned by this method.  Callers can use the methods on SBMLDocument such
as
@if python @link libsbml.SBMLDocument.getNumErrors() SBMLDocument.getNumErrors()@endlink@endif,
@if conly SBMLDocument_getNumErrors() @else SBMLDocument.getNumErrors()@endif@~
and
@if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
@if java SBMLDocument.getError()@endif
@if cpp SBMLDocument.getError()@endif
@if csharp SBMLDocument.getError()@endif
@if conly SBMLDocument_getError()@endif@~
to get the errors.  The object returned by
@if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
@if java SBMLDocument.getError()@endif
@if cpp SBMLDocument.getError()@endif
@if csharp SBMLDocument.getError()@endif
@if conly SBMLDocument_getError()@endif@~
is an SBMLError object, and it has methods to get the error code,
category, and severity level of the problem, as well as a textual
description of the problem.  The possible severity levels range from
informational messages to fatal errors; see the documentation for
@if conly SBMLError_t @else SBMLError@endif@~
for more information.

If the file @p filename could not be read, the file-reading error will
appear first.  The error code @if clike (a value drawn from the
enumeration #XMLErrorCode_t)@endif@~ can provide a clue about what
happened.  For example, a file might be unreadable (either because it does
not actually exist or because the user does not have the necessary access
privileges to read it) or some sort of file operation error may have been
reported by the underlying operating system.  Callers can check for these
situations using a program fragment such as the following:
@if cpp
@code{.cpp}
SBMLReader reader;
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc->getNumErrors() > 0)
{
  if (doc->getError(0)->getErrorId() == XMLError::XMLFileUnreadable)
  {
    // Handle case of unreadable file here.
  }
  else if (doc->getError(0)->getErrorId() == XMLError::XMLFileOperationError)
  {
    // Handle case of other file operation error here.
  }
  else
  {
    // Handle other cases -- see error codes defined in XMLErrorCode_t
    // for other possible cases to check.
  }
}
@endcode
@endif
@if conly
@code{.c}
SBMLReader_t   *sr;
SBMLDocument_t *d;

sr = SBMLReader_create();

d = SBMLReader_readSBML(sr, filename);

if (SBMLDocument_getNumErrors(d) > 0)
{
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_FILE_NOT_FOUND)
  {
     ...
  }
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_NOT_SBML)
  {
     ...
  }
}
@endcode
@endif
@if java
@code{.java}
SBMLReader reader = new SBMLReader();
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileUnreadable)
    {
        // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileOperationError)
    {
        // Handle case of other file operation error here.
    }
    else
    {
        // Handle other error cases.
    }
}
@endcode
@endif
@if python
@code{.py}
reader = SBMLReader()
if reader == None:
  # Handle the truly exceptional case of no object created here.

doc = reader.readSBMLFromFile(filename)
if doc.getNumErrors() > 0:
  if doc.getError(0).getErrorId() == XMLFileUnreadable:
    # Handle case of unreadable file here.
  elif doc.getError(0).getErrorId() == XMLFileOperationError:
    # Handle case of other file error here.
  else:
    # Handle other error cases here.
@endcode
@endif
@if csharp
@code{.cs}
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileUnreadable)
    {
         // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileOperationError)
    {
         // Handle case of other file operation error here.
    }
    else
    {
         // Handle other cases -- see error codes defined in XMLErrorCode_t
         // for other possible cases to check.
    }
 }
@endcode
@endif@~

@par
If the given filename ends with the suffix @c \'.gz\' (for example,
@c \'myfile.xml.gz\'), the file is assumed to be compressed in @em gzip
format and will be automatically decompressed upon reading.
Similarly, if the given filename ends with @c \'.zip\' or @c \'.bz2\', the
file is assumed to be compressed in @em zip or @em bzip2 format
(respectively).  Files whose names lack these suffixes will be read
uncompressed.  Note that if the file is in @em zip format but the
archive contains more than one file, only the first file in the
archive will be read and the rest ignored.

@par
To read a gzip/zip file, libSBML needs to be configured and linked with the
<a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
time.  It also needs to be linked with the <a target=\'_blank\'
href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
these are the default configurations for libSBML.)  Errors about unreadable
files will be logged if a compressed filename is given and libSBML was
<em>not</em> linked with the corresponding required library.

This method is identical to
@if python @link SBMLReader.readSBMLFromFile() SBMLReader.readSBMLFromFile()@endlink@endif@if java @link SBMLReader.readSBMLFromFile() SBMLReader.readSBMLFromFile(String)@endlink@endif@if cpp SBMLReader.readSBMLFromFile()@endif@if csharp SBMLReader.readSBMLFromFile()@endif.

@param filename the name or full pathname of the file to be read.

@return a pointer to the SBMLDocument object created from the SBML
content in @p filename.

@note LibSBML versions 2.x and later versions behave differently in
error handling in several respects.  One difference is how early some
errors are caught and whether libSBML continues processing a file in
the face of some early errors.  In general, libSBML versions after 2.x
stop parsing SBML inputs sooner than libSBML version 2.x in the face
of XML errors, because the errors may invalidate any further SBML
content.  For example, a missing XML declaration at the beginning of
the file was ignored by libSBML 2.x but in version 3.x and later, it
will cause libSBML to stop parsing the rest of the input altogether.
While this behavior may seem more severe and intolerant, it was
necessary in order to provide uniform behavior regardless of which
underlying XML parser (Expat, Xerces, libxml2) is being used by
libSBML.  The XML parsers themselves behave differently in their error
reporting, and sometimes libSBML has to resort to the lowest common
denominator.

@see readSBMLFromString()
@see SBMLError
@see SBMLDocument
";


%feature("docstring") SBMLReader::readSBMLFromFile "
Reads an SBML document from the given file.

If the file named @p filename does not exist or its content is not valid
SBML, one or more errors will be logged with the SBMLDocument object
returned by this method.  Callers can use the methods on SBMLDocument such
as
@if python @link libsbml.SBMLDocument.getNumErrors() SBMLDocument.getNumErrors()@endlink@endif,
@if conly SBMLDocument_getNumErrors() @else SBMLDocument.getNumErrors()@endif@~
and
@if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
@if java SBMLDocument.getError()@endif
@if cpp SBMLDocument.getError()@endif
@if csharp SBMLDocument.getError()@endif
@if conly SBMLDocument_getError()@endif@~
to get the errors.  The object returned by
@if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
@if java SBMLDocument.getError()@endif
@if cpp SBMLDocument.getError()@endif
@if csharp SBMLDocument.getError()@endif
@if conly SBMLDocument_getError()@endif@~
is an SBMLError object, and it has methods to get the error code,
category, and severity level of the problem, as well as a textual
description of the problem.  The possible severity levels range from
informational messages to fatal errors; see the documentation for
@if conly SBMLError_t @else SBMLError@endif@~
for more information.

If the file @p filename could not be read, the file-reading error will
appear first.  The error code @if clike (a value drawn from the
enumeration #XMLErrorCode_t)@endif@~ can provide a clue about what
happened.  For example, a file might be unreadable (either because it does
not actually exist or because the user does not have the necessary access
privileges to read it) or some sort of file operation error may have been
reported by the underlying operating system.  Callers can check for these
situations using a program fragment such as the following:
@if cpp
@code{.cpp}
SBMLReader reader;
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc->getNumErrors() > 0)
{
  if (doc->getError(0)->getErrorId() == XMLError::XMLFileUnreadable)
  {
    // Handle case of unreadable file here.
  }
  else if (doc->getError(0)->getErrorId() == XMLError::XMLFileOperationError)
  {
    // Handle case of other file operation error here.
  }
  else
  {
    // Handle other cases -- see error codes defined in XMLErrorCode_t
    // for other possible cases to check.
  }
}
@endcode
@endif
@if conly
@code{.c}
SBMLReader_t   *sr;
SBMLDocument_t *d;

sr = SBMLReader_create();

d = SBMLReader_readSBML(sr, filename);

if (SBMLDocument_getNumErrors(d) > 0)
{
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_FILE_NOT_FOUND)
  {
     ...
  }
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_NOT_SBML)
  {
     ...
  }
}
@endcode
@endif
@if java
@code{.java}
SBMLReader reader = new SBMLReader();
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileUnreadable)
    {
        // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileOperationError)
    {
        // Handle case of other file operation error here.
    }
    else
    {
        // Handle other error cases.
    }
}
@endcode
@endif
@if python
@code{.py}
reader = SBMLReader()
if reader == None:
  # Handle the truly exceptional case of no object created here.

doc = reader.readSBMLFromFile(filename)
if doc.getNumErrors() > 0:
  if doc.getError(0).getErrorId() == XMLFileUnreadable:
    # Handle case of unreadable file here.
  elif doc.getError(0).getErrorId() == XMLFileOperationError:
    # Handle case of other file error here.
  else:
    # Handle other error cases here.
@endcode
@endif
@if csharp
@code{.cs}
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileUnreadable)
    {
         // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileOperationError)
    {
         // Handle case of other file operation error here.
    }
    else
    {
         // Handle other cases -- see error codes defined in XMLErrorCode_t
         // for other possible cases to check.
    }
 }
@endcode
@endif@~

@par
If the given filename ends with the suffix @c \'.gz\' (for example,
@c \'myfile.xml.gz\'), the file is assumed to be compressed in @em gzip
format and will be automatically decompressed upon reading.
Similarly, if the given filename ends with @c \'.zip\' or @c \'.bz2\', the
file is assumed to be compressed in @em zip or @em bzip2 format
(respectively).  Files whose names lack these suffixes will be read
uncompressed.  Note that if the file is in @em zip format but the
archive contains more than one file, only the first file in the
archive will be read and the rest ignored.

@par
To read a gzip/zip file, libSBML needs to be configured and linked with the
<a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
time.  It also needs to be linked with the <a target=\'_blank\'
href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
these are the default configurations for libSBML.)  Errors about unreadable
files will be logged if a compressed filename is given and libSBML was
<em>not</em> linked with the corresponding required library.

This method is identical to
@if python @link SBMLReader.readSBML() SBMLReader.readSBML()@endlink@endif@if java @link SBMLReader.readSBML() SBMLReader.readSBML(String)@endlink@endif@if cpp SBMLReader.readSBML()@endif@if csharp SBMLReader.readSBML()@endif.

@param filename the name or full pathname of the file to be read.

@return a pointer to the SBMLDocument object created from the SBML
content in @p filename.

@note LibSBML versions 2.x and later versions behave differently in
error handling in several respects.  One difference is how early some
errors are caught and whether libSBML continues processing a file in
the face of some early errors.  In general, libSBML versions after 2.x
stop parsing SBML inputs sooner than libSBML version 2.x in the face
of XML errors, because the errors may invalidate any further SBML
content.  For example, a missing XML declaration at the beginning of
the file was ignored by libSBML 2.x but in version 3.x and later, it
will cause libSBML to stop parsing the rest of the input altogether.
While this behavior may seem more severe and intolerant, it was
necessary in order to provide uniform behavior regardless of which
underlying XML parser (Expat, Xerces, libxml2) is being used by
libSBML.  The XML parsers themselves behave differently in their error
reporting, and sometimes libSBML has to resort to the lowest common
denominator.

@see readSBMLFromString()
@see SBMLError
@see SBMLDocument
";


%feature("docstring") SBMLReader::readSBMLFromString "
Reads an SBML document from a text string.

This method is flexible with respect to the presence of an XML
declaration at the beginning of the string.  In particular, if the
string in @p xml does not begin with the XML declaration
@verbatim
<?xml version=\'1.0\' encoding=\'UTF-8\'?>
@endverbatim
then this method will automatically prepend the declaration
to @p xml.

This method will log a fatal error if the content given in the parameter
@p xml is not in SBML format.  See the method documentation for
@if conly SBMLReader_readSBML() 
@elseif java SBMLReader.readSBML()
@else SBMLReader.readSBML()
@endif
for an example of code for testing the returned error code.

@param xml a string containing a full SBML model.

@return a pointer to the SBMLDocument created from the SBML content,
or a null pointer if @p xml is @c None.

@note When using this method to read an SBMLDocument that uses the SBML
Level&nbsp;3 Hierarchical %Model Composition package (comp) the document
location cannot be set automatically. Thus, if the model contains
references to ExternalModelDefinition objects, it will be necessary to
manually set the document URI location
(@if conly SBMLDocument_setLocationURI()
@elseif java SBMLDocument.setLocationURI() 
@else SBMLDocument.setLocationURI()
@endif
) in order to facilitate resolving these models.

@see SBMLReader.readSBML()
";


%feature("docstring") SBMLReader::hasZlib "
Static method; returns @c True if this copy of libSBML supports
<i>gzip</I> and <i>zip</i> format compression.

@return @c True if libSBML has been linked with the <i>zlib</i>
library, @c False otherwise.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike hasBzip2() @else SBMLReader.hasBzip2()@endif@~
";


%feature("docstring") SBMLReader::hasBzip2 "
Static method; returns @c True if this copy of libSBML supports
<i>bzip2</i> format compression.

@return @c True if libSBML is linked with the <i>bzip2</i>
libraries, @c False otherwise.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike hasZlib() @else SBMLReader.hasZlib()@endif@~
";


%feature("docstring") SBMLReader::readInternal "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") readSBML "
Reads an SBML document from the given file.

If the file named @p filename does not exist or its content is not valid
SBML, one or more errors will be logged with the SBMLDocument object
returned by this method.  Callers can use the methods on SBMLDocument such
as
@if python @link libsbml.SBMLDocument.getNumErrors() SBMLDocument.getNumErrors()@endlink@endif,
@if conly SBMLDocument_getNumErrors() @else SBMLDocument.getNumErrors()@endif@~
and
@if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
@if java SBMLDocument.getError()@endif
@if cpp SBMLDocument.getError()@endif
@if csharp SBMLDocument.getError()@endif
@if conly SBMLDocument_getError()@endif@~
to get the errors.  The object returned by
@if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
@if java SBMLDocument.getError()@endif
@if cpp SBMLDocument.getError()@endif
@if csharp SBMLDocument.getError()@endif
@if conly SBMLDocument_getError()@endif@~
is an SBMLError object, and it has methods to get the error code,
category, and severity level of the problem, as well as a textual
description of the problem.  The possible severity levels range from
informational messages to fatal errors; see the documentation for
@if conly SBMLError_t @else SBMLError@endif@~
for more information.

If the file @p filename could not be read, the file-reading error will
appear first.  The error code @if clike (a value drawn from the
enumeration #XMLErrorCode_t)@endif@~ can provide a clue about what
happened.  For example, a file might be unreadable (either because it does
not actually exist or because the user does not have the necessary access
privileges to read it) or some sort of file operation error may have been
reported by the underlying operating system.  Callers can check for these
situations using a program fragment such as the following:
@if cpp
@code{.cpp}
SBMLReader reader;
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc->getNumErrors() > 0)
{
  if (doc->getError(0)->getErrorId() == XMLError::XMLFileUnreadable)
  {
    // Handle case of unreadable file here.
  }
  else if (doc->getError(0)->getErrorId() == XMLError::XMLFileOperationError)
  {
    // Handle case of other file operation error here.
  }
  else
  {
    // Handle other cases -- see error codes defined in XMLErrorCode_t
    // for other possible cases to check.
  }
}
@endcode
@endif
@if conly
@code{.c}
SBMLReader_t   *sr;
SBMLDocument_t *d;

sr = SBMLReader_create();

d = SBMLReader_readSBML(sr, filename);

if (SBMLDocument_getNumErrors(d) > 0)
{
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_FILE_NOT_FOUND)
  {
     ...
  }
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_NOT_SBML)
  {
     ...
  }
}
@endcode
@endif
@if java
@code{.java}
SBMLReader reader = new SBMLReader();
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileUnreadable)
    {
        // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileOperationError)
    {
        // Handle case of other file operation error here.
    }
    else
    {
        // Handle other error cases.
    }
}
@endcode
@endif
@if python
@code{.py}
reader = SBMLReader()
if reader == None:
  # Handle the truly exceptional case of no object created here.

doc = reader.readSBMLFromFile(filename)
if doc.getNumErrors() > 0:
  if doc.getError(0).getErrorId() == XMLFileUnreadable:
    # Handle case of unreadable file here.
  elif doc.getError(0).getErrorId() == XMLFileOperationError:
    # Handle case of other file error here.
  else:
    # Handle other error cases here.
@endcode
@endif
@if csharp
@code{.cs}
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileUnreadable)
    {
         // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileOperationError)
    {
         // Handle case of other file operation error here.
    }
    else
    {
         // Handle other cases -- see error codes defined in XMLErrorCode_t
         // for other possible cases to check.
    }
 }
@endcode
@endif@~

@par
If the given filename ends with the suffix @c \'.gz\' (for example,
@c \'myfile.xml.gz\'), the file is assumed to be compressed in @em gzip
format and will be automatically decompressed upon reading.
Similarly, if the given filename ends with @c \'.zip\' or @c \'.bz2\', the
file is assumed to be compressed in @em zip or @em bzip2 format
(respectively).  Files whose names lack these suffixes will be read
uncompressed.  Note that if the file is in @em zip format but the
archive contains more than one file, only the first file in the
archive will be read and the rest ignored.

@par
To read a gzip/zip file, libSBML needs to be configured and linked with the
<a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
time.  It also needs to be linked with the <a target=\'_blank\'
href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
these are the default configurations for libSBML.)  Errors about unreadable
files will be logged if a compressed filename is given and libSBML was
<em>not</em> linked with the corresponding required library.

@if conly
@memberof SBMLReader_t
@endif
";


%feature("docstring") readSBMLFromFile "
Reads an SBML document from the given file.

If the file named @p filename does not exist or its content is not valid
SBML, one or more errors will be logged with the SBMLDocument object
returned by this method.  Callers can use the methods on SBMLDocument such
as
@if python @link libsbml.SBMLDocument.getNumErrors() SBMLDocument.getNumErrors()@endlink@endif,
@if conly SBMLDocument_getNumErrors() @else SBMLDocument.getNumErrors()@endif@~
and
@if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
@if java SBMLDocument.getError()@endif
@if cpp SBMLDocument.getError()@endif
@if csharp SBMLDocument.getError()@endif
@if conly SBMLDocument_getError()@endif@~
to get the errors.  The object returned by
@if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
@if java SBMLDocument.getError()@endif
@if cpp SBMLDocument.getError()@endif
@if csharp SBMLDocument.getError()@endif
@if conly SBMLDocument_getError()@endif@~
is an SBMLError object, and it has methods to get the error code,
category, and severity level of the problem, as well as a textual
description of the problem.  The possible severity levels range from
informational messages to fatal errors; see the documentation for
@if conly SBMLError_t @else SBMLError@endif@~
for more information.

If the file @p filename could not be read, the file-reading error will
appear first.  The error code @if clike (a value drawn from the
enumeration #XMLErrorCode_t)@endif@~ can provide a clue about what
happened.  For example, a file might be unreadable (either because it does
not actually exist or because the user does not have the necessary access
privileges to read it) or some sort of file operation error may have been
reported by the underlying operating system.  Callers can check for these
situations using a program fragment such as the following:
@if cpp
@code{.cpp}
SBMLReader reader;
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc->getNumErrors() > 0)
{
  if (doc->getError(0)->getErrorId() == XMLError::XMLFileUnreadable)
  {
    // Handle case of unreadable file here.
  }
  else if (doc->getError(0)->getErrorId() == XMLError::XMLFileOperationError)
  {
    // Handle case of other file operation error here.
  }
  else
  {
    // Handle other cases -- see error codes defined in XMLErrorCode_t
    // for other possible cases to check.
  }
}
@endcode
@endif
@if conly
@code{.c}
SBMLReader_t   *sr;
SBMLDocument_t *d;

sr = SBMLReader_create();

d = SBMLReader_readSBML(sr, filename);

if (SBMLDocument_getNumErrors(d) > 0)
{
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_FILE_NOT_FOUND)
  {
     ...
  }
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_NOT_SBML)
  {
     ...
  }
}
@endcode
@endif
@if java
@code{.java}
SBMLReader reader = new SBMLReader();
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileUnreadable)
    {
        // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileOperationError)
    {
        // Handle case of other file operation error here.
    }
    else
    {
        // Handle other error cases.
    }
}
@endcode
@endif
@if python
@code{.py}
reader = SBMLReader()
if reader == None:
  # Handle the truly exceptional case of no object created here.

doc = reader.readSBMLFromFile(filename)
if doc.getNumErrors() > 0:
  if doc.getError(0).getErrorId() == XMLFileUnreadable:
    # Handle case of unreadable file here.
  elif doc.getError(0).getErrorId() == XMLFileOperationError:
    # Handle case of other file error here.
  else:
    # Handle other error cases here.
@endcode
@endif
@if csharp
@code{.cs}
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileUnreadable)
    {
         // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileOperationError)
    {
         // Handle case of other file operation error here.
    }
    else
    {
         // Handle other cases -- see error codes defined in XMLErrorCode_t
         // for other possible cases to check.
    }
 }
@endcode
@endif@~

@par
If the given filename ends with the suffix @c \'.gz\' (for example,
@c \'myfile.xml.gz\'), the file is assumed to be compressed in @em gzip
format and will be automatically decompressed upon reading.
Similarly, if the given filename ends with @c \'.zip\' or @c \'.bz2\', the
file is assumed to be compressed in @em zip or @em bzip2 format
(respectively).  Files whose names lack these suffixes will be read
uncompressed.  Note that if the file is in @em zip format but the
archive contains more than one file, only the first file in the
archive will be read and the rest ignored.

@par
To read a gzip/zip file, libSBML needs to be configured and linked with the
<a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
time.  It also needs to be linked with the <a target=\'_blank\'
href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
these are the default configurations for libSBML.)  Errors about unreadable
files will be logged if a compressed filename is given and libSBML was
<em>not</em> linked with the corresponding required library.

@param filename the name or full pathname of the file to be read.

@return a pointer to the SBMLDocument structure created from the SBML
content in @p filename.

@if conly
@memberof SBMLReader_t
@endif
";


%feature("docstring") readSBMLFromString "
Reads an SBML document from a text string.

This method is flexible with respect to the presence of an XML
declaration at the beginning of the string.  In particular, if the
string in @p xml does not begin with the XML declaration
@verbatim
<?xml version=\'1.0\' encoding=\'UTF-8\'?>
@endverbatim
then this method will automatically prepend the declaration
to @p xml.

This method will log a fatal error if the content given in the parameter
@p xml is not in SBML format.  See the method documentation for
@if conly SBMLReader_readSBML() 
@elseif java SBMLReader.readSBML()
@else SBMLReader.readSBML()
@endif
for an example of code for testing the returned error code.

@param xml a string containing a full SBML model.

@return a pointer to the SBMLDocument structure created from the SBML
content in @p xml.

@note When using this method to read an SBMLDocument that uses the SBML
Level&nbsp;3 Hierarchical %Model Composition package (comp) the document
location cannot be set automatically. Thus, if the model contains
references to ExternalModelDefinition objects, it will be necessary to
manually set the document URI location
(@if conly SBMLDocument_setLocationURI()
@elseif java SBMLDocument.setLocationURI() 
@else SBMLDocument.setLocationURI()
@endif
) in order to facilitate resolving these models.

@if conly
@memberof SBMLReader_t
@endif
";


%feature("docstring") SBMLWriter "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html File and text-string SBML writer.

@htmlinclude not-sbml-warning.html

The SBMLWriter class is the converse of SBMLReader, and provides the
main interface for serializing SBML models into XML and writing the
result to an output stream or to files and text strings.  The methods 
for writing SBML all take an SBMLDocument object and a destination.  
They return a boolean or integer value to indicate success or failure.

@section sbmlwriter-compression Support for writing compressed files

LibSBML provides support for writing (as well as reading) compressed
SBML files.  The process is transparent to the calling
application---the application does not need to do anything
deliberate to invoke the functionality.  If a given SBML filename ends
with an extension for the @em gzip, @em zip or @em bzip2 compression
formats (respectively, <code>&quot;.gz&quot;</code>,
<code>&quot;.zip&quot;</code>, or <code>&quot;.bz2&quot;</code>),
then the methods
SBMLWriter.writeSBML()
and SBMLReader.readSBML()
will automatically compress and decompress the file while writing and
reading it.  If the filename has no such extension, it
will be written and read uncompressed as normal.

The compression feature requires that the @em zlib (for @em gzip and @em
zip formats) and/or @em bzip2 (for @em bzip2 format) be available on the
system running libSBML, and that libSBML was configured with their
support compiled-in.  Please see the libSBML @if clike <a href=\'libsbml-installation.html\'>installation instructions</a>@endif@if python <a href=\'libsbml-installation.html\'>installation instructions</a>@endif@if java  <a href=\'../../../libsbml-installation.html\'>installation instructions</a>@endif@~ for 
more information about this.  The methods
SBMLWriter.hasZlib() and
SBMLWriter.hasBzip2()
can be used by an application to query at run-time whether support
for the compression libraries is available in the present copy of
libSBML.

Support for compression is not mandated by the SBML standard, but
applications may find it helpful, particularly when large SBML models
are being communicated across data links of limited bandwidth.
";


%feature("docstring") SBMLWriter::SBMLWriter "
Creates a new SBMLWriter.

The libSBML SBMLWriter objects offer methods for writing SBML in
XML form to files and text strings.
";


%feature("docstring") SBMLWriter::setProgramName "
Sets the name of this program, i.e., the program that is about to
write out the SBMLDocument.

If the program name and version are set (see
SBMLWriter.setProgramVersion()), the
following XML comment, intended for human consumption, will be written
at the beginning of the XML document:
@verbatim
 <!-- Created by <program name> version <program version>
 on yyyy-MM-dd HH:mm with libSBML version <libsbml version>. -->
 @endverbatim

If the program name and version are not set at some point before
calling the writeSBML() methods, no such comment is written out.

@param name the name of this program (where \'this program\' refers to
program in which libSBML is embedded, not libSBML itself!)

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@see setProgramVersion()
";


%feature("docstring") SBMLWriter::setProgramVersion "
Sets the version of this program, i.e., the program that is about to
write out the SBMLDocument.

If the program version and name are set (see
SBMLWriter.setProgramName()), the
following XML comment, intended for human consumption, will be written
at the beginning of the document:
@verbatim
 <!-- Created by <program name> version <program version>
 on yyyy-MM-dd HH:mm with libSBML version <libsbml version>. -->
 @endverbatim

If the program version and name are not set at some point before
calling the writeSBML() methods, no such comment is written out.

@param version the version of this program (where \'this program\'
refers to program in which libSBML is embedded, not libSBML itself!)

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@see setProgramName()
";


%feature("docstring") SBMLWriter::writeSBML "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>writeSBML(SBMLDocument d, string filename)</pre>

Writes the given SBML document to filename.

@htmlinclude assuming-compressed-file.html

@param d the SBML document to be written.

@param filename the name or full pathname of the file where the SBML
is to be written. 

@return @c True on success and @c False if the filename could not be
opened for writing.

@note @htmlinclude note-writing-zipped-files.html

@see setProgramVersion()
@see setProgramName()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>writeSBML(SBMLDocument d, std::ostream stream)</pre>

Writes the given SBML document to the output stream.

@param d the SBML document to be written.

@param stream the stream object where the SBML is to be written.

@return @c True on success and @c False if one of the underlying
parser components fail (rare).

@see setProgramVersion()
@see setProgramName()
";


%feature("docstring") SBMLWriter::writeToString "
@internal

@internal
";


%feature("docstring") SBMLWriter::writeSBMLToFile "
Writes the given SBML document to filename.

@htmlinclude assuming-compressed-file.html

@param d the SBML document to be written.

@param filename the name or full pathname of the file where the SBML
is to be written. 

@return @c True on success and @c False if the filename could not be
opened for writing.

@note @htmlinclude note-writing-zipped-files.html

@see setProgramVersion()
@see setProgramName()
";


%feature("docstring") SBMLWriter::writeSBMLToString "
Writes the given SBML document to an in-memory string and returns a
pointer to it.

The string is owned by the caller and should be freed (with @c free())
when no longer needed.

@param d the SBML document to be written.

@return the string on success and @c 0 if one of the underlying parser
components fail.

@see setProgramVersion()
@see setProgramName()
";


%feature("docstring") SBMLWriter::hasZlib "
Predicate returning @c True if this copy of libSBML has been linked
with the <em>zlib</em> library.

LibSBML supports reading and writing files compressed with either
bzip2 or zip/gzip compression.  The facility depends on libSBML having
been compiled with the necessary support libraries.  This method
allows a calling program to inquire whether that is the case for the
copy of libSBML it is using.

@return @c True if libSBML is linked with zlib, @c False otherwise.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike hasBzip2() @else SBMLWriter.hasBzip2() @endif@~
";


%feature("docstring") SBMLWriter::hasBzip2 "
Predicate returning @c True if this copy of libSBML has been linked
with the <em>bzip2</em> library.

LibSBML supports reading and writing files compressed with either
bzip2 or zip/gzip compression.  The facility depends on libSBML having
been compiled with the necessary support libraries.  This method
allows a calling program to inquire whether that is the case for the
copy of libSBML it is using.

@return @c True if libSBML is linked with bzip2, @c False otherwise.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike hasZlib() @else SBMLWriter.hasZlib() @endif@~
";


%feature("docstring") writeSBML "
Writes the given SBML document @p d to the file named by @p filename.
This convenience function is functionally equivalent to:

SBMLWriter_writeSBML(SBMLWriter_create(), d, filename);

@htmlinclude assuming-compressed-file.html

@param d the SBMLDocument object to be written out in XML format.

@param filename a string giving the path to a file where the XML
content is to be written.

@return @c 1 (true) on success and @c 0 (false) if @p filename could not be
written.  Some possible reasons for failure include (a) being unable to
open the file, and (b) using a filename that indicates a compressed SBML
file (i.e., a filename ending in <code>&quot;.zip&quot;</code> or
similar) when the compression functionality has not been enabled in
the underlying copy of libSBML.

@see @if conly SBMLWriter_hasZlib() @else SBMLWriter.hasZlib() @endif
@see @if conly SBMLWriter_hasBzip2() @else SBMLWriter.hasBzip2() @endif

@if conly
@memberof SBMLWriter_t
@endif
";


%feature("docstring") writeSBMLToString "
Writes the given SBML document @p d to an in-memory string and returns a
pointer to it.  The string is owned by the caller and should be freed
(with free()) when no longer needed.  This convenience function is
functionally equivalent to:

SBMLWriter_writeSBMLToString(SBMLWriter_create(), d);

but does not require the caller to create an SBMLWriter object first.

@param d an SBMLDocument object to be written out in XML format.

@return the string on success and @c None if one of the underlying parser
components fail.

@if clike @warning Note that the string is owned by the caller and
should be freed after it is no longer needed.@endif@~

@if conly
@memberof SBMLWriter_t
@endif
";


%feature("docstring") writeSBMLToFile "
Writes the given SBML document @p d to the file @p filename.
This convenience function is functionally equivalent to:

SBMLWriter_writeSBMLToFile(SBMLWriter_create(), d, filename);

but that does not require the caller to create an SBMLWriter object first.

@htmlinclude assuming-compressed-file.html

@param d an SBMLDocument object to be written out in XML format.

@param filename a string giving the path to a file where the XML
content is to be written.

@return @c 1 (true) on success and @c 0 (false) if @p filename could not be
written.  Some possible reasons for failure include (a) being unable to
open the file, and (b) using a filename that indicates a compressed SBML
file (i.e., a filename ending in <code>&quot;.zip&quot;</code> or
similar) when the compression functionality has not been enabled in
the underlying copy of libSBML.

@if clike @warning Note that the string is owned by the caller and
should be freed (with the normal string <code>free()</code> C++
function) after it is no longer needed.@endif@~

@see SBMLWriter.hasZlib()
@see SBMLWriter.hasBzip2()

@if conly
@memberof SBMLWriter_t
@endif
";


%feature("docstring") SBMLTypeCode_toString "
This method takes an SBML type code and returns a string representing
the code.

@if clike LibSBML attaches an identifying code to every kind of SBML
object.  These are known as <em>SBML type codes</em>.  The set of
possible type codes is defined in the enumeration #SBMLTypeCode_t.
The names of the type codes all begin with the characters
@c SBML_. @endif@if java LibSBML attaches an identifying code to every
kind of SBML object.  These are known as <em>SBML type codes</em>.  In
other languages, the set of type codes is stored in an enumeration; in
the Java language interface for libSBML, the type codes are defined as
static integer constants in the interface class {@link
libsbmlConstants}.  The names of the type codes all begin with the
characters @c SBML_. @endif@if python LibSBML attaches an identifying
code to every kind of SBML object.  These are known as <em>SBML type
codes</em>.  In the Python language interface for libSBML, the type
codes are defined as static integer constants in the interface class
@link libsbml@endlink.  The names of the type codes all begin with the
characters @c SBML_. @endif@if csharp LibSBML attaches an identifying
code to every kind of SBML object.  These are known as <em>SBML type
codes</em>.  In the C# language interface for libSBML, the type codes
are defined as static integer constants in the interface class @link
libsbml@endlink.  The names of the type codes all begin with
the characters @c SBML_. @endif@~

@return a human readable name for the given
@if clike #SBMLTypeCode_t value@else SBML type code@endif.

@note The caller does not own the returned string and is therefore not
allowed to modify it.
";


%feature("docstring") SBase "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html SBML\'s <em>%SBase</em> class, base class of most SBML objects.

Most components in SBML are derived from a single abstract base type,
SBase.  In addition to serving as the parent class for most other
classes of objects in SBML, this base type is designed to allow a
modeler or a software package to attach arbitrary information to each
major element or list in an SBML model.

SBase has an optional subelement called \'notes\'.  It is intended to
serve as a place for storing optional information intended to be seen by
humans.  An example use of the \'notes\' element would be to contain
formatted user comments about the model element in which the \'notes\'
element is enclosed.  There are certain conditions on the XHTML content
permitted inside the \'notes\' element; please consult the <a
target=\'_blank\' href=\'http://sbml.org/Documents/Specifications\'>SBML
specification document</a> corresponding to the SBML Level and Version
of your model for more information about the requirements for \'notes\'
content.

SBase has another optional subelement called \'annotation\'.  Whereas the
\'notes\' element described above is a container for content to be shown
directly to humans, the \'annotation\' element is a container for optional
software-generated content @em not meant to be shown to humans.  The
element\'s content type is <a target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
\'any\'</a>, allowing essentially arbitrary data content.  SBML places
only a few restrictions on the organization of the content; these are
intended to help software tools read and write the data as well as help
reduce conflicts between annotations added by different tools.  As is
the case with \'notes\', it is important to refer to the <a
target=\'_blank\' href=\'http://sbml.org/Documents/Specifications\'>SBML
specification document</a> corresponding to the SBML Level and Version
of your model for more information about the requirements for
\'annotation\' content.

It is worth pointing out that the \'annotation\' element in the definition
of SBase exists in order that software developers may attach optional
application-specific data to the elements in an SBML model.  However, it
is important that this facility not be misused.  In particular, it is
<em>critical</em> that data essential to a model definition or that can
be encoded in existing SBML elements is <em>not</em> stored in
\'annotation\'. Parameter values, functional dependencies between model
elements, etc., should not be recorded as annotations.  It is crucial to
keep in mind the fact that data placed in annotations can be freely
ignored by software applications.  If such data affects the
interpretation of a model, then software interoperability is greatly
impeded.

SBML Level 2 introduced an optional SBase attribute named \'metaid\' for
supporting metadata annotations using RDF (<a target=\'_blank\'
href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>). The
attribute value has the data type <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML identifier
type, which means each \'metaid\' value must be globally unique within an
SBML file.  (Importantly, this uniqueness criterion applies across any
attribute with type <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML
ID</a>, not just the \'metaid\' attribute used by SBML---something to
be aware of if your application-specific XML content inside the
\'annotation\' subelement happens to use <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)  The \'metaid\' value
serves to identify a model component for purposes such as referencing
that component from metadata placed within \'annotation\' subelements.

Beginning with SBML Level 2 Version 2, SBase has an optional
attribute named \'sboTerm\' for supporting the use of the Systems Biology
Ontology.  In SBML proper, the data type of the attribute is a string of
the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a seven digit integer number;
libSBML simplifies the representation by only storing the \'NNNNNNN\'
integer portion.  Thus, in libSBML, the \'sboTerm\' attribute on SBase has
data type @c int, and SBO identifiers are stored simply as integers.
(For convenience, SBase offers methods for returning both the integer
form and a text-string form of the SBO identifier.)  SBO terms are a
type of optional annotation, and each different class of SBML object
derived from SBase imposes its own requirements about the values
permitted for \'sboTerm\'.  More details can be found in SBML specifications
for Level&nbsp;2 Version&nbsp;2 and above.

Finally, note that, in the list of methods on SBase, there is no public
constructor because SBase is an abstract class.  The constructors reside
in the subclasses derived from SBase.


@section sbase-miriam Standard format for annotations linking data resources

SBML Level 2 Versions 2, 3 and 4, and Level&nbsp;3, define a proposed
regular format for encoding two particular categories of annotations:
(a) references to controlled vocabulary terms and database identifiers
which define and describe biological and biochemical entities in a
model; and (b) descriptions of the provenance of a model, including its
author(s) and modification history.
";


%feature("docstring") SBase::accept "
@internal

@internal
";


%feature("docstring") SBase::clone "
Creates and returns a deep copy of this SBase object.

@return the (deep) copy of this SBase object.
";


%feature("docstring") SBase::getElementBySId "
Returns the first child element found that has the given @p id in the
model-wide @c SId namespace, or @c None if no such object is found.

@param id string representing the \'id\' attribute value of the object
to find.

@return pointer to the first element found with the given identifier.
";


%feature("docstring") SBase::getElementByMetaId "
Returns the first child element it can find with a specific \'metaid\'
attribute value, or @c None if no such object is found.

@par
The optional attribute named \'metaid\', present on every major SBML
component type, is for supporting metadata annotations using RDF (<a
href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
attribute value has the data type <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
identifier type, which means each \'metaid\' value must be globally unique
within an SBML file.  The latter point is important, because the
uniqueness criterion applies across <em>any</em> attribute with type
<code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
by SBML---something to be aware of if your application-specific XML
content inside the \'annotation\' subelement happens to use the XML
<code>ID</code> type.  Although SBML itself specifies the use of <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
the \'metaid\' attribute, SBML-compatible applications should be careful if
they use XML <code>ID</code>\'s in XML portions of a model that are not
defined by SBML, such as in the application-specific content of the
\'annotation\' subelement.  Finally, note that LibSBML does not provide an
explicit XML <code>ID</code> data type; it uses ordinary character
strings, which is easier for applications to support.

@param metaid string representing the \'metaid\' attribute value of the
object to find.

@return pointer to the first element found with the given meta-identifier.
";


%feature("docstring") SBase::getAllElements "
Returns a List of all child SBase objects, including those nested to
an arbitrary depth.

@param filter a pointer to an ElementFilter, which causes the function 
to return only elements that match a particular set of constraints.  
If None (the default), the function will return all child objects.

@return a pointer to a List of pointers to all objects that are children
of this object.
";


%feature("docstring") SBase::renameSIdRefs "
Replaces all uses of a given @c SIdRef type attribute value with another
value.

@par
In SBML, object identifiers are of a data type called <code>SId</code>.
In SBML Level&nbsp;3, an explicit data type called <code>SIdRef</code> was
introduced for attribute values that refer to <code>SId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to an identifier\', but the effective
data type was the same as <code>SIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>SIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all attributes and (if appropriate)
mathematical formulas in MathML content, comparing the referenced
identifiers to the value of @p oldid.  If any matches are found, the
matching values are replaced with @p newid.  The method does @em not
descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") SBase::renameMetaIdRefs "
Replaces all uses of a given meta identifier attribute value with
another value.

@par
In SBML, object \'meta\' identifiers are of the XML data type <code>ID</code>;
the SBML object attribute itself is typically named <code>metaid</code>.  All
attributes that hold values <em>referring</em> to values of type
<code>ID</code> are of the XML data type <code>IDREF</code>.  They are also
sometimes informally referred to as \'metaid refs\', in analogy to the
SBML-defined type <code>SIdRef</code>.

This method works by looking at all meta-identifier attribute values,
comparing the identifiers to the value of @p oldid.  If any matches are
found, the matching identifiers are replaced with @p newid.  The method
does @em not descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") SBase::renameUnitSIdRefs "
Replaces all uses of a given @c UnitSIdRef type attribute value with
another value.

@par
In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
introduced for attribute values that refer to <code>UnitSId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to a unit identifier\', but the effective
data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all unit identifier attribute values
(including, if appropriate, inside mathematical formulas), comparing the
referenced unit identifiers to the value of @p oldid.  If any matches
are found, the matching values are replaced with @p newid.  The method
does @em not descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") SBase::replaceSIDWithFunction "
@internal

@internal
";


%feature("docstring") SBase::divideAssignmentsToSIdByFunction "
@internal

@internal
";


%feature("docstring") SBase::multiplyAssignmentsToSIdByFunction "
@internal

@internal
";


%feature("docstring") SBase::getElementFromPluginsBySId "
@internal

@internal
";


%feature("docstring") SBase::getElementFromPluginsByMetaId "
@internal

@internal
";


%feature("docstring") SBase::hasNonstandardIdentifierBeginningWith "
@internal

@internal
";


%feature("docstring") SBase::prependStringToAllIdentifiers "
@internal

@internal
";


%feature("docstring") SBase::transformIdentifiers "
@internal

@internal
";


%feature("docstring") SBase::getAllElementsFromPlugins "
Returns a List of all child SBase objects contained in SBML package
plug-ins.

@par
SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
that it uses one or more SBML packages, and likewise, a software tool may be
able to support one or more packages.  LibSBML does not come preconfigured
with all possible packages included and enabled, in part because not all
package specifications have been finalized.  To support the ability for
software systems to enable support for the Level&nbsp;3 packages they choose,
libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
package is implemented in a separate code plug-in that can be enabled by the
application to support working with that SBML package.  A given SBML model
may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
objects created by libSBML plug-ins supporting additional Level&nbsp;3
packages.

This method walks down the list of all SBML Level&nbsp;3 packages used
by this object and returns all child objects defined by those packages.

@param filter a pointer to an ElementFilter, which causes the function 
to return only elements that match a particular set of constraints.  
If None (the default), the function will return all child objects.

@return a pointer to a List of pointers to all children objects from
plug-ins.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~
";


%feature("docstring") SBase::getMetaId "
Returns the value of the \'metaid\' attribute of this SBML object.

@par
The optional attribute named \'metaid\', present on every major SBML
component type, is for supporting metadata annotations using RDF (<a
href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
attribute value has the data type <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
identifier type, which means each \'metaid\' value must be globally unique
within an SBML file.  The latter point is important, because the
uniqueness criterion applies across <em>any</em> attribute with type
<code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
by SBML---something to be aware of if your application-specific XML
content inside the \'annotation\' subelement happens to use the XML
<code>ID</code> type.  Although SBML itself specifies the use of <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
the \'metaid\' attribute, SBML-compatible applications should be careful if
they use XML <code>ID</code>\'s in XML portions of a model that are not
defined by SBML, such as in the application-specific content of the
\'annotation\' subelement.  Finally, note that LibSBML does not provide an
explicit XML <code>ID</code> data type; it uses ordinary character
strings, which is easier for applications to support.

@return the meta-identifier of this SBML object.

@see isSetMetaId()
@see setMetaId()
";


%feature("docstring") SBase::getId "
Returns the value of the \'id\' attribute of this SBML object, if it has one, 
or the \'variable\' attribute of a Rule, or the \'symbol\' attribute of
an InitialAssignment.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this SBML object, or the \'variable\' if the object is a
Rule, or the \'symbol\' if the object is an InitialAssignment.

@note Because of the inconsistent behavior of this function with respect
to assignments and rules, callers should use getIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") SBase::getIdAttribute "
Returns the value of the \'id\' attribute of this SBML object.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this SBML object, if set and valid for this
level and version of SBML; an empty string otherwise.

@note Because of the inconsistent behavior of this function with respect
to assignments and rules, callers should use getIdAttribute() instead.

@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") SBase::getName "
Returns the value of the \'name\' attribute of this SBML object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") SBase::getNotes "
Returns the content of the \'notes\' subelement of this object as
a tree of XMLNode objects.

@par
The optional SBML element named \'notes\', present on every major SBML
component type (and in SBML Level&nbsp;3, the \'message\' subelement of
Constraint), is intended as a place for storing optional information
intended to be seen by humans.  An example use of the \'notes\' element
would be to contain formatted user comments about the model element in
which the \'notes\' element is enclosed.  Every object derived directly or
indirectly from type SBase can have a separate value for \'notes\', allowing
users considerable freedom when adding comments to their models.

The format of \'notes\' elements conform to the definition of <a
target=\'_blank\' href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.
However, the content cannot be @em entirely free-form; it must satisfy
certain requirements defined in the <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specifications</a>
for specific SBML Levels.  To help verify the formatting of \'notes\'
content, libSBML provides the static utility method
SyntaxChecker.hasExpectedXHTMLSyntax(); this
method implements a verification process that lets callers check whether
the content of a given XMLNode object conforms to the SBML requirements
for \'notes\' and \'message\' structure.  Developers are urged to consult the
appropriate <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specification
document</a> for the Level and Version of their model for more in-depth
explanations of using \'notes\' in SBML.  The SBML Level&nbsp;2 and&nbsp;3
specifications have considerable detail about how \'notes\' element content
must be structured. 

The \'notes\' element content returned by this method will be in XML
form, but libSBML does not provide an object model specifically for
the content of notes.  Callers will need to traverse the XML tree
structure using the facilities available on XMLNode and related
objects.  For an alternative method of accessing the notes, see
getNotesString().

@return the content of the \'notes\' subelement of this SBML object as a
tree structure composed of XMLNode objects.

@see getNotesString()
@see isSetNotes()
@see setNotes()
@see setNotes()
@see appendNotes()
@see appendNotes()
@see unsetNotes()
@see SyntaxChecker.hasExpectedXHTMLSyntax()
";


%feature("docstring") SBase::getNotesString "
Returns the content of the \'notes\' subelement of this object as a
string.

@par
The optional SBML element named \'notes\', present on every major SBML
component type (and in SBML Level&nbsp;3, the \'message\' subelement of
Constraint), is intended as a place for storing optional information
intended to be seen by humans.  An example use of the \'notes\' element
would be to contain formatted user comments about the model element in
which the \'notes\' element is enclosed.  Every object derived directly or
indirectly from type SBase can have a separate value for \'notes\', allowing
users considerable freedom when adding comments to their models.

The format of \'notes\' elements conform to the definition of <a
target=\'_blank\' href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.
However, the content cannot be @em entirely free-form; it must satisfy
certain requirements defined in the <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specifications</a>
for specific SBML Levels.  To help verify the formatting of \'notes\'
content, libSBML provides the static utility method
SyntaxChecker.hasExpectedXHTMLSyntax(); this
method implements a verification process that lets callers check whether
the content of a given XMLNode object conforms to the SBML requirements
for \'notes\' and \'message\' structure.  Developers are urged to consult the
appropriate <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specification
document</a> for the Level and Version of their model for more in-depth
explanations of using \'notes\' in SBML.  The SBML Level&nbsp;2 and&nbsp;3
specifications have considerable detail about how \'notes\' element content
must be structured. 

For an alternative method of accessing the notes, see getNotes(),
which returns the content as an XMLNode tree structure.  Depending on
an application\'s needs, one or the other method may be more
convenient.

@return the content of the \'notes\' subelement of this SBML object as a
string.

@see getNotes()
@see isSetNotes()
@see setNotes()
@see setNotes()
@see appendNotes()
@see appendNotes()
@see unsetNotes()
@see SyntaxChecker.hasExpectedXHTMLSyntax()
";


%feature("docstring") SBase::getAnnotation "
Returns the content of the \'annotation\' subelement of this object as
a tree of XMLNode objects.

@par
Whereas the SBML \'notes\' subelement is a container for content to be
shown directly to humans, the \'annotation\' element is a container for
optional software-generated content @em not meant to be shown to
humans.  Every object derived from SBase can have its own value for
\'annotation\'.  The element\'s content type is <a target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
\'any\'</a>, allowing essentially arbitrary well-formed XML data
content.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details. 

The annotations returned by this method will be in XML form.  LibSBML
provides an object model and related interfaces for certain specific
kinds of annotations, namely model history information and RDF
content.  See the ModelHistory, CVTerm and RDFAnnotationParser classes
for more information about the facilities available.

@return the annotation of this SBML object as a tree of XMLNode objects.

@see getAnnotationString()
@see isSetAnnotation()
@see setAnnotation()
@see setAnnotation()
@see appendAnnotation()
@see appendAnnotation()
@see unsetAnnotation()
";


%feature("docstring") SBase::getAnnotationString "
Returns the content of the \'annotation\' subelement of this object as a
character string.

@par
Whereas the SBML \'notes\' subelement is a container for content to be
shown directly to humans, the \'annotation\' element is a container for
optional software-generated content @em not meant to be shown to
humans.  Every object derived from SBase can have its own value for
\'annotation\'.  The element\'s content type is <a target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
\'any\'</a>, allowing essentially arbitrary well-formed XML data
content.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details. 

The annotations returned by this method will be in string form.  See the
method getAnnotation() for a version that returns annotations in XML form.

@return the annotation of this SBML object as a character string.

@see getAnnotation()
@see isSetAnnotation()
@see setAnnotation()
@see setAnnotation()
@see appendAnnotation()
@see appendAnnotation()
@see unsetAnnotation()
";


%feature("docstring") SBase::getNamespaces "
Returns a list of the XML Namespaces declared on the SBML document
owning this object.

The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.

@return the XML Namespaces associated with this SBML object, or @c None
in certain very usual circumstances where a namespace is not set.

@see getLevel()
@see getVersion()
";


%feature("docstring") SBase::getSBMLDocument "
Returns the SBMLDocument object containing this object instance.

@par
LibSBML uses the class SBMLDocument as a top-level container for
storing SBML content and data associated with it (such as warnings and
error messages).  An SBML model in libSBML is contained inside an
SBMLDocument object.  SBMLDocument corresponds roughly to the class
<i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
specifications, but it does not have a direct correspondence in SBML
Level&nbsp;1.  (But, it is created by libSBML no matter whether the
model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)

This method allows the caller to obtain the SBMLDocument for the
current object.

@return the parent SBMLDocument object of this SBML object.

@see getParentSBMLObject()
@see getModel()
";


%feature("docstring") SBase::getParentSBMLObject "
Returns the parent SBML object containing this object.

This returns the immediately-containing object.  This method is
convenient when holding an object nested inside other objects in an
SBML model.  

@return the parent SBML object of this SBML object.

@see getSBMLDocument()
@see getModel()
";


%feature("docstring") SBase::getAncestorOfType "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getAncestorOfType(int type, string pkgName = \'core\')</pre>

Returns the first ancestor object that has the given SBML type code from the given package.

@if clike LibSBML attaches an identifying code to every kind of SBML
object.  These are known as <em>SBML type codes</em>.  The set of
possible type codes is defined in the enumeration #SBMLTypeCode_t.
The names of the type codes all begin with the characters
@c SBML_. @endif@if java LibSBML attaches an identifying code to every
kind of SBML object.  These are known as <em>SBML type codes</em>.  In
other languages, the set of type codes is stored in an enumeration; in
the Java language interface for libSBML, the type codes are defined as
static integer constants in the interface class {@link
libsbmlConstants}.  The names of the type codes all begin with the
characters @c SBML_. @endif@if python LibSBML attaches an identifying
code to every kind of SBML object.  These are known as <em>SBML type
codes</em>.  In the Python language interface for libSBML, the type
codes are defined as static integer constants in the interface class
@link libsbml@endlink.  The names of the type codes all begin with the
characters @c SBML_. @endif@if csharp LibSBML attaches an identifying
code to every kind of SBML object.  These are known as <em>SBML type
codes</em>.  In the C# language interface for libSBML, the type codes
are defined as static integer constants in the interface class @link
libsbmlcs.libsbml libsbml@endlink.  The names of the type codes all begin with
the characters @c SBML_. @endif@~

This method searches the tree of objects that are parents of this
object, and returns the first one that has the given SBML type code from 
the given @p pkgName.

@param type the SBML type code of the object sought.

@param pkgName (optional) the short name of an SBML Level&nbsp;3
package to which the sought-after object must belong.

@return the ancestor SBML object of this SBML object that corresponds
to the given @if clike #SBMLTypeCode_t value@else SBML object type
code@endif, or @c None if no ancestor exists.

@warning The optional argument @p pkgName must be used for all type codes
from SBML Level&nbsp;3 packages.  Otherwise, the function will search the 
\'core\' namespace alone, not find any corresponding elements, and return 
None.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.


@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getAncestorOfType(int type, string pkgName = \'core\')</pre>

Returns the first ancestor object that has the given SBML type code from the given package.

@if clike LibSBML attaches an identifying code to every kind of SBML
object.  These are known as <em>SBML type codes</em>.  The set of
possible type codes is defined in the enumeration #SBMLTypeCode_t.
The names of the type codes all begin with the characters
@c SBML_. @endif@if java LibSBML attaches an identifying code to every
kind of SBML object.  These are known as <em>SBML type codes</em>.  In
other languages, the set of type codes is stored in an enumeration; in
the Java language interface for libSBML, the type codes are defined as
static integer constants in the interface class {@link
libsbmlConstants}.  The names of the type codes all begin with the
characters @c SBML_. @endif@if python LibSBML attaches an identifying
code to every kind of SBML object.  These are known as <em>SBML type
codes</em>.  In the Python language interface for libSBML, the type
codes are defined as static integer constants in the interface class
@link libsbml@endlink.  The names of the type codes all begin with the
characters @c SBML_. @endif@if csharp LibSBML attaches an identifying
code to every kind of SBML object.  These are known as <em>SBML type
codes</em>.  In the C# language interface for libSBML, the type codes
are defined as static integer constants in the interface class @link
libsbmlcs.libsbml libsbml@endlink.  The names of the type codes all begin with
the characters @c SBML_. @endif@~

This method searches the tree of objects that are parents of this
object, and returns the first one that has the given SBML type code from 
the given @p pkgName.

@param type the SBML type code of the object sought.

@param pkgName (optional) the short name of an SBML Level&nbsp;3
package to which the sought-after object must belong.

@return the ancestor SBML object of this SBML object that corresponds
to the given @if clike #SBMLTypeCode_t value@else SBML object type
code@endif, or @c None if no ancestor exists.

@warning The optional argument @p pkgName must be used for all type codes
from SBML Level&nbsp;3 packages.  Otherwise, the function will search the 
\'core\' namespace alone, not find any corresponding elements, and return 
None.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBase::getSBOTerm "
Returns the integer portion of the value of the \'sboTerm\' attribute of
this object.

@par
Beginning with SBML Level 2 Version 2, objects derived from SBase have
an optional attribute named \'sboTerm\' for supporting the use of the
Systems Biology Ontology.  In SBML proper, the data type of the
attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
seven digit integer number; libSBML simplifies the representation by
only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
\'sboTerm\' attribute on SBase has data type @c int, and SBO identifiers
are stored simply as integers.
@par
SBO terms are a type of optional annotation, and each different class
of SBML object derived from SBase imposes its own requirements about
the values permitted for \'sboTerm\'. More details can be found in SBML
specifications for Level&nbsp;2 Version&nbsp;2 and above.

@return the value of the \'sboTerm\' attribute as an integer, or @c -1
if the value is not set.
";


%feature("docstring") SBase::getSBOTermID "
Returns the string representation of the \'sboTerm\' attribute of
this object.

@par
Beginning with SBML Level 2 Version 2, objects derived from SBase have
an optional attribute named \'sboTerm\' for supporting the use of the
Systems Biology Ontology.  In SBML proper, the data type of the
attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
seven digit integer number; libSBML simplifies the representation by
only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
\'sboTerm\' attribute on SBase has data type @c int, and SBO identifiers
are stored simply as integers.
@par
SBO terms are a type of optional annotation, and each different class
of SBML object derived from SBase imposes its own requirements about
the values permitted for \'sboTerm\'. More details can be found in SBML
specifications for Level&nbsp;2 Version&nbsp;2 and above.

@return the value of the \'sboTerm\' attribute as a string (its value
will be of the form \'SBO:NNNNNNN\'), or an empty string if
the value is not set.
";


%feature("docstring") SBase::getSBOTermAsURL "
Returns the URL representation of the \'sboTerm\' attribute of this
object.

This method returns the entire SBO identifier as a text string in the
form <code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://identifiers.org/biomodels.sbo/SBO:NNNNNNN\'</code>.

@par
SBO terms are a type of optional annotation, and each different class
of SBML object derived from SBase imposes its own requirements about
the values permitted for \'sboTerm\'. More details can be found in SBML
specifications for Level&nbsp;2 Version&nbsp;2 and above.

@return the value of the \'sboTerm\' attribute as an identifiers.org URL,
or an empty string if the value is not set.
";


%feature("docstring") SBase::getLine "
Returns the line number where this object first appears in the XML
representation of the SBML document.

@return the line number of this SBML object.  If this object was
created programmatically and not read from a file, this method will
return the value @c 0.

@note The line number for each construct in an SBML model is set upon
reading the model.  The accuracy of the line number depends on the
correctness of the XML representation of the model, and on the
particular XML parser library being used.  The former limitation
relates to the following problem: if the model is actually invalid
XML, then the parser may not be able to interpret the data correctly
and consequently may not be able to establish the real line number.
The latter limitation is simply that different parsers seem to have
their own accuracy limitations, and out of all the parsers supported
by libSBML, none have been 100% accurate in all situations. (At this
time, libSBML supports the use of <a target=\'_blank\'
href=\'http://xmlsoft.org\'>libxml2</a>, <a target=\'_blank\'
href=\'http://expat.sourceforge.net/\'>Expat</a> and <a target=\'_blank\'
href=\'http://xerces.apache.org/xerces-c/\'>Xerces</a>.)

@see getColumn()
";


%feature("docstring") SBase::getColumn "
Returns the column number where this object first appears in the XML
representation of the SBML document.

@return the column number of this SBML object.  If this object was
created programmatically and not read from a file, this method will
return the value @c 0.

@note The column number for each construct in an SBML model is set
upon reading the model.  The accuracy of the column number depends on
the correctness of the XML representation of the model, and on the
particular XML parser library being used.  The former limitation
relates to the following problem: if the model is actually invalid
XML, then the parser may not be able to interpret the data correctly
and consequently may not be able to establish the real column number.
The latter limitation is simply that different parsers seem to have
their own accuracy limitations, and out of all the parsers supported
by libSBML, none have been 100% accurate in all situations. (At this
time, libSBML supports the use of <a target=\'_blank\'
href=\'http://xmlsoft.org\'>libxml2</a>, <a target=\'_blank\'
href=\'http://expat.sourceforge.net/\'>Expat</a> and <a target=\'_blank\'
href=\'http://xerces.apache.org/xerces-c/\'>Xerces</a>.)

@see getLine()
";


%feature("docstring") SBase::getModelHistory "
Returns the ModelHistory object, if any, attached to this object.

@return the ModelHistory object attached to this object, or @c None if
none exist.

@note In SBML Level&nbsp;2, model history annotations were only
permitted on the Model element.  In SBML Level&nbsp;3, they are
permitted on all SBML components derived from SBase.
";


%feature("docstring") SBase::isSetMetaId "
Predicate returning @c True if this object\'s \'metaid\' attribute is set.

@par
The optional attribute named \'metaid\', present on every major SBML
component type, is for supporting metadata annotations using RDF (<a
href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
attribute value has the data type <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
identifier type, which means each \'metaid\' value must be globally unique
within an SBML file.  The latter point is important, because the
uniqueness criterion applies across <em>any</em> attribute with type
<code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
by SBML---something to be aware of if your application-specific XML
content inside the \'annotation\' subelement happens to use the XML
<code>ID</code> type.  Although SBML itself specifies the use of <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
the \'metaid\' attribute, SBML-compatible applications should be careful if
they use XML <code>ID</code>\'s in XML portions of a model that are not
defined by SBML, such as in the application-specific content of the
\'annotation\' subelement.  Finally, note that LibSBML does not provide an
explicit XML <code>ID</code> data type; it uses ordinary character
strings, which is easier for applications to support. 

@return @c True if the \'metaid\' attribute of this SBML object is
set, @c False otherwise.

@see getMetaId()
@see setMetaId()
";


%feature("docstring") SBase::isSetId "
Predicate returning @c True if a call to getId() returns a
non-empty string.

For most objects, this function will return @c True if its \'id\'
attribute is set, and @c False if it is not, or if the object has no
\'id\' attribute at all.  However, for an EventAssignment or a Rule,
isSetId() checks whether the \'variable\' attribute is set, and for an
InitialAssignment, it checks whether the \'symbol\' attribute is set.
Because those elements will also have an \'id\' attribute in SBML
Level&nbsp;3 Version&nbsp;2 which isSetId() will not check, the function
itself is deprecated, and it is recommended to use isSetIdAttribute() in
all cases where one needs to know whether the \'id\' attribute is set, and
to use EventAssignment.isSetVariable(), Rule.isSetVariable() and
InitialAssignment.isSetSymbol() when the status of the \'variable\' or
\'symbol\' attributes need to be checked.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") SBase::isSetIdAttribute "
Predicate returning @c True if this object\'s \'id\' attribute is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") SBase::isSetName "
Predicate returning @c True if this
object\'s \'name\' attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") SBase::isSetNotes "
Predicate returning @c True if this
object\'s \'notes\' subelement exists and has content.

The optional SBML element named \'notes\', present on every major SBML
component type, is intended as a place for storing optional
information intended to be seen by humans.  An example use of the
\'notes\' element would be to contain formatted user comments about the
model element in which the \'notes\' element is enclosed.  Every object
derived directly or indirectly from type SBase can have a separate
value for \'notes\', allowing users considerable freedom when adding
comments to their models.

The format of \'notes\' elements must be <a target=\'_blank\'
href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
verify the formatting of \'notes\' content, libSBML provides the static
utility method SyntaxChecker.hasExpectedXHTMLSyntax(); however,
readers are urged to consult the appropriate <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specification
document</a> for the Level and Version of their model for more
in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
specifications have considerable detail about how \'notes\' element
content must be structured.

@return @c True if a \'notes\' subelement exists, @c False otherwise.

@see getNotes()
@see getNotesString()
@see setNotes()
@see setNotes()
@see appendNotes()
@see appendNotes()
@see unsetNotes()
@see SyntaxChecker.hasExpectedXHTMLSyntax()
";


%feature("docstring") SBase::isSetAnnotation "
Predicate returning @c True if this
object\'s \'annotation\' subelement exists and has content.

Whereas the SBase \'notes\' subelement is a container for content to be
shown directly to humans, the \'annotation\' element is a container for
optional software-generated content @em not meant to be shown to
humans.  Every object derived from SBase can have its own value for
\'annotation\'.  The element\'s content type is <a target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
\'any\'</a>, allowing essentially arbitrary well-formed XML data
content.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

@return @c True if a \'annotation\' subelement exists, @c False
otherwise.

@see getAnnotation()
@see getAnnotationString()
@see setAnnotation()
@see setAnnotation()
@see appendAnnotation()
@see appendAnnotation()
@see unsetAnnotation()
";


%feature("docstring") SBase::isSetSBOTerm "
Predicate returning @c True if this
object\'s \'sboTerm\' attribute is set.

@return @c True if the \'sboTerm\' attribute of this SBML object is
set, @c False otherwise.
";


%feature("docstring") SBase::setMetaId "
Sets the value of the meta-identifier attribute of this SBML object.

@par
The optional attribute named \'metaid\', present on every major SBML
component type, is for supporting metadata annotations using RDF (<a
href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
attribute value has the data type <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
identifier type, which means each \'metaid\' value must be globally unique
within an SBML file.  The latter point is important, because the
uniqueness criterion applies across <em>any</em> attribute with type
<code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
by SBML---something to be aware of if your application-specific XML
content inside the \'annotation\' subelement happens to use the XML
<code>ID</code> type.  Although SBML itself specifies the use of <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
the \'metaid\' attribute, SBML-compatible applications should be careful if
they use XML <code>ID</code>\'s in XML portions of a model that are not
defined by SBML, such as in the application-specific content of the
\'annotation\' subelement.  Finally, note that LibSBML does not provide an
explicit XML <code>ID</code> data type; it uses ordinary character
strings, which is easier for applications to support. 

The string @p metaid is copied.  

@param metaid the identifier string to use as the value of the
\'metaid\' attribute.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getMetaId()
@see isSetMetaId()
";


%feature("docstring") SBase::isSetModelHistory "
Predicate returning @c True if this
object has a ModelHistory object attached to it.

@return @c True if the ModelHistory of this object is set,
@c False otherwise.

@note In SBML Level&nbsp;2, model history annotations were only
permitted on the Model element.  In SBML Level&nbsp;3, they are
permitted on all SBML components derived from SBase.
";


%feature("docstring") SBase::setId "
Sets the value of the \'id\' attribute of this SBML object.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") SBase::setIdAttribute "
Sets the value of the \'id\' attribute of this SBML object.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") SBase::setName "
Sets the value of the \'name\' attribute of this SBML object.

@par

The string in @p name is copied.

@param name the new name for the SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") SBase::setAnnotation "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setAnnotation(XMLNode annotation)</pre>

Sets the value of the \'annotation\' subelement of this SBML object.

The content of @p annotation is copied, and any previous content of
this object\'s \'annotation\' subelement is deleted.

Whereas the SBase \'notes\' subelement is a container for content to be
shown directly to humans, the \'annotation\' element is a container for
optional software-generated content @em not meant to be shown to
humans.  Every object derived from SBase can have its own value for
\'annotation\'.  The element\'s content type is <a target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
\'any\'</a>, allowing essentially arbitrary well-formed XML data
content.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

Call this method will result in any existing content of the
\'annotation\' subelement to be discarded.  Unless you have taken steps
to first copy and reconstitute any existing annotations into the @p
annotation that is about to be assigned, it is likely that performing
such wholesale replacement is unfriendly towards other software
applications whose annotations are discarded.  An alternative may be
to use SBase.appendAnnotation() or
SBase.appendAnnotation().

@param annotation an XML structure that is to be used as the new content
of the \'annotation\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@see getAnnotationString()
@see isSetAnnotation()
@see setAnnotation()
@see appendAnnotation()
@see appendAnnotation()
@see unsetAnnotation()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setAnnotation(string annotation)</pre>

Sets the value of the \'annotation\' subelement of this SBML object.

The content of @p annotation is copied, and any previous content of
this object\'s \'annotation\' subelement is deleted.

Whereas the SBase \'notes\' subelement is a container for content to be
shown directly to humans, the \'annotation\' element is a container for
optional software-generated content @em not meant to be shown to
humans.  Every object derived from SBase can have its own value for
\'annotation\'.  The element\'s content type is <a target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
\'any\'</a>, allowing essentially arbitrary well-formed XML data
content.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

Call this method will result in any existing content of the
\'annotation\' subelement to be discarded.  Unless you have taken steps
to first copy and reconstitute any existing annotations into the @p
annotation that is about to be assigned, it is likely that performing
such wholesale replacement is unfriendly towards other software
applications whose annotations are discarded.  An alternative may be
to use SBase.appendAnnotation() or
SBase.appendAnnotation().

@param annotation an XML string that is to be used as the content
of the \'annotation\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getAnnotationString()
@see isSetAnnotation()
@see setAnnotation()
@see appendAnnotation()
@see appendAnnotation()
@see unsetAnnotation()
";


%feature("docstring") SBase::appendAnnotation "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>appendAnnotation(XMLNode annotation)</pre>

Appends the given @p annotation to the \'annotation\' subelement of this
object.

Whereas the SBase \'notes\' subelement is a container for content to be
shown directly to humans, the \'annotation\' element is a container for
optional software-generated content @em not meant to be shown to
humans.  Every object derived from SBase can have its own value for
\'annotation\'.  The element\'s content type is <a
target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type \'any\'</a>,
allowing essentially arbitrary well-formed XML data content.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

Unlike SBase.setAnnotation() or
SBase.setAnnotation(), this method
allows other annotations to be preserved when an application adds its
own data.

@param annotation an XML structure that is to be copied and appended
to the content of the \'annotation\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
(if the parent element does not have the \'metaid\' attribute set)
@li @link libsbml#LIBSBML_DUPLICATE_ANNOTATION_NS LIBSBML_DUPLICATE_ANNOTATION_NS@endlink
(if the parent was already annotated with the annotation in question)

@see getAnnotationString()
@see isSetAnnotation()
@see setAnnotation()
@see setAnnotation()
@see appendAnnotation()
@see unsetAnnotation()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>appendAnnotation(string annotation)</pre>

Appends the given @p annotation to the \'annotation\' subelement of this
object.

Whereas the SBase \'notes\' subelement is a container for content to be
shown directly to humans, the \'annotation\' element is a container for
optional software-generated content @em not meant to be shown to
humans.  Every object derived from SBase can have its own value for
\'annotation\'.  The element\'s content type is <a
target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type \'any\'</a>,
allowing essentially arbitrary well-formed XML data content.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

Unlike SBase.setAnnotation() or
SBase.setAnnotation(), this method
allows other annotations to be preserved when an application adds its
own data.

@param annotation an XML string that is to be copied and appended
to the content of the \'annotation\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
(if the parent element does not have the \'metaid\' attribute set)
@li @link libsbml#LIBSBML_DUPLICATE_ANNOTATION_NS LIBSBML_DUPLICATE_ANNOTATION_NS@endlink
(if the parent was already annotated with the annotation in question)

@see getAnnotationString()
@see isSetAnnotation()
@see setAnnotation()
@see setAnnotation()
@see appendAnnotation()
@see unsetAnnotation()
";


%feature("docstring") SBase::removeTopLevelAnnotationElement "
Removes the top-level element within the \'annotation\' subelement of this
SBML object with the given name and optional URI.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

Calling this method allows a particular annotation element to be removed
whilst the remaining annotations remain intact.

@param elementName a string representing the name of the top level
annotation element that is to be removed.
@param elementURI an optional string that is used to check both the name
and URI of the top level element to be removed.
@param removeEmpty if after removing of the element, the annotation is 
empty, and the removeEmpty argument is true, the annotation node will be 
deleted (default). 

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_ANNOTATION_NAME_NOT_FOUND LIBSBML_ANNOTATION_NAME_NOT_FOUND@endlink
@li @link libsbml#LIBSBML_ANNOTATION_NS_NOT_FOUND LIBSBML_ANNOTATION_NS_NOT_FOUND@endlink

@see replaceTopLevelAnnotationElement()
@see replaceTopLevelAnnotationElement()
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBase::replaceTopLevelAnnotationElement "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>replaceTopLevelAnnotationElement(XMLNode annotation)</pre>

Replaces the given top-level element within the \'annotation\' 
subelement of this SBML object and with the annotation element supplied.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

This method determines the name of the element to be replaced from the
annotation argument. Functionally it is equivalent to calling <code>
removeTopLevelAnnotationElement(name)</code> followed by calling
<code>appendAnnotation(annotation_with_name)</code>, with the exception
that the placement of the annotation element remains the same.

@param annotation XMLNode representing the replacement top level annotation.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@see removeTopLevelAnnotationElement()
@see replaceTopLevelAnnotationElement()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>replaceTopLevelAnnotationElement(string annotation)</pre>

Replaces the given top-level element within the \'annotation\' 
subelement of this SBML object and with the annotation element supplied.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

This method determines the name of the element to be replaced from the
annotation argument. Functionally it is equivalent to calling <code>
removeTopLevelAnnotationElement(name)</code> followed by calling
<code>appendAnnotation(annotation_with_name)</code>, with the exception
that the placement of the annotation element remains the same.

@param annotation string representing the replacement top level annotation.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@see removeTopLevelAnnotationElement()
@see replaceTopLevelAnnotationElement()
";


%feature("docstring") SBase::setNotes "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setNotes(string notes, bool addXHTMLMarkup = false)</pre>

Sets the value of the \'notes\' subelement of this SBML object to a copy
of the string @p notes.

The content of @p notes is copied, and any existing content of this
object\'s \'notes\' subelement is deleted.

The optional SBML element named \'notes\', present on every major SBML
component type, is intended as a place for storing optional
information intended to be seen by humans.  An example use of the
\'notes\' element would be to contain formatted user comments about the
model element in which the \'notes\' element is enclosed.  Every object
derived directly or indirectly from type SBase can have a separate
value for \'notes\', allowing users considerable freedom when adding
comments to their models.

The format of \'notes\' elements must be <a target=\'_blank\'
href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
verify the formatting of \'notes\' content, libSBML provides the static
utility method SyntaxChecker.hasExpectedXHTMLSyntax(); however,
readers are urged to consult the appropriate <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specification
document</a> for the Level and Version of their model for more
in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
specifications have considerable detail about how \'notes\' element
content must be structured.

The following code illustrates a very simple way of setting the notes
using this method.  Here, the object being annotated is the whole SBML
document, but that is for illustration purposes only; you could of
course use this same approach to annotate any other SBML component.
@if cpp
@code{.cpp}
SBMLDocument s = new SBMLDocument(3, 1);
s->setNotes(\'<body xmlns=\'http://www.w3.org/1999/xhtml\'><p>here is my note</p></body>\');
@endcode
@endif
@if java
@code{.java}
SBMLDocument s = new SBMLDocument(3, 1);
s.setNotes(\'<body xmlns=\'http://www.w3.org/1999/xhtml\'><p>here is my note</p></body>\');
@endcode
@endif
@if python
@code{.py}
try:
  sbmlDoc = SBMLDocument(3, 1)
except ValueError:
  print(\'Could not create SBMLDocument object\')
  sys.exit(1)

note = \'<body xmlns=\'http://www.w3.org/1999/xhtml\'><p>here is my note</p></body>\'

status = sbmlDoc.setNotes(note)
if status != LIBSBML_OPERATION_SUCCESS:
  # Do something to handle the error here.
  print(\'Unable to set notes on the SBML document object\')
  sys.exit(1)
@endcode
@endif
@if csharp
@code
SBMLDocument s = new SBMLDocument(3, 1);
s.setNotes(\'<body xmlns=\'http://www.w3.org/1999/xhtml\'><p>here is my note</p></body>\');
@endcode
@endif@~

@param notes an XML string that is to be used as the content of the
\'notes\' subelement of this object.

@param addXHTMLMarkup a boolean indicating whether to wrap the contents
of the @p notes argument with XHTML paragraph (<code>&lt;p&gt;</code>)
tags.  This is appropriate when the string in @p notes does not already
containg the appropriate XHTML markup.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getNotesString()
@see isSetNotes()
@see setNotes()
@see appendNotes()
@see appendNotes()
@see unsetNotes()
@see SyntaxChecker.hasExpectedXHTMLSyntax()
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.


@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setNotes(XMLNode notes)</pre>

Sets the value of the \'notes\' subelement of this SBML object.

The content of @p notes is copied, and any existing content of this
object\'s \'notes\' subelement is deleted.

The optional SBML element named \'notes\', present on every major SBML
component type, is intended as a place for storing optional
information intended to be seen by humans.  An example use of the
\'notes\' element would be to contain formatted user comments about the
model element in which the \'notes\' element is enclosed.  Every object
derived directly or indirectly from type SBase can have a separate
value for \'notes\', allowing users considerable freedom when adding
comments to their models.

The format of \'notes\' elements must be <a target=\'_blank\'
href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
verify the formatting of \'notes\' content, libSBML provides the static
utility method SyntaxChecker.hasExpectedXHTMLSyntax(); however,
readers are urged to consult the appropriate <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specification
document</a> for the Level and Version of their model for more
in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
specifications have considerable detail about how \'notes\' element
content must be structured.

@param notes an XML structure that is to be used as the content of the
\'notes\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@see getNotesString()
@see isSetNotes()
@see setNotes()
@see appendNotes()
@see appendNotes()
@see unsetNotes()
@see SyntaxChecker.hasExpectedXHTMLSyntax()
";


%feature("docstring") SBase::appendNotes "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>appendNotes(string notes)</pre>

Appends the given @p notes to the \'notes\' subelement of this object.

The content of the parameter @p notes is copied.

The optional SBML element named \'notes\', present on every major SBML
component type, is intended as a place for storing optional
information intended to be seen by humans.  An example use of the
\'notes\' element would be to contain formatted user comments about the
model element in which the \'notes\' element is enclosed.  Every object
derived directly or indirectly from type SBase can have a separate
value for \'notes\', allowing users considerable freedom when adding
comments to their models.

The format of \'notes\' elements must be <a target=\'_blank\'
href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
verify the formatting of \'notes\' content, libSBML provides the static
utility method SyntaxChecker.hasExpectedXHTMLSyntax(); however,
readers are urged to consult the appropriate <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specification
document</a> for the Level and Version of their model for more
in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
specifications have considerable detail about how \'notes\' element
content must be structured.

@param notes an XML string that is to appended to the content of
the \'notes\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getNotesString()
@see isSetNotes()
@see setNotes()
@see setNotes()
@see appendNotes()
@see unsetNotes()
@see SyntaxChecker.hasExpectedXHTMLSyntax()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>appendNotes(XMLNode notes)</pre>

Appends the given @p notes to the \'notes\' subelement of this object.

The content of @p notes is copied.

The optional SBML element named \'notes\', present on every major SBML
component type, is intended as a place for storing optional
information intended to be seen by humans.  An example use of the
\'notes\' element would be to contain formatted user comments about the
model element in which the \'notes\' element is enclosed.  Every object
derived directly or indirectly from type SBase can have a separate
value for \'notes\', allowing users considerable freedom when adding
comments to their models.

The format of \'notes\' elements must be <a target=\'_blank\'
href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
verify the formatting of \'notes\' content, libSBML provides the static
utility method SyntaxChecker.hasExpectedXHTMLSyntax(); however,
readers are urged to consult the appropriate <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specification
document</a> for the Level and Version of their model for more
in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
specifications have considerable detail about how \'notes\' element
content must be structured.

@param notes an XML node structure that is to appended to the content
of the \'notes\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getNotesString()
@see isSetNotes()
@see setNotes()
@see setNotes()
@see appendNotes()
@see unsetNotes()
@see SyntaxChecker.hasExpectedXHTMLSyntax()
";


%feature("docstring") SBase::setModelHistory "
Sets the ModelHistory of this object.

The content of @p history is copied, and this object\'s existing model
history content is deleted.

@param history ModelHistory of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@note In SBML Level&nbsp;2, model history annotations were only
permitted on the Model element.  In SBML Level&nbsp;3, they are
permitted on all SBML components derived from SBase.
";


%feature("docstring") SBase::setSBMLDocument "
@internal

@internal
";


%feature("docstring") SBase::connectToParent "
@internal

@internal
";


%feature("docstring") SBase::setSBOTerm "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setSBOTerm(int value)</pre>

Sets the value of the \'sboTerm\' attribute.

@par
Beginning with SBML Level 2 Version 2, objects derived from SBase have
an optional attribute named \'sboTerm\' for supporting the use of the
Systems Biology Ontology.  In SBML proper, the data type of the
attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
seven digit integer number; libSBML simplifies the representation by
only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
\'sboTerm\' attribute on SBase has data type @c int, and SBO identifiers
are stored simply as integers.
@par
SBO terms are a type of optional annotation, and each different class
of SBML object derived from SBase imposes its own requirements about
the values permitted for \'sboTerm\'. More details can be found in SBML
specifications for Level&nbsp;2 Version&nbsp;2 and above.

@param value the NNNNNNN integer portion of the SBO identifier.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see setSBOTerm()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setSBOTerm(string &sboid)</pre>

Sets the value of the \'sboTerm\' attribute by string.

@par
Beginning with SBML Level 2 Version 2, objects derived from SBase have
an optional attribute named \'sboTerm\' for supporting the use of the
Systems Biology Ontology.  In SBML proper, the data type of the
attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
seven digit integer number; libSBML simplifies the representation by
only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
\'sboTerm\' attribute on SBase has data type @c int, and SBO identifiers
are stored simply as integers.
@par
SBO terms are a type of optional annotation, and each different class
of SBML object derived from SBase imposes its own requirements about
the values permitted for \'sboTerm\'. More details can be found in SBML
specifications for Level&nbsp;2 Version&nbsp;2 and above.

@param sboid the SBO identifier string of the form \'SBO:NNNNNNN\'.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see setSBOTerm()
";


%feature("docstring") SBase::setNamespaces "
Sets the namespaces relevant of this SBML object.

The content of @p xmlns is copied, and this object\'s existing
namespace content is deleted.

The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and
(in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.

@param xmlns the namespaces to set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
";


%feature("docstring") SBase::unsetMetaId "
Unsets the value of the \'metaid\' attribute of this SBML object.

@par
The optional attribute named \'metaid\', present on every major SBML
component type, is for supporting metadata annotations using RDF (<a
href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
attribute value has the data type <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
identifier type, which means each \'metaid\' value must be globally unique
within an SBML file.  The latter point is important, because the
uniqueness criterion applies across <em>any</em> attribute with type
<code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
by SBML---something to be aware of if your application-specific XML
content inside the \'annotation\' subelement happens to use the XML
<code>ID</code> type.  Although SBML itself specifies the use of <a
href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
the \'metaid\' attribute, SBML-compatible applications should be careful if
they use XML <code>ID</code>\'s in XML portions of a model that are not
defined by SBML, such as in the application-specific content of the
\'annotation\' subelement.  Finally, note that LibSBML does not provide an
explicit XML <code>ID</code> data type; it uses ordinary character
strings, which is easier for applications to support. 

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") SBase::unsetId "
Unsets the value of the \'id\' attribute of this SBML object.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") SBase::unsetIdAttribute "
Unsets the value of the \'id\' attribute of this SBML object.

Most (but not all) objects in SBML include two common attributes: \'id\'
and \'name\'.  The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") SBase::unsetName "
Unsets the value of the \'name\' attribute of this SBML object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") SBase::unsetNotes "
Unsets the value of the \'notes\' subelement of this SBML object.

The optional SBML element named \'notes\', present on every major SBML
component type, is intended as a place for storing optional
information intended to be seen by humans.  An example use of the
\'notes\' element would be to contain formatted user comments about the
model element in which the \'notes\' element is enclosed.  Every object
derived directly or indirectly from type SBase can have a separate
value for \'notes\', allowing users considerable freedom when adding
comments to their models.

The format of \'notes\' elements must be <a target=\'_blank\'
href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
verify the formatting of \'notes\' content, libSBML provides the static
utility method SyntaxChecker.hasExpectedXHTMLSyntax(); however,
readers are urged to consult the appropriate <a target=\'_blank\'
href=\'http://sbml.org/Documents/Specifications\'>SBML specification
document</a> for the Level and Version of their model for more
in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
specifications have considerable detail about how \'notes\' element
content must be structured.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@see getNotesString()
@see isSetNotes()
@see setNotes()
@see setNotes()
@see appendNotes()
@see appendNotes()
@see SyntaxChecker.hasExpectedXHTMLSyntax()
";


%feature("docstring") SBase::unsetAnnotation "
Unsets the value of the \'annotation\' subelement of this SBML object.

Whereas the SBase \'notes\' subelement is a container for content to be
shown directly to humans, the \'annotation\' element is a container for
optional software-generated content @em not meant to be shown to
humans.  Every object derived from SBase can have its own value for
\'annotation\'.  The element\'s content type is <a target=\'_blank\'
href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
\'any\'</a>, allowing essentially arbitrary well-formed XML data
content.

SBML places a few restrictions on the organization of the content of
annotations; these are intended to help software tools read and write
the data as well as help reduce conflicts between annotations added by
different tools.  Please see the SBML specifications for more details.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@see getAnnotation()
@see getAnnotationString()
@see isSetAnnotation()
@see setAnnotation()
@see setAnnotation()
@see appendAnnotation()
@see appendAnnotation()
";


%feature("docstring") SBase::unsetSBOTerm "
Unsets the value of the \'sboTerm\' attribute of this SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
";


%feature("docstring") SBase::addCVTerm "
Adds a copy of the given CVTerm object to this SBML object.

@param term the CVTerm to assign.

@param newBag if @c True, creates a new RDF bag with the same identifier
as a previous bag, and if @c False, adds the term to an existing
RDF bag with the same type of qualifier as the term being added.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink, if
this object lacks a \'metaid\' attribute
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@note Since the CV Term uses the \'metaid\' attribute of the object as a
reference, if the object has no \'metaid\' attribute value set, then the
CVTerm will not be added.

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBase::getCVTerms "
Returns a list of CVTerm objects in the annotations of this SBML
object.

@return the list of CVTerms for this SBML object.
";


%feature("docstring") SBase::getNumCVTerms "
Returns the number of CVTerm objects in the annotations of this SBML
object.

@return the number of CVTerms for this SBML object.
";


%feature("docstring") SBase::getCVTerm "
Returns the nth CVTerm in the list of CVTerms of this SBML
object.

@param n long the index of the CVTerm to retrieve.

@return the nth CVTerm in the list of CVTerms for this SBML object.
If the index @p n is invalid, @c None is returned.
";


%feature("docstring") SBase::unsetCVTerms "
Clears the list of CVTerm objects attached to this SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") SBase::unsetModelHistory "
Unsets the ModelHistory object attached to this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note In SBML Level&nbsp;2, model history annotations were only
permitted on the Model element.  In SBML Level&nbsp;3, they are
permitted on all SBML components derived from SBase.
";


%feature("docstring") SBase::getResourceBiologicalQualifier "
Returns the MIRIAM <em>biological qualifier</em> associated with the
given resource.

In <a target=\'_blank\' href=\'http://biomodels.net/miriam\'>MIRIAM</a>,
qualifiers are an optional means of indicating the relationship
between a model component and its annotations.  There are two broad
kinds of annotations: <em>model</em> and <em>biological</em>.  The
latter kind is used to qualify the relationship between a model
component and a biological entity which it represents.  Examples of
relationships include \'is\' and \'has part\', but many others are
possible.  MIRIAM defines <a target=\'_blank\'
href=\'http://www.ebi.ac.uk/miriam/main/qualifiers/\'>numerous
relationship qualifiers</a> to enable different software tools to
qualify biological annotations in the same standardized way.  In
libSBML, the MIRIAM controlled-vocabulary annotations on an SBML model
element are represented using lists of CVTerm objects, and the
the MIRIAM biological qualifiers are represented using
values @if clike from the enumeration
type #BiolQualifierType_t.@endif@if python whose
names begin with <code>BQB_</code> in the interface class
@link libsbml libsbml@endlink.@endif@if java whose
names begin with <code>BQB_</code> in the interface class
{@link libsbmlConstants}.@endif@if csharp whose
names begin with <code>BQB_</code> in the interface class
@link libsbmlcs.libsbml libsbml@endlink.@endif@~

This method searches the controlled-vocabulary annotations
(i.e., the list of CVTerm objects) on the present object, then out of
those that have biological qualifiers, looks for an annotation to the
given @p resource.  If such an annotation is found, it returns the
type of biological qualifier associated with that resource as a 
value @if clike from the enumeration type
#BiolQualifierType_t.@endif@if python whose name begins with
<code>BQB_</code> from the interface
class @link libsbml libsbml@endlink.@endif@if java whose name
begins with <code>BQB_</code> from the interface
class {@link libsbmlConstants}.@endif@if csharp whose
names begin with <code>BQB_</code> in the interface class
@link libsbmlcs.libsbml libsbml@endlink.@endif@~

@param resource string representing the resource; e.g.,
<code>\'http://www.geneontology.org/#GO:0005892\'</code>.

@return the qualifier associated with the resource,
or @link libsbml#BQB_UNKNOWN BQB_UNKNOWN@endlink if the
resource does not exist.

@if clike
@note The set of MIRIAM biological qualifiers grows over
time, although relatively slowly.  The values in the enumeration
#BiolQualifierType_t are up to date with MIRIAM at the time of a given
libSBML release.  The set of values may be expanded in later libSBML
releases, to match the values defined by MIRIAM at that later time.
@endif@if python
@note The set of MIRIAM biological qualifiers grows over
time, although relatively slowly.  The values are up to date with
MIRIAM at the time of a given libSBML release.  The set of values in
list of <code>BQB_</code> constants defined in @link libsbml
libsbml@endlink may be expanded in later libSBML releases, to match
the values defined by MIRIAM at that later time.
@endif@if java
@note The set of MIRIAM biological qualifiers grows over
time, although relatively slowly.  The values are up to date with
MIRIAM at the time of a given libSBML release.  The set of values in
list of <code>BQB_</code> constants defined in {@link libsbmlConstants}
may be expanded in later libSBML releases, to match
the values defined by MIRIAM at that later time.
@endif@if csharp
@note The set of MIRIAM biological qualifiers grows over
time, although relatively slowly.  The values are up to date with
MIRIAM at the time of a given libSBML release.  The set of values in
list of <code>BQB_</code> constants defined in @link libsbmlcs.libsbml libsbml@endlink
may be expanded in later libSBML releases, to match
the values defined by MIRIAM at that later time.
@endif@~
";


%feature("docstring") SBase::getResourceModelQualifier "
Returns the MIRIAM <em>model qualifier</em> associated with the
given resource.

In <a target=\'_blank\' href=\'http://biomodels.net/miriam\'>MIRIAM</a>,
qualifiers are an optional means of indicating the relationship
between a model component and its annotations.  There are two broad
kinds of annotations: <em>model</em> and <em>biological</em>.  The
former kind is used to qualify the relationship between a model
component and another modeling object.  An example qualifier is
\'isDerivedFrom\', to indicate that a given component of the model is
derived from the modeling object represented by the referenced
resource.  MIRIAM defines <a target=\'_blank\'
href=\'http://www.ebi.ac.uk/miriam/main/qualifiers/\'>numerous
relationship qualifiers</a> to enable different software tools to
qualify model annotations in the same standardized way.  In libSBML,
the MIRIAM controlled-vocabulary annotations on an SBML model element
are represented using lists of CVTerm objects, and the 
the MIRIAM model qualifiers are represented using
values @if clike from the enumeration
type #ModelQualifierType_t.@endif@if python whose
names begin with <code>BQM_</code> in the interface class
@link libsbml libsbml@endlink.@endif@if java whose
names begin with <code>BQM_</code> in the interface class
{@link libsbmlConstants}.@endif@if csharp whose
names begin with <code>BQB_</code> in the interface class
@link libsbmlcs.libsbml libsbml@endlink.@endif@~

This method method searches the controlled-vocabulary annotations
(i.e., the list of CVTerm objects) on the present object, then out of
those that have model qualifiers, looks for an annotation to the given
@p resource.  If such an annotation is found, it returns the type of
type of model qualifier associated with that resource as a 
value @if clike from the enumeration type
#ModelQualifierType_t.@endif@if python whose name begins with
<code>BQM_</code> from the interface
class @link libsbml libsbml@endlink.@endif@if java whose name
begins with <code>BQM_</code> from the interface
class {@link libsbmlConstants}.@endif@if csharp whose
names begin with <code>BQB_</code> in the interface class
@link libsbmlcs.libsbml libsbml@endlink.@endif@~

@param resource string representing the resource; e.g.,
<code>\'http://www.geneontology.org/#GO:0005892\'</code>.

@return the @if clike #ModelQualifierType_t value@else model qualifier
type@endif@~ associated with the resource, or @link libsbml#BQM_UNKNOWN BQM_UNKNOWN@endlink if the resource does not exist.

@if clike
@note The set of MIRIAM biological qualifiers grows over
time, although relatively slowly.  The values in the enumeration
#ModelQualifierType_t are up to date with MIRIAM at the time of a given
libSBML release.  The set of values may be expanded in later libSBML
releases, to match the values defined by MIRIAM at that later time.
@endif@if python
@note The set of MIRIAM model qualifiers grows over
time, although relatively slowly.  The values are up to date with
MIRIAM at the time of a given libSBML release.  The set of values in
list of <code>BQM_</code> constants defined in @link libsbml
libsbml@endlink may be expanded in later libSBML releases, to match
the values defined by MIRIAM at that later time.
@endif@if java
@note The set of MIRIAM model qualifiers grows over
time, although relatively slowly.  The values are up to date with
MIRIAM at the time of a given libSBML release.  The set of values in
list of <code>BQM_</code> constants defined in {@link libsbmlConstants}
may be expanded in later libSBML releases, to match
the values defined by MIRIAM at that later time.
@endif@if csharp
@note The set of MIRIAM model qualifiers grows over
time, although relatively slowly.  The values are up to date with
MIRIAM at the time of a given libSBML release.  The set of values in
list of <code>BQM_</code> constants defined in @link libsbmlcs.libsbml libsbml@endlink
may be expanded in later libSBML releases, to match
the values defined by MIRIAM at that later time.
@endif@~
";


%feature("docstring") SBase::getModel "
Returns the Model object for the SBML Document in which the current object is located.

@return the Model object for the SBML Document of this SBML object.

@see getParentSBMLObject()
@see getSBMLDocument()
";


%feature("docstring") SBase::getLevel "
Returns the SBML Level of the SBMLDocument object containing @em this
object.

@par
LibSBML uses the class SBMLDocument as a top-level container for
storing SBML content and data associated with it (such as warnings and
error messages).  An SBML model in libSBML is contained inside an
SBMLDocument object.  SBMLDocument corresponds roughly to the class
<i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
specifications, but it does not have a direct correspondence in SBML
Level&nbsp;1.  (But, it is created by libSBML no matter whether the
model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)

@return the SBML level of this SBML object.

@see getVersion()
@see getNamespaces()
@see getPackageVersion()
";


%feature("docstring") SBase::getVersion "
Returns the Version within the SBML Level of the SBMLDocument object
containing @em this object.

@par
LibSBML uses the class SBMLDocument as a top-level container for
storing SBML content and data associated with it (such as warnings and
error messages).  An SBML model in libSBML is contained inside an
SBMLDocument object.  SBMLDocument corresponds roughly to the class
<i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
specifications, but it does not have a direct correspondence in SBML
Level&nbsp;1.  (But, it is created by libSBML no matter whether the
model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)

@return the SBML version of this SBML object.

@see getLevel()
@see getNamespaces()
";


%feature("docstring") SBase::getPackageCoreVersion "
Returns the SBML Core Version within the SBML Level of the actual object.

@par
LibSBML uses the class SBMLDocument as a top-level container for
storing SBML content and data associated with it (such as warnings and
error messages).  An SBML model in libSBML is contained inside an
SBMLDocument object.  SBMLDocument corresponds roughly to the class
<i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
specifications, but it does not have a direct correspondence in SBML
Level&nbsp;1.  (But, it is created by libSBML no matter whether the
model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)

@return the SBML core version of this SBML object.
";


%feature("docstring") SBase::getPackageVersion "
Returns the Version of the SBML Level&nbsp;3 package to which this
element belongs to.

@return the version of the SBML Level&nbsp;3 package to which this
element belongs. The value @c 0 will be returned if this element
belongs to the SBML Level&nbsp;3 Core package.

@see getLevel()
@see getVersion()
";


%feature("docstring") SBase::getPackageName "
Returns the name of the SBML Level&nbsp;3 package in which this element
is defined.

@return the name of the SBML package in which this element is defined.
The string <code>&quot;core&quot;</code> will be returned if this
element is defined in SBML Level&nbsp;3 Core. The string
<code>&quot;unknown&quot;</code> will be returned if this element is
not defined in any SBML package.
";


%feature("docstring") SBase::getTypeCode "
Returns the libSBML type code for this object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@par
Here follow some additional general information about libSBML type codes:

@li The codes are the possible return values (integers) for the following
functions:
<ul>
<li> SBase.getTypeCode()
<li> ListOf.getItemTypeCode()
</ul>
(Compatibility note: in libSBML 5, the type of return values of these
functions changed from an enumeration to an integer for extensibility
in the face of different programming languages.)

@li Each package extension must define similar sets of values for each
SBase subclass (e.g. <code>SBMLLayoutTypeCode_t</code> for the SBML Level&nbsp;3
%Layout extension, <code>SBMLFbcTypeCode_t</code> for the SBML Level&nbsp;3 Flux
Balance Constraints extension, etc.).

@li The value of each package-specific type code can be duplicated between
those of different packages.  (This is necessary because the development
of libSBML extensions for different SBML packages may be undertaken by
different developers at different times; requiring the developers to
coordinate their use of type codes would be nettlesome and probably
doomed to failure.)

@li To distinguish between the type codes of different packages, both the
return value of SBase.getTypeCode() and SBase.getPackageName() must be
checked.  This is particularly important for functions that take an SBML
type code as an argument, such as
SBase.getAncestorOfType(), which by
default assumes you are handing it a core type, and will return @c None if
the value you give it is actually from a package.

The following example code illustrates the combined use of
SBase.getPackageName() and SBase.getTypeCode():
@if cpp
@code{.cpp}
 void example (SBase sb)
 {
   cons string pkgName = sb->getPackageName();
   if (pkgName == \'core\')
   {
     switch (sb->getTypeCode())
     {
       case SBML_MODEL:
          ....
          break;
       case SBML_REACTION:
          ....
     }
   }
   else if (pkgName == \'layout\')
   {
     switch (sb->getTypeCode())
     {
       case SBML_LAYOUT_LAYOUT:
          ....
          break;
       case SBML_LAYOUT_REACTIONGLYPH:
          ....
     }
   }
   ...
 }
@endcode
@endif
@if python
@code{.py}
def example(item):
  pkg_name  = item.getPackageName()
  type_code = item.getTypeCode()
  if pkg_name == \'core\':
    print(\'Got a \' + SBMLTypeCode_toString(type_code, \'core\') + \' object\')
    if type_code == SBML_MODEL:
      print(\'This is a very, very nice model\')
      # Do whatever the application wants to do with the model.
    elif type_code == SBML_COMPARTMENT:
      print(\'This is a very, very nice compartment\')
      # Do whatever the application wants to do with the compartment.
    elif type_code == SBML_SPECIES:
      print(\'This is a very, very nice species\')
      # Do whatever the application wants to do with the species.
    elif ...
      ...
  elif pkg_name == \'layout\':
    print(\'Got a \' + SBMLTypeCode_toString(type_code, \'layout\') + \' object\')
    if type_code == SBML_LAYOUT_POINT:
      print(\'This is a very, very nice layout point\')
      # Do whatever the application wants to do with the layout point.
    elif type_code == SBML_LAYOUT_BOUNDINGBOX:
      print(\'This is a very, very nice layout bounding box\')
      # Do whatever the application wants to do with the layout bounding box.
    elif ...
      ...
  elif pkg_name == \'unknown\':
    print(\'Something went wrong -- libSBML did not recognize the object type\')
    # Handle errors
@endcode
@endif
@if java
@code{.java}
void example (SBase sb)
{
  String pkgName = sb.getPackageName();
  if (pkgName.equals(\'core\'))
  {
    switch (sb.getTypeCode())
    {
      case libsbml.SBML_MODEL:
         ....
         break;
      case libsbml.SBML_REACTION:
         ....
    }
  }
  else if (pkgName.equals(\'layout\'))
  {
    switch (sb.getTypeCode())
    {
      case libsbml.SBML_LAYOUT_LAYOUT:
         ....
         break;
      case libsbml.SBML_LAYOUT_REACTIONGLYPH:
         ....
    }
  }
  ...
}
@endcode
@endif

@return the @if clike #SBMLTypeCode_t value@else SBML object type
code@endif@~ of this SBML object, or @link libsbml#SBML_UNKNOWN SBML_UNKNOWN@endlink (the default).  The latter is possible because
subclasses of SBase are not required to implement this method to return
a type code.

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getPackageName()
@see getElementName()
";


%feature("docstring") SBase::getElementName "
Returns the XML element name of this object.

This is overridden by subclasses to return a string appropriate to the
SBML component.  For example, Model defines it as returning
@c \'model\', CompartmentType defines it as returning @c \'compartmentType\',
and so on.
";


%feature("docstring") SBase::toSBML "
Returns a string consisting of a partial SBML corresponding to just
this object.

@return the partial SBML that describes this SBML object.

@warning <span class=\'warning\'>This is primarily provided for testing
and debugging purposes.  It may be removed in a future version of
libSBML.</span>
";


%feature("docstring") SBase::toXMLNode "
Returns this element as an XMLNode.

@return this element as an XMLNode.

@warning <span class=\'warning\'>This operation is computationally
expensive, because the element has to be fully serialized to a string
and then parsed into the XMLNode structure.  Attempting to convert a
large tree structure (e.g., a large Model) may consume significant
computer memory and time.</span>
";


%feature("docstring") SBase::getPlugin "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getPlugin(long n)</pre>

Returns the nth plug-in object (extension interface) for an SBML Level&nbsp;3
package extension.
The returned plug-in will be the appropriate type of plugin requested:
calling Model.getPlugin() will return an FbcModelPlugin; calling
Parameter.getPlugin() will return CompSBasePlugin, etc.

If no such plugin exists, None is returned.

@par
SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
that it uses one or more SBML packages, and likewise, a software tool may be
able to support one or more packages.  LibSBML does not come preconfigured
with all possible packages included and enabled, in part because not all
package specifications have been finalized.  To support the ability for
software systems to enable support for the Level&nbsp;3 packages they choose,
libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
package is implemented in a separate code plug-in that can be enabled by the
application to support working with that SBML package.  A given SBML model
may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
objects created by libSBML plug-ins supporting additional Level&nbsp;3
packages.

@param n the index of the plug-in to return.

@return the nth plug-in object (the libSBML extension interface) of a
package extension.
If the index @p n is invalid, @c None is returned.

@see getNumPlugins()
@see getPlugin()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getPlugin(string package)</pre>

Returns a plug-in object (extension interface) for an SBML Level&nbsp;3
package extension with the given package name or URI.
The returned plug-in will be the appropriate type of plugin requested:
calling Model.getPlugin() will return an FbcModelPlugin; calling
Parameter.getPlugin() will return CompSBasePlugin, etc.

If no such plugin exists, None is returned.

@par
SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
that it uses one or more SBML packages, and likewise, a software tool may be
able to support one or more packages.  LibSBML does not come preconfigured
with all possible packages included and enabled, in part because not all
package specifications have been finalized.  To support the ability for
software systems to enable support for the Level&nbsp;3 packages they choose,
libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
package is implemented in a separate code plug-in that can be enabled by the
application to support working with that SBML package.  A given SBML model
may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
objects created by libSBML plug-ins supporting additional Level&nbsp;3
packages.

@param package the name or URI of the package.

@return the plug-in object (the libSBML extension interface) of a
package extension with the given package name or URI.

@see getPlugin()
";


%feature("docstring") SBase::getDisabledPlugin "
Returns the nth disabled plug-in object (extension interface) for an SBML Level&nbsp;3
package extension.  If no such plugin exists, None is returned.

@par
SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
that it uses one or more SBML packages, and likewise, a software tool may be
able to support one or more packages.  LibSBML does not come preconfigured
with all possible packages included and enabled, in part because not all
package specifications have been finalized.  To support the ability for
software systems to enable support for the Level&nbsp;3 packages they choose,
libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
package is implemented in a separate code plug-in that can be enabled by the
application to support working with that SBML package.  A given SBML model
may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
objects created by libSBML plug-ins supporting additional Level&nbsp;3
packages.

@par
If a plugin is <em>disabled</em>, the package information it contains is
no longer considered to be part of the SBML document for the purposes of
searching the document or writing out the document.  However, the information
is still retained, so if the plugin is enabled again, the same information
will once again be available, and will be written out to the final model.

@param n the index of the disabled plug-in to return.

@return the nth disabled plug-in object (the libSBML extension interface) of a
package extension.
If the index @p n is invalid, @c None is returned.

@see getNumDisabledPlugins()
@see getPlugin()
";


%feature("docstring") SBase::getNumPlugins "
Returns the number of plug-in objects (extenstion interfaces) for SBML
Level&nbsp;3 package extensions known.

@par
SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
that it uses one or more SBML packages, and likewise, a software tool may be
able to support one or more packages.  LibSBML does not come preconfigured
with all possible packages included and enabled, in part because not all
package specifications have been finalized.  To support the ability for
software systems to enable support for the Level&nbsp;3 packages they choose,
libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
package is implemented in a separate code plug-in that can be enabled by the
application to support working with that SBML package.  A given SBML model
may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
objects created by libSBML plug-ins supporting additional Level&nbsp;3
packages.

@return the number of plug-in objects (extension interfaces) of
package extensions known by this instance of libSBML.

@see getPlugin()
";


%feature("docstring") SBase::getNumDisabledPlugins "
Returns the number of disabled plug-in objects (extension interfaces) 
for SBML Level&nbsp;3 package extensions known.

@par
SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
that it uses one or more SBML packages, and likewise, a software tool may be
able to support one or more packages.  LibSBML does not come preconfigured
with all possible packages included and enabled, in part because not all
package specifications have been finalized.  To support the ability for
software systems to enable support for the Level&nbsp;3 packages they choose,
libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
package is implemented in a separate code plug-in that can be enabled by the
application to support working with that SBML package.  A given SBML model
may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
objects created by libSBML plug-ins supporting additional Level&nbsp;3
packages.

@par
If a plugin is <em>disabled</em>, the package information it contains is
no longer considered to be part of the SBML document for the purposes of
searching the document or writing out the document.  However, the information
is still retained, so if the plugin is enabled again, the same information
will once again be available, and will be written out to the final model.

@return the number of disabled plug-in objects (extension interfaces) 
of package extensions known by this instance of libSBML.
";


%feature("docstring") SBase::deleteDisabledPlugins "
Deletes all information stored in disabled plugins.  If the plugin is 
re-enabled later, it will then not have any previously-stored information.

@par
SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
that it uses one or more SBML packages, and likewise, a software tool may be
able to support one or more packages.  LibSBML does not come preconfigured
with all possible packages included and enabled, in part because not all
package specifications have been finalized.  To support the ability for
software systems to enable support for the Level&nbsp;3 packages they choose,
libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
package is implemented in a separate code plug-in that can be enabled by the
application to support working with that SBML package.  A given SBML model
may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
objects created by libSBML plug-ins supporting additional Level&nbsp;3
packages.

@par
If a plugin is <em>disabled</em>, the package information it contains is
no longer considered to be part of the SBML document for the purposes of
searching the document or writing out the document.  However, the information
is still retained, so if the plugin is enabled again, the same information
will once again be available, and will be written out to the final model.

@param recursive if @c True, the disabled information will be deleted
also from all child elements, otherwise only from this SBase element.

@see getNumDisabledPlugins()
";


%feature("docstring") SBase::enablePackage "
Enables or disables the given SBML Level&nbsp;3 package on this object.

This method enables the specified package on this object and other
objects connected by child-parent links in the same SBMLDocument object.
This method is the converse of
SBase.disablePackage().

@param pkgURI the URI of the package.

@param pkgPrefix the XML prefix of the package.

@param flag whether to enable (@c True) or disable (@c False) the package.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_PKG_UNKNOWN LIBSBML_PKG_UNKNOWN@endlink
@li @link libsbml#LIBSBML_PKG_VERSION_MISMATCH LIBSBML_PKG_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_PKG_CONFLICTED_VERSION LIBSBML_PKG_CONFLICTED_VERSION@endlink

@see disablePackage()
";


%feature("docstring") SBase::disablePackage "
Disables the given SBML Level&nbsp;3 package on this object.

This method disables the specified package on this object
and other objects connected by child-parent links in the same
SBMLDocument object.

An example of when this may be useful is during construction of model
components when mixing existing and new models.  Suppose your
application read an SBML document containing a model that used the SBML
Hierarchical %Model Composition (&ldquo;comp&rdquo;) package, and
extracted parts of that model in order to construct a new model in
memory.  The new, in-memory model will not accept a component drawn from
an other SBMLDocument with different package namespace declarations.
You could reconstruct the same namespaces in the in-memory model first,
but as a shortcut, you could also disable the package namespace on the
object being added.  Here is a code example to help clarify this:
@if cpp
@code{.cpp}
// We read in an SBML L3V1 model that uses the \'comp\'
// package namespace.
doc = readSBML(\'sbml-file-with-comp-elements.xml\');

// We extract one of the species from the model.
Species s1 = doc->getModel()->getSpecies(0);

// We construct a new model.  This model does not use the
// \'comp\' package.
Model  newModel = new Model(3,1);

// The following will fail with an error, because addSpecies()
// will first check that the parent of the given object has
// namespaces declared, and will discover that s1 does but
// newModel does not.

// newModel->addSpecies(s1);

// However, if we disable the \'comp\' package on s1, then
// the call to addSpecies will work.

s1->disablePackage(\'http://www.sbml.org/sbml/level3/version1/comp/version1\',
                   \'comp\');
newModel->addSpecies(s1);
@endcode
@endif
@if python
@code{.py}
import sys
import os.path
from libsbml import *

# We read an SBML L3V1 model that uses the \'comp\' package.

doc = readSBML(\'sbml-file-with-comp-elements.xml\');
if doc.getNumErrors() > 0:
  print(\'readSBML encountered errors while reading the file.\')
  doc.printErrors()
  sys.exit(1)

# We extract one of the species from the model.

model = doc.getModel()
if model == None:
  print(\'Unable to retrieve Model object\')
  sys.exit(1)

s1 = model.getSpecies(0)
if s1 == None:
  print(\'Unable to retrieve Species object\')
  sys.exit(1)

# We construct a new model.
# This model does not use the \'comp\' package.

try:
  newDoc = SBMLDocument(3, 1)
except ValueError:
  print(\'Could not create SBMLDocument object\')
  sys.exit(1)

newModel = newDoc.createModel()
if newModel == None:
  print(\'Unable to create new Model object\')
  sys.exit(1)

# The following would normally fail with an error, because
# addSpecies() would first check that the parent of the given
# object has namespaces declared, and will discover that s1
# does but newModel does not.

#   newModel.addSpecies(s1)

# However, if we disable the \'comp\' package on s1, then the
# call to addSpecies will work.

compNS = \'http://www.sbml.org/sbml/level3/version1/comp/version1\'
status = s1.disablePackage(compNS, \'comp\')
if status != LIBSBML_OPERATION_SUCCESS:
  print(\'Unable to disable package.\')
  sys.exit(1)

newSpecies = newModel.addSpecies(s1)   # This will work now.
if newSpecies == None:
  print(\'Could not add Species\')       # (This will not happen,
  sys.exit(1)                          # but always check errors.)
@endcode
@endif
@if java
@code{.java}
// We read in an SBML L3V1 model that uses the \'comp\' package namespace
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBML(\'sbml-file-with-comp-elements.xml\');

// We extract one of the species from the model we just read in.
Species s1 = doc.getModel().getSpecies(0);

// We construct a new model.  This model does not use the \'comp\' package.
Model newModel = new Model(3,1);

// The following will fail with an error, because addSpecies() will
// first check that the parent of the given object has namespaces
// declared, and will discover that s1 does but newModel does not.

// newModel->addSpecies(s1);

// However, if we disable the \'comp\' package on s1, then the call
// to addSpecies will work.

s1->disablePackage(\'http://www.sbml.org/sbml/level3/version1/comp/version1\',
                   \'comp\');
newModel.addSpecies(s1);
@endcode
@endif

@param pkgURI the URI of the package.

@param pkgPrefix the XML prefix of the package.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_PKG_UNKNOWN LIBSBML_PKG_UNKNOWN@endlink
@li @link libsbml#LIBSBML_PKG_VERSION_MISMATCH LIBSBML_PKG_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_PKG_CONFLICTED_VERSION LIBSBML_PKG_CONFLICTED_VERSION@endlink

@see enablePackage()
";


%feature("docstring") SBase::enablePackageInternal "
@internal

@internal
";


%feature("docstring") SBase::isPackageURIEnabled "
Predicate returning @c True if an SBML Level&nbsp;3 package with the
given URI is enabled with this object.

@param pkgURI the URI of the package.

@return @c True if the given package is enabled within this object, @c False otherwise.

@see isPackageEnabled()
";


%feature("docstring") SBase::isPackageEnabled "
Predicate returning @c True if the given SBML Level&nbsp;3 package is
enabled with this object.

The search ignores the package version.

@param pkgName the name of the package.

@return @c True if the given package is enabled within this object,
@c False otherwise.

@see isPackageURIEnabled()
";


%feature("docstring") SBase::isPkgURIEnabled "
Predicate returning @c True if an SBML Level&nbsp;3 package with the
given URI is enabled with this object.

@param pkgURI the URI of the package.

@return @c True if the given package is enabled within this object, @c False otherwise.

@deprecated Replaced in libSBML 5.2.0 by
SBase.isPackageURIEnabled().

@see isPkgEnabled()
";


%feature("docstring") SBase::isPkgEnabled "
Predicate returning @c True if the given SBML Level&nbsp;3 package is
enabled with this object.

The search ignores the package version.

@param pkgName the name of the package.

@return @c True if the given package is enabled within this object,
@c False otherwise.

@deprecated Replaced in libSBML 5.2.0 by
SBase.isPackageEnabled().

@see isPkgURIEnabled()
";


%feature("docstring") SBase::writeExtensionElements "
@internal

@internal
";


%feature("docstring") SBase::write "
@internal

@internal
";


%feature("docstring") SBase::writeElements "
@internal

@internal
";


%feature("docstring") SBase::hasRequiredAttributes "
@internal

@internal
";


%feature("docstring") SBase::hasRequiredElements "
@internal

@internal
";


%feature("docstring") SBase::checkCompatibility "
@internal

@internal
";


%feature("docstring") SBase::setSBMLNamespaces "
@internal

@internal
";


%feature("docstring") SBase::setSBMLNamespacesAndOwn "
@internal

@internal
";


%feature("docstring") SBase::getSBMLNamespaces "
@internal

@internal
";


%feature("docstring") SBase::removeDuplicateAnnotations "
@internal

@internal
";


%feature("docstring") SBase::checkMathMLNamespace "
@internal

@internal
";


%feature("docstring") SBase::getDerivedUnitDefinition "
@internal

@internal
";


%feature("docstring") SBase::containsUndeclaredUnits "
@internal

@internal
";


%feature("docstring") SBase::removeFromParentAndDelete "
Removes this object from its parent.

If the parent was storing this object as a pointer, it is deleted.  If
not, it is simply cleared (as in ListOf objects).  This is a pure
virtual method, as every SBase element has different parents, and
therefore different methods of removing itself.  Will fail (and not
delete itself) if it has no parent object.  This function is designed to
be overridden, but for all objects whose parent is of the class ListOf,
the default implementation will work.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") SBase::matchesSBMLNamespaces "
Returns @c True if this object\'s set of XML namespaces are the same
as the given object\'s XML namespaces.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments.

@param sb an object to compare with respect to namespaces.

@return boolean, @c True if this object\'s collection of namespaces is
the same as @p sb\'s, @c False otherwise.
";


%feature("docstring") SBase::matchesRequiredSBMLNamespacesForAddition "
Returns @c True if this object\'s set of XML namespaces are a subset
of the given object\'s XML namespaces.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments.

@param sb an object to compare with respect to namespaces.

@return boolean, @c True if this object\'s collection of namespaces is
a subset of @p sb\'s, @c False otherwise.
";


%feature("docstring") SBase::setUserData "
Sets the user data of this element.

@par
The user data associated with an SBML object can be used by an application
developer to attach custom information to that object in the model.  In case
of a deep copy, this data will passed as-is.  The data attribute will never
be interpreted by libSBML.

@param userData specifies the new user data.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") SBase::*getUserData "
Returns the user data that has been previously set via setUserData().

@par
The user data associated with an SBML object can be used by an application
developer to attach custom information to that object in the model.  In case
of a deep copy, this data will passed as-is.  The data attribute will never
be interpreted by libSBML.

@return the user data of this node, or @c None if no user data has been set.

@if clike
@see ASTNode.setUserData()
@endif@~
";


%feature("docstring") SBase::isSetUserData "
Predicate returning true or false depending on whether
the user data of this element has been set.

@par
The user data associated with an SBML object can be used by an application
developer to attach custom information to that object in the model.  In case
of a deep copy, this data will passed as-is.  The data attribute will never
be interpreted by libSBML.

@return boolean, @c True if this object\'s user data has been set,
@c False otherwise.
";


%feature("docstring") SBase::unsetUserData "
Unsets the user data of this element.

@par
The user data associated with an SBML object can be used by an application
developer to attach custom information to that object in the model.  In case
of a deep copy, this data will passed as-is.  The data attribute will never
be interpreted by libSBML.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") SBase::getURI "
Returns the namespace URI to which this element belongs to.

For example, all elements that belong to SBML Level&nbsp;3 Version&nbsp;1 Core
must would have the URI <code>\'http://www.sbml.org/sbml/level3/version1/core\'</code>;
all elements that belong to Layout Extension Version 1 for SBML Level&nbsp;3
Version&nbsp;1 Core must would have the URI
<code>\'http://www.sbml.org/sbml/level3/version1/layout/version1\'</code>.

This function first returns the URI for this element by looking into the
SBMLNamespaces object of the document with the its package name.  If not
found, it will then look for the namespace associated with the element
itself.

@return the URI of this element, as a text string.

@see getSBMLDocument()
@see getPackageName()
";


%feature("docstring") SBase::getPrefix "
Returns the XML namespace prefix of this element.

This reports the XML namespace prefix chosen for this class of object in
the current SBML document.  This may be an empty string if the component
has no explicit prefix (for instance, if it is a core SBML object placed
in the default SBML namespace of the document).  If it is not empty, then
it corresponds to the XML namespace prefix used set the object, whatever
that may be in a given SBML document.

@return a text string representing the XML namespace prefix.
";


%feature("docstring") SBase::hasOptionalAttributes "
@internal

@internal
";


%feature("docstring") SBase::hasOptionalElements "
@internal

@internal
";


%feature("docstring") SBase::updateSBMLNamespace "
@internal

@internal
";


%feature("docstring") SBase::setElementText "
@internal
When overridden allows SBase elements to use the text included in between
the elements tags. The default implementation does nothing.

@param text the text string found between the element tags.

@internal
";


%feature("docstring") SBase::matchesCoreSBMLNamespace "
@internal

@internal
";


%feature("docstring") SBase::createObject "
@internal

@internal
";


%feature("docstring") SBase::readOtherXML "
@internal

@internal
";


%feature("docstring") SBase::getElementPosition "
@internal

@internal
";


%feature("docstring") SBase::getErrorLog "
@internal

@internal
";


%feature("docstring") SBase::logError "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") SBase::logUnknownAttribute "
@internal

@internal
";


%feature("docstring") SBase::logUnknownElement "
@internal

@internal
";


%feature("docstring") SBase::logEmptyString "
@internal

@internal
";


%feature("docstring") SBase::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") SBase::readAttributes "
@internal

@internal
";


%feature("docstring") SBase::writeAttributes "
@internal

@internal
";


%feature("docstring") SBase::writeXMLNS "
@internal

@internal
";


%feature("docstring") SBase::syncAnnotation "
@internal

@internal
";


%feature("docstring") SBase::reconstructRDFAnnotation "
@internal

@internal
";


%feature("docstring") SBase::checkOrderAndLogError "
@internal

@internal
";


%feature("docstring") SBase::checkListOfPopulated "
@internal

@internal
";


%feature("docstring") SBase::checkDefaultNamespace "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") SBase::checkAnnotation "
@internal

@internal
";


%feature("docstring") SBase::checkXHTML "
@internal

@internal
";


%feature("docstring") SBase::loadPlugins "
@internal

@internal
";


%feature("docstring") SBase::createExtensionObject "
@internal

@internal
";


%feature("docstring") SBase::setElementNamespace "
@internal

@internal
";


%feature("docstring") SBase::getElementNamespace "
@internal

@internal
";


%feature("docstring") SBase::readExtensionAttributes "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") SBase::writeExtensionAttributes "
@internal

@internal
";


%feature("docstring") SBase::storeUnknownExtAttribute "
@internal

@internal
";


%feature("docstring") SBase::storeUnknownExtElement "
@internal

@internal
";


%feature("docstring") SBase::getSBMLPrefix "
@internal

@internal
";


%feature("docstring") SBase::getRootElement "
@internal

@internal
";


%feature("docstring") SBase::getHasBeenDeleted "
@internal

@internal
";


%feature("docstring") SBase::setSBaseFields "
@internal

@internal
";


%feature("docstring") SBase::readAnnotation "
@internal

@internal
";


%feature("docstring") SBase::removeDuplicatedResources "
@internal

@internal
";


%feature("docstring") SBase::addTermToExistingBag "
@internal

@internal
";


%feature("docstring") SBase::readNotes "
@internal

@internal
";


%feature("docstring") ListOf "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html Parent class for libSBML\'s \'ListOfXYZ\' classes.

@htmlinclude not-sbml-warning.html

The ListOf class in libSBML is a utility class that serves as the parent
class for implementing the ListOf__ classes.  It provides methods for
working generically with the various SBML lists of objects in a program.
LibSBML uses this separate list class rather than ordinary
@if conly C@endif@if cpp C++; @endif@if java Java@endif@if python Python@endif@~ lists,
so that it can provide the methods and features associated with SBase.

Whether a given ListOf element may be empty or not depends on the 
element in question, and on what level and version of SBML it
is being used for.  For ListOf elements in SBML Level&nbsp;3
Version&nbsp;1 and prior, no core list and few package lists could
be empty.  As of SBML Level&nbsp;3 Version&nbsp;2, the rules
were relaxed, and lists were allowed to be empty.  In libSBML,
documents created for Level&nbsp;3 Version&nbsp;2 will be written
with empty ListOf\'s if that ListOf contains some other \'extra\'
information: an attribute such as metaid or sboTerm, a child
\'&lt;notes&gt;\' or \'&lt;annotation&gt;\', or information from a SBML 
Level&nbsp;3 package.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") ListOf::ListOf "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOf(SBMLNamespaces sbmlns)</pre>

Creates a new ListOf with a given SBMLNamespaces object.

@param sbmlns the set of SBML namespaces that this ListOf should
contain.

@note Bare ListOf @if conly structures @else objects@endif@~ are
impossible to add to SBML models.  The ListOf
@if conly structure type@else class@endif@~ is simply the base
of <em>other</em> @if conly structure types @else classes@endif@~ in
libSBML.  Calling programs are not intended to create bare ListOf
@if conly structures @else objects@endif@~ themselves.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOf(long level   = SBML_DEFAULT_LEVEL, long version = SBML_DEFAULT_VERSION)</pre>

Creates a new ListOf object.

@param level the SBML Level; if not assigned, defaults to the
value of SBMLDocument.getDefaultLevel().

@param version the Version within the SBML Level; if not assigned,
defaults to the value of SBMLDocument.getDefaultVersion().

@note Bare ListOf @if conly structures @else objects@endif@~ are
impossible to add to SBML models.  The ListOf
@if conly structure type@else class@endif@~ is simply the base
of <em>other</em> @if conly structure types @else classes@endif@~ in
libSBML.  Calling programs are not intended to create bare ListOf
@if conly structures @else objects@endif@~ themselves.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.


@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOf(ListOf orig)</pre>

Copy constructor; creates a copy of this ListOf.

@param orig the ListOf instance to copy.
";


%feature("docstring") ListOf::accept "
@internal

@internal
";


%feature("docstring") ListOf::clone "
Creates and returns a deep copy of this ListOf object.

@return the (deep) copy of this ListOf object.
";


%feature("docstring") ListOf::append "
Adds an item to the end of this ListOf\'s list of items.

This method makes a clone of the @p item handed to it.  This means that
when the ListOf object is destroyed, the original items will not be
destroyed.  For a method with an alternative ownership behavior, see the
ListOf.appendAndOwn() method.

@param item the item to be added to the list.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@see appendAndOwn()
@see appendFrom()
";


%feature("docstring") ListOf::appendAndOwn "
Adds an item to the end of this ListOf\'s list of items.

This method does not clone the @p disownedItem handed to it; instead, it assumes
ownership of it.  This means that when the ListOf is destroyed, the item
will be destroyed along with it.  For a method with an alternative
ownership behavior, see the ListOf.append() method.

@param disownedItem the item to be added to the list.
Will become a child of the parent list.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@see append()
@see appendFrom()
";


%feature("docstring") ListOf::appendFrom "
Adds a clone of a list of items to this ListOf\'s list.

Note that because this clones the objects handed to it, the original
items will not be destroyed when this ListOf object is destroyed.

@param list a list of items to be added.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@see append()
@see appendAndOwn()
";


%feature("docstring") ListOf::insert "
Inserts an item at a given position in this ListOf\'s list of items.

This variant of the method makes a clone of the @p item handed to it.
This means that when the ListOf is destroyed, the original @p item will
<em>not</em> be destroyed.

@param location the location in the list where to insert the item.
@param item the item to be inserted to the list.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@see insertAndOwn()
";


%feature("docstring") ListOf::insertAndOwn "
Inserts an item at a given position in this ListOf\'s list of items.

This variant of the method does not make a clone of the @p disownedItem handed to it.
This means that when the ListOf is destroyed, the original @p item
<em>will</em> be destroyed.

@param location the location where to insert the item.
@param disownedItem the item to be inserted to the list.
Will become a child of the parent list.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink

@see insert()
";


%feature("docstring") ListOf::get "
Get an item from the list.

@param n the index number of the item to get.

@return the <em>n</em>th item in this ListOf items, or a null pointer if
the index number @p n refers to a nonexistent position in this list.

@see size()
";


%feature("docstring") ListOf::getElementBySId "
Returns the first child element found that has the given identifier.

This method searches this ListOf\'s list of items for SBML objects based
on their \'id\' attribute value in the model-wide <code>SId</code>
identifier namespace.

@param id string representing the id of the object to find.

@return the first element found with the given @p id, or @c None if no
such object is found.
";


%feature("docstring") ListOf::getElementByMetaId "
Returns the first child element found with the given meta-identifier.

@param metaid string representing the \'metaid\' attribute of the object
to find.

@return the first element found with the given @p metaid, or @c None if
no such object is found.
";


%feature("docstring") ListOf::getAllElements "
Returns a List of all child SBase objects.

The values returned include all children of the objects in this ListOf
list, nested to an arbitrary depth.

@param filter a pointer to an ElementFilter, which causes the function 
to return only elements that match a particular set of constraints.  
If None (the default), the function will return all child objects.

@return a List of pointers to all child objects.
";


%feature("docstring") ListOf::clear "
Removes all items in this ListOf object.

If parameter @p doDelete is @c True (default), all items in this ListOf
object are deleted and cleared, and thus the caller doesn\'t have to
delete those items.  Otherwise, all items are cleared only from this
ListOf object; the caller is still responsible for deleting the actual
items.  (In the latter case, callers are advised to store pointers to
all items elsewhere before calling this function.)

@param doDelete if @c True (default), all items are deleted and cleared.
Otherwise, all items are just cleared and not deleted.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") ListOf::removeFromParentAndDelete "
Removes all items in this ListOf object and deletes its properties too.

This performs a call to clear() with an argument of @c True (thus removing
all the child objects in the list), followed by calls to various libSBML
<code>unset<em>Foo</em></code> methods to delete everything else: CVTerm
objects, model history objects, etc.

@if cpp Implementations of subclasses of ListOf may need to override
this method if different handling of child objects is needed.@endif@~

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
";


%feature("docstring") ListOf::remove "
Removes the <em>n</em>th item from this ListOf list of items and returns
it.

The caller owns the returned item and is responsible for deleting it.

@param n the index of the item to remove.

@see size()
";


%feature("docstring") ListOf::size "
Returns number of items in this ListOf list.

@return the number of items in this ListOf items.
";


%feature("docstring") ListOf::setSBMLDocument "
@internal

@internal
";


%feature("docstring") ListOf::getTypeCode "
Returns the libSBML type code for this object, namely,
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink (default).

@note The various ListOf classes mostly differ from each other in what they
contain.  Hence, one must call getItemTypeCode() to fully determine the
class of this SBML object.

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getItemTypeCode()
@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOf::getItemTypeCode "
Get the type code of the objects contained in this ListOf.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

Classes that inherit from the ListOf class should override this method
to return the SBML type code for the objects contained in this ListOf.
If they do not, this method will return
@link libsbml#SBML_UNKNOWN SBML_UNKNOWN@endlink

@return The ListOf base class contains no SBML objects, and therefore
this method returns @link libsbml#SBML_UNKNOWN SBML_UNKNOWN@endlink.

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOf::getElementName "
Returns the XML element name of this object, which for ListOf, is
always @c \'listOf\'.

@return the XML name of this element.
";


%feature("docstring") ListOf::writeElements "
@internal

@internal
";


%feature("docstring") ListOf::enablePackageInternal "
@internal

@internal
";


%feature("docstring") ListOf::updateSBMLNamespace "
@internal

@internal
";


%feature("docstring") ListOf::hasOptionalElements "
@internal

@internal
";


%feature("docstring") ListOf::isExplicitlyListed "
@internal

@internal
";


%feature("docstring") ListOf::setExplicitlyListed "
@internal

@internal
";


%feature("docstring") ListOf::sort "
@internal

@internal
";


%feature("docstring") ListOf::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") ListOf::readAttributes "
@internal

@internal
";


%feature("docstring") ListOf::writeAttributes "
@internal

@internal
";


%feature("docstring") ListOf::isValidTypeForList "
@internal

@internal
";


%feature("docstring") Model "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html An SBML model.

In an SBML model definition, a single object of class Model serves as
the overall container for the lists of the various model components.
All of the lists are optional, but if a given list container is present
within the model, the list must not be empty; that is, it must have
length one or more.  The following are the components and lists
permitted in different Levels and Versions of SBML in
version @htmlinclude libsbml-version.html
of libSBML:
<ul>
<li> In SBML Level 1, the components are: UnitDefinition, Compartment,
Species, Parameter, Rule, and Reaction.  Instances of the classes are
placed inside instances of classes ListOfUnitDefinitions,
ListOfCompartments, ListOfSpecies, ListOfParameters, ListOfRules, and
ListOfReactions.

<li> In SBML Level 2 Version 1, the components are: FunctionDefinition,
UnitDefinition, Compartment, Species, Parameter, Rule, Reaction and
Event.  Instances of the classes are placed inside instances of classes
ListOfFunctionDefinitions, ListOfUnitDefinitions, ListOfCompartments,
ListOfSpecies, ListOfParameters, ListOfRules, ListOfReactions, and
ListOfEvents.

<li> In SBML Level 2 Versions 2, 3 and 4, the components are:
FunctionDefinition, UnitDefinition, CompartmentType, SpeciesType,
Compartment, Species, Parameter, InitialAssignment, Rule, Constraint,
Reaction and Event.  Instances of the classes are placed inside
instances of classes ListOfFunctionDefinitions, ListOfUnitDefinitions,
ListOfCompartmentTypes, ListOfSpeciesTypes, ListOfCompartments,
ListOfSpecies, ListOfParameters, ListOfInitialAssignments, ListOfRules,
ListOfConstraints, ListOfReactions, and ListOfEvents.

<li> In SBML Level 3 Version 1, the components are: FunctionDefinition,
UnitDefinition, Compartment, Species, Parameter, InitialAssignment,
Rule, Constraint, Reaction and Event.  Instances of the classes are
placed inside instances of classes ListOfFunctionDefinitions,
ListOfUnitDefinitions, ListOfCompartments, ListOfSpecies,
ListOfParameters, ListOfInitialAssignments, ListOfRules,
ListOfConstraints, ListOfReactions, and ListOfEvents.  
</ul>

Although all the lists are optional, there are dependencies between SBML
components such that defining some components requires defining others.
An example is that defining a species requires defining a compartment,
and defining a reaction requires defining a species.  The dependencies
are explained in more detail in the SBML specifications.

In addition to the above lists and attributes, the Model class in both
SBML Level&nbsp;2 and Level&nbsp;3 has the usual two attributes of \'id\'
and \'name\', and both are optional.  As is the case for other SBML
components with \'id\' and \'name\' attributes, they must be used according
to the guidelines described in the SBML specifications.  (Within the
frameworks of SBML Level&nbsp;2 and Level&nbsp;3, a
Model object identifier has no assigned meaning, but extension packages
planned for SBML Level&nbsp;3 are likely to make use of this
identifier.)

Finally, SBML Level&nbsp;3 has introduced a number of additional Model
attributes.  They are discussed in a separate section below.


@section approaches Approaches to creating objects using the libSBML API

LibSBML provides two main mechanisms for creating objects: class
constructors
(e.g., @if java <a href=\'org/sbml/libsbml/Species.html\'>Species()</a> @else Species.Species() @endif), 
and <code>create<span class=\'placeholder-nospace\'><em>Object</em></span>()</code>
methods (such as Model.createSpecies()) provided by certain <span
class=\'placeholder-nospace\'><em>Object</em></span> classes such as Model.  These
multiple mechanisms are provided by libSBML for flexibility and to
support different use-cases, but they also have different implications
for the overall model structure.

In general, the recommended approach is to use the <code>create<span
class=\'placeholder-nospace\'><em>Object</em></span>()</code> methods.  These
methods both create an object @em and link it to the parent in one step.
Here is an example:@if clike
@code{.cpp}
// Create an SBMLDocument object in Level 3 Version 1 format:

SBMLDocument sbmlDoc = new SBMLDocument(3, 1);

// Create a Model object inside the SBMLDocument object and set
// its identifier.  The call returns a pointer to the Model object
// created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

Model model = sbmlDoc->createModel();
model->setId(\'BestModelEver\');

// Create a Species object inside the Model and set its identifier.
// Similar to the lines above, this call returns a pointer to the Species
// object created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

Species sp = model->createSpecies();
sp->setId(\'MySpecies\');
@endcode
@endif@if java
@code{.java}
// Create an SBMLDocument object in Level 3 Version 1 format:

SBMLDocument sbmlDoc = new SBMLDocument(3, 1);

// Create a Model object inside the SBMLDocument object and set
// its identifier.  The call returns a pointer to the Model object
// created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).  Note that
// the call to setId() returns a status code, and a real program
// should check this status code to make sure everything went okay.

Model model = sbmlDoc.createModel();
model.setId(&#34;BestModelEver&#34;);

// Create a Species object inside the Model and set its identifier.
// Similar to the lines above, this call returns a pointer to the Species
// object created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).  Note that, like
// with Model, the call to setId() returns a status code, and a real program
// should check this status code to make sure everything went okay.

Species sp = model.createSpecies();
sp.setId(&#34;BestSpeciesEver&#34;);
@endcode
@endif@if python
@code{.py}
# Create an SBMLDocument object in Level 3 Version 1 format.
# Make sure to check for possible failures.

try:
  sbmlDoc = SBMLDocument(3, 1)
except ValueError:
  print(\'Could not create SBMLDocument object\')
  sys.exit(1)

# Create a Model object inside the SBMLDocument object and set its
# identifier, checking the returned values.  The call to setId() returns a
# status code to indicate whether the assignment was successful.

model = sbmlDoc.createModel()
if model == None:
  # Do something to handle the error here.
  print(\'Unable to create Model object.\')
  sys.exit(1)

status = model.setId(\'BestModelEver\')
if status != LIBSBML_OPERATION_SUCCESS:
  # Do something to handle the error here.
  print(\'Unable to set identifier on the Model object\')
  sys.exit(1)

# Create a Species object inside the Model and set its identifier.
# Again, the setId() returns a status code to indicate whether the
# assignment was successful.

sp = model.createSpecies()
if sp == None:
  # Do something to handle the error here.
  print(\'Unable to create Species object.\')
  sys.exit(1)

status = sp.setId(\'BestSpeciesEver\')
if status != LIBSBML_OPERATION_SUCCESS:
  # Do something to handle the error here.
  print(\'Unable to set identifier on the Species object\')
  sys.exit(1)
@endcode
@endif@if csharp
@code
// Create an SBMLDocument object in Level 3 Version 1 format:

SBMLDocument sbmlDoc = new SBMLDocument(3, 1);

// Create a Model object inside the SBMLDocument object and set
// its identifier.  The call returns a pointer to the Model object
// created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

Model model = sbmlDoc.createModel();
model.setId(\'BestModelEver\');

// Create a Species object inside the Model and set its identifier.
// Similar to the lines above, this call returns a pointer to the Species
// object created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

Species sp = model.createSpecies();
sp.setId(\'MySpecies\');
@endcode
@endif@~

The <code>create<span
class=\'placeholder-nospace\'><em>Object</em></span>()</code> methods return a
pointer to the object created, but they also add the object to the
relevant list of object instances contained in the parent.  (These lists
become the <code>&lt;listOf<span
class=\'placeholder-nospace\'><em>Object</em></span>s&gt;</code> elements in the
finished XML rendition of SBML.)  In the example above,
Model.createSpecies() adds the created species directly to the
<code>&lt;listOfSpeciesgt;</code> list in the model.  Subsequently,
methods called on the species change the species in the model (which is
what is expected in most situations).

@section model-checking Consistency and adherence to SBML specifications

To make it easier for applications to do whatever they need,
libSBML version @htmlinclude libsbml-version.html
is relatively lax when it comes to enforcing correctness and
completeness of models @em during model construction and editing.
Essentially, libSBML @em will @em not in most cases check automatically
that a model\'s components have valid attribute values, or that the
overall model is consistent and free of errors---even obvious errors
such as duplication of identifiers.  This allows applications great
leeway in how they build their models, but it means that software
authors must take deliberate steps to ensure that the model will be, in
the end, valid SBML.  These steps include such things as keeping track
of the identifiers used in a model, manually performing updates in
certain situations where an entity is referenced in more than one place
(e.g., a species that is referenced by multiple SpeciesReference
objects), and so on.

That said, libSBML does provide powerful features for deliberately
performing validation of SBML when an application decides it is time to
do so.  The interfaces to these facilities are on the SBMLDocument
class, in the form of SBMLDocument.checkInternalConsistency() and
SBMLDocument.checkConsistency().  Please refer to the documentation for
SBMLDocument for more information about this.

While applications may play fast and loose and live like free spirits
during the construction and editing of SBML models, they should always
make sure to call SBMLDocument.checkInternalConsistency() and/or
SBMLDocument.checkConsistency() before writing out the final version of
an SBML model.


@section model-l3-attrib Model attributes introduced in SBML Level&nbsp;3

As mentioned above, the Model class has a number of optional attributes
in SBML Level&nbsp;3.  These are \'substanceUnits\',
\'timeUnits\', \'volumeUnits\', \'areaUnits\', \'lengthUnits\', \'extentUnits\',
and \'conversionFactor.  The following provide more information about
them.

@subsection model-l3-substanceunits The \'substanceUnits\' attribute

The \'substanceUnits\' attribute is used to specify the unit of
measurement associated with substance quantities of Species objects that
do not specify units explicitly.  If a given Species object definition
does not specify its unit of substance quantity via the \'substanceUnits\'
attribute on the Species object instance, then that species inherits the
value of the Model \'substanceUnits\' attribute.  If the Model does not
define a value for this attribute, then there is no unit to inherit, and
all species that do not specify individual \'substanceUnits\' attribute
values then have <em>no</em> declared units for their quantities.  The
SBML Level&nbsp;3 specifications provide more details.

Note that when the identifier of a species appears in a model\'s
mathematical expressions, the unit of measurement associated with that
identifier is <em>not solely determined</em> by setting \'substanceUnits\'
on Model or Species.  Please see the discussion about units given in
the documentation for the Species class.


@subsection model-l3-timeunits The \'timeUnits\' attribute

The \'timeUnits\' attribute on SBML Level&nbsp;3\'s Model object is used to
specify the unit in which time is measured in the model.  This attribute
on Model is the <em>only</em> way to specify a unit for time in a model.
It is a global attribute; time is measured in the model everywhere in
the same way.  This is particularly relevant to Reaction and RateRule
objects in a model: all Reaction and RateRule objects in SBML define
per-time values, and the unit of time is given by the \'timeUnits\'
attribute on the Model object instance.  If the Model \'timeUnits\'
attribute has no value, it means that the unit of time is not defined
for the model\'s reactions and rate rules.  Leaving it unspecified in an
SBML model does not result in an invalid model in SBML Level&nbsp;3;
however, as a matter of best practice, we strongly recommend that all
models specify units of measurement for time.


@subsection model-l3-voletc The \'volumeUnits\', \'areaUnits\', and \'lengthUnits\' attributes

The attributes \'volumeUnits\', \'areaUnits\' and \'lengthUnits\' together are
used to set the units of measurements for the sizes of Compartment
objects in an SBML Level&nbsp;3 model when those objects do not
otherwise specify units.  The three attributes correspond to the most
common cases of compartment dimensions: \'volumeUnits\' for compartments
having a \'spatialDimensions\' attribute value of @c \'3\', \'areaUnits\' for
compartments having a \'spatialDimensions\' attribute value of @c \'2\', and
\'lengthUnits\' for compartments having a \'spatialDimensions\' attribute
value of @c \'1\'.  The attributes are not applicable to compartments
whose \'spatialDimensions\' attribute values are @em not one of @c \'1\',
@c \'2\' or @c \'3\'.

If a given Compartment object instance does not provide a value for its
\'units\' attribute, then the unit of measurement of that compartment\'s
size is inherited from the value specified by the Model \'volumeUnits\',
\'areaUnits\' or \'lengthUnits\' attribute, as appropriate based on the
Compartment object\'s \'spatialDimensions\' attribute value.  If the Model
object does not define the relevant attribute, then there are no units
to inherit, and all Compartment objects that do not set a value for
their \'units\' attribute then have <em>no</em> units associated with
their compartment sizes.

The use of three separate attributes is a carry-over from SBML
Level&nbsp;2.  Note that it is entirely possible for a model to define a
value for two or more of the attributes \'volumeUnits\', \'areaUnits\' and
\'lengthUnits\' simultaneously, because SBML models may contain
compartments with different numbers of dimensions.


@subsection model-l3-extentunits The \'extentUnits\' attribute

Reactions are processes that occur over time.  These processes involve
events of some sort, where a single ``reaction event\'\' is one in which
some set of entities (known as reactants, products and modifiers in
SBML) interact, once.  The <em>extent</em> of a reaction is a measure of
how many times the reaction has occurred, while the time derivative of
the extent gives the instantaneous rate at which the reaction is
occurring.  Thus, what is colloquially referred to as the \'rate of the
reaction\' is in fact equal to the rate of change of reaction extent.

In SBML Level&nbsp;3, the combination of \'extentUnits\' and \'timeUnits\'
defines the units of kinetic laws in SBML and establishes how the
numerical value of each KineticLaw object\'s mathematical formula is
meant to be interpreted in a model.  The units of the kinetic laws are
taken to be \'extentUnits\' divided by \'timeUnits\'.

Note that this embodies an important principle in SBML Level&nbsp;3
models: <em>all reactions in an SBML model must have the same units</em>
for the rate of change of extent.  In other words, the units of all
reaction rates in the model <em>must be the same</em>.  There is only
one global value for \'extentUnits\' and one global value for \'timeUnits\'.


@subsection model-l3-convfactor The \'conversionFactor\' attribute

The attribute \'conversionFactor\' in SBML Level&nbsp;3\'s Model object
defines a global value inherited by all Species object instances that do
not define separate values for their \'conversionFactor\' attributes.  The
value of this attribute must refer to a Parameter object instance
defined in the model.  The Parameter object in question must be a
constant; ie it must have its \'constant\' attribute value set to
@c \'True\'.

If a given Species object definition does not specify a conversion
factor via the \'conversionFactor\' attribute on Species, then the species
inherits the conversion factor specified by the Model \'conversionFactor\'
attribute.  If the Model does not define a value for this attribute,
then there is no conversion factor to inherit.  More information about
conversion factors is provided in the SBML Level&nbsp;3
specifications.
";


%feature("docstring") Model::Model "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Model(SBMLNamespaces sbmlns)</pre>

Creates a new Model using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments.

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Model(long level, long version)</pre>

Creates a new Model using the given SBML @p level and @p version
values.

@param level a long integer, the SBML Level to assign to this Model.

@param version a long integer, the SBML Version to assign to this
Model.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Model(Model orig)</pre>

Copy constructor; creates a (deep) copy of the given Model object.

@param orig the object to copy.
";


%feature("docstring") Model::accept "
@internal

@internal
";


%feature("docstring") Model::clone "
Creates and returns a deep copy of this Model object.

@return the (deep) copy of this Model object.
";


%feature("docstring") Model::getElementBySId "
Returns the first child element found that has the given @p id.

This operation searches the model-wide <code>SId</code> identifier
type namespace

@param id string representing the id of the object to find.

@return pointer to the first element found with the given @p id, or
@c None if no such object is found.
";


%feature("docstring") Model::getElementByMetaId "
Returns the first child element it can find with the given @p metaid.

@param metaid string representing the meta-identifier of the object to
find.

@return pointer to the first element found with the given @p metaid, or
None if no such object is found.
";


%feature("docstring") Model::getAllElements "
Returns a List of all child objects.

This returns all child objects nested to an arbitrary depth.  If an
optional element filter is provided, only those objects for which the
@p filter\'s <code>filter()</code> method returns true will be added to 
the list.

@param filter a pointer to an ElementFilter, which causes the function 
to return only elements that match a particular set of constraints.  
If None (the default), the function will return all child objects.

@return a List of pointers to all children objects.
";


%feature("docstring") Model::getId "
Returns the value of the \'id\' attribute of this Model.

@note Because of the inconsistent behavior of this function with 
respect to assignments and rules, it is now recommended to
use the getIdAttribute() function instead.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this Model.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") Model::getName "
Returns the value of the \'name\' attribute of this Model object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") Model::getSubstanceUnits "
Returns the value of the \'substanceUnits\' attribute of this Model.

@return the substanceUnits of this Model.

@note The \'substanceUnits\' attribute is available in
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::getTimeUnits "
Returns the value of the \'timeUnits\' attribute of this Model.

@return the timeUnits of this Model.

@note The \'timeUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::getVolumeUnits "
Returns the value of the \'volumeUnits\' attribute of this Model.

@return the volumeUnits of this Model.

@note The \'volumeUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::getAreaUnits "
Returns the value of the \'areaUnits\' attribute of this Model.

@return the areaUnits of this Model.

@note The \'areaUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::getLengthUnits "
Returns the value of the \'lengthUnits\' attribute of this Model.

@return the lengthUnits of this Model.

@note The \'lengthUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::getExtentUnits "
Returns the value of the \'extentUnits\' attribute of this Model.

@return the extentUnits of this Model.

@note The \'extentUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::getConversionFactor "
Returns the value of the \'conversionFactor\' attribute of this Model.

@return the conversionFactor of this Model.

@note The \'conversionFactor\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::isSetId "
Predicate returning @c True if this
Model\'s \'id\' attribute is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") Model::isSetName "
Predicate returning @c True if this
Model\'s \'name\' attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") Model::isSetSubstanceUnits "
Predicate returning @c True if this
Model\'s \'substanceUnits\' attribute is set.

@return @c True if the \'substanceUnits\' attribute of this Model is
set, @c False otherwise.

@note The \'substanceUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::isSetTimeUnits "
Predicate returning @c True if this
Model\'s \'timeUnits\' attribute is set.

@return @c True if the \'timeUnits\' attribute of this Model is
set, @c False otherwise.

@note The \'substanceUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::isSetVolumeUnits "
Predicate returning @c True if this
Model\'s \'volumeUnits\' attribute is set.

@return @c True if the \'volumeUnits\' attribute of this Model is
set, @c False otherwise.

@note The \'volumeUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::isSetAreaUnits "
Predicate returning @c True if this
Model\'s \'areaUnits\' attribute is set.

@return @c True if the \'areaUnits\' attribute of this Model is
set, @c False otherwise.

@note The \'areaUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::isSetLengthUnits "
Predicate returning @c True if this
Model\'s \'lengthUnits\' attribute is set.

@return @c True if the \'lengthUnits\' attribute of this Model is
set, @c False otherwise.

@note The \'lengthUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::isSetExtentUnits "
Predicate returning @c True if this
Model\'s \'extentUnits\' attribute is set.

@return @c True if the \'extentUnits\' attribute of this Model is
set, @c False otherwise.

@note The \'extentUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::isSetConversionFactor "
Predicate returning @c True if this
Model\'s \'conversionFactor\' attribute is set.

@return @c True if the \'conversionFactor\' attribute of this Model is
set, @c False otherwise.

@note The \'conversionFactor\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::setId "
Sets the value of the \'id\' attribute of this Model.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") Model::setName "
Sets the value of the \'name\' attribute of this Model.

@par

The string in @p name is copied.

@param name the new name for the SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Model::setSubstanceUnits "
Sets the value of the \'substanceUnits\' attribute of this Model.

The string in @p units is copied.

@param units the new substanceUnits for the Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'substanceUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::setTimeUnits "
Sets the value of the \'timeUnits\' attribute of this Model.

The string in @p units is copied.

@param units the new timeUnits for the Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'timeUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::setVolumeUnits "
Sets the value of the \'volumeUnits\' attribute of this Model.

The string in @p units is copied.

@param units the new volumeUnits for the Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'volumeUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::setAreaUnits "
Sets the value of the \'areaUnits\' attribute of this Model.

The string in @p units is copied.

@param units the new areaUnits for the Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'areaUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::setLengthUnits "
Sets the value of the \'lengthUnits\' attribute of this Model.

The string in @p units is copied.

@param units the new lengthUnits for the Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'lengthUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::setExtentUnits "
Sets the value of the \'extentUnits\' attribute of this Model.

The string in @p units is copied.

@param units the new extentUnits for the Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'extentUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::setConversionFactor "
Sets the value of the \'conversionFactor\' attribute of this Model.

The string in @p units is copied.

@param units the new conversionFactor for the Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'conversionFactor\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::unsetId "
Unsets the value of the \'id\' attribute of this Model.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") Model::unsetName "
Unsets the value of the \'name\' attribute of this Model.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") Model::unsetSubstanceUnits "
Unsets the value of the \'substanceUnits\' attribute of this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'substanceUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::unsetTimeUnits "
Unsets the value of the \'timeUnits\' attribute of this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'timeUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::unsetVolumeUnits "
Unsets the value of the \'volumeUnits\' attribute of this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'volumeUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::unsetAreaUnits "
Unsets the value of the \'areaUnits\' attribute of this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'areaUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::unsetLengthUnits "
Unsets the value of the \'lengthUnits\' attribute of this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'lengthUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::unsetExtentUnits "
Unsets the value of the \'extentUnits\' attribute of this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'extentUnits\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::unsetConversionFactor "
Unsets the value of the \'conversionFactor\' attribute of this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'conversionFactor\' attribute is available in 
SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
";


%feature("docstring") Model::addFunctionDefinition "
Adds a copy of the given FunctionDefinition object to this Model.

@param fd the FunctionDefinition to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createFunctionDefinition()
";


%feature("docstring") Model::addUnitDefinition "
Adds a copy of the given UnitDefinition object to this Model.

@param ud the UnitDefinition object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createUnitDefinition()
";


%feature("docstring") Model::addCompartmentType "
Adds a copy of the given CompartmentType object to this Model.

@param ct the CompartmentType object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@note The CompartmentType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.

@see createCompartmentType()
";


%feature("docstring") Model::addSpeciesType "
Adds a copy of the given SpeciesType object to this Model.

@param st the SpeciesType object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@note The SpeciesType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.

@see createSpeciesType()
";


%feature("docstring") Model::addCompartment "
Adds a copy of the given Compartment object to this Model.

@param c the Compartment object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createCompartment()
";


%feature("docstring") Model::addSpecies "
Adds a copy of the given Species object to this Model.

@param s the Species object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createSpecies()
";


%feature("docstring") Model::addParameter "
Adds a copy of the given Parameter object to this Model.

@param p the Parameter object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createParameter()
";


%feature("docstring") Model::addInitialAssignment "
Adds a copy of the given InitialAssignment object to this Model.

@param ia the InitialAssignment object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createInitialAssignment()
";


%feature("docstring") Model::addRule "
Adds a copy of the given Rule object to this Model.

@param r the Rule object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createAlgebraicRule()
@see createAssignmentRule()
@see createRateRule()
";


%feature("docstring") Model::addConstraint "
Adds a copy of the given Constraint object to this Model.

@param c the Constraint object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createConstraint()
";


%feature("docstring") Model::addReaction "
Adds a copy of the given Reaction object to this Model.

@param r the Reaction object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createReaction()
";


%feature("docstring") Model::addEvent "
Adds a copy of the given Event object to this Model.

@param e the Event object to add.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createEvent()
";


%feature("docstring") Model::createFunctionDefinition "
Creates a new FunctionDefinition inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the FunctionDefinition object created.

@see addFunctionDefinition()
";


%feature("docstring") Model::createUnitDefinition "
Creates a new UnitDefinition inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the UnitDefinition object created.

@see addUnitDefinition()
";


%feature("docstring") Model::createUnit "
Creates a new Unit object within the last UnitDefinition object
created in this model and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

The mechanism by which the UnitDefinition was created is not
significant.  If a UnitDefinition object does not exist in this model,
a new Unit is @em not created and @c None is returned instead.

@return the Unit object created.

@see addUnitDefinition()
";


%feature("docstring") Model::createCompartmentType "
Creates a new CompartmentType inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the CompartmentType object created.

@note The CompartmentType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.

@see addCompartmentType()
";


%feature("docstring") Model::createSpeciesType "
Creates a new SpeciesType inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the SpeciesType object created.

@note The SpeciesType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.

@see addSpeciesType()
";


%feature("docstring") Model::createCompartment "
Creates a new Compartment inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the Compartment object created.

@see addCompartment()
";


%feature("docstring") Model::createSpecies "
Creates a new Species inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the Species object created.

@see addSpecies()
";


%feature("docstring") Model::createParameter "
Creates a new Parameter inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the Parameter object created.

@see addParameter()
";


%feature("docstring") Model::createInitialAssignment "
Creates a new InitialAssignment inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the InitialAssignment object created.

@see addInitialAssignment()
";


%feature("docstring") Model::createAlgebraicRule "
Creates a new AlgebraicRule inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the AlgebraicRule object created.

@see addRule()
";


%feature("docstring") Model::createAssignmentRule "
Creates a new AssignmentRule inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the AssignmentRule object created.

@see addRule()
";


%feature("docstring") Model::createRateRule "
Creates a new RateRule inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the RateRule object created.

@see addRule()
";


%feature("docstring") Model::createConstraint "
Creates a new Constraint inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the Constraint object created.

@see addConstraint()
";


%feature("docstring") Model::createReaction "
Creates a new Reaction inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the Reaction object created.

@see addReaction()
";


%feature("docstring") Model::createReactant "
Creates a new SpeciesReference object for a reactant inside the last
Reaction object in this Model, and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The mechanism by which the last Reaction object was created and added
to this Model is not significant.  It could have been created in a
variety of ways, for example using createReaction().  If a Reaction
does not exist for this model, a new SpeciesReference is @em not
created and @c None is returned instead.

@return the SpeciesReference object created.  If a Reaction does not
exist for this model, a new SpeciesReference is @em not created and
@c None is returned.
";


%feature("docstring") Model::createProduct "
Creates a new SpeciesReference object for a product inside the last
Reaction object in this Model, and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The mechanism by which the last Reaction object was created and added
to this Model is not significant.  It could have been created in a
variety of ways, for example using createReaction().  If a Reaction
does not exist for this model, a new SpeciesReference is @em not
created and @c None is returned instead.

@return the SpeciesReference object created. If a Reaction does not
exist for this model, a new SpeciesReference is @em not created and
@c None is returned.
";


%feature("docstring") Model::createModifier "
Creates a new ModifierSpeciesReference object for a modifier species
inside the last Reaction object in this Model, and returns a pointer
to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The mechanism by which the last Reaction object was created and added
to this Model is not significant.  It could have been created in a
variety of ways, for example using createReaction().  If a Reaction
does not exist for this model, a new SpeciesReference is @em not
created and @c None is returned instead.

@return the SpeciesReference object created.  If a Reaction does not
exist for this model, a new SpeciesReference is @em not created and
@c None is returned.
";


%feature("docstring") Model::createKineticLaw "
Creates a new KineticLaw inside the last Reaction object created in
this Model, and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The mechanism by which the last Reaction object was created and added
to this Model is not significant.  It could have been created in a
variety of ways, for example using createReaction().  If a Reaction
does not exist for this model, a new SpeciesReference is @em not
created and @c None is returned instead.

@return the KineticLaw object created.  If a Reaction does not exist for
this model, or a Reaction does exist but already has a KineticLaw, a new
KineticLaw is @em not created and @c None is returned.
";


%feature("docstring") Model::createKineticLawParameter "
Creates a new local Parameter inside the KineticLaw object of the last
Reaction created inside this Model, and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The last KineticLaw object in this Model could have been created in a
variety of ways.  For example, it could have been added using
createKineticLaw(), or it could be the result of using
Reaction.createKineticLaw() on the Reaction object created by a
createReaction().  If a Reaction does not exist for this model, or the
last Reaction does not contain a KineticLaw object, a new Parameter is
@em not created and @c None is returned instead.

@return the Parameter object created.  If a Reaction does not exist for
this model, or a KineticLaw for the Reaction does not exist, a new
Parameter is @em not created and @c None is returned.
";


%feature("docstring") Model::createKineticLawLocalParameter "
Creates a new LocalParameter inside the KineticLaw object of the last
Reaction created inside this Model, and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The last KineticLaw object in this Model could have been created in a
variety of ways.  For example, it could have been added using
createKineticLaw(), or it could be the result of using
Reaction.createKineticLaw() on the Reaction object created by a
createReaction().  If a Reaction does not exist for this model, or the
last Reaction does not contain a KineticLaw object, a new Parameter is
@em not created and @c None is returned instead.

@return the Parameter object created.  If a Reaction does not exist for
this model, or a KineticLaw for the Reaction does not exist, a new
Parameter is @em not created and @c None is returned.
";


%feature("docstring") Model::createEvent "
Creates a new Event inside this Model and returns it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@return the Event object created.
";


%feature("docstring") Model::createEventAssignment "
Creates a new EventAssignment inside the last Event object created in
this Model, and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The mechanism by which the last Event object in this model was created
is not significant.  It could have been created in a variety of ways,
for example by using createEvent().  If no Event object exists in this
Model object, a new EventAssignment is @em not created and @c None is
returned instead.

@return the EventAssignment object created.
";


%feature("docstring") Model::createTrigger "
Creates a new Trigger inside the last Event object created in
this Model, and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The mechanism by which the last Event object in this model was created
is not significant.  It could have been created in a variety of ways,
for example by using createEvent().  If no Event object exists in this
Model object, a new EventAssignment is @em not created and @c None is
returned instead.

@return the Trigger object created.
";


%feature("docstring") Model::createDelay "
Creates a new Delay inside the last Event object created in
this Model, and returns a pointer to it.

The SBML Level and Version of the enclosing Model object, as well as
any SBML package namespaces, are used to initialize this
object\'s corresponding attributes.

@par
The mechanism by which the last Event object in this model was created
is not significant.  It could have been created in a variety of ways,
for example by using createEvent().  If no Event object exists in this
Model object, a new EventAssignment is @em not created and @c None is
returned instead.

@return the Delay object created.
";


%feature("docstring") Model::setAnnotation "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setAnnotation(XMLNode annotation)</pre>

Sets the value of the \'annotation\' subelement of this SBML object to a
copy of @p annotation.

Any existing content of the \'annotation\' subelement is discarded.
Unless you have taken steps to first copy and reconstitute any
existing annotations into the @p annotation that is about to be
assigned, it is likely that performing such wholesale replacement is
unfriendly towards other software applications whose annotations are
discarded.  An alternative may be to use appendAnnotation().

@param annotation an XML structure that is to be used as the content
of the \'annotation\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@see appendAnnotation()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setAnnotation(string annotation)</pre>

Sets the value of the \'annotation\' subelement of this SBML object to a
copy of @p annotation.

Any existing content of the \'annotation\' subelement is discarded.
Unless you have taken steps to first copy and reconstitute any
existing annotations into the @p annotation that is about to be
assigned, it is likely that performing such wholesale replacement is
unfriendly towards other software applications whose annotations are
discarded.  An alternative may be to use appendAnnotation().

@param annotation an XML string that is to be used as the content
of the \'annotation\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see appendAnnotation()
";


%feature("docstring") Model::appendAnnotation "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>appendAnnotation(XMLNode annotation)</pre>

Appends annotation content to any existing content in the \'annotation\'
subelement of this object.

The content in @p annotation is copied.  Unlike setAnnotation(), this
method allows other annotations to be preserved when an application
adds its own data.

@param annotation an XML structure that is to be copied and appended
to the content of the \'annotation\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see setAnnotation()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>appendAnnotation(string annotation)</pre>

Appends annotation content to any existing content in the \'annotation\'
subelement of this object.

The content in @p annotation is copied.  Unlike setAnnotation(), this 
method allows other annotations to be preserved when an application
adds its own data.

@param annotation an XML string that is to be copied and appended
to the content of the \'annotation\' subelement of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see setAnnotation()
";


%feature("docstring") Model::getListOfFunctionDefinitions "
Get the ListOfFunctionDefinitions object in this Model.

@return the list of FunctionDefinitions for this Model.
";


%feature("docstring") Model::getListOfUnitDefinitions "
Get the ListOfUnitDefinitions object in this Model.

@return the list of UnitDefinitions for this Model.
";


%feature("docstring") Model::getListOfCompartmentTypes "
Get the ListOfCompartmentTypes object in this Model.

@return the list of CompartmentTypes for this Model.

@note The CompartmentType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.
";


%feature("docstring") Model::getListOfSpeciesTypes "
Get the ListOfSpeciesTypes object in this Model.

@return the list of SpeciesTypes for this Model.

@note The SpeciesType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.
";


%feature("docstring") Model::getListOfCompartments "
Get the ListOfCompartments object in this Model.

@return the list of Compartments for this Model.
";


%feature("docstring") Model::getListOfSpecies "
Get the ListOfSpecies object in this Model.

@return the list of Species for this Model.
";


%feature("docstring") Model::getListOfParameters "
Get the ListOfParameters object in this Model.

@return the list of Parameters for this Model.
";


%feature("docstring") Model::getListOfInitialAssignments "
Get the ListOfInitialAssignments object in this Model.

@return the list of InitialAssignments for this Model.
";


%feature("docstring") Model::getListOfRules "
Get the ListOfRules object in this Model.

@return the list of Rules for this Model.
";


%feature("docstring") Model::getListOfConstraints "
Get the ListOfConstraints object in this Model.

@return the list of Constraints for this Model.
";


%feature("docstring") Model::getListOfReactions "
Get the ListOfReactions object in this Model.

@return the list of Reactions for this Model.
";


%feature("docstring") Model::getListOfEvents "
Get the ListOfEvents object in this Model.

@return the list of Events for this Model.
";


%feature("docstring") Model::getFunctionDefinition "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getFunctionDefinition(long n)</pre>

Get the nth FunctionDefinitions object in this Model.

@param n the index of the object to return.

@return the nth FunctionDefinition of this Model.
If the index @p n is invalid, @c None is returned.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getFunctionDefinition(string sid)</pre>

Get a FunctionDefinition object based on its identifier.

@param sid the identifier to search for.

@return the FunctionDefinition in this Model with the identifier
@p sid or @c None if no such FunctionDefinition exists.
";


%feature("docstring") Model::getUnitDefinition "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getUnitDefinition(long n)</pre>

Get the nth UnitDefinition object in this Model.

@param n the index of the object to return.

@return the nth UnitDefinition of this Model.
If the index @p n is invalid, @c None is returned.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getUnitDefinition(string sid)</pre>

Get a UnitDefinition based on its identifier.

@param sid the identifier to search for.

@return the UnitDefinition in this Model with the identifier @p sid or
@c None if no such UnitDefinition exists.
";


%feature("docstring") Model::getCompartmentType "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getCompartmentType(long n)</pre>

Get the nth CompartmentType object in this Model.

@param n the index of the object to return.

@return the nth CompartmentType of this Model.
If the index @p n is invalid, @c None is returned.

@note The CompartmentType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getCompartmentType(string sid)</pre>

Get a CompartmentType object based on its identifier.

@param sid the identifier to search for.

@return the CompartmentType in this Model with the identifier @p sid
or @c None if no such CompartmentType exists.

@note The CompartmentType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.
";


%feature("docstring") Model::getSpeciesType "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getSpeciesType(long n)</pre>

Get the nth SpeciesType object in this Model.

@param n the index of the object to return.

@return the nth SpeciesType of this Model.
If the index @p n is invalid, @c None is returned.

@note The SpeciesType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getSpeciesType(string sid)</pre>

Get a SpeciesType object based on its identifier.

@param sid the identifier to search for.

@return the SpeciesType in this Model with the identifier @p sid or
@c None if no such SpeciesType exists.

@note The SpeciesType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.
";


%feature("docstring") Model::getCompartment "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getCompartment(long n)</pre>

Get the nth Compartment object in this Model.

@param n the index of the object to return.

@return the nth Compartment of this Model.
If the index @p n is invalid, @c None is returned.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getCompartment(string sid)</pre>

Get a Compartment object based on its identifier.

@param sid the identifier to search for.

@return the Compartment in this Model with the identifier @p sid or
@c None if no such Compartment exists.
";


%feature("docstring") Model::getSpecies "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getSpecies(long n)</pre>

Get the nth Species object in this Model.

@param n the index of the object to return.

@return the nth Species of this Model.
If the index @p n is invalid, @c None is returned.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getSpecies(string sid)</pre>

Get a Species object based on its identifier.

@param sid the identifier to search for.

@return the Species in this Model with the identifier @p sid or @c None
if no such Species exists.
";


%feature("docstring") Model::getParameter "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getParameter(long n)</pre>

Get the nth Parameter object in this Model.

@param n the index of the object to return.

@return the nth Parameter of this Model.
If the index @p n is invalid, @c None is returned.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getParameter(string sid)</pre>

Get a Parameter object based on its identifier.

@param sid the identifier to search for.

@return the Parameter in this Model with the identifier @p sid or @c None
if no such Parameter exists.
";


%feature("docstring") Model::getInitialAssignment "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getInitialAssignment(string symbol)</pre>

Get an InitialAssignment object based on the symbol to which it
assigns a value.

@param symbol the symbol to search for.

@return the InitialAssignment in this Model with the given \'symbol\'
attribute value or @c None if no such InitialAssignment exists.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getInitialAssignment(long n)</pre>

Get the nth InitialAssignment object in this Model.

@param n the index of the object to return.

@return the nth InitialAssignment of this Model.
If the index @p n is invalid, @c None is returned.
";


%feature("docstring") Model::getInitialAssignmentBySymbol "
Get an InitialAssignment object based on the symbol to which it
assigns a value.

@param symbol the symbol to search for.

@return the InitialAssignment in this Model with the given \'symbol\'
attribute value or @c None if no such InitialAssignment exists.
";


%feature("docstring") Model::getRule "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getRule(string variable)</pre>

Get a Rule object based on the variable to which it assigns a value.

@param variable the variable to search for.

@return the Rule in this Model with the given \'variable\' attribute
value or @c None if no such Rule exists.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getRule(long n)</pre>

Get the nth Rule object in this Model.

@param n the index of the object to return.

@return the nth Rule of this Model.
If the index @p n is invalid, @c None is returned.
";


%feature("docstring") Model::getRuleByVariable "
Get a Rule object based on the variable to which it assigns a value.

@param variable the variable to search for.

@return the Rule in this Model with the given \'variable\' attribute
value or @c None if no such Rule exists.
";


%feature("docstring") Model::getAssignmentRule "
Get a Rule object based on the variable to which it assigns a value.

@param variable the variable to search for.

@return the Rule in this Model with the given \'variable\' attribute
value or @c None if no such Rule exists.
";


%feature("docstring") Model::getRateRule "
Get a Rule object based on the variable to which it assigns a value.

@param variable the symbol to search for.

@return the Rule in this Model with the given \'variable\' attribute
value or @c None if no such Rule exists.
";


%feature("docstring") Model::getAssignmentRuleByVariable "
Get a Rule object based on the variable to which it assigns a value.

@param variable the variable to search for.

@return the Rule in this Model with the given \'variable\' attribute
value or @c None if no such Rule exists.
";


%feature("docstring") Model::getRateRuleByVariable "
Get a Rule object based on the variable to which it assigns a value.

@param variable the variable to search for.

@return the Rule in this Model with the given \'variable\' attribute
value or @c None if no such Rule exists.
";


%feature("docstring") Model::getConstraint "
Get the nth Constraint object in this Model.

@param n the index of the object to return.

@return the nth Constraint of this Model.
If the index @p n is invalid, @c None is returned.
";


%feature("docstring") Model::getReaction "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getReaction(long n)</pre>

Get the nth Reaction object in this Model.

@param n the index of the object to return.

@return the nth Reaction of this Model.
If the index @p n is invalid, @c None is returned.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getReaction(string sid)</pre>

Get a Reaction object based on its identifier.

@param sid the identifier to search for.

@return the Reaction in this Model with the identifier @p sid or @c None
if no such Reaction exists.
";


%feature("docstring") Model::getSpeciesReference "
Get a SpeciesReference object based on its identifier.

@param sid the identifier to search for.

@return the SpeciesReference in this Model with the identifier @p sid or @c None
if no such SpeciesReference exists.
";


%feature("docstring") Model::getModifierSpeciesReference "
Get a ModifierSpeciesReference object based on its identifier.

@param sid the identifier to search for.

@return the ModifierSpeciesReference in this Model with the 
identifier @p sid or @c None
if no such ModifierSpeciesReference exists.
";


%feature("docstring") Model::getEvent "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getEvent(long n)</pre>

Get the nth Event object in this Model.

@param n the index of the object to return.

@return the nth Event of this Model.
If the index @p n is invalid, @c None is returned.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getEvent(string sid)</pre>

Get an Event object based on its identifier.

@param sid the identifier to search for.

@return the Event in this Model with the identifier @p sid or @c None if
no such Event exists.
";


%feature("docstring") Model::getNumFunctionDefinitions "
Get the number of FunctionDefinition objects in this Model.

@return the number of FunctionDefinitions in this Model.
";


%feature("docstring") Model::getNumUnitDefinitions "
Get the number of UnitDefinition objects in this Model.

@return the number of UnitDefinitions in this Model.
";


%feature("docstring") Model::getNumCompartmentTypes "
Get the number of CompartmentType objects in this Model.

@return the number of CompartmentTypes in this Model.

@note The CompartmentType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.
";


%feature("docstring") Model::getNumSpeciesTypes "
Get the number of SpeciesType objects in this Model.

@return the number of SpeciesTypes in this Model.

@note The SpeciesType object class is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
Level&nbsp;1 nor Level&nbsp;3.
";


%feature("docstring") Model::getNumCompartments "
Get the number of Compartment objects in this Model.

@return the number of Compartments in this Model.
";


%feature("docstring") Model::getNumSpecies "
Get the number of Species objects in this Model.

@return the number of Species in this Model.
";


%feature("docstring") Model::getNumSpeciesWithBoundaryCondition "
Get the number of Species in this Model having their
\'boundaryCondition\' attribute value set to @c True.

@return the number of Species in this Model with boundaryCondition set
to true.
";


%feature("docstring") Model::getNumParameters "
Get the number of Parameter objects in this Model.

@return the number of Parameters in this Model.  Parameters defined in
KineticLaws are not included.
";


%feature("docstring") Model::getNumInitialAssignments "
Get the number of InitialAssignment objects in this Model.

@return the number of InitialAssignments in this Model.
";


%feature("docstring") Model::getNumRules "
Get the number of Rule objects in this Model.

@return the number of Rules in this Model.
";


%feature("docstring") Model::getNumConstraints "
Get the number of Constraint objects in this Model.

@return the number of Constraints in this Model.
";


%feature("docstring") Model::getNumReactions "
Get the number of Reaction objects in this Model.

@return the number of Reactions in this Model.
";


%feature("docstring") Model::getNumEvents "
Get the number of Event objects in this Model.

@return the number of Events in this Model.
";


%feature("docstring") Model::removeFromParentAndDelete "
Remove this Model from its parent SBMLDocument object.

This works by finding this Model\'s parent SBMLDocument and then calling
<code>setModel(None)</code> on it, indirectly deleting itself.
Overridden from the SBase function since the parent is not a ListOf.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") Model::renameAllIds "
@internal

@internal
";


%feature("docstring") Model::renameIDs "
@internal

@internal
";


%feature("docstring") Model::renameSIdRefs "
Replaces all uses of a given @c SIdRef type attribute value with another
value.

@par
In SBML, object identifiers are of a data type called <code>SId</code>.
In SBML Level&nbsp;3, an explicit data type called <code>SIdRef</code> was
introduced for attribute values that refer to <code>SId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to an identifier\', but the effective
data type was the same as <code>SIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>SIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all attributes and (if appropriate)
mathematical formulas in MathML content, comparing the referenced
identifiers to the value of @p oldid.  If any matches are found, the
matching values are replaced with @p newid.  The method does @em not
descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") Model::renameUnitSIdRefs "
Replaces all uses of a given @c UnitSIdRef type attribute value with
another value.

@par
In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
introduced for attribute values that refer to <code>UnitSId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to a unit identifier\', but the effective
data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all unit identifier attribute values
(including, if appropriate, inside mathematical formulas), comparing the
referenced unit identifiers to the value of @p oldid.  If any matches
are found, the matching values are replaced with @p newid.  The method
does @em not descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") Model::isBoolean "
@internal

@internal
";


%feature("docstring") Model::convertL1ToL2 "
@internal

@internal
";


%feature("docstring") Model::convertL1ToL3 "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") Model::convertL2ToL3 "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") Model::convertL2ToL1 "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") Model::convertL3ToL1 "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") Model::convertL3ToL2 "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") Model::convertFromL3V2 "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") Model::dealWithFast "
@internal

@internal
";


%feature("docstring") Model::dealWithL3Fast "
@internal

@internal
";


%feature("docstring") Model::addModifiers "
@internal

@internal
";


%feature("docstring") Model::addConstantAttribute "
@internal

@internal
";


%feature("docstring") Model::setSpatialDimensions "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") Model::addDefinitionsForDefaultUnits "
@internal

@internal
";


%feature("docstring") Model::dealWithDefaultValues "
@internal

@internal
";


%feature("docstring") Model::convertParametersToLocals "
@internal

@internal
";


%feature("docstring") Model::setSpeciesReferenceConstantValueAndStoichiometry "
@internal

@internal
";


%feature("docstring") Model::removeMetaId "
@internal

@internal
";


%feature("docstring") Model::removeSBOTerms "
@internal

@internal
";


%feature("docstring") Model::removeHasOnlySubstanceUnits "
@internal

@internal
";


%feature("docstring") Model::removeSBOTermsNotInL2V2 "
@internal

@internal
";


%feature("docstring") Model::removeDuplicateTopLevelAnnotations "
@internal

@internal
";


%feature("docstring") Model::removeParameterRuleUnits "
@internal

@internal
";


%feature("docstring") Model::convertStoichiometryMath "
@internal

@internal
";


%feature("docstring") Model::assignRequiredValues "
@internal

@internal
";


%feature("docstring") Model::dealWithModelUnits "
@internal
 

@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.

@internal
";


%feature("docstring") Model::dealWithStoichiometry "
@internal

@internal
";


%feature("docstring") Model::dealWithEvents "
@internal

@internal
";


%feature("docstring") Model::removeSpeciesTypes "
@internal

@internal
";


%feature("docstring") Model::removeCompartmentTypes "
@internal

@internal
";


%feature("docstring") Model::setSBMLDocument "
@internal

@internal
";


%feature("docstring") Model::getTypeCode "
Returns the libSBML type code for this %SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_MODEL SBML_MODEL@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getElementName()
@see getPackageName()
";


%feature("docstring") Model::getElementName "
Returns the XML element name of this object, which for Model, is
always @c \'model\'.

@return the name of this element, i.e., @c \'model\'.
";


%feature("docstring") Model::getElementPosition "
@internal

@internal
";


%feature("docstring") Model::writeElements "
@internal

@internal
";


%feature("docstring") Model::populateListFormulaUnitsData "
Populates the internal list of derived units for this Model object.

This method tells libSBML to (re)calculate all units for all components
of the enclosing Model object.  The result is stored in an internal list
of unit data.  Users can access the resulting data by calling the method
SBase.getDerivedUnitDefinition() available on most objects.  (The name
\'formula units data\' is drawn from the name of the internal objects
libSBML uses to store the data; note that these internal objects are not
exposed to callers, because callers can interact with the results using
the ordinary SBML unit objects.)

This method is used by libSBML itself in the validator concerned with
unit consistency.  The unit consistency validator (like all other
validators in libSBML) is invoked by using
SBMLDocument.checkConsistency(), with the consistency checks for the
category @link libsbml#LIBSBML_CAT_UNITS_CONSISTENCY LIBSBML_CAT_UNITS_CONSISTENCY@endlink turned on.  The method
populateListFormulaUnitsData() does not need to be called prior to
invoking the validator if unit consistency checking has not been turned
off.  This method is only provided for cases when callers have a special
need to force the unit data to be recalculated.  For instance, during
construction of a model, a caller may want to interrogate libSBML\'s
inferred units without invoking full-blown model validation; this is a
scenario in which calling populateListFormulaUnitsData() may be useful.

@warning Computing and inferring units is a time-consuming operation.
Callers may want to call isPopulatedListFormulaUnitsData() to determine
whether the units may already have been computed, to save themselves the
need of invoking unit inference unnecessarily.

@see isPopulatedListFormulaUnitsData()
";


%feature("docstring") Model::isPopulatedListFormulaUnitsData "
Predicate returning @c True if libSBML has derived units for the
components of this model.

LibSBML can infer the units of measurement associated with different
elements of a model.  When libSBML does that, it builds a complex
internal structure during a resource-intensive operation.  This is done
automatically only when callers invoke validation (via
SBMLDocument.checkConsistency()) and have not turned off the unit
validation option.

Callers can force units to be recalculated by calling
populateListFormulaUnitsData().  To avoid calling that method
unnecessarily, calling programs may first want to invoke this method
(isPopulatedListFormulaUnitsData()) to determine whether it is even
necessary.

@return @c True if the units have already been computed, @c False
otherwise.
";


%feature("docstring") Model::addFormulaUnitsData "
@internal

@internal
";


%feature("docstring") Model::getFormulaUnitsDataForVariable "
@internal

@internal
";


%feature("docstring") Model::getFormulaUnitsDataForAssignment "
@internal

@internal
";


%feature("docstring") Model::getNumFormulaUnitsData "
@internal

@internal
";


%feature("docstring") Model::getListFormulaUnitsData "
@internal

@internal
";


%feature("docstring") Model::populateAllElementIdList "
Populates the internal list of the identifiers of all elements within this Model object.

This method tells libSBML to retrieve the identifiers of all elements
of the enclosing Model object.  The result is stored in an internal list
of ids.  Users can access the resulting data by calling the method
getAllElementIdList().

@warning Retrieving all elements within a model is a time-consuming operation.
Callers may want to call isPopulatedAllElementIdList() to determine
whether the id list may already have been populated.

@see isPopulatedAllElementIdList()
";


%feature("docstring") Model::isPopulatedAllElementIdList "
Predicate returning @c True if libSBML has a list of the ids of all 
components of this model.

@return @c True if the id list has already been populated, @c False
otherwise.
";


%feature("docstring") Model::getAllElementIdList "
Returns the internal list of the identifiers of all elements within this Model object.

@return an IdList of all the identifiers in the model.

@see populateAllElementIdList()
@see isPopulatedAllElementIdList()
";


%feature("docstring") Model::clearAllElementIdList "
Clears the internal list of the identifiers of all elements within this Model object.

@see populateAllElementIdList()
@see isPopulatedAllElementIdList()
";


%feature("docstring") Model::populateAllElementMetaIdList "
Populates the internal list of the metaids of all elements within this Model object.

This method tells libSBML to retrieve the identifiers of all elements
of the enclosing Model object.  The result is stored in an internal list
of metaids.  Users can access the resulting data by calling the method
getAllElementMetaIdList().

@warning Retrieving all elements within a model is a time-consuming operation.
Callers may want to call isPopulatedAllElementMetaIdList() to determine
whether the metaid list may already have been populated.

@see isPopulatedAllElementMetaIdList()
";


%feature("docstring") Model::isPopulatedAllElementMetaIdList "
Predicate returning @c True if libSBML has a list of the metaids of all 
components of this model.

@return @c True if the metaid list has already been populated, @c False
otherwise.
";


%feature("docstring") Model::getAllElementMetaIdList "
Returns the internal list of the metaids of all elements within this Model object.

@return an IdList of all the metaids in the model.

@see populateAllElementMetaIdList()
@see isPopulatedAllElementMetaIdList()
";


%feature("docstring") Model::clearAllElementMetaIdList "
Clears the internal list of the metaids of all elements within this Model object.

@see populateAllElementMetaIdList()
@see isPopulatedAllElementMetaIdList()
";


%feature("docstring") Model::hasRequiredElements "
Predicate returning @c True if all the required elements for this Model
object have been set.

@return a boolean value indicating whether all the required
elements for this object have been defined.
";


%feature("docstring") Model::removeFunctionDefinition "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeFunctionDefinition(long n)</pre>

Removes the nth FunctionDefinition object from this Model object and 
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the FunctionDefinition object to remove.

@return the FunctionDefinition object removed, or @c None if the given
index is out of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeFunctionDefinition(string sid)</pre>

Removes the FunctionDefinition object with the given identifier from this Model 
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the FunctionDefinition object to remove.

@return the FunctionDefinition object removed, or @c None if no
FunctionDefinition object with the identifier exists in this Model
object.
";


%feature("docstring") Model::removeUnitDefinition "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeUnitDefinition(long n)</pre>

Removes the nth UnitDefinition object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the UnitDefinition object to remove.

@return the UnitDefinition object removed., or @c None if the given
index is out of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeUnitDefinition(string sid)</pre>

Removes the UnitDefinition object with the given identifier from this Model
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the UnitDefinition object to remove.

@return the UnitDefinition object removed, or @c None if no
UnitDefinition object with the identifier exists in this Model object.
";


%feature("docstring") Model::removeCompartmentType "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeCompartmentType(long n)</pre>

Removes the nth CompartmentType object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the CompartmentType object to remove.

@return the ComapartmentType object removed, or @c None if the given
index is out of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeCompartmentType(string sid)</pre>

Removes the CompartmentType object with the given identifier from this Model
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the object to remove.

@return the CompartmentType object removed, or @c None if no
CompartmentType object with the identifier exists in this Model object.
";


%feature("docstring") Model::removeSpeciesType "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeSpeciesType(long n)</pre>

Removes the nth SpeciesType object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the SpeciesType object to remove.

@return the SpeciesType object removed, or @c None if the given index is
out of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeSpeciesType(string sid)</pre>

Removes the SpeciesType object with the given identifier from this Model
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the SpeciesType object to remove.

@return the SpeciesType object removed, or @c None if no SpeciesType
object with the identifier exists in this Model object.
";


%feature("docstring") Model::removeCompartment "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeCompartment(long n)</pre>

Removes the nth Compartment object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the Compartment object to remove.

@return the Compartment object removed, or @c None if the given index is
out of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeCompartment(string sid)</pre>

Removes the Compartment object with the given identifier from this Model
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the Compartment object to remove.

@return the Compartment object removed, or @c None if no Compartment
object with the identifier exists in this Model object.
";


%feature("docstring") Model::removeSpecies "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeSpecies(long n)</pre>

Removes the nth Species object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the Species object to remove.

@return the Species object removed, or @c None if the given index is out
of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeSpecies(string sid)</pre>

Removes the Species object with the given identifier from this Model
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the Species object to remove.

@return the Species object removed, or @c None if no Species object with
the identifier exists in this Model object.
";


%feature("docstring") Model::removeParameter "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeParameter(long n)</pre>

Removes the nth Parameter object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the Parameter object to remove.

@return the Parameter object removed, or @c None if the given index is
out of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeParameter(string sid)</pre>

Removes the Parameter object with the given identifier from this Model
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the Parameter object to remove.

@return the Parameter object removed, or @c None if no Parameter object
with the identifier exists in this Model object.
";


%feature("docstring") Model::removeInitialAssignment "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeInitialAssignment(string symbol)</pre>

Removes the InitialAssignment object with the given \'symbol\' attribute 
from this Model object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param symbol the \'symbol\' attribute of the InitialAssignment object to remove.

@return the InitialAssignment object removed, or @c None if no
InitialAssignment object with the \'symbol\' attribute exists in this
Model object.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeInitialAssignment(long n)</pre>

Removes the nth InitialAssignment object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the InitialAssignment object to remove.

@return the InitialAssignment object removed, or @c None if the given
index is out of range.
";


%feature("docstring") Model::removeRule "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeRule(string variable)</pre>

Removes the Rule object with the given \'variable\' attribute from this Model 
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param variable the \'variable\' attribute of the Rule object to remove.

@return the Rule object removed, or @c None if no Rule object with the
\'variable\' attribute exists in this Model object.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeRule(long n)</pre>

Removes the nth Rule object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the Rule object to remove.

@return the Rule object removed, or @c None if the given index is out of
range.
";


%feature("docstring") Model::removeRuleByVariable "
Removes the Rule object with the given \'variable\' attribute from this Model 
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param variable the \'variable\' attribute of the Rule object to remove.

@return the Rule object removed, or @c None if no Rule object with the
\'variable\' attribute exists in this Model object.
";


%feature("docstring") Model::removeConstraint "
Removes the nth Constraint object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the Constraint object to remove.

@return the Constraint object removed, or @c None if the given index is
out of range.
";


%feature("docstring") Model::removeReaction "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeReaction(long n)</pre>

Removes the nth Reaction object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the Reaction object to remove.

@return the Reaction object removed, or @c None if the given index is
out of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeReaction(string sid)</pre>

Removes the Reaction object with the given identifier from this Model
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the Reaction object to remove.

@return the Reaction object removed, or @c None if no Reaction object
with the identifier exists in this Model object.
";


%feature("docstring") Model::removeEvent "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeEvent(long n)</pre>

Removes the nth Event object from this Model object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the Event object to remove.

@return the Event object removed, or @c None if the given index is out
of range.

   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>removeEvent(string sid)</pre>

Removes the Event object with the given identifier from this Model
object and returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param sid the identifier of the Event object to remove.

@return the Event object removed, or @c None if no Event object with the
identifier exists in this Model object.
";


%feature("docstring") Model::appendFrom "
Copies a given Model object\'s subcomponents and appends the copies to
the appropriate places in this Model.

This method also calls the <code>appendFrom</code> method on all libSBML
plug-in objects.

@par
SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
that it uses one or more SBML packages, and likewise, a software tool may be
able to support one or more packages.  LibSBML does not come preconfigured
with all possible packages included and enabled, in part because not all
package specifications have been finalized.  To support the ability for
software systems to enable support for the Level&nbsp;3 packages they choose,
libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
package is implemented in a separate code plug-in that can be enabled by the
application to support working with that SBML package.  A given SBML model
may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
objects created by libSBML plug-ins supporting additional Level&nbsp;3
packages.

@param model the Model to merge with this one.
";


%feature("docstring") Model::enablePackageInternal "
@internal

@internal
";


%feature("docstring") Model::updateSBMLNamespace "
@internal

@internal
";


%feature("docstring") Model::readOtherXML "
@internal

@internal
";


%feature("docstring") Model::createObject "
@internal

@internal
";


%feature("docstring") Model::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") Model::readAttributes "
@internal

@internal
";


%feature("docstring") Model::readL1Attributes "
@internal

@internal
";


%feature("docstring") Model::readL2Attributes "
@internal

@internal
";


%feature("docstring") Model::readL3Attributes "
@internal

@internal
";


%feature("docstring") Model::writeAttributes "
@internal

@internal
";


%feature("docstring") Model::syncAnnotation "
@internal

@internal
";


%feature("docstring") Model::populatePerTimeUnitDefinition "
@internal

@internal
";


%feature("docstring") Model::createSpeciesReferenceUnitsData "
@internal

@internal
";


%feature("docstring") Model::removeListFormulaUnitsData "
@internal

@internal
";


%feature("docstring") Model::createSubstanceUnitsData "
@internal

@internal
";


%feature("docstring") Model::getL3SubstanceUD "
@internal

@internal
";


%feature("docstring") Model::getSubstanceUD "
@internal

@internal
";


%feature("docstring") Model::createTimeUnitsData "
@internal

@internal
";


%feature("docstring") Model::getL3TimeUD "
@internal

@internal
";


%feature("docstring") Model::getTimeUD "
@internal

@internal
";


%feature("docstring") Model::createVolumeUnitsData "
@internal

@internal
";


%feature("docstring") Model::getL3VolumeUD "
@internal

@internal
";


%feature("docstring") Model::getVolumeUD "
@internal

@internal
";


%feature("docstring") Model::createAreaUnitsData "
@internal

@internal
";


%feature("docstring") Model::getL3AreaUD "
@internal

@internal
";


%feature("docstring") Model::getAreaUD "
@internal

@internal
";


%feature("docstring") Model::createLengthUnitsData "
@internal

@internal
";


%feature("docstring") Model::getL3LengthUD "
@internal

@internal
";


%feature("docstring") Model::getLengthUD "
@internal

@internal
";


%feature("docstring") Model::createExtentUnitsData "
@internal

@internal
";


%feature("docstring") Model::getL3ExtentUD "
@internal

@internal
";


%feature("docstring") Model::createSubstancePerTimeUnitsData "
@internal

@internal
";


%feature("docstring") Model::getL3SubstancePerTimeUD "
@internal

@internal
";


%feature("docstring") Model::getSubstancePerTimeUD "
@internal

@internal
";


%feature("docstring") Model::createCompartmentUnitsData "
@internal

@internal
";


%feature("docstring") Model::createSpeciesUnitsData "
@internal

@internal
";


%feature("docstring") Model::createL3SpeciesUnitsData "
@internal

@internal
";


%feature("docstring") Model::createParameterUnitsData "
@internal

@internal
";


%feature("docstring") Model::createUnitsDataFromMath "
@internal

@internal
";


%feature("docstring") Model::createInitialAssignmentUnitsData "
@internal

@internal
";


%feature("docstring") Model::createConstraintUnitsData "
@internal

@internal
";


%feature("docstring") Model::createRuleUnitsData "
@internal

@internal
";


%feature("docstring") Model::createReactionUnitsData "
@internal

@internal
";


%feature("docstring") Model::createEventUnitsData "
@internal

@internal
";


%feature("docstring") Model::createDelayUnitsData "
@internal

@internal
";


%feature("docstring") Model::createTriggerUnitsData "
@internal

@internal
";


%feature("docstring") Model::createPriorityUnitsData "
@internal

@internal
";


%feature("docstring") Model::createEventAssignmentUnitsData "
@internal

@internal
";


%feature("docstring") Model::createLocalParameterUnitsData "
@internal

@internal
";


%feature("docstring") SBMLDocument "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html Overall SBML container object.

@if clike LibSBML uses the class SBMLDocument as a
top-level container for storing SBML content and data associated with it
(such as warnings and error messages).  The two primary means of reading
an SBML model, SBMLReader.readSBML() and
SBMLReader.readSBMLFromString(), both return a pointer to an
SBMLDocument object.  From there, callers can inquire about any errors
encountered (e.g., using SBMLDocument.getNumErrors()), access the Model
object, and perform other actions such as consistency-checking and model
translation.
@endif@if python LibSBML uses the class SBMLDocument as a
top-level container for storing SBML content and data associated with it
(such as warnings and error messages).  The two primary means of reading
an SBML model, SBMLReader.readSBML() and
SBMLReader.readSBMLFromString(), both return a pointer to an
SBMLDocument object.  From there, callers can inquire about any errors
encountered (e.g., using SBMLDocument.getNumErrors()), access the Model
object, and perform other actions such as consistency-checking and model
translation.
@endif@if java LibSBML uses the class SBMLDocument as a top-level
container for storing SBML content and data associated with it (such as
warnings and error messages).  The two primary means of reading an SBML
model, SBMLReader.readSBML() and
SBMLReader.readSBMLFromString(), both return an SBMLDocument
object.  From there, callers can inquire about any errors encountered
(e.g., using SBMLDocument.getNumErrors()), access the Model object, and
perform other actions such as consistency-checking and model
translation.
@endif@~

When creating fresh models programmatically, the starting point is
typically the creation of an SBMLDocument object instance.  The
SBMLDocument constructor accepts arguments for the SBML Level and
Version of the model to be created.  After creating the SBMLDocument
object, calling programs then typically call SBMLDocument.createModel()
almost immediately, and then proceed to call the methods on the Model
object to fill out the model\'s contents.

SBMLDocument corresponds roughly to the class <i>Sbml</i> defined in the
SBML Level&nbsp;2 specification and <i>SBML</i> in the Level&nbsp;3
specification.  It does not have a direct correspondence in SBML
Level&nbsp;1.  (However, to make matters simpler for applications,
libSBML creates an SBMLDocument no matter whether the model is
Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)  In its barest form, when written out in
XML format for (e.g.) SBML Level&nbsp;2 Version&nbsp;4, the corresponding
structure is the following:
@verbatim
<sbml xmlns=\'http://www.sbml.org/sbml/level2/version4\' level=\'2\' version=\'4\'>
  ...
</sbml>@endverbatim

SBMLDocument is derived from SBase, and therefore contains the usual SBase
attributes (in SBML Level&nbsp;2 and Level&nbsp;3) of \'metaid\' and \'sboTerm\', as
well as the subelements \'notes\' and \'annotation\'.  It also contains the
attributes \'level\' and \'version\' indicating the Level and Version of the
SBML data structure.  These can be accessed using the methods defined by
the SBase class for that purpose.

@section checking Checking consistency and adherence to SBML specifications

One of the most important features of libSBML is its ability to perform
SBML validation to ensure that a model adheres to the SBML specification
for whatever Level+Version combination the model uses.  SBMLDocument
provides the methods for running consistency-checking and validation
rules on the SBML content.

First, a brief explanation of the rationale is in order.  In libSBML
versions up to and including the version&nbsp;3.3.x series, the
individual methods for creating and setting attributes and other
components were quite lenient, and allowed a caller to compose SBML
entities that might not, in the end, represent valid SBML.  This allowed
applications the freedom to do things such as save incomplete models
(which is useful when models are being developed over long periods of
time).  In the version&nbsp;4.x series, libSBML is somewhat stricter,
but still permits structures to be created independently and the results
to be combined in a separate step.  In all these cases, it means that a
separate validation step is necessary when a calling program finally
wants to finish a complete SBML document.

The primary interface to this validation facility is SBMLDocument\'s
SBMLDocument.checkInternalConsistency() and
SBMLDocument.checkConsistency().  The former verifies the basic
internal consistency and syntax of an SBML document, and the latter
implements more elaborate validation rules (both those defined by the
SBML specifications, as well as additional rules offered by libSBML).

@if clike The checks performed by SBMLDocument.checkInternalConsistency() are
hardwired and cannot be changed by calling programs, but the validation
performed by SBMLDocument.checkConsistency() is under program control
using the method SBMLDocument.setConsistencyChecks().  Applications can
selectively disable specific kinds of checks that they may not be
interested in, by calling SBMLDocument.setConsistencyChecks() with
appropriate parameters.
@endif@if python The checks performed by SBMLDocument.checkInternalConsistency() are
hardwired and cannot be changed by calling programs, but the validation
performed by SBMLDocument.checkConsistency() is under program control
using the method SBMLDocument.setConsistencyChecks().  Applications can
selectively disable specific kinds of checks that they may not be
interested in, by calling SBMLDocument.setConsistencyChecks() with
appropriate parameters.
@endif@if java The checks performed by SBMLDocument.checkInternalConsistency() are
hardwired and cannot be changed by calling programs, but the validation
performed by SBMLDocument.checkConsistency() is under program control
using the method SBMLDocument.setConsistencyChecks().  Applications can selectively disable specific kinds of checks
that they may not be interested by calling
SBMLDocument.setConsistencyChecks() with
appropriate parameters.
@endif@~

These methods have slightly different relevance depending on whether a
model is created programmaticaly from scratch, or whether it is read in
from a file or data stream.  The following list summarizes the possible
scenarios.

<em>Scenario 1: Creating a model from scratch</em>.  Before writing out
the model, 

@li Call SBMLDocument.checkInternalConsistency(), then inquire about
the results by calling SBMLDocument.getNumErrors()

@li Call @if java SBMLDocument.setConsistencyChecks() @else SBMLDocument.setConsistencyChecks() @endif@~ to configure
which checks will be performed by SBMLDocument.checkConsistency()

@li Call SBMLDocument.checkConsistency(), then inquire about the results by
calling SBMLDocument.getNumErrors()

<em>Scenario 2: Reading a model from a file or data stream.</em> After
reading the model,

@li Basic consistency checks will have been performed automatically by
libSBML upon reading the content, so you only need to inquire about the
results by using SBMLDocument.getNumErrors()

@li Call @if java SBMLDocument.setConsistencyChecks() @else SBMLDocument.setConsistencyChecks() @endif@~ to configure
which checks are performed by SBMLDocument.checkConsistency()

@li Call SBMLDocument.checkConsistency(), then inquire about the results
by calling SBMLDocument.getNumErrors()

@if clike An example of using the consistency-checking
and validation facilities is provided in this manual in the
section @ref libsbml-example. @endif@~

It should be noted that as of SBML Level&nbsp;3 Version&nbsp;2, the Model
became an optional child of SBMLDocument, instead of being required.  This
means that one can no longer use SBMLDocument.getModel() as a cheap method
of checking if an SBML document was read in properly: the more robust
getError methods detailed above must be used instead.

@section converting Converting documents between Levels and Versions of SBML

LibSBML provides facilities for limited translation of SBML between
Levels and Versions of the SBML specifications.  The method for doing is
is @if java SBMLDocument.setLevelAndVersion() @else setLevelAndVersion() @endif.  In 
general, models can be converted upward without difficulty (e.g., from
SBML Level&nbsp;1 to Level&nbsp;2, or from an earlier Version of
Level&nbsp;2 to the latest Version of Level&nbsp;2).  Sometimes models
can be translated downward as well, if they do not use constructs
specific to more advanced Levels of SBML.

Calling @if java SBMLDocument.setLevelAndVersion() @else SBMLDocument.setLevelAndVersion() @endif@~ will not @em necessarily lead
to a successful conversion.  The method will return a boolean value
to indicate success or failure.  Callers must check the error log (see 
next section) attached to the SBMLDocument object after calling
@if java SBMLDocument.setLevelAndVersion() @else SBMLDocument.setLevelAndVersion() @endif@~ in order to assess whether any
problems arose.

If an application is interested in translating to a lower Level and/or
Version of SBML within a Level, the following methods allow for prior
assessment of whether there is sufficient compatibility to make a
translation possible:

@li SBMLDocument.checkL1Compatibility(),
@li SBMLDocument.checkL2v1Compatibility(),
@li SBMLDocument.checkL2v2Compatibility(),
@li SBMLDocument.checkL2v3Compatibility(), 
@li SBMLDocument.checkL2v4Compatibility(),
@li SBMLDocument.checkL2v5Compatibility(), and
@li SBMLDocument.checkL3v1Compatibility().

Some changes between Versions of SBML Level&nbsp;2 may lead to
unexpected behaviors when attempting conversions in either direction.
For example, SBML Level&nbsp;2 Version&nbsp;4 relaxed the requirement
for consistency in units of measurement between expressions annd
quantities in a model.  As a result, a model written in Version&nbsp;4,
if converted to Version&nbsp;3 with no other changes, may fail
validation as a Version&nbsp;3 model because Version&nbsp;3 imposed
stricter requirements on unit consistency.

Other changes between SBML Level 2 and Level 3 make downward conversions
challenging.  In some cases, it means that a model converted to
Level&nbsp;2 from Level&nbsp;3 will contain attributes that were not
explicitly given in the Level&nbsp;3 model, because in Level&nbsp;2
these attributes may have been optional or have default values.

@section errors Error handling

Upon reading a model, SBMLDocument logs any problems encountered while
reading the model from the file or data stream.  The log contains
objects that record diagnostic information about any notable issues that
arose.  Whether the problems are warnings or errors, they are both
reported through a single common interface involving the object class
SBMLError.

The methods SBMLDocument.getNumErrors(), @if java SBMLDocument.getError() @else SBMLDocument.getError() @endif@~ and
SBMLDocument.printErrors() allow callers to interact with the warnings
or errors logged.  Alternatively, callers may retrieve the entire log as
an SBMLErrorLog object using the method SBMLDocument.getErrorLog().
The SBMLErrorLog object provides some alternative methods for
interacting with the set of errors and warnings.  In either case,
applications typically should first call SBMLDocument.getNumErrors() to
find out if any issues have been logged after specific libSBML
operations such as the ones discussed in the sections above.  If they
have, then an application will should proceed to inspect the individual
reports using either the direct interfaces on SBMLDocument or using the
methods on the SBMLErrorLog object.

@if clike An example of using the error facility is
provided in this manual in the
section @ref libsbml-example. @endif@~
";


%feature("docstring") SBMLDocument::getDefaultLevel "
The default SBML Level of new SBMLDocument objects.

@par
This \'default Level\' corresponds to the most recent SBML specification
Level available at the time libSBML version @htmlinclude
libsbml-version.html was released.  The default Level is used by
SBMLDocument if no Level is explicitly specified at the time of the
construction of an SBMLDocument instance.

@return an integer indicating the most recent SBML specification Level.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike getDefaultVersion() @else SBMLDocument.getDefaultVersion() @endif@~
";


%feature("docstring") SBMLDocument::getDefaultVersion "
The default Version of new SBMLDocument objects.

@par
This \'default Version\' corresponds to the most recent Version within the
most recent Level of SBML available at the time libSBML version
@htmlinclude libsbml-version.html was released.  The default Version is
used by SBMLDocument if no Version is explicitly specified at the time of
the construction of an SBMLDocument instance. 

@return an integer indicating the most recent SBML specification
Version.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike getDefaultLevel() @else SBMLDocument.getDefaultLevel() @endif@~
";


%feature("docstring") SBMLDocument::SBMLDocument "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>SBMLDocument(SBMLNamespaces sbmlns)</pre>

Creates a new SBMLDocument using the given SBMLNamespaces object 
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments. 

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>SBMLDocument(long level = 0, long version = 0)</pre>

Creates a new SBMLDocument, optionally with given values for the SBML
Level and Version.

If <em>both</em> the SBML Level and Version attributes are not
specified, the SBML document is treated as having the latest Level and
Version of SBML as determined by SBMLDocument.getDefaultLevel() and
SBMLDocument.getDefaultVersion(); <em>however</em>, the SBMLDocument
object is otherwise left blank.  In particular, the blank SBMLDocument
object has no associated XML attributes, including (but not limited
to) an XML namespace declaration.  The XML namespace declaration is
not added until the model is written out, <em>or</em> the method
SBMLDocument.setLevelAndVersion()
is called.  This may be important to keep in mind
if an application needs to add additional XML namespace declarations
on the <code>&lt;sbml&gt;</code> element.  Application writers should
either provide values for @p level and @p version on the call to this
constructor, or else call
SBMLDocument.setLevelAndVersion()
shortly after creating the SBMLDocument object.

@param level an integer for the SBML Level.

@param version an integer for the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~

@see SBMLDocument.setLevelAndVersion()
@see getDefaultLevel()
@see getDefaultVersion()
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.


@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>SBMLDocument(SBMLDocument orig)</pre>

Copy constructor; creates a copy of this SBMLDocument.

@param orig the object to copy.
";


%feature("docstring") SBMLDocument::accept "
@internal

@internal
";


%feature("docstring") SBMLDocument::clone "
Creates and returns a deep copy of this SBMLDocument object.

@return the (deep) copy of this SBMLDocument object.
";


%feature("docstring") SBMLDocument::isSetModel "
Returns @c True if the Model object has been set, otherwise 
returns @c False.

@return @c True if the Model object has been set
";


%feature("docstring") SBMLDocument::getModel "
Returns the Model object stored in this SBMLDocument.

It is important to note that this method <em>does not create</em> a
Model instance.  The model in the SBMLDocument must have been created
at some prior time, for example using SBMLDocument.createModel() 
or SBMLDocument.setModel().
This method returns @c None if a model does not yet exist.

@return the Model contained in this SBMLDocument, or @c None if no such model exists.

@see createModel()
";


%feature("docstring") SBMLDocument::getElementBySId "
Returns the first child element found that has the given @p id in the
model-wide SId namespace, or @c None if no such object is found.

@param id string representing the id of the object to find.

@return pointer to the first element found with the given @p id.
";


%feature("docstring") SBMLDocument::getElementByMetaId "
Returns the first child element it can find with the given @p metaid, or
itself if it has the given @p metaid, or @c None if no such object is
found.

@param metaid string representing the metaid of the object to find.

@return pointer to the first element found with the given @p metaid.
";


%feature("docstring") SBMLDocument::getAllElements "
Returns a List of all child SBase objects, including those nested to an
arbitrary depth

@param filter a pointer to an ElementFilter, which causes the function 
to return only elements that match a particular set of constraints.  
If None (the default), the function will return all child objects.

@return a List of pointers to all children objects.
";


%feature("docstring") SBMLDocument::expandFunctionDefinitions "
Removes FunctionDefinition constructs from the document and expands
any instances of their use within <code>&lt;math&gt;</code> elements.

For example, suppose a Model contains a FunctionDefinition with
identifier @c \'f\' representing the math expression: <em>f(x, y) = x *
y</em>.  Suppose further that there is a reaction in which the
<code>&lt;math&gt;</code> element of the KineticLaw object contains
<code>f(s, p)</code>, where @c s and @c p are other identifiers
defined in the model.  The outcome of invoking this method is that the
<code>&lt;math&gt;</code> of the KineticLaw now represents the
expression <em>s * p</em> and the model no longer contains any
FunctionDefinition objects.

@return bool @c True if the transformation was successful, 
@c False, otherwise.

@note This function will check the consistency of a model before
attemptimg the transformation.  If the model is not valid SBML, the
transformation will not be performed and the function will return
@c False.
";


%feature("docstring") SBMLDocument::expandInitialAssignments "
Removes InitialAssignment constructs from the document and
replaces them with appropriate values.

For example, suppose a Model contains a InitialAssignment to a symbol
@c \'k\' where @c \'k\' is the identifier of a Parameter.  The outcome of
invoking this method is that the \'value\' attribute of the Parameter
definition is set to the result calculated using the InitialAssignment
object\'s <code>&lt;math&gt;</code> formula, and the corresponding
InitialAssignment is then removed from the Model.

@return bool @c True if the transformation was successful, 
@c False, otherwise.

@note This function will check the consistency of a model before
attemptimg the transformation.  If the model is not valid SBML, the
transformation will not be performed and the function will return
@c False.  As part of that process, this method will check that it has
values for any components referred to by the <code>&lt;math&gt;</code>
elements of InitialAssignment objects.  In cases where not all of the
values have been declared (e.g., if the mathematical expression refers
to model entities that have no declared values), the InitialAssignment
in question will @em not be removed and this method will return
@c False.
";


%feature("docstring") SBMLDocument::setLevelAndVersion "
Sets the SBML Level and Version of this SBMLDocument instance,
attempting to convert the model as needed.

This method is the principal way in libSBML to convert models between
Levels and Versions of SBML.  Generally, models can be converted
upward without difficulty (e.g., from SBML Level&nbsp;1 to
Level&nbsp;2, or from an earlier Version of Level&nbsp;2 to the latest
Version of Level&nbsp;2).  Sometimes models can be translated downward
as well, if they do not use constructs specific to more advanced
Levels of SBML.

Before calling this method, callers may check compatibility directly
using the methods SBMLDocument.checkL1Compatibility(),
SBMLDocument.checkL2v1Compatibility(),
SBMLDocument.checkL2v2Compatibility(),
SBMLDocument.checkL2v3Compatibility(),
SBMLDocument.checkL2v4Compatibility(),
SBMLDocument.checkL2v5Compatibility(),
SBMLDocument.checkL3v1Compatibility(), and
SBMLDocument.checkL3v2Compatibility().

The valid combinations of SBML Level and Version as of this release
of libSBML are the following: 
<ul>
<li> Level&nbsp;1 Version&nbsp;2
<li> Level&nbsp;2 Version&nbsp;1
<li> Level&nbsp;2 Version&nbsp;2
<li> Level&nbsp;2 Version&nbsp;3
<li> Level&nbsp;2 Version&nbsp;4
<li> Level&nbsp;2 Version&nbsp;5
<li> Level&nbsp;3 Version&nbsp;1
<li> Level&nbsp;3 Version&nbsp;2
</ul>

Strict conversion applies the additional criteria that both the
source and the target model must be consistent SBML.  Users can
control the consistency checks that are applied using the
SBMLDocument.setConsistencyChecksForConversion() method.  If either
the source or the potential target model have validation errors, the
conversion is not performed.  When a strict conversion is successful,
the underlying SBML object model is altered to reflect the new level
and version.  Thus, information that cannot be converted
(e.g. sboTerms) will be lost.

@param level the desired SBML Level.

@param version the desired Version within the SBML Level.

@param strict boolean indicating whether to check consistency
of both the source and target model when performing
conversion (defaults to <code> true </code>).

@param ignorePackages boolean indicating whether the presence of
packages should be ignored by the conversion routine 
(defaults to <code> false </code>).

@return @c True if the level and version of the document were
successfully set to the requested values (which may have required
conversion of the model), @c False otherwise.

@note Calling this method will not @em necessarily lead to a successful
conversion.  If the conversion fails, it will be logged in the error
list associated with this SBMLDocument.  Callers should consult
getNumErrors() to find out if the conversion succeeded without
problems.  For conversions from Level&nbsp;2 to Level&nbsp;1, callers
can also check the Level of the model after calling this method to
find out whether it is Level&nbsp;1.  (If the conversion to
Level&nbsp;1 failed, the Level of this model will be left unchanged.)

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~

@see checkL1Compatibility()
@see checkL2v1Compatibility()
@see checkL2v2Compatibility()
@see checkL2v3Compatibility()
@see checkL2v4Compatibility()
@see checkL2v5Compatibility()
@see checkL3v1Compatibility()
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBMLDocument::updateSBMLNamespace "
@internal

@internal
";


%feature("docstring") SBMLDocument::setModel "
Sets the Model for this SBMLDocument to a copy of the given Model.

@param m the new Model to use.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink

@see createModel()
@see getModel()
";


%feature("docstring") SBMLDocument::createModel "
Creates a new Model inside this SBMLDocument, and returns a pointer to
it.

In SBML Level&nbsp;2, the use of an identifier on a Model object is
optional.  This method takes an optional argument, @p sid, for setting
the identifier.  If not supplied, the identifier attribute on the
Model instance is not set.

@param sid the identifier of the new Model to create.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~

@see getModel()
@see SBMLDocument.setModel()
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBMLDocument::setLocationURI "
Sets the location of this SBMLDocument.

Called automatically when readSBMLFromFile is used, but may be set
manually as well.
";


%feature("docstring") SBMLDocument::getLocationURI "
Get the location of this SBMLDocument.

If this document was read from a file or had its location set manually,
that filename or set location will be returned, otherwise, an empty
string is returned.
";


%feature("docstring") SBMLDocument::setConsistencyChecks "
Controls the consistency checks that are performed when
SBMLDocument.checkConsistency() is called.

This method works by adding or subtracting consistency checks from the
set of all possible checks that SBMLDocument.checkConsistency() knows
how to perform.  This method may need to be called multiple times in
order to achieve the desired combination of checks.  The first
argument (@p category) in a call to this method indicates the category
of consistency/error checks that are to be turned on or off, and the
second argument (@p apply, a boolean) indicates whether to turn it on
(value of @c True) or off (value of @c False).

@if clike
The possible categories (values to the argument @p category) are the
set of values from the enumeration #SBMLErrorCategory_t.
The following are the possible choices:
@endif@if java
The possible categories (values to the argument @p category) are the
set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
in the interface class {@link libsbmlConstants}.
The following are the possible choices:
@endif@if python 
The possible categories (values to the argument @p category) are the
set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
in the interface class @link libsbml libsbml@endlink.
The following are the possible choices:
@endif@~
<ul>
<li> @link libsbml#LIBSBML_CAT_GENERAL_CONSISTENCY LIBSBML_CAT_GENERAL_CONSISTENCY@endlink:
Correctness and consistency of specific SBML language constructs.
Performing this set of checks is highly recommended.  With respect to
the SBML specification, these concern failures in applying the
validation rules numbered 2xxxx in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_IDENTIFIER_CONSISTENCY LIBSBML_CAT_IDENTIFIER_CONSISTENCY@endlink:
Correctness and consistency of identifiers used for model entities.  An
example of inconsistency would be using a species identifier in a
reaction rate formula without first having declared the species.  With
respect to the SBML specification, these concern failures in applying
the validation rules numbered 103xx in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_UNITS_CONSISTENCY LIBSBML_CAT_UNITS_CONSISTENCY@endlink:
Consistency of measurement units associated with quantities in a model.
With respect to the SBML specification, these concern failures in
applying the validation rules numbered 105xx in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_MATHML_CONSISTENCY LIBSBML_CAT_MATHML_CONSISTENCY@endlink:
Syntax of MathML constructs.  With respect to the SBML specification,
these concern failures in applying the validation rules numbered 102xx
in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3
Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_SBO_CONSISTENCY LIBSBML_CAT_SBO_CONSISTENCY@endlink:
Consistency and validity of %SBO identifiers (if any) used in the model.
With respect to the SBML specification, these concern failures in
applying the validation rules numbered 107xx in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_OVERDETERMINED_MODEL LIBSBML_CAT_OVERDETERMINED_MODEL@endlink:
Static analysis of whether the system of equations implied by a model is
mathematically overdetermined.  With respect to the SBML specification,
this is validation rule #10601 in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_MODELING_PRACTICE LIBSBML_CAT_MODELING_PRACTICE@endlink:
Additional checks for recommended good modeling practice. (These are
tests performed by libSBML and do not have equivalent SBML validation
rules.)  </ul>

<em>By default, all validation checks are applied</em> to the model in
an SBMLDocument object @em unless
SBMLDocument.setConsistencyChecks()
is called to indicate that only a subset should be applied.  Further,
this default (i.e., performing all checks) applies separately to
<em>each new SBMLDocument object</em> created.  In other words, each
time a model is read using SBMLReader.readSBML(),
SBMLReader.readSBMLFromString(),
or the global functions readSBML() and readSBMLFromString(), a new
SBMLDocument is created and for that document, a call to
SBMLDocument.checkConsistency() will default to applying all possible checks.
Calling programs must invoke
SBMLDocument.setConsistencyChecks()
for each such new model if they wish to change the consistency checks
applied.

@param category a value drawn from @if clike #SBMLErrorCategory_t@else
the set of SBML error categories@endif@~ indicating the
consistency checking/validation to be turned on or off.

@param apply a boolean indicating whether the checks indicated by
@p category should be applied or not.

@see SBMLDocument.checkConsistency()
";


%feature("docstring") SBMLDocument::setConsistencyChecksForConversion "
Controls the consistency checks that are performed when
SBMLDocument.setLevelAndVersion() is called.

This method works by adding or subtracting consistency checks from the
set of all possible checks that may be performed to avoid conversion
to or from an invalid document.  This method may need to be called 
multiple times in
order to achieve the desired combination of checks.  The first
argument (@p category) in a call to this method indicates the category
of consistency/error checks that are to be turned on or off, and the
second argument (@p apply, a boolean) indicates whether to turn it on
(value of @c True) or off (value of @c False).

@if clike
The possible categories (values to the argument @p category) are the
set of values from the enumeration #SBMLErrorCategory_t.
The following are the possible choices:
@endif@if java
The possible categories (values to the argument @p category) are the
set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
in the interface class {@link libsbmlConstants}.
The following are the possible choices:
@endif@if python 
The possible categories (values to the argument @p category) are the
set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
in the interface class @link libsbml libsbml@endlink.
The following are the possible choices:
@endif@~
<ul>
<li> @link libsbml#LIBSBML_CAT_GENERAL_CONSISTENCY LIBSBML_CAT_GENERAL_CONSISTENCY@endlink:
Correctness and consistency of specific SBML language constructs.
Performing this set of checks is highly recommended.  With respect to
the SBML specification, these concern failures in applying the
validation rules numbered 2xxxx in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_IDENTIFIER_CONSISTENCY LIBSBML_CAT_IDENTIFIER_CONSISTENCY@endlink:
Correctness and consistency of identifiers used for model entities.  An
example of inconsistency would be using a species identifier in a
reaction rate formula without first having declared the species.  With
respect to the SBML specification, these concern failures in applying
the validation rules numbered 103xx in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_UNITS_CONSISTENCY LIBSBML_CAT_UNITS_CONSISTENCY@endlink:
Consistency of measurement units associated with quantities in a model.
With respect to the SBML specification, these concern failures in
applying the validation rules numbered 105xx in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_MATHML_CONSISTENCY LIBSBML_CAT_MATHML_CONSISTENCY@endlink:
Syntax of MathML constructs.  With respect to the SBML specification,
these concern failures in applying the validation rules numbered 102xx
in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3
Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_SBO_CONSISTENCY LIBSBML_CAT_SBO_CONSISTENCY@endlink:
Consistency and validity of %SBO identifiers (if any) used in the model.
With respect to the SBML specification, these concern failures in
applying the validation rules numbered 107xx in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_OVERDETERMINED_MODEL LIBSBML_CAT_OVERDETERMINED_MODEL@endlink:
Static analysis of whether the system of equations implied by a model is
mathematically overdetermined.  With respect to the SBML specification,
this is validation rule #10601 in the Level&nbsp;2
Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.

<li> @link libsbml#LIBSBML_CAT_MODELING_PRACTICE LIBSBML_CAT_MODELING_PRACTICE@endlink:
Additional checks for recommended good modeling practice. (These are
tests performed by libSBML and do not have equivalent SBML validation
rules.)
</ul>

<em>By default, all validation checks are applied</em> to the model in
an SBMLDocument object @em unless
SBMLDocument.setConsistencyChecks()
is called to indicate that only a subset should be applied.  Further,
this default (i.e., performing all checks) applies separately to
<em>each new SBMLDocument object</em> created.  In other words, each
time a model is read using SBMLReader.readSBML(),
SBMLReader.readSBMLFromString(),
or the global functions readSBML() and readSBMLFromString(), a new
SBMLDocument is created and for that document, a call to
SBMLDocument.checkConsistency() will default to applying all possible checks.
Calling programs must invoke
SBMLDocument.setConsistencyChecks()
for each such new model if they wish to change the consistency checks
applied.

@param category a value drawn from @if clike #SBMLErrorCategory_t@else
the set of SBML error categories@endif@~ indicating the consistency
checking/validation to be turned on or off.

@param apply a boolean indicating whether the checks indicated by
@p category should be applied or not.

@see SBMLDocument.setLevelAndVersion()
";


%feature("docstring") SBMLDocument::checkConsistency "
Performs consistency checking and validation on this SBML document.

If this method returns a nonzero value (meaning, one or more
consistency checks have failed for SBML document), the failures may be
due to warnings @em or errors.  Callers should inspect the severity
flag in the individual SBMLError objects returned by
SBMLDocument.getError() to determine the nature of the failures.

@return the number of failed checks (errors) encountered.

@see SBMLDocument.checkInternalConsistency()
";


%feature("docstring") SBMLDocument::checkConsistencyWithStrictUnits "
Performs consistency checking and validation on this SBML document
using the ultra strict units validator that assumes that there
are no hidden numerical conversion factors.

If this method returns a nonzero value (meaning, one or more
consistency checks have failed for SBML document), the failures may be
due to warnings @em or errors.  Callers should inspect the severity
flag in the individual SBMLError objects returned by
SBMLDocument.getError() to determine the nature of the failures.

@return the number of failed checks (errors) encountered.

@see SBMLDocument.checkInternalConsistency()
";


%feature("docstring") SBMLDocument::validateSBML "
Performs consistency checking and validation on this SBML document.

If this method returns a nonzero value (meaning, one or more
consistency checks have failed for SBML document), the failures may be
due to warnings @em or errors.  Callers should inspect the severity
flag in the individual SBMLError objects returned by
SBMLDocument.getError() to determine the nature of the failures.

@note unlike checkConsistency this method will write the document
in order to determine all errors for the document. This will 
also clear the error log. 

@return the number of failed checks (errors) encountered.

@see SBMLDocument.checkConsistency()
";


%feature("docstring") SBMLDocument::checkInternalConsistency "
Performs consistency checking on libSBML\'s internal representation of 
an SBML Model.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.

The distinction between this method and
SBMLDocument.checkConsistency() is that this method reports on
fundamental syntactic and structural errors that violate the XML
Schema for SBML; by contrast, SBMLDocument.checkConsistency()
performs more elaborate model verifications and also validation
according to the validation rules written in the appendices of the
SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 specification documents.

@see SBMLDocument.checkConsistency()
";


%feature("docstring") SBMLDocument::checkL1Compatibility "
Performs a set of consistency checks on the document to establish
whether it is compatible with SBML Level&nbsp;1 and can be converted
to Level&nbsp;1.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBMLDocument::checkL2v1Compatibility "
Performs a set of consistency checks on the document to establish
whether it is compatible with SBML Level&nbsp;2 Version&nbsp;1 and can
be converted to Level&nbsp;2 Version&nbsp;1.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBMLDocument::checkL2v2Compatibility "
Performs a set of consistency checks on the document to establish
whether it is compatible with SBML Level&nbsp;2 Version&nbsp;2 and can
be converted to Level&nbsp;2 Version&nbsp;2.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBMLDocument::checkL2v3Compatibility "
Performs a set of consistency checks on the document to establish
whether it is compatible with SBML Level&nbsp;2 Version&nbsp;3 and can
be converted to Level&nbsp;2 Version&nbsp;3.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBMLDocument::checkL2v4Compatibility "
Performs a set of consistency checks on the document to establish
whether it is compatible with SBML Level&nbsp;2 Version&nbsp;4 and can
be converted to Level&nbsp;2 Version&nbsp;4.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.
";


%feature("docstring") SBMLDocument::checkL2v5Compatibility "
Performs a set of consistency checks on the document to establish
whether it is compatible with SBML Level&nbsp;2 Version&nbsp;5 and can
be converted to Level&nbsp;2 Version&nbsp;5.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.
";


%feature("docstring") SBMLDocument::checkL3v1Compatibility "
Performs a set of consistency checks on the document to establish
whether it is compatible with SBML Level&nbsp;3 Version&nbsp;1 and can
be converted to Level&nbsp;3 Version&nbsp;1.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.
";


%feature("docstring") SBMLDocument::checkL3v2Compatibility "
Performs a set of consistency checks on the document to establish
whether it is compatible with SBML Level&nbsp;3 Version&nbsp;2 and can
be converted to Level&nbsp;3 Version&nbsp;2.

Callers should query the results of the consistency check by calling
SBMLDocument.getError().

@return the number of failed checks (errors) encountered.
";


%feature("docstring") SBMLDocument::getError "
Returns the nth error or warning encountered during parsing,
consistency checking, or attempted translation of this model.

Callers can use method XMLError.getSeverity() on the result to assess
the severity of the problem.  The possible severity levels range from
informational messages to fatal errors.

@return the error or warning indexed by integer @p n, or return
@c None if <code>n &gt; (getNumErrors() - 1)</code>.

@param n the integer index of the error sought.

@see SBMLDocument.getNumErrors()
";


%feature("docstring") SBMLDocument::getErrorWithSeverity "
Returns the nth error or warning with the given severity
encountered during parsing, consistency checking, or attempted
translation of this model.

@return the error or warning indexed by integer @p n, or return
@c None if <code>n &gt; (getNumErrors(severity) - 1)</code>.

@param n the integer index of the error sought.
@param severity the severity of the error sought.

@see SBMLDocument.getNumErrors()
";


%feature("docstring") SBMLDocument::getNumErrors "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getNumErrors(long severity)</pre>

Returns the number of errors or warnings encountered with the given 
severity during parsing,
consistency checking, or attempted translation of this model.

@param severity the severity of the error sought.

@return the number of errors or warnings encountered.

@see SBMLDocument.getError()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getNumErrors()</pre>

Returns the number of errors or warnings encountered during parsing,
consistency checking, or attempted translation of this model.

@return the number of errors or warnings encountered.

@see SBMLDocument.getError()
";


%feature("docstring") SBMLDocument::printErrors "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>printErrors(std::ostream stream, long severity)</pre>

Prints all the errors or warnings with the given severity encountered 
trying to parse, check, or translate this SBML document.

It prints the text to the stream given by the parameter @p
stream.  

If no errors have occurred, i.e., <code>getNumErrors(severity) == 0</code>, no
output will be sent to the stream.

The format of the output is:
@verbatim
    N error(s):
      line NNN: (id) message
@endverbatim

@param stream the ostream or ostringstream object indicating where
the output should be printed.
@param severity of the errors sought.

@see getNumErrors()
@see getErrorLog()
@see SBMLDocument.getErrorWithSeverity()
    

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>printErrors(std::ostream stream = std::cerr)</pre>

Prints all the errors or warnings encountered trying to parse,
check, or translate this SBML document.

It prints the text to the stream given by the optional parameter @p
stream.  If no parameter is given, it prints the output to the
standard error stream.

If no errors have occurred, i.e., <code>getNumErrors() == 0</code>, no
output will be sent to the stream.

The format of the output is:
@verbatim
   N error(s):
     line NNN: (id) message
 @endverbatim

@param stream the ostream or ostringstream object indicating where
the output should be printed.

@ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif@~

@see getNumErrors()
@see getErrorLog()
@see SBMLDocument.getError()
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") SBMLDocument::setSBMLDocument "
@internal

@internal
";


%feature("docstring") SBMLDocument::convert "
Converts this document using the converter that best matches
the given conversion properties. 

@param props the conversion properties to use.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_CONV_CONVERSION_NOT_AVAILABLE LIBSBML_CONV_CONVERSION_NOT_AVAILABLE@endlink
";


%feature("docstring") SBMLDocument::enablePackageInternal "
@internal

@internal
";


%feature("docstring") SBMLDocument::getTypeCode "
Returns the libSBML type code for this %SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_DOCUMENT SBML_DOCUMENT@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see SBMLDocument.getElementName()
@see getPackageName()
";


%feature("docstring") SBMLDocument::getElementName "
Returns the XML element name of this object, which for SBMLDocument,
is always @c \'sbml\'.

@return the name of this element, i.e., @c \'sbml\'.
";


%feature("docstring") SBMLDocument::getErrorLog "
Returns the list of errors or warnings logged during parsing, 
consistency checking, or attempted translation of this model.

@return the SBMLErrorLog used for this SBMLDocument.

@see SBMLDocument.getNumErrors()
";


%feature("docstring") SBMLDocument::getNamespaces "
Returns a list of XML Namespaces associated with the XML content
of this SBML document.

@return the XML Namespaces associated with this SBML object.
";


%feature("docstring") SBMLDocument::enableDefaultNS "
Set/unset default namespace to each top-level element defined in the
given package extension.

This works by adding a <code>xmlns=&quot;...&quot;</code> attribute.  No
prefix will be written when writing elements defined in the given
package extension if @c True is given as second argument.

@param package the name or URI of the package extension.
@param flag boolean value to indicate whether to write a namespace
prefix.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION@endlink
";


%feature("docstring") SBMLDocument::isEnabledDefaultNS "
Returns @c True if a default namespace is added to each top-level
element defined in the given package extension, otherwise returns
@c False.

This basically checks if the attribute
<code>xmlns=&quot;...&quot;</code> is present.

@param package the name or URI of the package extension.

@return a boolean indicating whether the given package\'s default namespace is enabled.
";


%feature("docstring") SBMLDocument::setPackageRequired "
Sets the <code>required</code> attribute value of the given package
extension.

@param package the name or URI of the package extension.
@param flag Boolean value indicating whether the package is required.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION@endlink
";


%feature("docstring") SBMLDocument::getPackageRequired "
Returns the <code>required</code> attribute of the given package
extension.

@param package the name or URI of the package extension.

@return Boolean flag indicating whether the package is flagged as
being required.
";


%feature("docstring") SBMLDocument::isSetPackageRequired "
Returns @c True if the required attribute of the given package extension
is defined, otherwise returns @c False.

@param package the name or URI of the package extension.

@return a Boolean indicating whether the package\'s \'required\' flag is set.
";


%feature("docstring") SBMLDocument::isIgnoredPackage "
Returns @c True if the given package extension is one of an ignored
packages, otherwise returns @c False.

An ignored package is one that is defined to be used in this SBML
document, but the package is not enabled in this copy of libSBML.

@param pkgURI the URI of the package extension.

@return a Boolean, @c True if the package is being ignored and
@c False otherwise.
";


%feature("docstring") SBMLDocument::isDisabledIgnoredPackage "
Returns @c True if the given package extension is one of an ignored
packages that has been disabled, otherwise returns @c False.

An ignored package is one that is defined to be used in this SBML
document, but the package is not enabled in this copy of libSBML.
It may have been disabled to avoid reproducing the package
information when writing out the file.

@param pkgURI the URI of the package extension.

@return a Boolean, @c True if the package is being ignored and
@c False otherwise.
";


%feature("docstring") SBMLDocument::setPkgRequired "
Sets the value of the <code>required</code> attribute for the given
package.

@param package the name or URI of the package extension.
@param flag a Boolean value.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION@endlink

@deprecated Replaced in libSBML 5.2.0 by
setPackageRequired(@if java String, boolean@endif)
";


%feature("docstring") SBMLDocument::getPkgRequired "
Returns the <code>required</code> attribute of the given package
extension.

@param package the name or URI of the package extension.

@return a Boolean value indicating whether the package is flagged as
being required in this SBML document.

@deprecated Replaced in libSBML 5.2.0 by
getPackageRequired(@if java String@endif)
";


%feature("docstring") SBMLDocument::isSetPkgRequired "
Returns @c True if the required attribute of the given package extension
is defined, otherwise returns @c False.

@param package the name or URI of the package extension.

@return a Boolean value.

@deprecated Replaced in libSBML 5.2.0 by
isSetPackageRequired(@if java String@endif)
";


%feature("docstring") SBMLDocument::isIgnoredPkg "
Returns @c True if the given package extension is one of ignored
packages, otherwise returns @c False.

An ignored package is one that is defined to be used in this SBML
document, but the package is not enabled in this copy of libSBML.

@param pkgURI the URI of the package extension.

@return a boolean indicating whether the given package is being ignored.

@deprecated Replaced in libSBML 5.2.0 by
isIgnoredPackage(@if java String@endif)
";


%feature("docstring") SBMLDocument::getElementPosition "
@internal

@internal
";


%feature("docstring") SBMLDocument::writeElements "
@internal

@internal
";


%feature("docstring") SBMLDocument::getApplicableValidators "
@internal

@internal
";


%feature("docstring") SBMLDocument::getConversionValidators "
@internal

@internal
";


%feature("docstring") SBMLDocument::setApplicableValidators "
@internal

@internal
";


%feature("docstring") SBMLDocument::setConversionValidators "
@internal

@internal
";


%feature("docstring") SBMLDocument::getNumValidators "
@internal

@internal
";


%feature("docstring") SBMLDocument::clearValidators "
@internal

@internal
";


%feature("docstring") SBMLDocument::addValidator "
@internal

@internal
";


%feature("docstring") SBMLDocument::getValidator "
@internal

@internal
";


%feature("docstring") SBMLDocument::addUnknownPackageRequired "
@internal

@internal
";


%feature("docstring") SBMLDocument::hasUnknownPackage "
@internal

@internal
";


%feature("docstring") SBMLDocument::getNumUnknownPackages "
@internal

@internal
";


%feature("docstring") SBMLDocument::getUnknownPackageURI "
@internal

@internal
";


%feature("docstring") SBMLDocument::getUnknownPackagePrefix "
@internal

@internal
";


%feature("docstring") SBMLDocument::createObject "
@internal

@internal
";


%feature("docstring") SBMLDocument::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") SBMLDocument::readAttributes "
@internal

@internal
";


%feature("docstring") SBMLDocument::writeAttributes "
@internal

@internal
";


%feature("docstring") SBMLDocument::writeXMLNS "
@internal

@internal
";


%feature("docstring") SBMLDocument::setInvalidLevel "
@internal

@internal
";


%feature("docstring") FunctionDefinition "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A user-defined function in an SBML model.

The FunctionDefinition structure associates an identifier with a
function definition.  This identifier can then be used as the function
called in subsequent MathML content elsewhere in an SBML model.

FunctionDefinition has one required attribute, \'id\', to give the
function a unique identifier by which other parts of an SBML model
definition can refer to it.  A FunctionDefinition instance can also have
an optional \'name\' attribute of type @c string.  Identifiers and names
must be used according to the guidelines described in the %SBML
specification (e.g., Section 3.3 in the Level 2 Version 4
specification).

FunctionDefinition has a \'math\' subelement containing a MathML
expression defining the function body.  In SBML Level&nbsp;2 and SBML
Level&nbsp;3 Version&nbsp;1, that \'math\' subelement is required;
in SBML Level&nbsp;3 Version&nbsp;2, this restriction was relaxed,
making the \'math\' subelement optional.  The content of this element can
only be a MathML \'lambda\' element.  The \'lambda\' element must begin with
zero or more \'bvar\' elements, followed by any other of the elements in
the MathML subset allowed in SBML Level 2 @em except \'lambda\' (i.e., a
\'lambda\' element cannot contain another \'lambda\' element).  This is the
only place in SBML where a \'lambda\' element can be used.  The function
defined by a FunctionDefinition is only available for use in other
MathML elements that @em follow the FunctionDefinition definition in the
model.  (These restrictions prevent recursive and mutually-recursive
functions from being expressed.)

A further restriction on the content of \'math\' is that it cannot contain
references to variables other than the variables declared to the
\'lambda\' itself.  That is, the contents of MathML \'ci\' elements inside
the body of the \'lambda\' can only be the variables declared by its
\'bvar\' elements, or the identifiers of other FunctionDefinition
instances in the model.  This means must be written so that all
variables or parameters used in the MathML content are passed to them
via their function parameters.  In SBML Level&nbsp;2, this restriction
applies also to the MathML @c csymbol elements for @em time and @em
delay; in SBML Level&nbsp;3, it additionally applies to the @c csymbol
element for @em avogadro.

In SBML Level&nbsp;3 Version&nbsp;2, if no math element is present in 
the FunctionDefinition, the function has no mathematical meaning 
defined in SBML Level&nbsp;3 Core. This situation may arise when models 
are incomplete, or when additional meanings are provided by an SBML 
Level&nbsp;3 package.

@note Function definitions (also informally known as user-defined
functions) were introduced in SBML Level 2.  They have purposefully
limited capabilities.  A function cannot reference parameters or other
model quantities outside of itself; values must be passed as parameters
to the function.  Moreover, recursive and mutually-recursive functions
are not permitted.  The purpose of these limitations is to balance power
against complexity of implementation.  With the restrictions as they
are, function definitions could be implemented as textual
substitutions---they are simply macros.  Software implementations
therefore do not need the full function-definition machinery typically
associated with programming languages.
<br><br>
Another important point to note is FunctionDefinition does not
have a separate attribute for defining the units of the value returned
by the function.  The units associated with the function\'s return value,
when the function is called from within MathML expressions elsewhere in
SBML, are simply the overall units of the expression in
FunctionDefinition\'s \'math\' subelement when applied to the arguments
supplied in the call to the function.  Ascertaining these units requires
performing dimensional analysis on the expression.  (Readers may wonder
why there is no attribute.  The reason is that having a separate
attribute for declaring the units would not only be redundant, but also
lead to the potential for having conflicting information.  In the case
of a conflict between the declared units and those of the value actually
returned by the function, the only logical resolution rule would be to
assume that the correct units are those of the expression anyway.)
";


%feature("docstring") ListOfFunctionDefinitions "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of FunctionDefinition objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") FunctionDefinition::FunctionDefinition "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>FunctionDefinition(SBMLNamespaces sbmlns)</pre>

Creates a new FunctionDefinition using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments. 

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>FunctionDefinition(long level, long version)</pre>

Creates a new FunctionDefinition using the given SBML @p level and @p version
values.

@param level a long integer, the SBML Level to assign to this FunctionDefinition.

@param version a long integer, the SBML Version to assign to this
FunctionDefinition.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>FunctionDefinition(FunctionDefinition orig)</pre>

Copy constructor; creates a copy of this FunctionDefinition.

@param orig the object to copy.
";


%feature("docstring") FunctionDefinition::accept "
@internal

@internal
";


%feature("docstring") FunctionDefinition::clone "
Creates and returns a deep copy of this FunctionDefinition object.

@return the (deep) copy of this FunctionDefinition object.
";


%feature("docstring") FunctionDefinition::getId "
Returns the value of the \'id\' attribute of this FunctionDefinition.

@note Because of the inconsistent behavior of this function with 
respect to assignments and rules, it is now recommended to
use the getIdAttribute() function instead.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this FunctionDefinition.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") FunctionDefinition::getName "
Returns the value of the \'name\' attribute of this FunctionDefinition object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") FunctionDefinition::getMath "
Get the mathematical formula of this FunctionDefinition.

@return an ASTNode, the value of the \'math\' subelement of this
FunctionDefinition, or @c None if the math is not set.
";


%feature("docstring") FunctionDefinition::isSetId "
Predicate returning @c True if this
FunctionDefinition\'s \'id\' attribute is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") FunctionDefinition::isSetName "
Predicate returning @c True if this
FunctionDefinition\'s \'name\' attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") FunctionDefinition::isSetMath "
Predicate returning @c True if this
FunctionDefinition\'s \'math\' subelement contains a value.

@return @c True if the \'math\' for this FunctionDefinition is set,
@c False otherwise.
";


%feature("docstring") FunctionDefinition::setId "
Sets the value of the \'id\' attribute of this FunctionDefinition.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") FunctionDefinition::setName "
Sets the value of the \'name\' attribute of this FunctionDefinition.

@par

The string in @p name is copied.

@param name the new name for the SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") FunctionDefinition::setMath "
Sets the \'math\' subelement of this FunctionDefinition to the Abstract
Syntax Tree given in @p math.

@param math an AST containing the mathematical expression to
be used as the formula for this FunctionDefinition.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
";


%feature("docstring") FunctionDefinition::unsetName "
Unsets the value of the \'name\' attribute of this FunctionDefinition.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") FunctionDefinition::getArgument "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getArgument(long n)</pre>

Get the <code>n</code>th argument to this function.

Callers should first find out the number of arguments to the function
by calling getNumArguments().

@param n an integer index for the argument sought.

@return the nth argument (bound variable) passed to this
FunctionDefinition.
If the index @p n is invalid, @c None is returned.

@see getNumArguments()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>getArgument(string name)</pre>

Get the argument named @p name to this FunctionDefinition.

@param name the exact name (case-sensitive) of the sought-after
argument.

@return the argument (bound variable) having the given name, or @c None if
no such argument exists.
";


%feature("docstring") FunctionDefinition::getBody "
Get the mathematical expression that is the body of this
FunctionDefinition object.

@return the body of this FunctionDefinition as an Abstract Syntax
Tree, or @c None if no body is defined.
";


%feature("docstring") FunctionDefinition::isSetBody "
Predicate returning @c True if the body of this
FunctionDefinition has set.

@return @c True if the body of this FunctionDefinition is 
set, @c False otherwise.
";


%feature("docstring") FunctionDefinition::getNumArguments "
Get the number of arguments (bound variables) taken by this
FunctionDefinition.

@return the number of arguments (bound variables) that must be passed
to this FunctionDefinition.
";


%feature("docstring") FunctionDefinition::getTypeCode "
Returns the libSBML type code for this %SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_FUNCTION_DEFINITION SBML_FUNCTION_DEFINITION@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getElementName()
@see getPackageName()
";


%feature("docstring") FunctionDefinition::getElementName "
Returns the XML element name of this object, which for
FunctionDefinition, is always @c \'functionDefinition\'.

@return the name of this element, i.e., @c \'functionDefinition\'.
";


%feature("docstring") FunctionDefinition::writeElements "
@internal

@internal
";


%feature("docstring") FunctionDefinition::hasRequiredAttributes "
Predicate returning @c True if
all the required attributes for this FunctionDefinition object
have been set.

The required attributes for a FunctionDefinition object are:
@li \'id\'

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") FunctionDefinition::hasRequiredElements "
Predicate returning @c True if
all the required elements for this FunctionDefinition object
have been set.

@note The required elements for a FunctionDefinition object are:
@li \'math\' inSBML Level&nbsp;2 and Level&nbsp;3 Version&nbsp;1.  
(In SBML Level&nbsp;3 Version&nbsp;2+, it is no longer required.)

@return a boolean value indicating whether all the required
elements for this object have been defined.
";


%feature("docstring") FunctionDefinition::renameUnitSIdRefs "
Renames all the @c UnitSIdRef attributes on this element.

@par
In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
introduced for attribute values that refer to <code>UnitSId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to a unit identifier\', but the effective
data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all unit identifier attribute values
(including, if appropriate, inside mathematical formulas), comparing the
unit identifiers to the value of @p oldid.  If any matches are found,
the matching identifiers are replaced with @p newid.  The method does
@em not descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") FunctionDefinition::readOtherXML "
@internal

@internal
";


%feature("docstring") FunctionDefinition::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") FunctionDefinition::readAttributes "
@internal

@internal
";


%feature("docstring") FunctionDefinition::readL2Attributes "
@internal

@internal
";


%feature("docstring") FunctionDefinition::readL3Attributes "
@internal

@internal
";


%feature("docstring") FunctionDefinition::writeAttributes "
@internal

@internal
";


%feature("docstring") ListOfFunctionDefinitions::ListOfFunctionDefinitions "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfFunctionDefinitions(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfFunctionDefinitions object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfFunctionDefinitions object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfFunctionDefinitions(long level, long version)</pre>

Creates a new ListOfFunctionDefinitions object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfFunctionDefinitions::clone "
Creates and returns a deep copy of this ListOfFunctionDefinitions object.

@return the (deep) copy of this ListOfFunctionDefinitions object.
";


%feature("docstring") ListOfFunctionDefinitions::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., FunctionDefinition objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for the objects contained in this ListOf:
@link libsbml#SBML_FUNCTION_DEFINITION SBML_FUNCTION_DEFINITION@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfFunctionDefinitions::getElementName "
Returns the XML element name of this object.

For ListOfFunctionDefinitions, the XML element name is
@c \'listOfFunctionDefinitions\'.

@return the name of this element, i.e., @c \'listOfFunctionDefinitions\'.
";


%feature("docstring") ListOfFunctionDefinitions::get "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(long n)</pre>

Get a FunctionDefinition from the ListOfFunctionDefinitions.

@param n the index number of the FunctionDefinition to get.

@return the nth FunctionDefinition in this ListOfFunctionDefinitions.
If the index @p n is invalid, @c None is returned.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(string sid)</pre>

Get a FunctionDefinition from the ListOfFunctionDefinitions
based on its identifier.

@param sid a string representing the identifier 
of the FunctionDefinition to get.

@return FunctionDefinition in this ListOfFunctionDefinitions
with the given @p sid or @c None if no such
FunctionDefinition exists.

@see get()
@see size()
";


%feature("docstring") ListOfFunctionDefinitions::remove "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(long n)</pre>

Removes the nth item from this ListOfFunctionDefinitions items and returns a pointer to
it.

The caller owns the returned item and is responsible for deleting it.

@param n the index of the item to remove.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(string sid)</pre>

Removes item in this ListOfFunctionDefinitions items with the given identifier.

The caller owns the returned item and is responsible for deleting it.
If none of the items in this list have the identifier @p sid, then
@c None is returned.

@param sid the identifier of the item to remove.

@return the item removed.  As mentioned above, the caller owns the
returned item.
";


%feature("docstring") ListOfFunctionDefinitions::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfFunctionDefinitions::createObject "
@internal

@internal
";


%feature("docstring") UnitKind_equals "
Tests for logical equality between two given <code>UNIT_KIND_</code>
code values.

This function behaves exactly like C\'s <code>==</code> operator, except
for the following two cases:
<ul>
<li>@link libsbml#UNIT_KIND_LITER UNIT_KIND_LITER@endlink <code>==</code> @link libsbml#UNIT_KIND_LITRE UNIT_KIND_LITRE@endlink
<li>@link libsbml#UNIT_KIND_METER UNIT_KIND_METER@endlink <code>==</code> @link libsbml#UNIT_KIND_METRE UNIT_KIND_METRE@endlink
</ul>

In the two cases above, C equality comparison would yield @c 0 (false)
(because each of the above is a distinct enumeration value), but
this function returns @c True.

@param uk1 a <code>UNIT_KIND_</code> value.
@param uk2 a second <code>UNIT_KIND_</code> value to compare to @p uk1.

@return @c 1 (true) if @p uk1 is logically equivalent to @p
uk2, @c 0 (false) otherwise.

@note For more information about the libSBML unit codes, please refer to
the class documentation for Unit.

@if conly
@memberof Unit_t
@endif
";


%feature("docstring") UnitKind_forName "
Converts a text string naming a kind of unit to its corresponding
libSBML <code>UNIT_KIND_</code> constant/enumeration value.

@param name a string, the name of a predefined base unit in SBML.

@return @if clike a value from UnitKind_t corresponding to the given
string @p name (determined in a case-insensitive manner).
@endif@if python a value the set of <code>UNIT_KIND_</code> codes
defined in class @link libsbml libsbml@endlink, corresponding to the
string @p name (determined in a case-insensitive
manner).@endif@if java a value the set of <code>UNIT_KIND_</code> codes
defined in class {@link libsbmlConstants}, corresponding to the string
@p name (determined in a case-insensitive manner).@endif@~

@note For more information about the libSBML unit codes, please refer to
the class documentation for Unit.

@if conly
@memberof Unit_t
@endif
";


%feature("docstring") UnitKind_toString "
Converts a unit code to a text string equivalent.

@param uk @if clike a value from the UnitKind_t enumeration
@endif@if python a value from the set of <code>UNIT_KIND_</code> codes
defined in the class @link libsbml libsbml@endlink
@endif@if java a value from the set of <code>UNIT_KIND_</code> codes
defined in the class {@link libsbmlConstants}
@endif@~

@return the name corresponding to the given unit code.

@note For more information about the libSBML unit codes, please refer to
the class documentation for Unit.

@warning The string returned is a static data value.  The caller does not
own the returned string and is therefore not allowed to modify it.

@if conly
@memberof Unit_t
@endif
";


%feature("docstring") UnitKind_isValidUnitKindString "
Predicate for testing whether a given string corresponds to a
predefined libSBML unit code.

@param str a text string naming a base unit defined by SBML.
@param level the Level of SBML.
@param version the Version within the Level of SBML.

@return 1 (true) if string is the name of a valid
<code>UNIT_KIND_</code> value, @c 0 (false) otherwise.

@note For more information about the libSBML unit codes, please refer to
the class documentation for Unit.

@if conly
@memberof Unit_t
@endif
";


%feature("docstring") Unit "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A single unit referenced in an SBML <em>unit definition</em>.

The SBML unit definition facility uses two classes of objects,
UnitDefinition and Unit.  The approach to defining units in %SBML is
compositional; for example, <em>meter second<sup> &ndash;2</sup></em> is
constructed by combining a Unit object representing <em>meter</em> with
another Unit object representing <em>second<sup> &ndash;2</sup></em>.
The combination is wrapped inside a UnitDefinition, which provides for
assigning an identifier and optional name to the combination.  The
identifier can then be referenced from elsewhere in a model.  Thus, the
UnitDefinition class is the container, and Unit instances are placed
inside UnitDefinition instances.

A Unit has four attributes named \'kind\', \'exponent\', \'scale\'
and \'multiplier\'.  It represents a (possibly transformed) reference to a
base unit.  The attribute \'kind\' on Unit indicates the chosen base unit.
Its value must be one of the text strings listed below; this list
corresponds to SBML Level&nbsp;3:

@par
<table border=\'0\' class=\'centered text-table width80 normal-font code\'
       style=\'border: none !important\'>
<tr>
<td>ampere</td><td>farad</td><td>joule</td><td>lux</td><td>radian</td><td>volt</td>
</tr>
<tr>
<td>avogadro</td><td>gram</td><td>katal</td><td>metre</td><td>second</td><td>watt</td>
</tr>
<tr>
<td>becquerel</td><td>gray</td><td>kelvin</td><td>mole</td><td>siemens</td><td>weber</td>
</tr>
<tr>
<td>candela</td><td>henry</td><td>kilogram</td><td>newton</td><td>sievert</td>
</tr>
<tr>
<td>coulomb</td><td>hertz</td><td>litre</td><td>ohm</td><td>steradian</td>
</tr>
<tr>
<td>dimensionless</td><td>item</td><td>lumen</td><td>pascal</td><td>tesla</td>
</tr>
</table>

A few small differences exist between the Level&nbsp;3 list of base
units and the list defined in other Level/Version combinations of SBML.
Specifically, Levels of SBML before Level&nbsp;3 do not define
@c avogadro; conversely, Level&nbsp;2 Version&nbsp;1 defines @c Celsius,
and Level&nbsp;1 defines @c celsius, @c meter, and @c liter, none of
which are available in Level&nbsp;3.  In libSBML, each of the predefined
base unit names is represented by an enumeration value @if clike in
#UnitKind_t@else whose name begins with the characters
<code>UNIT_KIND_</code>@endif, discussed in a separate section below.

The attribute named \'exponent\' on Unit represents an exponent on the
unit.  In SBML Level&nbsp;2, the attribute is optional and has a default
value of @c 1 (one); in SBML Level&nbsp;3, the attribute is mandatory
and there is no default value.  A Unit also has an attribute
called \'scale\'; its value must be an integer exponent for a power-of-ten
multiplier used to set the scale of the unit.  For example, a unit
having a \'kind\' value of @c gram and a \'scale\' value of @c -3 signifies
10<sup>&nbsp;&ndash;3</sup> * gram, or milligrams.  In SBML
Level&nbsp;2, the attribute is optional and has a default value of @c 0
(zero), because 10<sup> 0</sup> = 1; in SBML Level&nbsp;3, the attribute
is mandatory and has no default value.  Lastly, the attribute named
\'multiplier\' can be used to multiply the unit by a real-numbered factor;
this enables the definition of units that are not power-of-ten multiples
of SI units.  For instance, a multiplier of 0.3048 could be used to
define @c foot as a measure of length in terms of a @c metre.  The
\'multiplier\' attribute is optional in SBML Level&nbsp;2, where it has a
default value of @c 1 (one); in SBML Level&nbsp;3, the attribute is
mandatory and has no default value.

@if clike
<h3><a class=\'anchor\' name=\'UnitKind_t\'>UnitKind_t</a></h3>
@else
<h3><a class=\'anchor\' name=\'UnitKind_t\'>%Unit identification codes</a></h3>
@endif@~

As discussed above, SBML defines a set of base units which serves as the
starting point for new unit definitions.  This set of base units
consists of the SI units and a small number of additional convenience
units.

@if clike Until SBML Level&nbsp;2 Version&nbsp;3, there
existed a data type in the SBML specifications called @c UnitKind,
enumerating the possible SBML base units.  Although SBML Level&nbsp;2
Version&nbsp;3 removed this type from the language specification,
libSBML maintains the corresponding enumeration type #UnitKind_t as a
convenience and as a way to provide backward compatibility to previous
SBML Level/Version specifications.  (The removal in SBML Level&nbsp;2
Version&nbsp;3 of the enumeration @c UnitKind was also accompanied by
the redefinition of the data type @c UnitSId to include the previous
@c UnitKind values as reserved symbols in the @c UnitSId space.  This
change has no net effect on permissible models, their representation or
their syntax.  The purpose of the change in the SBML specification was
simply to clean up an inconsistency about the contexts in which these
values were usable.)
@endif@if java In SBML Level&nbsp;2 Versions before
Version&nbsp;3, there existed an enumeration of units called
@c UnitKind.  In Version&nbsp;3, this enumeration was removed and the
identifier class @c UnitSId redefined to include the previous
@c UnitKind values as reserved symbols.  This change has no net effect on
permissible models, their representation or their syntax.  The purpose
of the change in the SBML specification was simply to clean up an
inconsistency about the contexts in which these values were usable.
However, libSBML maintains UnitKind in the form of of a set of static
integer constants whose names begin with the characters
<code>UNIT_KIND_</code>.  These constants are defined in the class
<code><a href=\'libsbmlConstants.html\'>libsbmlConstants</a></code>.
@endif@if python In SBML Level&nbsp;2 Versions before
Version&nbsp;3, there existed an enumeration of units called
@c UnitKind.  In Version&nbsp;3, this enumeration was removed and the
identifier class @c UnitSId redefined to include the previous
@c UnitKind values as reserved symbols.  This change has no net effect on
permissible models, their representation or their syntax.  The purpose
of the change in the SBML specification was simply to clean up an
inconsistency about the contexts in which these values were usable.
However, libSBML maintains UnitKind in the form of of a set of static
integer constants whose names begin with the characters
<code>UNIT_KIND_</code>.  These constants are defined in the class
@link libsbml libsbml@endlink.
@endif@~

As a consequence of the fact that libSBML supports models in all Levels
and Versions of SBML, libSBML\'s set of @c UNIT_KIND_ values is a union
of all the possible base unit names defined in the different SBML
specifications.  However, not every base unit is allowed in every
Level+Version combination of SBML.  Note in particular the following
exceptions:
<ul>
<li> The alternate spelling @c \'meter\' is included in
addition to the official SI spelling @c \'metre\'.  This spelling is only
permitted in SBML Level&nbsp;1 models.

<li> The alternate spelling @c \'liter\' is included in addition to the
official SI spelling @c \'litre\'.  This spelling is only permitted in
SBML Level&nbsp;1 models.

<li> The unit @c \'Celsius\' is included because of its presence in
specifications of SBML prior to SBML Level&nbsp;2 Version&nbsp;2.

<li> The unit @c avogadro was introduced in SBML Level&nbsp;3, and
is only permitted for use in SBML Level&nbsp;3 models.
</ul>

@if clike The table below lists the symbols defined in the
@c UnitKind_t enumeration, and their
meanings. @else The table below lists the unit
constants defined in libSBML, and their meanings. @endif@~

@htmlinclude unitkind-table.html
";


%feature("docstring") ListOfUnits "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of Unit objects.

ListOfUnits is entirely contained within UnitDefinition.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") Unit::Unit "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Unit(SBMLNamespaces sbmlns)</pre>

Creates a new Unit using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments.

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Unit(long level, long version)</pre>

Creates a new Unit using the given SBML @p level and @p version
values.

@param level a long integer, the SBML Level to assign to this Unit.

@param version a long integer, the SBML Version to assign to this
Unit.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Unit(Unit orig)</pre>

Copy constructor; creates a copy of this Unit.

@param orig the object to copy.
";


%feature("docstring") Unit::accept "
@internal

@internal
";


%feature("docstring") Unit::clone "
Creates and returns a deep copy of this Unit object.

@return the (deep) copy of this Unit object.
";


%feature("docstring") Unit::initDefaults "
Initializes the fields of this Unit object to \'typical\' default
values.

The SBML Unit component has slightly different aspects and default
attribute values in different SBML Levels and Versions.  This method
sets the values to certain common defaults, based mostly on what they
are in SBML Level&nbsp;2.  Specifically:
<ul>
<li> Sets attribute \'exponent\' to @c 1
<li> Sets attribute \'scale\' to @c 0
<li> Sets attribute \'multiplier\' to @c 1.0
</ul>

The \'kind\' attribute is left unchanged.
";


%feature("docstring") Unit::getKind "
Returns the \'kind\' of Unit this is.

@if clike
@return the value of the \'kind\' attribute of this Unit as a
value from the <a class=\'el\' href=\'#UnitKind_t\'>UnitKind_t</a> enumeration.
@endif@if java
@return the value of the \'kind\' attribute of this Unit as a
value from the set of constants whose names begin
with <code>UNIT_KIND_</code> defined in the class
<code><a href=\'libsbmlConstants.html\'>libsbmlConstants</a></code>.
@endif@if python
@return the value of the \'kind\' attribute of this Unit as a
value from the set of constants whose names begin
with <code>UNIT_KIND_</code> defined in the class
@link libsbml libsbml@endlink.
@endif@~
";


%feature("docstring") Unit::getExponent "
Returns the value of the \'exponent\' attribute of this unit.

@return the \'exponent\' value of this Unit, as an integer.
";


%feature("docstring") Unit::getExponentAsDouble "
Returns the value of the \'exponent\' attribute of this unit.

@return the \'exponent\' value of this Unit, as a double.
";


%feature("docstring") Unit::getScale "
Returns the value of the \'scale\' attribute of this unit.

@return the \'scale\' value of this Unit, as an integer.
";


%feature("docstring") Unit::getMultiplier "
Returns the value of the \'multiplier\' attribute of this Unit.

@return the \'multiplier\' value of this Unit, as a double.
";


%feature("docstring") Unit::getOffset "
Returns the value of the \'offset\' attribute of this Unit.

@return the \'offset\' value of this Unit, as a double.

@warning <span class=\'warning\'>The \'offset\' attribute is only available in
SBML Level&nbsp;2 Version&nbsp;1.  This attribute is not present in SBML
Level&nbsp;2 Version&nbsp;2 or above.  When producing SBML models using
these later specifications, modelers and software tools need to account
for units with offsets explicitly.  The %SBML specification document
offers a number of suggestions for how to achieve this.  LibSBML methods
such as this one related to \'offset\' are retained for compatibility with
earlier versions of SBML Level&nbsp;2, but their use is strongly
discouraged.</span>
";


%feature("docstring") Unit::isAmpere "
Predicate for testing whether this Unit is of the kind @c ampere.

@return @c True if the kind of this Unit is @c ampere, @c False
otherwise.
";


%feature("docstring") Unit::isAvogadro "
Predicate for testing whether this Unit is of the kind @c avogadro.

@return @c True if the kind of this Unit is @c avogadro, @c False
otherwise.

@note The unit @c avogadro was introduced in SBML Level&nbsp;3, and
is only permitted for use in SBML Level&nbsp;3 models.
";


%feature("docstring") Unit::isBecquerel "
Predicate for testing whether this Unit is of the kind @c becquerel

@return @c True if the kind of this Unit is @c becquerel, @c False
otherwise.
";


%feature("docstring") Unit::isCandela "
Predicate for testing whether this Unit is of the kind @c candela

@return @c True if the kind of this Unit is @c candela, @c False
otherwise.
";


%feature("docstring") Unit::isCelsius "
Predicate for testing whether this Unit is of the kind @c Celsius

@return @c True if the kind of this Unit is @c Celsius, @c False
otherwise. 

@warning <span class=\'warning\'>The predefined unit @c Celsius was
removed from the list of predefined units in SBML Level&nbsp;2
Version&nbsp;2 at the same time that the \'offset\' attribute was removed
from Unit definitions.  LibSBML methods such as this one related to
@c Celsius are retained in order to support SBML Level&nbsp;2
Version&nbsp;1, but their use is strongly discouraged.</span>
";


%feature("docstring") Unit::isCoulomb "
Predicate for testing whether this Unit is of the kind @c coulomb

@return @c True if the kind of this Unit is @c coulomb, @c False
otherwise.
";


%feature("docstring") Unit::isDimensionless "
Predicate for testing whether this Unit is of the kind
@c dimensionless.

@return @c True if the kind of this Unit is @c dimensionless, @c False
otherwise.
";


%feature("docstring") Unit::isFarad "
Predicate for testing whether this Unit is of the kind @c farad

@return @c True if the kind of this Unit is @c farad, @c False
otherwise.
";


%feature("docstring") Unit::isGram "
Predicate for testing whether this Unit is of the kind @c gram

@return @c True if the kind of this Unit is @c gram, @c False
otherwise.
";


%feature("docstring") Unit::isGray "
Predicate for testing whether this Unit is of the kind @c gray

@return @c True if the kind of this Unit is @c gray, @c False
otherwise.
";


%feature("docstring") Unit::isHenry "
Predicate for testing whether this Unit is of the kind @c henry

@return @c True if the kind of this Unit is @c henry, @c False
otherwise.
";


%feature("docstring") Unit::isHertz "
Predicate for testing whether this Unit is of the kind @c hertz

@return @c True if the kind of this Unit is @c hertz, @c False
otherwise.
";


%feature("docstring") Unit::isItem "
Predicate for testing whether this Unit is of the kind @c item

@return @c True if the kind of this Unit is @c item, @c False
otherwise.
";


%feature("docstring") Unit::isJoule "
Predicate for testing whether this Unit is of the kind @c joule

@return @c True if the kind of this Unit is @c joule, @c False
otherwise.
";


%feature("docstring") Unit::isKatal "
Predicate for testing whether this Unit is of the kind @c katal

@return @c True if the kind of this Unit is @c katal, @c False
otherwise.
";


%feature("docstring") Unit::isKelvin "
Predicate for testing whether this Unit is of the kind @c kelvin

@return @c True if the kind of this Unit is @c kelvin, @c False
otherwise.
";


%feature("docstring") Unit::isKilogram "
Predicate for testing whether this Unit is of the kind @c kilogram

@return @c True if the kind of this Unit is @c kilogram, @c False
otherwise.
";


%feature("docstring") Unit::isLitre "
Predicate for testing whether this Unit is of the kind @c litre

@return @c True if the kind of this Unit is @c litre or \'liter\',
@c False otherwise.
";


%feature("docstring") Unit::isLumen "
Predicate for testing whether this Unit is of the kind @c lumen

@return @c True if the kind of this Unit is @c lumen, @c False
otherwise.
";


%feature("docstring") Unit::isLux "
Predicate for testing whether this Unit is of the kind @c lux

@return @c True if the kind of this Unit is @c lux, @c False
otherwise.
";


%feature("docstring") Unit::isMetre "
Predicate for testing whether this Unit is of the kind @c metre

@return @c True if the kind of this Unit is @c metre or \'meter\',
@c False 
otherwise.
";


%feature("docstring") Unit::isMole "
Predicate for testing whether this Unit is of the kind @c mole

@return @c True if the kind of this Unit is @c mole, @c False
otherwise.
";


%feature("docstring") Unit::isNewton "
Predicate for testing whether this Unit is of the kind @c newton

@return @c True if the kind of this Unit is @c newton, @c False
otherwise.
";


%feature("docstring") Unit::isOhm "
Predicate for testing whether this Unit is of the kind @c ohm

@return @c True if the kind of this Unit is @c ohm, @c False
otherwise.
";


%feature("docstring") Unit::isPascal "
Predicate for testing whether this Unit is of the kind @c pascal

@return @c True if the kind of this Unit is @c pascal, @c False
otherwise.
";


%feature("docstring") Unit::isRadian "
Predicate for testing whether this Unit is of the kind @c radian

@return @c True if the kind of this Unit is @c radian, @c False
otherwise.
";


%feature("docstring") Unit::isSecond "
Predicate for testing whether this Unit is of the kind @c second

@return @c True if the kind of this Unit is @c second, @c False
otherwise.
";


%feature("docstring") Unit::isSiemens "
Predicate for testing whether this Unit is of the kind @c siemens

@return @c True if the kind of this Unit is @c siemens, @c False
otherwise.
";


%feature("docstring") Unit::isSievert "
Predicate for testing whether this Unit is of the kind @c sievert

@return @c True if the kind of this Unit is @c sievert, @c False
otherwise.
";


%feature("docstring") Unit::isSteradian "
Predicate for testing whether this Unit is of the kind @c steradian

@return @c True if the kind of this Unit is @c steradian, @c False
otherwise.
";


%feature("docstring") Unit::isTesla "
Predicate for testing whether this Unit is of the kind @c tesla

@return @c True if the kind of this Unit is @c tesla, @c False
otherwise.
";


%feature("docstring") Unit::isVolt "
Predicate for testing whether this Unit is of the kind @c volt

@return @c True if the kind of this Unit is @c volt, @c False
otherwise.
";


%feature("docstring") Unit::isWatt "
Predicate for testing whether this Unit is of the kind @c watt

@return @c True if the kind of this Unit is @c watt, @c False
otherwise.
";


%feature("docstring") Unit::isWeber "
Predicate for testing whether this Unit is of the kind @c weber

@return @c True if the kind of this Unit is @c weber, @c False
otherwise.
";


%feature("docstring") Unit::isSetKind "
Predicate to test whether the \'kind\' attribute of this Unit is set.

@return @c True if the \'kind\' attribute of this Unit is set,
@c False otherwise.
";


%feature("docstring") Unit::isSetExponent "
Predicate to test whether the \'exponent\' attribute of this Unit 
is set.

@return @c True if the \'exponent\' attribute of this Unit is set, 
@c False otherwise.
";


%feature("docstring") Unit::isSetScale "
Predicate to test whether the \'scale\' attribute of this Unit 
is set.

@return @c True if the \'scale\' attribute of this Unit is set, 
@c False otherwise.
";


%feature("docstring") Unit::isSetMultiplier "
Predicate to test whether the \'multiplier\' attribute of this Unit 
is set.

@return @c True if the \'multiplier\' attribute of this Unit is set, 
@c False otherwise.
";


%feature("docstring") Unit::isSetOffset "
Predicate to test whether the \'offset\' attribute of this Unit 
is set.

@return @c True if the \'offset\' attribute of this Unit is set, 
@c False otherwise.

@warning <span class=\'warning\'>The \'offset\' attribute is only available in
SBML Level&nbsp;2 Version&nbsp;1.  This attribute is not present in SBML
Level&nbsp;2 Version&nbsp;2 or above.  When producing SBML models using
these later specifications, modelers and software tools need to account
for units with offsets explicitly.  The %SBML specification document
offers a number of suggestions for how to achieve this.  LibSBML methods
such as this one related to \'offset\' are retained for compatibility with
earlier versions of SBML Level&nbsp;2, but their use is strongly
discouraged.</span>
";


%feature("docstring") Unit::setKind "
Sets the \'kind\' attribute value of this Unit.

@if clike
@param kind a value from the <a class=\'el\'
href=\'#UnitKind_t\'>UnitKind_t</a> enumeration.
@endif@if java
@param kind a unit identifier chosen from the set of constants whose
names begin with <code>UNIT_KIND_</code> in <code><a
href=\'libsbmlConstants.html\'>libsbmlConstants</a></code>.
@endif@if python
@param kind a unit identifier chosen from the set of constants whose
names begin with <code>UNIT_KIND_</code> in @link libsbml libsbml@endlink.
@endif@~

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Unit::setExponent "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setExponent(int value)</pre>

Sets the \'exponent\' attribute value of this Unit.

@param value the integer to which the attribute \'exponent\' should be set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setExponent(double value)</pre>

Sets the \'exponent\' attribute value of this Unit.

@param value the double to which the attribute \'exponent\' should be set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
";


%feature("docstring") Unit::setScale "
Sets the \'scale\' attribute value of this Unit.

@param value the integer to which the attribute \'scale\' should be set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
";


%feature("docstring") Unit::setMultiplier "
Sets the \'multipler\' attribute value of this Unit.

@param value the floating-point value to which the attribute
\'multiplier\' should be set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
";


%feature("docstring") Unit::setOffset "
Sets the \'offset\' attribute value of this Unit.

@param value the floating point value to which the attribute \'offset\'
should set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@warning <span class=\'warning\'>The \'offset\' attribute is only available in
SBML Level&nbsp;2 Version&nbsp;1.  This attribute is not present in SBML
Level&nbsp;2 Version&nbsp;2 or above.  When producing SBML models using
these later specifications, modelers and software tools need to account
for units with offsets explicitly.  The %SBML specification document
offers a number of suggestions for how to achieve this.  LibSBML methods
such as this one related to \'offset\' are retained for compatibility with
earlier versions of SBML Level&nbsp;2, but their use is strongly
discouraged.</span>
";


%feature("docstring") Unit::unsetKind "
Unsets the \'kind\' attribute value of this Unit.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Unit::unsetExponent "
Unsets the \'exponent\' attribute value of this Unit.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Unit::unsetScale "
Unsets the \'scale\' attribute value of this Unit.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
";


%feature("docstring") Unit::unsetMultiplier "
Unsets the \'multipler\' attribute value of this Unit.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
";


%feature("docstring") Unit::unsetOffset "
Unsets the \'offset\' attribute value of this Unit.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@warning <span class=\'warning\'>The \'offset\' attribute is only available in
SBML Level&nbsp;2 Version&nbsp;1.  This attribute is not present in SBML
Level&nbsp;2 Version&nbsp;2 or above.  When producing SBML models using
these later specifications, modelers and software tools need to account
for units with offsets explicitly.  The %SBML specification document
offers a number of suggestions for how to achieve this.  LibSBML methods
such as this one related to \'offset\' are retained for compatibility with
earlier versions of SBML Level&nbsp;2, but their use is strongly
discouraged.</span>
";


%feature("docstring") Unit::getTypeCode "
Returns the libSBML type code of this object instance.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_UNIT SBML_UNIT@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getPackageName()
@see getElementName()
";


%feature("docstring") Unit::getElementName "
Returns the XML element name of this object, which for Unit, is
always @c \'unit\'.

@return the name of this element, i.e., @c \'unit\'.
";


%feature("docstring") Unit::writeElements "
@internal

@internal
";


%feature("docstring") Unit::isBuiltIn "
Predicate to test whether a given string is the name of a built-in SBML
unit, depending on the SBML level, since new predefined units were added
between level 2 versions 1 and 2, and then all predefined units were removed
again in SBML Level 3.

@param name a string to be tested against the built-in unit names.
@param level the level of SBML one is checking.

@return @c True if @p name is one of @c \'substance\', @c \'volume\',
or @c \'time\' and the @p level is @c 1; or if @p name is one of
@c \'substance\', @c \'volume\', @c \'area\', @c \'length\', or @c \'time\' and
the @p level is @c 2; @c False otherwise (including all values when
@p level is @c 3).


@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~
";


%feature("docstring") Unit::isUnitKind "
Predicate to test whether a given string is the name of a valid
base unit in SBML (such as @c \'gram\' or @c \'mole\').

This method exists because prior to SBML Level&nbsp;2 Version&nbsp;3,
an enumeration called @c UnitKind was defined by SBML.  This enumeration
was removed in SBML Level&nbsp;2 Version&nbsp;3 and its values were
folded into the space of values of a type called @c UnitSId.  This method
therefore has less significance in SBML Level&nbsp;2 Version&nbsp;3
and Level&nbsp;2 Version&nbsp;4, but remains for backward
compatibility and support for reading models in older Versions of
Level&nbsp;2.

@param name a string to be tested.

@param level a long integer representing the SBML specification
Level.

@param version a long integer representing the SBML specification
Version.

@return @c True if name is a valid SBML UnitKind, @c False otherwise.

@note The allowed unit names differ between SBML Levels&nbsp;1
and&nbsp;2 and again slightly between Level&nbsp;2 Versions&nbsp;1
and&nbsp;2.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~
";


%feature("docstring") Unit::areIdentical "
Predicate returning @c True if two
Unit objects are identical.

Two Unit objects are considered to be @em identical if they match in
all attributes.  (Contrast this to the method areEquivalent(@if java
Unit u1, %Unit u2@endif), which compares Unit objects only with respect
to certain attributes.)

@param unit1 the first Unit object to compare.
@param unit2 the second Unit object to compare.

@return @c True if all the attributes of unit1 are identical
to the attributes of unit2, @c False otherwise.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike areEquivalent() @else Unit.areEquivalent() @endif@~
";


%feature("docstring") Unit::areEquivalent "
Predicate returning @c True if 
Unit objects are equivalent.

Two Unit objects are considered to be @em equivalent either if (1) both
have a \'kind\' attribute value of @c dimensionless, or (2) their \'kind\',
\'exponent\' and (for SBML Level&nbsp;2 Version&nbsp;1) \'offset\'
attribute values are equal. (Contrast this to the method
areIdentical(@if java Unit, %Unit@endif), which compares Unit objects with respect to all
attributes, not just the \'kind\' and \'exponent\'.)

@param unit1 the first Unit object to compare.
@param unit2 the second Unit object to compare.

@return @c True if the \'kind\' and \'exponent\' attributes of unit1 are
identical to the kind and exponent attributes of unit2, or if the kind
attributes of both are @c dimensionless; @c False otherwise.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike areIdentical() @else Unit.areIdentical() @endif@~
";


%feature("docstring") Unit::removeScale "
Manipulates the attributes of the Unit to express the unit with the 
value of the scale attribute reduced to zero.

For example, 1 millimetre can be expressed as a Unit with kind=
@c \'metre\' multiplier=@c \'1\' scale=@c \'-3\' exponent=@c \'1\'. It can also be
expressed as a Unit with kind=@c \'metre\'
multiplier=<code>\'0.001\'</code> scale=@c \'0\' exponent=@c \'1\'.

@param unit the Unit object to manipulate.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike convertToSI() @else Unit.convertToSI() @endif@~
@see @if clike merge() @else Unit.merge() @endif@~
";


%feature("docstring") Unit::merge "
Merges two Unit objects with the same \'kind\' attribute value into a
single Unit.

For example, the following,
@verbatim
 <unit kind=\'metre\' exponent=\'2\'/>
 <unit kind=\'metre\' exponent=\'1\'/>
 @endverbatim
would be merged to become
@verbatim
 <unit kind=\'metre\' exponent=\'3\'/>
 @endverbatim

@param unit1 the first Unit object; the result of the operation is
left as a new version of this unit, modified in-place.  Not modified if
the two units have different kinds.

@param unit2 the second Unit object to merge with the first.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike convertToSI() @else Unit.convertToSI() @endif@~
@see @if clike removeScale() @else Unit.removeScale() @endif@~
";


%feature("docstring") Unit::convertToSI "
Returns a UnitDefinition object containing the given @p unit converted
to the appropriate SI unit.

This method exists because some units can be expressed in terms of
others when the same physical dimension is involved.  For example, one
hertz is identical to 1&nbsp;sec<sup>-1</sup>, one litre is equivalent
to 1 cubic decametre, and so on.

@param unit the Unit object to convert to SI.

@return a UnitDefinition object containing the SI unit.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see @if clike merge() @else Unit.merge() @endif@~
";


%feature("docstring") Unit::hasRequiredAttributes "
Predicate returning @c True if
all the required attributes for this Unit object
have been set.

The required attributes for a Unit object are:
@li \'kind\'
@li \'exponent\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
@li \'multiplier\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
@li \'scale\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") Unit::setExponentUnitChecking "
@internal

@internal
";


%feature("docstring") Unit::getExponentUnitChecking "
@internal

@internal
";


%feature("docstring") Unit::isUnitChecking "
@internal

@internal
";


%feature("docstring") Unit::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") Unit::readAttributes "
@internal

@internal
";


%feature("docstring") Unit::readL1Attributes "
@internal

@internal
";


%feature("docstring") Unit::readL2Attributes "
@internal

@internal
";


%feature("docstring") Unit::readL3Attributes "
@internal

@internal
";


%feature("docstring") Unit::writeAttributes "
@internal

@internal
";


%feature("docstring") Unit::isL1UnitKind "
@internal

@internal
";


%feature("docstring") Unit::isL2V1UnitKind "
@internal

@internal
";


%feature("docstring") Unit::isL2UnitKind "
@internal

@internal
";


%feature("docstring") Unit::isL3UnitKind "
@internal

@internal
";


%feature("docstring") Unit::isExplicitlySetExponent "
@internal

@internal
";


%feature("docstring") Unit::isExplicitlySetMultiplier "
@internal

@internal
";


%feature("docstring") Unit::isExplicitlySetScale "
@internal

@internal
";


%feature("docstring") Unit::isExplicitlySetOffset "
@internal

@internal
";


%feature("docstring") ListOfUnits::ListOfUnits "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfUnits(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfUnits object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfUnits object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfUnits(long level, long version)</pre>

Creates a new ListOfUnits object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfUnits::clone "
Creates and returns a deep copy of this ListOfUnits object.

@return the (deep) copy of this ListOfUnits object.
";


%feature("docstring") ListOfUnits::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., Unit objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for objects contained in this list:
@link libsbml#SBML_UNIT SBML_UNIT@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfUnits::getElementName "
Returns the XML element name of this object.

For ListOfUnits, the XML element name is @c \'listOfUnits\'.

@return the name of this element, i.e., @c \'listOfUnits\'.
";


%feature("docstring") ListOfUnits::get "
Get a Unit from the ListOfUnits.

@param n the index number of the Unit to get.

@return the nth Unit in this ListOfUnits.
If the index @p n is invalid, @c None is returned.

@see size()
";


%feature("docstring") ListOfUnits::remove "
Removes the nth item from this ListOfUnits items and returns a pointer to
it.

The caller owns the returned item and is responsible for deleting it.

@param n the index of the item to remove.

@see size()
";


%feature("docstring") ListOfUnits::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfUnits::createObject "
@internal

@internal
";


%feature("docstring") UnitDefinition "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A definition of a unit used in an SBML model.

Units of measurement may be supplied in a number of contexts in an SBML
model.  The SBML unit definition facility uses two classes of objects,
UnitDefinition and Unit.  The approach to defining units in SBML is
compositional; for example, <em>meter second<sup> &ndash;2</sup></em> is
constructed by combining a Unit object representing <em>meter</em> with
another Unit object representing <em>second<sup> &ndash;2</sup></em>.
The combination is wrapped inside a UnitDefinition, which provides for
assigning an identifier and optional name to the combination.  The
identifier can then be referenced from elsewhere in a model.  Thus, the
UnitDefinition class is the container, and Unit instances are placed
inside UnitDefinition instances.

Two points are worth discussing in the context of SBML units.  First,
unit declarations in SBML models are @em optional.  The consequence of
this is that a model must be numerically self-consistent independently
of unit declarations, for the benefit of software tools that cannot
interpret or manipulate units.  Unit declarations in SBML are thus more
akin to a type of annotation; they can indicate intentions, and can be
used by model readers for checking the consistency of the model,
labeling simulation output, etc., but any transformations of values
implied by different units must be incorporated @em explicitly into a
model.

Second, the vast majority of situations that require new SBML unit
definitions involve simple multiplicative combinations of base units and
factors.  An example is <em>moles per litre per second</em>.  What
distinguishes these sorts of unit definitions from more complex ones is
that they may be expressed without the use of an additive offset from a
zero point.  The use of offsets complicates all unit definition systems,
yet in the domain of SBML, the real-life cases requiring offsets are few
(and in fact, to the best of our knowledge, only involve temperature).
Consequently, the SBML unit system has been consciously designed to
simplify implementation of unit support for the most common cases in
systems biology.  The cost of this simplification is to require units
with offsets to be handled explicitly by the modeler.

@section unitdef-summary Summary of the UnitDefinition construct

UnitDefinition has two attributes and one subelement.  The two
attributes are \'id\' and \'name\', and the subelement is ListOfUnits.

The required attribute \'id\' and optional attribute \'name\' are both
strings.  The \'id\' attribute is used to give the defined unit a unique
identifier by which other parts of an SBML model definition can refer to
it.  The \'name\' attribute is intended to be used for giving the unit
definition an optional human-readable name.  Please see the <a
href=\'#unitdef-id\'>next section</a> for information about the values
permitted for \'id\'.

A UnitDefinition may contain exactly one ListOfUnits, and this list
may contain one or more Unit definitions; see the definitions of these
other object classes for more information about them.  In SBML 
Level&nbsp;2 and SBML Level&nbsp;3 Version&nbsp;1, if the ListOfUnits
was present, it must have one or more Unit definitions.  In SBML
Level&nbsp;3 Version&nbsp;2, this restriction was relaxed, and 
a ListOfUnits was allowed to be empty.  In either case, if a
UnitDefinition had no child Unit elements, the unit was considered
to be undefined.

The following
example illustrates a complete unit definition (when written in XML)
when all the pieces are combined together.  This defines \'mmls\'
to be millimoles per litre per second.
@verbatim
 <listOfUnitDefinitions>
     <unitDefinition id=\'mmls\'>
         <listOfUnits>
             <unit kind=\'mole\'   scale=\'-3\'/>
             <unit kind=\'litre\'  exponent=\'-1\'/>
             <unit kind=\'second\' exponent=\'-1\'/>
         </listOfUnits>
     </unitDefinition>
 </listOfUnitDefinitions>
 @endverbatim

@section unitdef-id Special considerations for Unit object identifiers

The attribute \'id\' in UnitDefinition cannot be given simply any value,
and the precise details of the values permitted differ slightly between
Levels of SBML:
<ul>

<li> The \'id\' of a UnitDefinition must @em not contain a value from the
list of SBML\'s predefined base unit names (i.e., the strings @c gram,
@c litre, etc.).  In SBML Level&nbsp;3, this list consists of the
following:

@par
<table border=\'0\' class=\'centered text-table width80 normal-font code\'
       style=\'border: none !important\'>
<tr>
<td>ampere</td><td>farad</td><td>joule</td><td>lux</td><td>radian</td><td>volt</td>
</tr>
<tr>
<td>avogadro</td><td>gram</td><td>katal</td><td>metre</td><td>second</td><td>watt</td>
</tr>
<tr>
<td>becquerel</td><td>gray</td><td>kelvin</td><td>mole</td><td>siemens</td><td>weber</td>
</tr>
<tr>
<td>candela</td><td>henry</td><td>kilogram</td><td>newton</td><td>sievert</td>
</tr>
<tr>
<td>coulomb</td><td>hertz</td><td>litre</td><td>ohm</td><td>steradian</td>
</tr>
<tr>
<td>dimensionless</td><td>item</td><td>lumen</td><td>pascal</td><td>tesla</td>
</tr>
</table>

This list of predefined base units is nearly identical in SBML
Level&nbsp;2 Version&nbsp;4, the exception being that Level&nbsp;2 does
not define @c avogadro.  SBML Level&nbsp;2 Version&nbsp;1 (and @em only
this Level+Version combination) provides an additional predefined unit
name, @c Celsius, not available in Level&nbsp;3.  Finally, SBML
Level&nbsp;1 Versions&nbsp;2&ndash;3 provide two more additional
predefined unit names, @c meter and @c liter.  This is explained in
somewhat greater detail in the description of the Unit class.

<ul>
<li> In SBML Level&nbsp;2 (all Versions), there is an additional set of
reserved identifiers: @c substance, @c volume, @c area, @c length, and
@c time.  Using one of these values for the attribute \'id\' of a
UnitDefinition has the effect of redefining the model-wide default units
for the corresponding quantities.  The list of special unit names in
SBML Level&nbsp;2 is given in the table below:
</ul>

@htmlinclude predefined-units.html

Also, SBML Level&nbsp;2 imposes two limitations on redefining the
predefined unit @c substance, @c volume, @c area, @c length, and
@c time: (1) The UnitDefinition of a predefined SBML unit can only contain
a single Unit object within it.  (2) The value of the \'kind\' attribute
in a Unit instance must be drawn from one of the values in the second
column of the table above.

The special unit names @c substance, @c volume, @c area, @c length, and
@c time are not defined by SBML Level&nbsp;3, which uses a different
approach to setting model-wide inherited units.


@section sbml-units-limits Further comments about SBML\'s unit definition system

The vast majority of modeling situations requiring new SBML unit
definitions involve simple multiplicative combinations of base units and
factors.  An example of this might be <em>moles per litre per
second</em>.  What distinguishes these sorts of simpler unit definitions
from more complex ones is that they may be expressed without the use of
an additive offset from a zero point.  The use of offsets complicates
all unit definition systems, yet in the domain of SBML the real-life
cases requiring offsets are few (and in fact, to the best of our
knowledge, only involve temperature).  Consequently, the SBML unit
system has been consciously designed in a way that attempts to simplify
implementation of unit support for the most common cases in systems
biology.

As of SBML Level&nbsp;2 Version&nbsp;2, Unit no longer has the
attribute called \'offset\' introduced in SBML Level&nbsp;2
Version&nbsp;1.  It turned out that the general case involving units
with offsets was incorrectly defined, and few (if any) developers even
attempted to support offset-based units in their software.  In the
development of Level&nbsp;2 Version&nbsp;2, a consensus among SBML
developers emerged that a fully generalized unit scheme is @em so
confusing and complicated that it actually @em impedes interoperability.
SBML Level&nbsp;2 Version&nbsp;2, Version&nbsp;3 and Version&nbsp;4 acknowledge this
reality by reducing and simplifying the unit system, specifically by
removing the \'offset\' attribute on Unit and @c Celsius as a pre-defined
unit.

The following guidelines suggest methods for handling units that do
require the use of zero offsets for their definitions:
<ul>
<li> <em>Handling Celsius</em>.  A model in which certain quantities are
temperatures measured in degrees Celsius can be converted
straightforwardly to a model in which those temperatures are in
kelvin.  A software tool could do this by performing a straightforward
substitution using the following relationship: <em>T<sub> kelvin</sub> =
T<sub>Celsius</sub> + 273.15</em>.  In every mathematical formula of the
model where a quantity (call it @em x) in degrees Celsius appears,
replace @em x with <em>x<sub>k</sub>+ 273.15</em>, where
<em>x<sub>k</sub></em> is now in kelvin.  An alternative approach would 
be to use a FunctionDefinition object to define a function encapsulating this
relationship above and then using that in the rest of the model as
needed.  Since Celsius is a commonly-used unit, software tools could
help users by providing users with the ability to express temperatures
in Celsius in the tools\' interfaces, and making substitutions
automatically when writing out the SBML.

<li> <em>Other units requiring offsets</em>.  One approach to handling
other kinds of units is to use a FunctionDefinition to define a function
encapsulating the necessary mathematical relationship, then
substituting a call to this function wherever the original quantity
appeared in the model.  For example, here is a possible definition for
converting Fahrenheit to Celsius degrees:
@verbatim
 <functionDefinition id=\'Fahrenheit_to_kelvin\'>
     <math xmlns=\'http://www.w3.org/1998/Math/MathML\'>
         <lambda>
             <bvar><ci> temp_in_fahrenheit </ci></bvar>
             <apply>
                 <divide/>
                 <apply>
                     <plus/>
                     <ci> temp_in_fahrenheit </ci>
                     <cn> 459.67 </cn>
                 </apply>
                 <cn> 1.8 </cn>
             </apply>
         </lambda>
     </math>
 </functionDefinition>
 @endverbatim

<li> An alternative approach not requiring the use of function definitions
is to use an AssignmentRule for each variable in Fahrenheit units.
The AssignmentRule could compute the conversion from Fahrenheit to
(say) kelvin, assign its value to a variable (in Kelvin units), and
then that variable could be used elsewhere in the model.

<li> Still another approach is to rewrite the mathematical formulas of a
model to directly incorporate the conversion formula wherever the
original quantity appeared.
</ul>

Please consult the SBML specifications for more information about this
and other issues involving units.
";


%feature("docstring") ListOfUnitDefinitions "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of UnitDefinition objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") UnitDefinition::UnitDefinition "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>UnitDefinition(SBMLNamespaces sbmlns)</pre>

Creates a new UnitDefinition using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments. 

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>UnitDefinition(long level, long version)</pre>

Creates a new UnitDefinition using the given SBML @p level and @p version
values.

@param level a long integer, the SBML Level to assign to this UnitDefinition.

@param version a long integer, the SBML Version to assign to this
UnitDefinition.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>UnitDefinition(UnitDefinition orig)</pre>

Copy constructor; creates a copy of this UnitDefinition.

@param orig the object to copy.
";


%feature("docstring") UnitDefinition::accept "
@internal

@internal
";


%feature("docstring") UnitDefinition::clone "
Creates and returns a deep copy of this UnitDefinition object.

@return the (deep) copy of this UnitDefinition object.
";


%feature("docstring") UnitDefinition::getElementBySId "
Returns the first child element found that has the given @p id in the
model-wide SId namespace, or @c None if no such object is found.

@param id string representing the id of the object to find.

@return pointer to the first element found with the given @p id.
";


%feature("docstring") UnitDefinition::getElementByMetaId "
Returns the first child element it can find with the given @p metaid, or
@c None if no such object is found.

@param metaid string representing the metaid of the object to find.

@return pointer to the first element found with the given @p metaid.
";


%feature("docstring") UnitDefinition::getAllElements "
Returns a List of all child SBase objects, including those nested to an
arbitrary depth

@param filter a pointer to an ElementFilter, which causes the function 
to return only elements that match a particular set of constraints.  
If None (the default), the function will return all child objects.

@return a List of pointers to all children objects.
";


%feature("docstring") UnitDefinition::getId "
Returns the value of the \'id\' attribute of this UnitDefinition.

@note Because of the inconsistent behavior of this function with 
respect to assignments and rules, it is now recommended to
use the getIdAttribute() function instead.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this UnitDefinition.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") UnitDefinition::getName "
Returns the value of the \'name\' attribute of this UnitDefinition object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") UnitDefinition::isSetId "
Predicate returning @c True if this
UnitDefinition\'s \'id\' attribute is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") UnitDefinition::isSetName "
Predicate returning @c True if this
UnitDefinition\'s \'name\' attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") UnitDefinition::setId "
Sets the value of the \'id\' attribute of this UnitDefinition.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") UnitDefinition::setName "
Sets the value of the \'name\' attribute of this UnitDefinition.

@par

The string in @p name is copied.

@param name the new name for the SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") UnitDefinition::unsetName "
Unsets the value of the \'name\' attribute of this UnitDefinition.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") UnitDefinition::isVariantOfArea "
Convenience function for testing if a given unit definition is a
variant of the predefined unit identifier @c \'area\'.

@return @c True if this UnitDefinition is a variant of the predefined
unit @c area, meaning square metres with only arbitrary variations
in scale or multiplier values; @c False otherwise.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::isVariantOfLength "
Convenience function for testing if a given unit definition is a
variant of the predefined unit identifier @c \'length\'.

@return @c True if this UnitDefinition is a variant of the predefined
unit @c length, meaning metres with only arbitrary variations in scale
or multiplier values; @c False otherwise.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::isVariantOfSubstance "
Convenience function for testing if a given unit definition is a
variant of the predefined unit identifier @c \'substance\'.

@return @c True if this UnitDefinition is a variant of the predefined
unit @c substance, meaning moles or items (and grams or kilograms from
SBML Level&nbsp;2 Version&nbsp;2 onwards) with only arbitrary variations
in scale or multiplier values; @c False otherwise.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::isVariantOfTime "
Convenience function for testing if a given unit definition is a
variant of the predefined unit identifier @c \'time\'.

@return @c True if this UnitDefinition is a variant of the predefined
unit @c time, meaning seconds with only arbitrary variations in scale or
multiplier values; @c False otherwise.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::isVariantOfVolume "
Convenience function for testing if a given unit definition is a
variant of the predefined unit identifier @c \'volume\'.

@return @c True if this UnitDefinition is a variant of the predefined
unit @c volume, meaning litre or cubic metre with only arbitrary
variations in scale or multiplier values; @c False otherwise.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::isVariantOfDimensionless "
Convenience function for testing if a given unit definition is a
variant of the unit @c \'dimensionless\'.

@return @c True if this UnitDefinition is a variant of
@c dimensionless, meaning dimensionless with only arbitrary variations in
scale or multiplier values; @c False otherwise.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::isVariantOfMass "
Convenience function for testing if a given unit definition is a
variant of the predefined unit identifier @c \'mass\'.

@return @c True if this UnitDefinition is a variant of mass units,
meaning gram or kilogram with only arbitrary variations in scale or
multiplier values; @c False otherwise.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::isVariantOfSubstancePerTime "
Convenience function for testing if a given unit definition is a
variant of the predefined unit @c \'substance\' divided by the predefined
unit @c \'time\'.

@return @c True if this UnitDefinition is a variant of the predefined
unit @c substance per predefined unit @c time, meaning it contains two
units one of which is a variant of substance and the other is a
variant of time which an exponent of -1; @c False otherwise.
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::addUnit "
Adds a copy of the given Unit to this UnitDefinition.

@param u the Unit instance to add to this UnitDefinition.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
@li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method should be used with some caution.  The fact that this
method @em copies the object passed to it means that the caller will be
left holding a physically different object instance than the one contained
inside this object.  Changes made to the original object instance (such as
resetting attribute values) will <em>not affect the instance in this
object</em>.  In addition, the caller should make sure to free the
original object if it is no longer being used, or else a memory leak will
result.  Please see other methods on this class (particularly a
corresponding method whose name begins with the word <code>create</code>)
for alternatives that do not lead to these issues. 

@see createUnit()
";


%feature("docstring") UnitDefinition::createUnit "
Creates a new and empty Unit, adds it to this UnitDefinition\'s list of
units, and returns it.

@return a newly constructed (and empty) Unit instance.

@note It is worth emphasizing that the attribute \'kind\' value of a
Unit is a required attribute for a valid Unit definition.  The
createUnit() method does not assign a valid kind to the constructed
unit (instead, it sets the \'kind\' to @link libsbml#UNIT_KIND_INVALID UNIT_KIND_INVALID@endlink).
Callers are cautioned to set the newly-constructed Unit\'s kind using
Unit.setKind() soon after calling this method.

@see addUnit()
";


%feature("docstring") UnitDefinition::getListOfUnits "
Returns the list of Units for this UnitDefinition instance.
@return the ListOfUnits value for this UnitDefinition.
";


%feature("docstring") UnitDefinition::getUnit "
Returns a specific Unit instance belonging to this UnitDefinition.

@param n an integer, the index of the Unit to be returned.

@return the nth Unit of this UnitDefinition.
If the index @p n is invalid, @c None is returned.

@see getNumUnits()
";


%feature("docstring") UnitDefinition::getNumUnits "
Returns the number of Unit objects contained within this
UnitDefinition.

@return an integer representing the number of Units in this
UnitDefinition.
";


%feature("docstring") UnitDefinition::removeUnit "
Removes the nth Unit object from this UnitDefinition object and
returns a pointer to it.

The caller owns the returned object and is responsible for deleting it.

@param n the index of the Unit object to remove.

@return the Unit object removed, or @c None if the given index 
is out of range.
";


%feature("docstring") UnitDefinition::setSBMLDocument "
@internal

@internal
";


%feature("docstring") UnitDefinition::enablePackageInternal "
@internal

@internal
";


%feature("docstring") UnitDefinition::updateSBMLNamespace "
@internal

@internal
";


%feature("docstring") UnitDefinition::getTypeCode "
Returns the libSBML type code for this object instance.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_UNIT_DEFINITION SBML_UNIT_DEFINITION@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getPackageName()
@see getElementName()
";


%feature("docstring") UnitDefinition::getElementName "
Returns the XML element name of this object, which for UnitDefinition,
is always @c \'unitDefinition\'.

@return the name of this element, i.e., @c \'unitDefinition\'.
";


%feature("docstring") UnitDefinition::simplify "
Simplifies the UnitDefinition such that any given kind of Unit object
occurs only once in the ListOfUnits.

For example, the following definition,
@verbatim
 <unitDefinition>
  <listOfUnits>
    <unit kind=\'metre\' exponent=\'1\'/>
    <unit kind=\'metre\' exponent=\'2\'/>
  </listOfUnits>
 <unitDefinition>
 @endverbatim
will be simplified to 
@verbatim
 <unitDefinition>
   <listOfUnits>
     <unit kind=\'metre\' exponent=\'3\'/>
   </listOfUnits>
 <unitDefinition>
 @endverbatim

@param ud the UnitDefinition object to be simplified.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~
";


%feature("docstring") UnitDefinition::reorder "
Alphabetically orders the Unit objects within the ListOfUnits of a
UnitDefinition.

@param ud the UnitDefinition object whose units are to be reordered.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~
";


%feature("docstring") UnitDefinition::convertToSI "
Convert a given UnitDefinition into a new UnitDefinition object
that uses SI units.

@param ud the UnitDefinition object to convert to SI.

@return a new UnitDefinition object representing the results of the
conversion.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~
";


%feature("docstring") UnitDefinition::areIdentical "
Predicate returning @c True if two
UnitDefinition objects are identical.

For the purposes of performing this comparison, two UnitDefinition
objects are considered identical when they contain identical lists of
Unit objects.  Pairs of Unit objects in the lists are in turn
considered identical if they satisfy the predicate
Unit.areIdentical().
The predicate compares every attribute of the
Unit objects.

@param ud1 the first UnitDefinition object to compare.
@param ud2 the second UnitDefinition object to compare.

@return @c True if all the Unit objects in @p ud1 are identical to the
Unit objects of @p ud2, @c False otherwise.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see UnitDefinition.areEquivalent()
@see Unit.areIdentical()
";


%feature("docstring") UnitDefinition::areEquivalent "
Predicate returning @c True if two
UnitDefinition objects are equivalent.

For the purposes of performing this comparison, two UnitDefinition
objects are considered equivalent when they contain @em equivalent
list of Unit objects.  Unit objects are in turn considered equivalent
if they satisfy the predicate
Unit.areEquivalent().
The predicate tests a subset of the objects\'s attributes.

@param ud1 the first UnitDefinition object to compare.

@param ud2 the second UnitDefinition object to compare.

@return @c True if all the Unit objects in @p ud1 are equivalent
to the Unit objects in @p ud2, @c False otherwise.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~

@see UnitDefinition.areIdentical()
@see Unit.areEquivalent()
";


%feature("docstring") UnitDefinition::areIdenticalSIUnits "
@internal

@internal
";


%feature("docstring") UnitDefinition::combine "
Combines two UnitDefinition objects into a single UnitDefinition.

This takes UnitDefinition objects @p ud1 and @p ud2, and creates a
UnitDefinition object that expresses the product of the units of @p
ud1 and @p ud2.

@param ud1 the first UnitDefinition object.
@param ud2 the second UnitDefinition object.

@return a UnitDefinition which represents the product of the 
units of the two argument UnitDefinitions.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~
";


%feature("docstring") UnitDefinition::divide "
Combines two UnitDefinition objects into a single UnitDefinition as
a division.

This takes UnitDefinition objects @p ud1 and @p ud2, and creates a
UnitDefinition object that expresses the division of the units of @p
ud1 and @p ud2.

@param ud1 the first UnitDefinition object.
@param ud2 the second UnitDefinition object.

@return a UnitDefinition which represents the division of the 
units of the two argument UnitDefinitions.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~
";


%feature("docstring") UnitDefinition::printUnits "
Expresses the given definition in a plain-text form.

For example,
UnitDefinition.printUnits()
applied to
@verbatim
 <unitDefinition>
  <listOfUnits>
    <unit kind=\'metre\' exponent=\'1\'/>
    <unit kind=\'second\' exponent=\'-2\'/>
  </listOfUnits>
 <unitDefinition>
 @endverbatim
will return the string <code>\'metre (exponent = 1, multiplier = 1,
scale = 0) second (exponent = -2, multiplier = 1, scale = 0)\'</code>
or, if the optional parameter @p compact is given the value @c True,
the string <code>\'(1 metre)^1 (1 second)^-2\'</code>.  This method may
be useful for printing unit information to human users, or in
debugging software, or other situations.

@param ud the UnitDefinition object.
@param compact boolean indicating whether the compact form
should be used (defaults to false).

@return a string expressing the unit definition defined by the given
UnitDefinition object @p ud.

@if python @note Because this is a static method on a class, the Python
language interface for libSBML will contain two variants.  One will be the
expected, normal static method on the class (i.e., a regular
<em>methodName</em>), and the other will be a standalone top-level
function with the name <em>ClassName_methodName()</em>. This is merely an
artifact of how the language interfaces are created in libSBML.  The
methods are functionally identical. @endif@~
   


@note Owing to the way that language interfaces are created in libSBML, this documentation may show methods that define default values for parameters with text that has the form <i><code>parameter</code></i> = <i><code>value</code></i>. This is <strong>not</strong> to be intepreted as a Python keyword argument; the use of a parameter name followed by an equals sign followed by a value is only meant to indicate a default value if the argument is not provided at all.  It is not a keyword in the Python sense.
";


%feature("docstring") UnitDefinition::writeElements "
@internal

@internal
";


%feature("docstring") UnitDefinition::hasRequiredAttributes "
Predicate returning @c True if
all the required attributes for this UnitDefinition object
have been set.

The required attributes for a UnitDefinition object are:
@li \'id\'

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") UnitDefinition::hasRequiredElements "
Predicate returning @c True if
all the required elements for this UnitDefinition object
have been set.

@note The required elements for a Constraint object are:
@li \'listOfUnits\' (required in SBML Level&nbsp;2 only, optional in Level&nbsp;3)

@return a boolean value indicating whether all the required
elements for this object have been defined.
";


%feature("docstring") UnitDefinition::createObject "
@internal

@internal
";


%feature("docstring") UnitDefinition::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") UnitDefinition::readAttributes "
@internal

@internal
";


%feature("docstring") UnitDefinition::readL1Attributes "
@internal

@internal
";


%feature("docstring") UnitDefinition::readL2Attributes "
@internal

@internal
";


%feature("docstring") UnitDefinition::readL3Attributes "
@internal

@internal
";


%feature("docstring") UnitDefinition::writeAttributes "
@internal

@internal
";


%feature("docstring") ListOfUnitDefinitions::ListOfUnitDefinitions "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfUnitDefinitions(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfUnitDefinitions object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfUnitDefinitions object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfUnitDefinitions(long level, long version)</pre>

Creates a new ListOfUnitDefinitions object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfUnitDefinitions::clone "
Creates and returns a deep copy of this ListOfUnitDefinitions object.

@return the (deep) copy of this ListOfUnitDefinitions object.
";


%feature("docstring") ListOfUnitDefinitions::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., UnitDefinition objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for objects contained in this list:
@link libsbml#SBML_UNIT_DEFINITION SBML_UNIT_DEFINITION@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfUnitDefinitions::getElementName "
Returns the XML element name of this object.

For ListOfUnitDefinitions, the XML element name is
@c \'listOfUnitDefinitions\'.

@return the name of this element, i.e., @c \'listOfUnitDefinitions\'.
";


%feature("docstring") ListOfUnitDefinitions::get "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(long n)</pre>

Get a UnitDefinition from the ListOfUnitDefinitions.

@param n the index number of the UnitDefinition to get.

@return the nth UnitDefinition in this ListOfUnitDefinitions.
If the index @p n is invalid, @c None is returned.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(string sid)</pre>

Get a UnitDefinition from the ListOfUnitDefinitions
based on its identifier.

@param sid a string representing the identifier 
of the UnitDefinition to get.

@return UnitDefinition in this ListOfUnitDefinitions
with the given @p sid or @c None if no such
UnitDefinition exists.

@see get()
@see size()
";


%feature("docstring") ListOfUnitDefinitions::getElementBySId "
Returns the first child element found that has the given @p id in the
model-wide SId namespace, or @c None if no such object is found.

Note that UnitDefinitions themselves are in the UnitId namespace, not
the SId namespace, so no UnitDefinition object will be returned from
this function (and is the reason we override the base
ListOf::getElementBySId function here).

@param id string representing the id of the object to find.

@return pointer to the first element found with the given @p id.
";


%feature("docstring") ListOfUnitDefinitions::remove "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(long n)</pre>

Removes the nth item from this ListOfUnitDefinitions items and returns a pointer to
it.

The caller owns the returned item and is responsible for deleting it.

@param n the index of the item to remove.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(string sid)</pre>

Removes item in this ListOfUnitDefinitions items with the given identifier.

The caller owns the returned item and is responsible for deleting it.
If none of the items in this list have the identifier @p sid, then
@c None is returned.

@param sid the identifier of the item to remove.

@return the item removed.  As mentioned above, the caller owns the
returned item.
";


%feature("docstring") ListOfUnitDefinitions::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfUnitDefinitions::createObject "
@internal

@internal
";


%feature("docstring") CompartmentType "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A <em>compartment type</em> in SBML Level&nbsp;2.

SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 provide the <em>compartment
type</em> as a grouping construct that can be used to establish a
relationship between multiple Compartment objects.  A CompartmentType
object only has an identity, and this identity can only be used to
indicate that particular Compartment objects in the model belong to this
type.  This may be useful for conveying a modeling intention, such as
when a model contains many similar compartments, either by their
biological function or the reactions they carry.  Without a compartment
type construct, it would be impossible within SBML itself to indicate
that all of the compartments share an underlying conceptual relationship
because each SBML compartment must be given a unique and separate
identity.  A CompartmentType has no mathematical meaning in
SBML---it has no effect on a model\'s mathematical interpretation.
Simulators and other numerical analysis software may ignore
CompartmentType definitions and references to them in a model.

There is no mechanism in SBML Level 2 for representing hierarchies of
compartment types.  One CompartmentType instance cannot be the subtype
of another CompartmentType instance; SBML provides no means of defining
such relationships.

As with other major structures in SBML, CompartmentType has a mandatory
attribute, \'id\', used to give the compartment type an identifier.  The
identifier must be a text %string conforming to the identifer syntax
permitted in SBML.  CompartmentType also has an optional \'name\'
attribute, of type @c string.  The \'id\' and \'name\' must be used
according to the guidelines described in the SBML specification (e.g.,
Section 3.3 in the Level 2 Version 4 specification).

CompartmentType was introduced in SBML Level 2 Version 2.  It is not
available in SBML Level&nbsp;1 nor in Level&nbsp;3.

@see Compartment
@see ListOfCompartmentTypes
@see SpeciesType
@see ListOfSpeciesTypes
";


%feature("docstring") ListOfCompartmentTypes "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of CompartmentType objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") CompartmentType::CompartmentType "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>CompartmentType(SBMLNamespaces sbmlns)</pre>

Creates a new CompartmentType object using the given SBMLNamespaces
object @p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments.

It is worth emphasizing that although this constructor does not take an
identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
(identifier) attribute of a CompartmentType object is required to have a
value.  Thus, callers are cautioned to assign a value after calling this
constructor.  Setting the identifier can be accomplished using the
method setId(@if java String@endif).

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>CompartmentType(long level, long version)</pre>

Creates a new CompartmentType object using the given SBML @p level and
@p version values.

@param level a long integer, the SBML Level to assign to this
CompartmentType.

@param version a long integer, the SBML Version to assign to this
CompartmentType.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>CompartmentType(CompartmentType orig)</pre>

Copy constructor; creates a copy of this CompartmentType object.

@param orig the object to copy.
";


%feature("docstring") CompartmentType::accept "
@internal

@internal
";


%feature("docstring") CompartmentType::clone "
Creates and returns a deep copy of this CompartmentType object.

@return the (deep) copy of this CompartmentType object.
";


%feature("docstring") CompartmentType::getId "
Returns the value of the \'id\' attribute of this CompartmentType.

@note Because of the inconsistent behavior of this function with 
respect to assignments and rules, it is now recommended to
use the getIdAttribute() function instead.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this CompartmentType.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") CompartmentType::getName "
Returns the value of the \'name\' attribute of this CompartmentType object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") CompartmentType::isSetId "
Predicate returning @c True if this CompartmentType object\'s \'id\'
attribute is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") CompartmentType::isSetName "
Predicate returning @c True if this CompartmentType object\'s \'name\'
attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") CompartmentType::setId "
Sets the value of the \'id\' attribute of this CompartmentType.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") CompartmentType::setName "
Sets the value of the \'name\' attribute of this CompartmentType.

@par

The string in @p name is copied.

@param name the new name for the SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") CompartmentType::unsetName "
Unsets the value of the \'name\' attribute of this CompartmentType object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") CompartmentType::getTypeCode "
Returns the libSBML type code for this SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_COMPARTMENT_TYPE SBML_COMPARTMENT_TYPE@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getElementName()
@see getPackageName()
";


%feature("docstring") CompartmentType::getElementName "
Returns the XML element name of this object

For CompartmentType, the element name is always @c \'compartmentType\'.

@return the name of this element.

@see getTypeCode()
@see getPackageName()
";


%feature("docstring") CompartmentType::writeElements "
@internal

@internal
";


%feature("docstring") CompartmentType::hasRequiredAttributes "
Predicate returning @c True if all the required attributes for this
CompartmentType object have been set.

The required attributes for a CompartmentType object are:
@li \'id\'

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") CompartmentType::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") CompartmentType::readAttributes "
@internal

@internal
";


%feature("docstring") CompartmentType::readL2Attributes "
@internal

@internal
";


%feature("docstring") CompartmentType::writeAttributes "
@internal

@internal
";


%feature("docstring") ListOfCompartmentTypes::ListOfCompartmentTypes "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfCompartmentTypes(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfCompartmentTypes object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfCompartmentTypes object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfCompartmentTypes(long level, long version)</pre>

Creates a new ListOfCompartmentTypes object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfCompartmentTypes::clone "
Creates and returns a deep copy of this ListOfCompartmentTypes object.

@return the (deep) copy of this ListOfCompartmentTypes object.
";


%feature("docstring") ListOfCompartmentTypes::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., CompartmentType objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for the objects contained in this ListOf
instance: @link libsbml#SBML_COMPARTMENT_TYPE SBML_COMPARTMENT_TYPE@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfCompartmentTypes::getElementName "
Returns the XML element name of this object.

For ListOfCompartmentTypes, the XML element name is
@c \'listOfCompartmentTypes\'.

@return the name of this element, i.e., @c \'listOfCompartmentTypes\'.
";


%feature("docstring") ListOfCompartmentTypes::get "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(long n)</pre>

Get a CompartmentType object from the ListOfCompartmentTypes.

@param n the index number of the CompartmentType object to get.

@return the nth CompartmentType object in this ListOfCompartmentTypes.
If the index @p n is invalid, @c None is returned.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(string sid)</pre>

Get a CompartmentType object from the ListOfCompartmentTypes
based on its identifier.

@param sid a string representing the identifier
of the CompartmentType object to get.

@return CompartmentType object in this ListOfCompartmentTypes
with the given @p sid or @c None if no such
CompartmentType object exists.

@see get()
@see size()
";


%feature("docstring") ListOfCompartmentTypes::remove "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(long n)</pre>

Removes the nth item from this ListOfCompartmentTypes items
and returns a pointer to it.

The caller owns the returned item and is responsible for deleting it.

@param n the index of the item to remove.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(string sid)</pre>

Removes item in this ListOfCompartmentTypes items with the given identifier.

The caller owns the returned item and is responsible for deleting it.
If none of the items in this list have the identifier @p sid, then
@c None is returned.

@param sid the identifier of the item to remove.

@return the item removed.  As mentioned above, the caller owns the
returned item.
";


%feature("docstring") ListOfCompartmentTypes::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfCompartmentTypes::createObject "
@internal

@internal
";


%feature("docstring") SpeciesType "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A <em>species type</em> in SBML Level 2.

The term @em species @em type refers to reacting entities independent of
location.  These include simple ions (e.g., protons, calcium), simple
molecules (e.g., glucose, ATP), large molecules (e.g., RNA,
polysaccharides, and proteins), and others.

SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 provide an explicit
SpeciesType class of object to enable Species objects of the same type
to be related together.  SpeciesType is a conceptual construct; the
existence of SpeciesType objects in a model has no effect on the model\'s
numerical interpretation.  Except for the requirement for uniqueness of
species/species type combinations located in compartments, simulators
and other numerical analysis software may ignore SpeciesType definitions
and references to them in a model.

There is no mechanism in SBML Level 2 for representing hierarchies of
species types.  One SpeciesType object cannot be the subtype of another
SpeciesType object; SBML provides no means of defining such
relationships.

As with other major structures in SBML, SpeciesType has a mandatory
attribute, \'id\', used to give the species type an identifier.  The
identifier must be a text string conforming to the identifer syntax
permitted in SBML.  SpeciesType also has an optional \'name\' attribute,
of type @c string.  The \'id\' and \'name\' must be used according to the
guidelines described in the SBML specification (e.g., Section 3.3 in
the Level&nbsp;2 Version&nbsp;4 specification).

SpeciesType was introduced in SBML Level 2 Version 2.  It is not
available in SBML Level&nbsp;1 nor in Level&nbsp;3.

@see Species
@see ListOfSpeciesTypes
@see CompartmentType
@see ListOfCompartmentTypes
";


%feature("docstring") ListOfSpeciesTypes "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of SpeciesType objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") SpeciesType::SpeciesType "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>SpeciesType(SBMLNamespaces sbmlns)</pre>

Creates a new SpeciesType using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments. 

It is worth emphasizing that although this constructor does not take
an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
(identifier) attribute of a SpeciesType object is required to have a value.
Thus, callers are cautioned to assign a value after calling this
constructor.  Setting the identifier can be accomplished using the
method SBase.setId().

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>SpeciesType(long level, long version)</pre>

Creates a new SpeciesType using the given SBML @p level and @p version
values.

@param level a long integer, the SBML Level to assign to this SpeciesType.

@param version a long integer, the SBML Version to assign to this
SpeciesType.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>SpeciesType(SpeciesType orig)</pre>

Copy constructor; creates a copy of this SpeciesType.

@param orig the object to copy.
";


%feature("docstring") SpeciesType::accept "
@internal

@internal
";


%feature("docstring") SpeciesType::clone "
Creates and returns a deep copy of this SpeciesType object.

@return the (deep) copy of this SpeciesType object.
";


%feature("docstring") SpeciesType::getId "
Returns the value of the \'id\' attribute of this SpeciesType.

@note Because of the inconsistent behavior of this function with 
respect to assignments and rules, it is now recommended to
use the getIdAttribute() function instead.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this SpeciesType.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") SpeciesType::getName "
Returns the value of the \'name\' attribute of this SpeciesType object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") SpeciesType::isSetId "
Predicate returning @c True if this
SpeciesType\'s \'id\' attribute is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") SpeciesType::isSetName "
Predicate returning @c True if this
SpeciesType\'s \'name\' attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") SpeciesType::setId "
Sets the value of the \'id\' attribute of this SpeciesType.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") SpeciesType::setName "
Sets the value of the \'name\' attribute of this SpeciesType.

@par

The string in @p name is copied.

@param name the new name for the SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") SpeciesType::unsetName "
Unsets the value of the \'name\' attribute of this SpeciesType.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") SpeciesType::getTypeCode "
Returns the libSBML type code for this SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_SPECIES_TYPE SBML_SPECIES_TYPE@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getElementName()
@see getPackageName()
";


%feature("docstring") SpeciesType::getElementName "
Returns the XML element name of this object, which for
SpeciesType, is always @c \'compartmentType\'.

@return the name of this element, i.e., @c \'compartmentType\'.
";


%feature("docstring") SpeciesType::writeElements "
@internal

@internal
";


%feature("docstring") SpeciesType::hasRequiredAttributes "
Predicate returning @c True if
all the required attributes for this SpeciesType object
have been set.

The required attributes for a SpeciesType object are:
@li \'id\'

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") SpeciesType::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") SpeciesType::readAttributes "
@internal

@internal
";


%feature("docstring") SpeciesType::readL2Attributes "
@internal

@internal
";


%feature("docstring") SpeciesType::writeAttributes "
@internal

@internal
";


%feature("docstring") ListOfSpeciesTypes::ListOfSpeciesTypes "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfSpeciesTypes(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfSpeciesTypes object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfSpeciesTypes object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfSpeciesTypes(long level, long version)</pre>

Creates a new ListOfSpeciesTypes object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfSpeciesTypes::clone "
Creates and returns a deep copy of this ListOfSpeciesTypes object.

@return the (deep) copy of this ListOfSpeciesTypes object.
";


%feature("docstring") ListOfSpeciesTypes::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., SpeciesType objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for objects contained in this list:
@link libsbml#SBML_SPECIES_TYPE SBML_SPECIES_TYPE@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfSpeciesTypes::getElementName "
Returns the XML element name of this object.

For ListOfSpeciesTypes, the XML element name is
@c \'listOfSpeciesTypes\'.

@return the name of this element, i.e., @c \'listOfSpeciesTypes\'.
";


%feature("docstring") ListOfSpeciesTypes::get "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(long n)</pre>

Get a SpeciesType from the ListOfSpeciesTypes.

@param n the index number of the SpeciesType to get.

@return the nth SpeciesType in this ListOfSpeciesTypes.
If the index @p n is invalid, @c None is returned.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(string sid)</pre>

Get a SpeciesType from the ListOfSpeciesTypes
based on its identifier.

@param sid a string representing the identifier 
of the SpeciesType to get.

@return SpeciesType in this ListOfSpeciesTypes
with the given @p sid or @c None if no such
SpeciesType exists.

@see get()
@see size()
";


%feature("docstring") ListOfSpeciesTypes::remove "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(long n)</pre>

Removes the nth item from this ListOfSpeciesTypes items and returns a pointer to
it.

The caller owns the returned item and is responsible for deleting it.

@param n the index of the item to remove.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(string sid)</pre>

Removes item in this ListOfSpeciesTypes items with the given identifier.

The caller owns the returned item and is responsible for deleting it.
If none of the items in this list have the identifier @p sid, then
@c None is returned.

@param sid the identifier of the item to remove.

@return the item removed.  As mentioned above, the caller owns the
returned item.
";


%feature("docstring") ListOfSpeciesTypes::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfSpeciesTypes::createObject "
@internal

@internal
";


%feature("docstring") Compartment "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html An SBML compartment, where species are located.

A compartment in SBML represents a bounded space in which species are
located.  Compartments do not necessarily have to correspond to actual
structures inside or outside of a biological cell.

It is important to note that although compartments are optional in the
overall definition of Model, every species in an SBML model must be
located in a compartment.  This in turn means that if a model defines
any species, the model must also define at least one compartment.  The
reason is simply that species represent physical things, and therefore
must exist @em somewhere.  Compartments represent the @em somewhere.

Compartment has one required attribute, \'id\', to give the compartment a
unique identifier by which other parts of an SBML model definition can
refer to it.  A compartment can also have an optional \'name\' attribute
of type @c string.  Identifiers and names must be used according to the
guidelines described in the SBML specifications.

Compartment also has an optional attribute \'spatialDimensions\' that is
used to indicate the number of spatial dimensions possessed by the
compartment.  Most modeling scenarios involve compartments with integer
values of \'spatialDimensions\' of @c 3 (i.e., a three-dimensional
compartment, which is to say, a volume), or 2 (a two-dimensional
compartment, a surface), or @c 1 (a one-dimensional compartment, a
line).  In SBML Level&nbsp;3, the type of this attribute is @c float,
there are no restrictions on the permitted values of the
\'spatialDimensions\' attribute, and there are no default values.  In SBML
Level&nbsp;2, the value must be a positive @c integer, and the default
value is @c 3; the permissible values in SBML Level&nbsp;2 are @c 3,
@c 2, @c 1, and @c 0 (for a point).

Another optional attribute on Compartment is \'size\', representing the @em
initial total size of that compartment in the model.  The \'size\' attribute
must be a floating-point value and may represent a volume (if the
compartment is a three-dimensional one), or an area (if the compartment is
two-dimensional), or a length (if the compartment is one-dimensional).
There is no default value of compartment size in SBML Level&nbsp;2 or
Level&nbsp;3.  In particular, a missing \'size\' value <em>does not imply
that the compartment size is 1</em>.  (This is unlike the definition of
compartment \'volume\' in SBML Level&nbsp;1.)  When the compartment\'s
\'spatialDimensions\' attribute does not have a value of @c 0, a missing
value of \'size\' for a given compartment signifies that the value either is
unknown, or to be obtained from an external source, or determined by an
InitialAssignment, AssignmentRule, AlgebraicRule or RateRule
@if conly structure @else object@endif@~ elsewhere in the model.  In SBML
Level&nbsp;2, there are additional special requirements on the values of
\'size\'; we discuss them in a <a href=\'#comp-l2\'>separate section
below</a>.

The units associated with a compartment\'s \'size\' attribute value may be
set using the optional attribute \'units\'.  The rules for setting and
using compartment size units differ between SBML Level&nbsp;2 and
Level&nbsp;3, and are discussed separately below.

Finally, the Compartment attribute named \'constant\' is used to
indicate whether the compartment\'s size stays constant after simulation
begins.  A value of @c True indicates the compartment\'s \'size\' cannot be
changed by any other construct except InitialAssignment; a value of
@c False indicates the compartment\'s \'size\' can be changed by other
constructs in SBML.  In SBML Level&nbsp;2, there is an additional
explicit restriction that if \'spatialDimensions\'=@c \'0\', the value
cannot be changed by InitialAssignment either.  Further, in
Level&nbsp;2, \'constant\' is optional, and has a default value of @c True.  In SBML
Level&nbsp;3, there is no default value for the \'constant\' attribute,
and it is required.


@section comp-l2 Additional considerations in SBML Level&nbsp;2

In SBML Level&nbsp;2, the default units of compartment size, and the kinds
of units allowed as values of the attribute \'units\', interact with the
number of spatial dimensions of the compartment.  The value of the \'units\'
attribute of a Compartment @if conly structure @else object@endif@~ must
be one of the base units (see Unit), or the predefined unit identifiers
@c volume, @c area, @c length or @c dimensionless, or a new unit defined by a
UnitDefinition @if conly structure @else object@endif@~ in the enclosing
Model, subject to the restrictions detailed in the following table:

<table border=\'0\' class=\'centered text-table width80 normal-font alt-row-colors\'
style=\'padding-bottom: 0.5em\'>
<caption class=\'top-caption\'>Restrictions on values permitted for
compartment <code>size</code> and <code>units</code> attributes.</caption>
<tr>
<th align=\'left\' valign=\'bottom\'>
Value of<br><code>spatialDimensions</code>
</th>
<th align=\'left\' valign=\'bottom\'>
<code>size</code><br>allowed?
</th>
<th align=\'left\' valign=\'bottom\'>
<code>units</code><br>allowed?
</th>
<th align=\'left\' valign=\'bottom\'>
Allowable kinds of units
</th>
<th align=\'left\' valign=\'bottom\'>
Default value of attribute <code>units</code>
</th>
</tr>
<tr>
<td><code>3</code></td>
<td>yes</td>
<td>yes</td>
<td>units of volume, or <code>dimensionless</code></td>
<td><code>volume</code></td>
</tr>
<tr>
<td><code>2</code></td>
<td>yes</td>
<td>yes</td>
<td>units of area, or <code>dimensionless</code></td>
<td><code>area</code></td>
</tr>
<tr>
<td><code>1</code></td>
<td>yes</td>
<td>yes</td>
<td>units of length, or <code>dimensionless</code></td>
<td><code>length</code></td>
</tr>
<tr>
<td><code>0</code></td>
<td>no</td>
<td>no</td>
<td>(no units allowed)</td>
<td></td>
</tr>
</tr>
</table>

In SBML Level&nbsp;2, the units of the compartment size, as defined by the
\'units\' attribute or (if \'units\' is not set) the default value listed in
the table above, are used in the following ways when the compartment has
a \'spatialDimensions\' value greater than @c 0:
<ul>
<li> The value of the \'units\' attribute is used as the units of the
compartment identifier when the identifier appears as a numerical
quantity in a mathematical formula expressed in MathML.

<li> The @c math element of an AssignmentRule or InitialAssignment
referring to this compartment @em must (in Level&nbsp;2 Versions&nbsp;1-3)
or @em should (in Level&nbsp;2 Version 4) have identical units.

<li> In RateRule objects that set the rate of change of the compartment\'s
size, the units of the rule\'s @c math element @em must (in Level&nbsp;2 
Versions&nbsp;1&ndash;3) or @em should (in Level&nbsp;2 Version 4) be identical to the
compartment\'s units (whether defined by the \'units\' attribute or by taking the 
default value from the Model) divided by the default @em time units.
(In other words, the units for the rate of change of compartment size
are <em>compartment size</em>/<em>time</em> units.

<li> When a Species is to be treated in terms of concentrations or
density, the units of the spatial size portion of the concentration
value (i.e., the denominator in the units formula @em substance/@em
size) are those indicated by the value of the \'units\' attribute on the
compartment in which the species is located.
</ul>

Compartments with \'spatialDimensions\'=@c 0 require special treatment in
this framework.  As implied above, the \'size\' attribute must not have a
value on an SBML Level&nbsp;2 Compartment
@if conly structure @else object@endif@~ if the \'spatialDimensions\'
attribute has a value of @c 0.  An additional related restriction is that
the \'constant\' attribute must default to or be set to @c True if the value
of the \'spatialDimensions\' attribute is @c 0, because a zero-dimensional
compartment cannot ever have a size.

If a compartment has no size or dimensional units, how should such a
compartment\'s identifier be interpreted when it appears in mathematical
formulas?  The answer is that such a compartment\'s identifier should not
appear in mathematical formulas in the first place---it has no
value, and its value cannot change.  Note also that a zero-dimensional
compartment is a point, and species located at points can only be
described in terms of amounts, not spatially-dependent measures such as
concentration.  Since SBML KineticLaw formulas are already in terms of
@em substance/@em time and not (say) @em concentration/@em time, volume
or other factors in principle are not needed for species located in
zero-dimensional compartments.

Finally, in SBML Level&nbsp;2 Versions 2&ndash;4, each compartment in a
model may optionally be designated as belonging to a particular
compartment @em type.  The optional attribute \'compartmentType\' is used
identify the compartment type represented by the Compartment structure.
The \'compartmentType\' attribute\'s value must be the identifier of a
CompartmentType instance defined in the model.  If the \'compartmentType\'
attribute is not present on a particular compartment definition, a
unique virtual compartment type is assumed for that compartment, and no
other compartment can belong to that compartment type.  The values of
\'compartmentType\' attributes on compartments have no effect on the
numerical interpretation of a model.  Simulators and other numerical
analysis software may ignore \'compartmentType\' attributes.  The
\'compartmentType\' attribute and the CompartmentType
@if conly structures @else class of objects@endif@~ are
not present in SBML Level&nbsp;3 Core nor in SBML Level&nbsp;1.


@section comp-l3 Additional considerations in SBML Level&nbsp;3

One difference between SBML Level&nbsp;3 and lower Levels of SBML is
that there are no restrictions on the permissible values of the
\'spatialDimensions\' attribute, and there is no default value defined for
the attribute.  The value of \'spatialDimensions\' does not have to be an
integer, either; this is to allow for the possibility of representing
structures with fractal dimensions.

The number of spatial dimensions possessed by a compartment cannot enter
into mathematical formulas, and therefore cannot directly alter the
numerical interpretation of a model.  However, the value of
\'spatialDimensions\' @em does affect the interpretation of the units
associated with a compartment\'s size.  Specifically, the value of
\'spatialDimensions\' is used to select among the Model attributes
\'volumeUnits\', \'areaUnits\' and \'lengthUnits\' when a Compartment
@if conly object @else structure@endif@~ does not define a value for its
\'units\' attribute.

The \'units\' attribute may be left unspecified for a given compartment in a
model; in that case, the compartment inherits the unit of measurement
specified by one of the attributes on the enclosing Model
@if conly structure @else object@endif@~ instance.  The applicable
attribute on Model depends on the value of the compartment\'s
\'spatialDimensions\' attribute; the relationship is shown in the table
below.  If the Model @if conly structure @else object@endif@~ does not
define the relevant attribute (\'volumeUnits\', \'areaUnits\' or
\'lengthUnits\') for a given \'spatialDimensions\' value, the unit associated
with that Compartment @if conly structure @else object@endif\'s size is
undefined.  If a given Compartment\'s \'units\' are left unset and 
the \'spatialDimensions\' either has a value other than @c 1, @c 2, or 
@c 3 or is left unset itself (as it has no default value),
then no unit can be chosen from among the Model\'s \'volumeUnits\',
\'areaUnits\' or \'lengthUnits\' attributes (even if the Model instance
provides values for those attributes), because there is no basis to select
between them.
Leaving the units of compartments\' sizes undefined in an SBML model does
not render the model invalid; however, as a matter of best practice, we
strongly recommend that all models specify the units of measurement for
all compartment sizes.

<table border=\'0\' class=\'centered text-table width80 normal-font alt-row-colors\'
style=\'padding-bottom: 0.5em\'>
<caption class=\'top-caption\'>Interpretation of the Compartment \'units\' attribute.</caption>
<tr>
<th align=\'left\' valign=\'bottom\'>
Value of attribute<br>\'spatialDimensions\'
</th>
<th align=\'left\' valign=\'bottom\'>
Attribute of Model used<br>for inheriting the unit
</th>
<th align=\'left\' valign=\'bottom\'>
Recommended candidate units
</th>
</tr>
<tr>
<td><code>3</code></td>
<td>&quot;volumeUnits&quot;</td>
<td>units of volume, or <code>dimensionless</code></td>
</tr>
<tr>
<td><code>2</code></td>
<td>&quot;areaUnits&quot;</td>
<td>units of area, or <code>dimensionless</code></td>
</tr>
<tr>
<td><code>1</code></td>
<td>&quot;lengthUnits&quot;</td>
<td>units of length, or <code>dimensionless</code></td>
</tr>
<tr>
<td><em>other</em></td>
<td><em>no units inherited</em></td>
<td><em>no specific recommendations</em></td>
</tr>
</tr>
</table>

The unit of measurement associated with a compartment\'s size, as defined
by the \'units\' attribute or (if \'units\' is not set) the inherited value
from Model according to the table above, is used in the following ways:

<ul>

<li> When the identifier of the compartment appears as a numerical
quantity in a mathematical formula expressed in MathML, it represents
the size of the compartment, and the unit associated with the size is
the value of the \'units\' attribute.

<li> When a Species is to be treated in terms of concentrations or
density, the unit associated with the spatial size portion of the
concentration value (i.e., the denominator in the formula
<em>amount</em>/<em>size</em>) is specified by the value of the \'units\'
attribute on the compartment in which the species is located.

<li> The \'math\' elements of AssignmentRule, InitialAssignment and
EventAssignment @if conly structures @else objects@endif@~ setting the
value of the compartment size should all have the same units as the unit
associated with the compartment\'s size.

<li> In a RateRule @if conly structure @else object@endif@~ that defines a
rate of change for a compartment\'s size, the unit of the rule\'s \'math\'
element should be identical to the compartment\'s \'units\' attribute divided
by the model-wide unit of <em>time</em>.  (In other words, {<em>unit of
compartment size</em>}/{<em>unit of time</em>}.)

</ul>


@section comp-other Other aspects of Compartment

In SBML Level&nbsp;1 and Level&nbsp;2, Compartment has an optional
attribute named \'outside\', whose value can be the identifier of another
Compartment @if conly structure @else object@endif@~ defined in the
enclosing Model @if conly structure @else object@endif@~.  Doing so means
that the other compartment contains it or is outside of it.  This enables
the representation of simple topological relationships between
compartments, for those simulation systems that can make use of the
information (e.g., for drawing simple diagrams of compartments).  It is
worth noting that in SBML, there is no relationship between compartment
sizes when compartment positioning is expressed using the \'outside\'
attribute.  The size of a given compartment does not in any sense include
the sizes of other compartments having it as the value of their \'outside\'
attributes.  In other words, if a compartment @em B has the identifier of
compartment @em A as its \'outside\' attribute value, the size of @em A does
not include the size of @em B.  The compartment sizes are separate.

In Level&nbsp;2, there are two restrictions on the \'outside\' attribute.
First, because a compartment with \'spatialDimensions\' of @c 0 has no
size, such a compartment cannot act as the container of any other
compartment @em except compartments that @em also have
\'spatialDimensions\' values of @c 0.  Second, the directed graph formed
by representing Compartment structures as vertexes and the \'outside\'
attribute values as edges must be acyclic.  The latter condition is
imposed to prevent a compartment from being contained inside itself.  In
the absence of a value for \'outside\', compartment definitions in SBML
Level&nbsp;2 do not have any implied spatial relationships between each
other.
";


%feature("docstring") ListOfCompartments "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of Compartment objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") Compartment::Compartment "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Compartment(SBMLNamespaces sbmlns)</pre>

Creates a new Compartment object using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments.

It is worth emphasizing that although this constructor does not take an
identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
(identifier) attribute of a Compartment object is required to have a
value.  Thus, callers are cautioned to assign a value after calling this
constructor.  Setting the identifier can be accomplished using the
method @if java Compartment.setId()@else setId()@endif.

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Compartment(long level, long version)</pre>

Creates a new Compartment object using the given SBML @p level and @p
version values.

@param level a long integer, the SBML Level to assign to this Compartment.

@param version a long integer, the SBML Version to assign to this
Compartment.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Compartment(Compartment orig)</pre>

Copy constructor.

This creates a copy of a Compartment object.

@param orig the Compartment instance to copy.
";


%feature("docstring") Compartment::accept "
@internal

@internal
";


%feature("docstring") Compartment::clone "
Creates and returns a deep copy of this Compartment object.

@return the (deep) copy of this Compartment object.
";


%feature("docstring") Compartment::initDefaults "
Initializes the fields of this Compartment object to \'typical\' default
values.

The SBML Compartment component has slightly different aspects and
default attribute values in different SBML Levels and Versions.
This method sets the values to certain common defaults, based
mostly on what they are in SBML Level&nbsp;2.  Specifically:

@li Sets attribute \'spatialDimensions\' to @c 3
@li Sets attribute \'constant\' to @c True
@li (Applies to Level&nbsp;1 models only) Sets attribute \'volume\' to @c 1.0
@li (Applies to Level&nbsp;3 models only) Sets attribute \'units\' to @c litre
";


%feature("docstring") Compartment::getId "
Returns the value of the \'id\' attribute of this Compartment.

@note Because of the inconsistent behavior of this function with 
respect to assignments and rules, it is now recommended to
use the getIdAttribute() function instead.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this Compartment.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") Compartment::getName "
Returns the value of the \'name\' attribute of this Compartment object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") Compartment::getCompartmentType "
Get the value of the \'compartmentType\' attribute of this Compartment
object.

@return the value of the \'compartmentType\' attribute of this
Compartment object as a string.

@note The \'compartmentType\' attribute is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.

@see isSetCompartmentType()
@see setCompartmentType()
@see unsetCompartmentType()
";


%feature("docstring") Compartment::getSpatialDimensions "
Get the number of spatial dimensions of this Compartment object.

@return the value of the \'spatialDimensions\' attribute of this
Compartment object as a long integereger.

@note In SBML Level&nbsp;3, the data type of the \'spatialDimensions\'
attribute is @c float, whereas in Level&nbsp;2, it is @c integer.  To
avoid backward compatibility issues, libSBML provides two separate methods
for obtaining the value as either an integer or a type @c float, for
models where it is relevant.

@see getSpatialDimensionsAsDouble()
@see setSpatialDimensions()
@see isSetSpatialDimensions()
@see unsetSpatialDimensions()
";


%feature("docstring") Compartment::getSpatialDimensionsAsDouble "
Get the number of spatial dimensions of this Compartment object,
as a double.

@return the value of the \'spatialDimensions\' attribute of this
Compartment object as a double, or @c NaN if this model is not in SBML
Level&nbsp;3 format.

@note In SBML Level&nbsp;3, the data type of the \'spatialDimensions\'
attribute is @c float, whereas in Level&nbsp;2, it is @c integer.  To
avoid backward compatibility issues, libSBML provides two separate methods
for obtaining the value as either an integer or a type @c float, for
models where it is relevant.

@see getSpatialDimensions()
@see setSpatialDimensions()
@see isSetSpatialDimensions()
@see unsetSpatialDimensions()
";


%feature("docstring") Compartment::getSize "
Get the size of this Compartment object.

@par
In SBML Level&nbsp;1, compartments are always three-dimensional
constructs and only have volumes, whereas in SBML Level&nbsp;2 and higher,
compartments may be other than three-dimensional, and therefore the
\'volume\' attribute is named \'size\' in Level&nbsp;2 and above.  LibSBML
provides both @if conly Compartment_getSize() and Compartment_getVolume()
@else getSize() and getVolume()@endif@~
for easier support of different SBML Levels.

@return the value of the \'size\' attribute (\'volume\' in Level&nbsp;1) of
this Compartment object as a floating-point number.

@note This method is identical to
@if java Compartment.getVolume()@else getVolume()@endif.

@see getVolume()
@see isSetSize()
@see setSize()
@see unsetSize()
";


%feature("docstring") Compartment::getVolume "
Get the volume of this Compartment object.

@par
In SBML Level&nbsp;1, compartments are always three-dimensional
constructs and only have volumes, whereas in SBML Level&nbsp;2 and higher,
compartments may be other than three-dimensional, and therefore the
\'volume\' attribute is named \'size\' in Level&nbsp;2 and above.  LibSBML
provides both @if conly Compartment_getSize() and Compartment_getVolume()
@else getSize() and getVolume()@endif@~
for easier support of different SBML Levels.

@return the value of the \'volume\' attribute (\'size\' in Level&nbsp;2) of
this Compartment object, as a floating-point number.

@note The attribute \'volume\' only exists by that name in SBML
Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a default
value (@c 1.0) and therefore methods such as
@if conly Compartment_isSetVolume() @else isSetVolume()@endif@~
will always return @c True for a Level&nbsp;1 model.  In Level&nbsp;2, a
compartment\'s size (the equivalent of SBML Level&nbsp;1\'s \'volume\') is
optional and has no default value, and therefore may or may not be set.

@note This method is identical to
@if java Compartment.getSize()@else getSize()@endif.

@see getSize()
@see isSetVolume()
@see setVolume()
@see unsetVolume()
";


%feature("docstring") Compartment::getUnits "
Get the units of this Compartment object\'s size.

The value of an SBML compartment\'s \'units\' attribute establishes the
unit of measurement associated with the compartment\'s size.

@return the value of the \'units\' attribute of this Compartment object,
as a string.  An empty string indicates that no units have been assigned
to the value of the size.

@note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

@see isSetUnits()
@see setUnits()
@see unsetUnits()
";


%feature("docstring") Compartment::getOutside "
Get the identifier, if any, of the Compartment object that is designated
as being outside of @em this one.

@return the value of the \'outside\' attribute of this Compartment object.

@note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
Level&nbsp;2, but does not exist in SBML Level&nbsp;3.

@see isSetOutside()
@see setOutside()
@see unsetOutside()
";


%feature("docstring") Compartment::getConstant "
Get the value of the \'constant\' attribute of this Compartment object.

@return @c True if this Compartment object\'s size is flagged as being
constant, @c False otherwise.

@see isSetConstant()
@see setConstant()
";


%feature("docstring") Compartment::isSetId "
Predicate returning @c True if this Compartment object\'s \'id\' attribute
is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") Compartment::isSetName "
Predicate returning @c True if this Compartment object\'s \'name\'
attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") Compartment::isSetCompartmentType "
Predicate returning @c True if this Compartment object\'s
\'compartmentType\' attribute is set.

@return @c True if the \'compartmentType\' attribute of this Compartment
is set, @c False otherwise.

@note The \'compartmentType\' attribute is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.

@see setCompartmentType()
@see getCompartmentType()
@see unsetCompartmentType()
";


%feature("docstring") Compartment::isSetSize "
Predicate returning @c True if this Compartment object\'s \'size\'
attribute is set.

This method is similar but not identical to
@if java Compartment.isSetVolume()@else isSetVolume()@endif.  The latter
should be used in the context of SBML Level&nbsp;1 models instead of
@if java Compartment.isSetSize()@else isSetSize()@endif@~
because @if java Compartment.isSetVolume()@else isSetVolume()@endif@~
performs extra processing to take into account the difference in
default values between SBML Levels&nbsp;1 and&nbsp;2.

@return @c True if the \'size\' attribute (\'volume\' in Level&nbsp;2) of
this Compartment object is set, @c False otherwise.

@see isSetVolume()
@see setSize()
@see getSize()
@see unsetSize()
";


%feature("docstring") Compartment::isSetVolume "
Predicate returning @c True if this Compartment object\'s \'volume\'
attribute is set.

This method is similar but not identical to
@if java Compartment.isSetSize()@else isSetSize()@endif.  The latter
should not be used in the context of SBML Level&nbsp;1 models because the
present method performs extra processing to take into account
the difference in default values between SBML Levels&nbsp;1 and&nbsp;2.

@return @c True if the \'volume\' attribute (\'size\' in Level&nbsp;2 and
above) of this Compartment object is set, @c False otherwise.

@note The attribute \'volume\' only exists by that name in SBML
Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a default
value (@c 1.0) and therefore methods such as
@if conly Compartment_isSetVolume() @else isSetVolume()@endif@~
will always return @c True for a Level&nbsp;1 model.  In Level&nbsp;2, a
compartment\'s size (the equivalent of SBML Level&nbsp;1\'s \'volume\') is
optional and has no default value, and therefore may or may not be set.

@see isSetSize()
@see getVolume()
@see setVolume()
@see unsetVolume()
";


%feature("docstring") Compartment::isSetUnits "
Predicate returning @c True if this Compartment object\'s \'units\'
attribute is set.

@return @c True if the \'units\' attribute of this Compartment object is
set, @c False otherwise.

@note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

@see setUnits()
@see getUnits()
@see unsetUnits()
";


%feature("docstring") Compartment::isSetOutside "
Predicate returning @c True if this Compartment object\'s \'outside\'
attribute is set.

@return @c True if the \'outside\' attribute of this Compartment object is
set, @c False otherwise.

@note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
Level&nbsp;2, but does not exist in SBML Level&nbsp;3.

@see getOutside()
@see setOutside()
@see unsetOutside()
";


%feature("docstring") Compartment::isSetSpatialDimensions "
Predicate returning @c True if this Compartment object\'s
\'spatialDimensions\' attribute is set.

@return @c True if the \'spatialDimensions\' attribute of this
Compartment object is set, @c False otherwise.

@see getSpatialDimensions()
@see setSpatialDimensions()
@see unsetSpatialDimensions()
";


%feature("docstring") Compartment::isSetConstant "
Predicate returning @c True if this Compartment object\'s \'constant\'
attribute is set.

@return @c True if the \'constant\' attribute of this Compartment object is
set, @c False otherwise.

@see getConstant()
@see setConstant()
";


%feature("docstring") Compartment::setId "
Sets the value of the \'id\' attribute of this Compartment object.

The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this Compartment object. If
the string is @c None, this method will return
@link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@see getId()
@see unsetId()
@see isSetId()
";


%feature("docstring") Compartment::setName "
Sets the value of the \'name\' attribute of this Compartment object.

@par

The string in @p name is copied.

@param name the new name for the SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Compartment::setCompartmentType "
Sets the \'compartmentType\' attribute of this Compartment object.

@param sid the identifier of a CompartmentType object defined elsewhere
in this Model. If the string is @c None, this method will return
@link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@note The \'compartmentType\' attribute is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.

@see isSetCompartmentType()
@see getCompartmentType()
@see unsetCompartmentType()
";


%feature("docstring") Compartment::setSpatialDimensions "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setSpatialDimensions(double value)</pre>

Sets the \'spatialDimensions\' attribute of this Compartment object as a double.

@param value a double indicating the number of dimensions
of this compartment.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getSpatialDimensions()
@see isSetSpatialDimensions()
@see unsetSpatialDimensions()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>setSpatialDimensions(long value)</pre>

Sets the \'spatialDimensions\' attribute of this Compartment object.

@param value a long integereger indicating the number of dimensions
of this compartment.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getSpatialDimensions()
@see isSetSpatialDimensions()
@see unsetSpatialDimensions()
";


%feature("docstring") Compartment::setSize "
Sets the \'size\' attribute (or \'volume\' in SBML Level&nbsp;1) of this
Compartment object.

@param value a @c float representing the size of this compartment
instance in whatever units are in effect for the compartment.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@note This method is identical to
@if java Compartment.setVolume()@else setVolume()@endif.

@see setVolume()
@see getSize()
@see isSetSize()
@see unsetSize()
";


%feature("docstring") Compartment::setVolume "
Sets the \'volume\' attribute (or \'size\' in SBML Level&nbsp;2) of this
Compartment object.

This method is identical to
@if java Compartment.setSize()@else setSize()@endif@~
and is provided for compatibility between SBML Level&nbsp;1 and
higher Levels of SBML.

@param value a @c float representing the volume of this compartment
instance in whatever units are in effect for the compartment.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

@note The attribute \'volume\' only exists by that name in SBML
Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a default
value (@c 1.0) and therefore methods such as
@if conly Compartment_isSetVolume() @else isSetVolume()@endif@~
will always return @c True for a Level&nbsp;1 model.  In Level&nbsp;2, a
compartment\'s size (the equivalent of SBML Level&nbsp;1\'s \'volume\') is
optional and has no default value, and therefore may or may not be set.

@see setSize()
@see getVolume()
@see isSetVolume()
@see unsetVolume()
";


%feature("docstring") Compartment::setUnits "
Sets the \'units\' attribute of this Compartment object.

@param sid the identifier of the defined units to use.  If @p sid is
@c None, then this method will return
@link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@see isSetUnits()
@see getUnits()
@see unsetUnits()
";


%feature("docstring") Compartment::setOutside "
Sets the \'outside\' attribute of this Compartment object.

@param sid the identifier of a compartment that encloses this one. If @p
sid is @c None, then this method will return
@link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
Level&nbsp;2, but does not exist in SBML Level&nbsp;3.

@see isSetOutside()
@see getOutside()
@see unsetOutside()
";


%feature("docstring") Compartment::setConstant "
Sets the value of the \'constant\' attribute of this Compartment object.

@param value a boolean indicating whether the size/volume of this
compartment should be considered constant (@c True) or variable
(@c False).

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see isSetConstant()
@see getConstant()
";


%feature("docstring") Compartment::renameSIdRefs "
Replaces all uses of a given @c SIdRef type attribute value with another
value.

@par
In SBML, object identifiers are of a data type called <code>SId</code>.
In SBML Level&nbsp;3, an explicit data type called <code>SIdRef</code> was
introduced for attribute values that refer to <code>SId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to an identifier\', but the effective
data type was the same as <code>SIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>SIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all attributes and (if appropriate)
mathematical formulas in MathML content, comparing the referenced
identifiers to the value of @p oldid.  If any matches are found, the
matching values are replaced with @p newid.  The method does @em not
descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") Compartment::renameUnitSIdRefs "
Replaces all uses of a given @c UnitSIdRef type attribute value with
another value.

@par
In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
introduced for attribute values that refer to <code>UnitSId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to a unit identifier\', but the effective
data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all unit identifier attribute values
(including, if appropriate, inside mathematical formulas), comparing the
referenced unit identifiers to the value of @p oldid.  If any matches
are found, the matching values are replaced with @p newid.  The method
does @em not descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") Compartment::unsetName "
Unsets the value of the \'name\' attribute of this Compartment object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") Compartment::unsetCompartmentType "
Unsets the value of the \'compartmentType\' attribute of this Compartment object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@note The \'compartmentType\' attribute is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.

@see setCompartmentType()
@see isSetCompartmentType()
@see getCompartmentType()
";


%feature("docstring") Compartment::unsetConstant "
Unsets the value of the \'constant\' attribute of this Compartment object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see isSetConstant()
@see setConstant()
@see getConstant()
";


%feature("docstring") Compartment::unsetSize "
Unsets the value of the \'size\' attribute of this Compartment object.

In SBML Level&nbsp;1, a compartment\'s volume has a default value
(@c 1.0) and therefore <em>should always be set</em>.  Calling this method
on a Level&nbsp;1 model resets the value to @c 1.0 rather than actually
unsetting it.  In Level&nbsp;2, a compartment\'s \'size\' is optional with
no default value, and unsetting it will result in the compartment having
no defined size.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note This method is identical to
@if java Compartment.unsetVolume()@else unsetVolume()@endif.

@see unsetVolume()
@see getSize()
@see isSetSize()
@see setSize()
";


%feature("docstring") Compartment::unsetVolume "
Unsets the value of the \'volume\' attribute of this Compartment object.

This method is identical to
@if java Compartment.unsetSize()@else unsetSize()@endif.  Please refer
to that method\'s documentation for more information about its behavior.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The attribute \'volume\' only exists by that name in SBML
Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a default
value (@c 1.0) and therefore methods such as
@if conly Compartment_isSetVolume() @else isSetVolume()@endif@~
will always return @c True for a Level&nbsp;1 model.  In Level&nbsp;2, a
compartment\'s size (the equivalent of SBML Level&nbsp;1\'s \'volume\') is
optional and has no default value, and therefore may or may not be set.

@see unsetSize()
@see getVolume()
@see setVolume()
@see isSetVolume()
";


%feature("docstring") Compartment::unsetUnits "
Unsets the value of the \'units\' attribute of this Compartment object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see isSetUnits()
@see setUnits()
@see getUnits()
";


%feature("docstring") Compartment::unsetOutside "
Unsets the value of the \'outside\' attribute of this Compartment object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
Level&nbsp;2, but does not exist in SBML Level&nbsp;3.

@see isSetOutside()
@see getOutside()
@see setOutside()
";


%feature("docstring") Compartment::unsetSpatialDimensions "
Unsets the value of the \'spatialDimensions\' attribute of this
Compartment object.

In SBML Levels prior to Level&nbsp;3, compartments must always have a
value for the number of dimensions.  Consequently, calling this method
on a model of SBML Level 1&ndash;2 will result in a return value of
@link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@note This function is only valid for SBML Level&nbsp;3.

@see getSpatialDimensions()
@see isSetSpatialDimensions()
@see setSpatialDimensions()
";


%feature("docstring") Compartment::getDerivedUnitDefinition "
Constructs and returns a UnitDefinition that corresponds to the units
of this Compartment object\'s designated size.

@par
Compartments in SBML have an attribute (\'units\') for declaring the
units of measurement intended for the value of the compartment\'s size.  In
the absence of a value given for this attribute, the units are inherited
from values either defined on the enclosing Model (in SBML Level&nbsp;3)
or in defaults (in SBML Level&nbsp;2).  This method returns a
@if conly UnitDefinition_t structure @else UnitDefinition object@endif@~
based on how this compartment\'s units are interpreted according to the
relevant SBML guidelines, or it returns @c None if no units have been
declared and no defaults are defined by the relevant SBML specification.

Note that unit declarations for
@if conly Compartment_t structures @else Compartment objects @endif@~
are specified in terms of the @em identifier of a unit,
but @em this method returns @if conly a structure @else an object @endif,
not a unit identifier.  It does this by constructing an appropriate
@if conly UnitDefinition_t structure @else UnitDefinition object@endif.  For
SBML Level&nbsp;2 models, it will do this even when the value of the
\'units\' attribute is one of the special SBML Level&nbsp;2 unit identifiers
@c \'substance\', @c \'volume\', @c \'area\', @c \'length\' or @c \'time\'.  Callers
may find this useful in conjunction with the helper methods provided by
the @if conly UnitDefinition_t structure @else UnitDefinition
class@endif@~ for comparing different UnitDefinition objects.

@return a UnitDefinition that expresses the units of this
Compartment object, or @c None if one cannot be constructed.

@note The libSBML system for unit analysis depends on the model as a
whole.  In cases where the
@if conly Compartment_t structure @else Compartment object@endif@~ has not
yet been added to a model, or the model itself is incomplete, unit
analysis is not possible, and consequently this method will return
@c None.

@see isSetUnits()
@see getUnits()
";


%feature("docstring") Compartment::getTypeCode "
Returns the libSBML type code for this SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_COMPARTMENT SBML_COMPARTMENT@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getElementName()
@see getPackageName()
";


%feature("docstring") Compartment::getElementName "
Returns the XML element name of this object

For Compartment, the XML element name is always @c \'compartment\'.

@return the name of this element.
";


%feature("docstring") Compartment::writeElements "
@internal

@internal
";


%feature("docstring") Compartment::hasRequiredAttributes "
Predicate returning @c True if all the required attributes for this
Compartment object have been set.

The required attributes for a Compartment object are:
@li \'id\' (or \'name\' in SBML Level&nbsp;1)
@li \'constant\' (in SBML Level&nbsp;3 only)

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") Compartment::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") Compartment::readAttributes "
@internal

@internal
";


%feature("docstring") Compartment::readL1Attributes "
@internal

@internal
";


%feature("docstring") Compartment::readL2Attributes "
@internal

@internal
";


%feature("docstring") Compartment::readL3Attributes "
@internal

@internal
";


%feature("docstring") Compartment::writeAttributes "
@internal

@internal
";


%feature("docstring") ListOfCompartments::ListOfCompartments "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfCompartments(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfCompartments object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfCompartments object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfCompartments(long level, long version)</pre>

Creates a new ListOfCompartments object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfCompartments::clone "
Creates and returns a deep copy of this ListOfCompartments object.

@return the (deep) copy of this ListOfCompartments object.
";


%feature("docstring") ListOfCompartments::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., Compartment objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for the objects contained in this ListOf
instance: @link libsbml#SBML_COMPARTMENT SBML_COMPARTMENT@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfCompartments::getElementName "
Returns the XML element name of this object.

For ListOfCompartments, the XML element name is always
@c \'listOfCompartments\'.

@return the name of this element.
";


%feature("docstring") ListOfCompartments::get "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(long n)</pre>

Get a Compartment object from the ListOfCompartments.

@param n the index number of the Compartment object to get.

@return the nth Compartment object in this ListOfCompartments.
If the index @p n is invalid, @c None is returned.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(string sid)</pre>

Get a Compartment object from the ListOfCompartments
based on its identifier.

@param sid a string representing the identifier
of the Compartment object to get.

@return Compartment object in this ListOfCompartments
with the given @p sid or @c None if no such
Compartment object exists.

@see get()
@see size()
";


%feature("docstring") ListOfCompartments::remove "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(long n)</pre>

Removes the nth item from this ListOfCompartments items and returns a pointer to
it.

The caller owns the returned item and is responsible for deleting it.

@param n the index of the item to remove.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(string sid)</pre>

Removes item in this ListOfCompartments items with the given identifier.

The caller owns the returned item and is responsible for deleting it.
If none of the items in this list have the identifier @p sid, then
@c None is returned.

@param sid the identifier of the item to remove.

@return the item removed.  As mentioned above, the caller owns the
returned item.
";


%feature("docstring") ListOfCompartments::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfCompartments::createObject "
@internal

@internal
";


%feature("docstring") Species "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html An SBML <em>species</em> &ndash; a pool of entities.

A @em species in SBML refers to a pool of entities that (a) are
considered indistinguishable from each other for the purposes of the
model, (b) participate in reactions, and (c) are located in a specific
@em compartment.  The SBML Species object class is intended to represent
these pools.

As with other major constructs in SBML, Species has a mandatory
attribute, \'id\', used to give the species type an identifier in the
model.  The identifier must be a text string conforming to the identifer
syntax permitted in SBML.  Species also has an optional \'name\'
attribute, of type @c string.  The \'id\' and \'name\' must be used
according to the guidelines described in the SBML specifications.

The required attribute \'compartment\' is used to identify the compartment
in which the species is located.  The attribute\'s value must be the
identifier of an existing Compartment object.  It is important to note
that there is no default value for the \'compartment\' attribute on
Species; every species in an SBML model must be assigned a compartment
@em explicitly.  (This also implies that every model with one or more
Species objects must define at least one Compartment object.)


@section species-amounts The initial amount and concentration of a species

The optional attributes \'initialAmount\' and \'initialConcentration\', both
having a data type of @c float, can be used to set the @em initial
quantity of the species in the compartment where the species is located.
These attributes are mutually exclusive; i.e., <em>only one</em> can
have a value on any given instance of a Species object.  Missing
\'initialAmount\' and \'initialConcentration\' values implies that their
values either are unknown, or to be obtained from an external source, or
determined by an InitialAssignment or other SBML construct elsewhere in
the model.

A species\' initial quantity in SBML is set by the \'initialAmount\' or
\'initialConcentration\' attribute exactly once.  If the \'constant\'
attribute is @c True, then the value of the species\' quantity is fixed
and cannot be changed except by an InitialAssignment.  These methods
differ in that the \'initialAmount\' and \'initialConcentration\' attributes
can only be used to set the species quantity to a literal floating-point
number, whereas the use of an InitialAssignment object allows the value
to be set using an arbitrary mathematical expression (which, thanks to
MathML\'s expressiveness, may evaluate to a rational number).  If the
species\' \'constant\' attribute is @c False, the species\' quantity value
may be overridden by an InitialAssignment or changed by AssignmentRule
or AlgebraicRule, and in addition, for <em>t &gt; 0</em>, it may also be
changed by a RateRule, Event objects, and as a result of being a
reactant or product in one or more Reaction objects.  (However, some
constructs are mutually exclusive; see the SBML specifications for the
precise details.)  It is not an error to define \'initialAmount\' or
\'initialConcentration\' on a species and also redefine the value using an
InitialAssignment, but the \'initialAmount\' or \'initialConcentration\'
setting in that case is ignored.  The SBML specifications provide
additional information about the semantics of assignments, rules and
values for simulation time <em>t</em> <= <em>0</em>.

SBML Level&nbsp;2 additionally stipulates that in cases where a species\'
compartment has a \'spatialDimensions\' value of @c 0 (zero), the species
cannot have a value for \'initialConcentration\' because the concepts of
concentration and density break down when a container has zero
dimensions.

@section species-units The units of a species\' amount or concentration

When the attribute \'initialAmount\' is set, the unit of measurement
associated with the value of \'initialAmount\' is specified by the Species
attribute \'substanceUnits\'.  When the \'initialConcentration\' attribute
is set, the unit of measurement associated with this concentration value
is {<em>unit of amount</em>} divided by {<em>unit of size</em>}, where
the {<em>unit of amount</em>} is specified by the Species
\'substanceUnits\' attribute, and the {<em>unit of size</em>} is specified
by the \'units\' attribute of the Compartment object in which the species
is located.  Note that in either case, a unit of <em>amount</em> is
involved and determined by the \'substanceUnits\' attribute.  Note
<strong>these two attributes alone do not determine the units of the
species when the species identifier appears in a mathematical
expression</strong>; <em>that</em> aspect is determined by the attribute
\'hasOnlySubstanceUnits\' discussed below.

In SBML Level&nbsp;3, if the \'substanceUnits\' attribute is not set on a
given Species object instance, then the unit of <em>amount</em> for that
species is inherited from the \'substanceUnits\' attribute on the
enclosing Model object instance.  If that attribute on Model is not set
either, then the unit associated with the species\' quantity is
undefined.

In SBML Level&nbsp;2, if the \'substanceUnits\' attribute is not set on a
given Species object instance, then the unit of <em>amount</em> for that
species is taken from the predefined SBML unit identifier
@c \'substance\'.  The value assigned to \'substanceUnits\' must be chosen from
one of the following possibilities: one of the base unit identifiers
defined in SBML, the built-in unit identifier @c \'substance\', or the
identifier of a new unit defined in the list of unit definitions in the
enclosing Model object.  The chosen units for \'substanceUnits\' must be
be @c \'dimensionless\', @c \'mole\', @c \'item\', @c \'kilogram\', @c \'gram\',
or units derived from these.

As noted at the beginning of this section, simply setting
\'initialAmount\' or \'initialConcentration\' alone does @em not determine
whether a species identifier represents an amount or a concentration
when it appears elsewhere in an SBML model.  The role of the attribute
\'hasOnlySubstanceUnits\' is to indicate whether the units of the species,
when the species identifier appears in mathematical formulas, are
intended to be concentration or amount.  The attribute takes on a
boolean value.  In SBML Level&nbsp;3, the attribute has no default value
and must always be set in a model; in SBML Level&nbsp;2, it has a
default value of @c False.

The <em>units of the species</em> are used in the following ways:
<ul>
<li> When the species\' identifier appears in a MathML formula, it
represents the species\' quantity, and the unit of measurement associated
with the quantity is as described above.

<li> The \'math\' elements of AssignmentRule, InitialAssignment and
EventAssignment objects referring to this species should all have the
same units as the unit of measurement associated with the species
quantity.

<li> In a RateRule object that defines the rate of change of the
species\' quantity, the unit associated with the rule\'s \'math\' element
should be equal to the unit of the species\' quantity divided by the
model-wide unit of <em>time</em>; in other words, {<em>unit of species
quantity</em>}/{<em>unit of time</em>}.

</ul>


@section species-constant The \'constant\' and \'boundaryCondition\' attributes

The Species object class has two boolean attributes named \'constant\' and
\'boundaryCondition\', used to indicate whether and how the quantity of
that species can vary during a simulation.  In SBML Level&nbsp;2 they
are optional; in SBML Level&nbsp;3 they are mandatory.  The following
table shows how to interpret the combined values of these attributes.

@htmlinclude species-boundarycondition.html

By default, when a species is a product or reactant of one or more
reactions, its quantity is determined by those reactions.  In SBML, it
is possible to indicate that a given species\' quantity is <em>not</em>
determined by the set of reactions even when that species occurs as a
product or reactant; i.e., the species is on the <em>boundary</em> of
the reaction system, and its quantity is not determined by the
reactions.  The boolean attribute \'boundaryCondition\' can be used to
indicate this.  A value of @c False indicates that the species @em is
part of the reaction system.  In SBML Level&nbsp;2, the attribute has a
default value of @c False, while in SBML Level&nbsp;3, it has no
default.

The \'constant\' attribute indicates whether the species\' quantity can be
changed at all, regardless of whether by reactions, rules, or constructs
other than InitialAssignment.  A value of @c False indicates that the
species\' quantity can be changed.  (This is also a common value because
the purpose of most simulations is precisely to calculate changes in
species quantities.)  In SBML Level&nbsp;2, the attribute has a default
value of @c False, while in SBML Level&nbsp;3, it has no default.  Note
that the initial quantity of a species can be set by an
InitialAssignment irrespective of the value of the \'constant\' attribute.

In practice, a \'boundaryCondition\' value of @c True means a differential
equation derived from the reaction definitions should not be generated
for the species.  However, the species\' quantity may still be changed by
AssignmentRule, RateRule, AlgebraicRule, Event, and InitialAssignment
constructs if its \'constant\' attribute is @c False.  Conversely, if the
species\' \'constant\' attribute is @c True, then its value cannot be
changed by anything except InitialAssignment.

A species having \'boundaryCondition\'=@c False and \'constant\'=@c False
can appear as a product and/or reactant of one or more reactions in the
model.  If the species is a reactant or product of a reaction, it must
@em not also appear as the target of any AssignmentRule or RateRule
object in the model.  If instead the species has \'boundaryCondition\'=
@c False and \'constant\'=@c True, then it cannot appear as a reactant or
product, or as the target of any AssignmentRule, RateRule or
EventAssignment object in the model.

Finally, it is worth clarifying that while the constant and 
boundaryCondition attributes restrict whether and how the species 
amount changes, the same is not true of a species\' concentration. In 
SBML, the concentration of a species is a quantity that depends on the 
size of the compartment in which it is located. A compartment\'s size 
may change, and therefore, so can the concentration of a species even 
if the amount of the species remains unchanged. A species\' concentration 
may therefore vary even if the Species object\'s constant attribute is 
set to @c True in a model.

@section species-l2-convfactor The conversionFactor attribute in SBML Level&nbsp;3

In SBML Level&nbsp;3, Species has an additional optional attribute,
\'conversionFactor\', that defines a conversion factor that applies to a
particular species.  The value must be the identifier of a Parameter
object instance defined in the model.  That Parameter object must be a
constant, meaning its \'constant\' attribute must be set to @c True.
If a given Species object definition defines a value for its
\'conversionFactor\' attribute, it takes precedence over any factor
defined by the Model object\'s \'conversionFactor\' attribute.

The unit of measurement associated with a species\' quantity can be
different from the unit of extent of reactions in the model.  SBML
Level&nbsp;3 avoids implicit unit conversions by providing an explicit
way to indicate any unit conversion that might be required.  The use of
a conversion factor in computing the effects of reactions on a species\'
quantity is explained in detail in the SBML Level&nbsp;3 specification
document.  Because the value of the \'conversionFactor\' attribute is the
identifier of a Parameter object, and because parameters can have units
attached to them, the transformation from reaction extent units to
species units can be completely specified using this approach.

Note that the unit conversion factor is <strong>only applied when
calculating the effect of a reaction on a species</strong>.  It is not
used in any rules or other SBML constructs that affect the species, and
it is also not used when the value of the species is referenced in a
mathematical expression.


@section species-l2-type The speciesType attribute in SBML Level&nbsp;2 Versions&nbsp;2&ndash;4

In SBML Level&nbsp;2 Versions&nbsp;2&ndash;4, each species in a model
may optionally be designated as belonging to a particular species type.
The optional attribute \'speciesType\' is used to identify the species
type of the chemical entities that make up the pool represented by the
Species objects.  The attribute\'s value must be the identifier of an
existing SpeciesType object in the model.  If the \'speciesType\'
attribute is not present on a particular species definition, it means
the pool contains chemical entities of a type unique to that pool; in
effect, a virtual species type is assumed for that species, and no other
species can belong to that species type.  The value of \'speciesType\'
attributes on species have no effect on the numerical interpretation of
a model; simulators and other numerical analysis software may ignore
\'speciesType\' attributes.

There can be only one species of a given species type in any given
compartment of a model.  More specifically, for all Species objects
having a value for the \'speciesType\' attribute, the pair
<center>
(\'speciesType\' attribute value, \'compartment\' attribute value)
</center>

must be unique across the set of all Species object in a model.


@section species-other The spatialSizeUnits attribute in SBML Level&nbsp;2 Versions&nbsp;1&ndash;2

In versions of SBML Level&nbsp;2 before Version&nbsp;3, the class
Species included an attribute called \'spatialSizeUnits\', which allowed
explicitly setting the units of size for initial concentration.  LibSBML
retains this attribute for compatibility with older definitions of
Level&nbsp;2, but its use is strongly discouraged because many software
tools do no properly interpret this unit declaration and it is
incompatible with all SBML specifications after Level&nbsp;2
Version&nbsp;3.


@section species-math Additional considerations for interpreting the numerical value of a species

Species are unique in SBML in that they have a kind of duality: a
species identifier may stand for either substance amount (meaning, a
count of the number of individual entities) or a concentration or
density (meaning, amount divided by a compartment size).  The previous
sections explain the meaning of a species identifier when it is
referenced in a mathematical formula or in rules or other SBML
constructs; however, it remains to specify what happens to a species
when the compartment in which it is located changes in size.

When a species definition has a \'hasOnlySubstanceUnits\' attribute value
of @c False and the size of the compartment in which the species is
located changes, the default in SBML is to assume that it is the
concentration that must be updated to account for the size change.  This
follows from the principle that, all other things held constant, if a
compartment simply changes in size, the size change does not in itself
cause an increase or decrease in the number of entities of any species
in that compartment.  In a sense, the default is that the @em amount of
a species is preserved across compartment size changes.  Upon such size
changes, the value of the concentration or density must be recalculated
from the simple relationship <em>concentration = amount / size</em> if
the value of the concentration is needed (for example, if the species
identifier appears in a mathematical formula or is otherwise referenced
in an SBML construct).  There is one exception: if the species\' quantity
is determined by an AssignmentRule, RateRule, AlgebraicRule, or an
EventAssignment and the species has a \'hasOnlySubstanceUnits\' attribute
value of @c False, it means that the <em>concentration</em> is assigned
by the rule or event; in that case, the <em>amount</em> must be
calculated when the compartment size changes.  (Events also require
additional care in this situation, because an event with multiple
assignments could conceivably reassign both a species quantity and a
compartment size simultaneously.  Please refer to the SBML
specifications for the details.)

Note that the above only matters if a species has a
\'hasOnlySubstanceUnits\' attribute value of @c False, meaning that the
species identifier refers to a concentration wherever the identifier
appears in a mathematical formula.  If instead the attribute\'s value is
@c True, then the identifier of the species <em>always</em> stands for
an amount wherever it appears in a mathematical formula or is referenced
by an SBML construct.  In that case, there is never a question about
whether an assignment or event is meant to affect the amount or
concentration: it is always the amount.

A particularly confusing situation can occur when the species has
\'constant\' attribute value of @c True in combination with a
\'hasOnlySubstanceUnits\' attribute value of @c False.  Suppose this
species is given a value for \'initialConcentration\'.  Does a \'constant\'
value of @c True mean that the concentration is held constant if the
compartment size changes?  No; it is still the amount that is kept
constant across a compartment size change.  The fact that the species
was initialized using a concentration value is irrelevant.
";


%feature("docstring") ListOfSpecies "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of Species objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") Species::Species "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Species(SBMLNamespaces sbmlns)</pre>

Creates a new Species using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments. 

It is worth emphasizing that although this constructor does not take
an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
(identifier) attribute of a Species is required to have a value.
Thus, callers are cautioned to assign a value after calling this
constructor.  Setting the identifier can be accomplished using the
method Species.setId().

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Species(long level, long version)</pre>

Creates a new Species using the given SBML @p level and @p version
values.

@param level a long integer, the SBML Level to assign to this Species.

@param version a long integer, the SBML Version to assign to this
Species.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Species(Species orig)</pre>

Copy constructor; creates a copy of this Species object.

@param orig the object to copy.
";


%feature("docstring") Species::accept "
@internal

@internal
";


%feature("docstring") Species::clone "
Creates and returns a deep copy of this Species object.

@return the (deep) copy of this Species object.
";


%feature("docstring") Species::initDefaults "
Initializes the fields of this Species object to \'typical\' defaults
values.

The SBML Species component has slightly different aspects and
default attribute values in different SBML Levels and Versions.
This method sets the values to certain common defaults, based
mostly on what they are in SBML Level&nbsp;2.  Specifically:
<ul>
<li> Sets \'boundaryCondition\' to @c False
<li> Sets \'constant\' to @c False
<li> sets \'hasOnlySubstanceUnits\' to @c False
<li> (Applies to Level&nbsp;3 models only) Sets attribute \'substanceUnits\' to @c mole
</ul>
";


%feature("docstring") Species::getId "
Returns the value of the \'id\' attribute of this Species.

@note Because of the inconsistent behavior of this function with 
respect to assignments and rules, it is now recommended to
use the getIdAttribute() function instead.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this Species.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") Species::getName "
Returns the value of the \'name\' attribute of this Species object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") Species::getSpeciesType "
Get the type of this Species object object.

@return the value of the \'speciesType\' attribute of this
Species as a string.

@note The \'speciesType\' attribute is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.
";


%feature("docstring") Species::getCompartment "
Get the compartment in which this species is located.

The compartment is designated by its identifier.

@return the value of the \'compartment\' attribute of this Species
object, as a string.
";


%feature("docstring") Species::getInitialAmount "
Get the value of the \'initialAmount\' attribute.

@return the initialAmount of this Species, as a floating point number.
";


%feature("docstring") Species::getInitialConcentration "
Get the value of the \'initialConcentration\' attribute.

@return the initialConcentration of this Species,, as a floating point
number.

@note The attribute \'initialConcentration\' is only available in SBML
Level&nbsp;2 and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
";


%feature("docstring") Species::getSubstanceUnits "
Get the value of the \'substanceUnits\' attribute.

@return the value of the \'substanceUnits\' attribute of this Species,
as a string.  An empty string indicates that no units have been
assigned.

@note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

@see isSetSubstanceUnits()
@see setSubstanceUnits()
";


%feature("docstring") Species::getSpatialSizeUnits "
Get the value of the \'spatialSizeUnits\' attribute.

@return the value of the \'spatialSizeUnits\' attribute of this Species
object, as a string.

@warning <span class=\'warning\'>In versions of SBML Level&nbsp;2 before
Version&nbsp;3, the class Species included an attribute called
\'spatialSizeUnits\', which allowed explicitly setting the units of size for
initial concentration.  This attribute was removed in SBML Level&nbsp;2
Version&nbsp;3.  LibSBML retains this attribute for compatibility with
older definitions of Level&nbsp;2, but its use is strongly discouraged
because it is incompatible with levels and versions of SBML beyond
Level&nbsp;2 Version&nbsp;2.</span>
";


%feature("docstring") Species::getUnits "
Get the value of the \'units\' attribute.

@return the units of this Species (L1 only).

@note The \'units\' attribute is defined only in SBML Level&nbsp;1.  In
SBML Level&nbsp;2 and Level&nbsp;3, it has been replaced by a
combination of \'substanceUnits\' and the units of the Compartment
object in which a species is located.  In SBML Level&nbsp;2
Versions&nbsp;1&ndash;2, an additional attribute \'spatialSizeUnits\'
helps determine the units of the species quantity, but this attribute
was removed in later versions of SBML Level&nbsp;2.
";


%feature("docstring") Species::getHasOnlySubstanceUnits "
Get the value of the \'hasOnlySubstanceUnits\' attribute.

@return @c True if this Species\' \'hasOnlySubstanceUnits\' attribute
value is @c True, @c False otherwise.

@note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
Level&nbsp;1.
";


%feature("docstring") Species::getBoundaryCondition "
Get the value of the \'boundaryCondition\' attribute.

@return @c True if this Species\' \'boundaryCondition\' attribute value
is @c True, @c False otherwise.
";


%feature("docstring") Species::getCharge "
Get the value of the \'charge\' attribute.

@return the charge of this Species object.

@note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
attribute on Species is deprecated and in SBML Level&nbsp;3 it does
not exist at all.  Its use strongly discouraged.  Its presence is
considered a misfeature in earlier definitions of SBML because its
implications for the mathematics of a model were never defined, and in
any case, no known modeling system ever used it.  Instead, models take
account of charge values directly in their definitions of species by
(for example) having separate species identities for the charged and
uncharged versions of the same species.  This allows the condition to
affect model mathematics directly.  LibSBML retains this method for
easier compatibility with SBML Level&nbsp;1.
";


%feature("docstring") Species::getConstant "
Get the value of the \'constant\' attribute.

@return @c True if this Species\'s \'constant\' attribute value is
@c True, @c False otherwise.

@note The attribute \'constant\' is only available in SBML Levels&nbsp;2
and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
";


%feature("docstring") Species::getConversionFactor "
Get the value of the \'conversionFactor\' attribute.

@return the conversionFactor of this Species, as a string.

@note The \'conversionFactor\' attribute was introduced in SBML
Level&nbsp;3.  It does not exist on Species in SBML Levels&nbsp;1
and&nbsp;2.
";


%feature("docstring") Species::isSetId "
Predicate returning @c True if this
Species object\'s \'id\' attribute is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") Species::isSetName "
Predicate returning @c True if this
Species object\'s \'name\' attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") Species::isSetSpeciesType "
Predicate returning @c True if this Species object\'s
\'speciesType\' attribute is set.

@return @c True if the \'speciesType\' attribute of this Species is
set, @c False otherwise.

@note The \'speciesType\' attribute is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.
";


%feature("docstring") Species::isSetCompartment "
Predicate returning @c True if this
Species object\'s \'compartment\' attribute is set.

@return @c True if the \'compartment\' attribute of this Species is
set, @c False otherwise.
";


%feature("docstring") Species::isSetInitialAmount "
Predicate returning @c True if this
Species object\'s \'initialAmount\' attribute is set.

@return @c True if the \'initialAmount\' attribute of this Species is
set, @c False otherwise.

@note In SBML Level&nbsp;1, Species\' \'initialAmount\' is required and
therefore <em>should always be set</em>.  (However, in Level&nbsp;1, the
attribute has no default value either, so this method will not return
@c True until a value has been assigned.)  In SBML Level&nbsp;2,
\'initialAmount\' is optional and as such may or may not be set.
";


%feature("docstring") Species::isSetInitialConcentration "
Predicate returning @c True if this
Species object\'s \'initialConcentration\' attribute is set.

@return @c True if the \'initialConcentration\' attribute of this Species is
set, @c False otherwise.

@note The attribute \'initialConcentration\' is only available in SBML
Level&nbsp;2 and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
";


%feature("docstring") Species::isSetSubstanceUnits "
Predicate returning @c True if this
Species object\'s \'substanceUnits\' attribute is set.

@return @c True if the \'substanceUnits\' attribute of this Species is
set, @c False otherwise.
";


%feature("docstring") Species::isSetSpatialSizeUnits "
Predicate returning @c True if this
Species object\'s \'spatialSizeUnits\' attribute is set.

@return @c True if the \'spatialSizeUnits\' attribute of this Species is
set, @c False otherwise.

@warning <span class=\'warning\'>In versions of SBML Level&nbsp;2 before
Version&nbsp;3, the class Species included an attribute called
\'spatialSizeUnits\', which allowed explicitly setting the units of size for
initial concentration.  This attribute was removed in SBML Level&nbsp;2
Version&nbsp;3.  LibSBML retains this attribute for compatibility with
older definitions of Level&nbsp;2, but its use is strongly discouraged
because it is incompatible with levels and versions of SBML beyond
Level&nbsp;2 Version&nbsp;2.</span>
";


%feature("docstring") Species::isSetUnits "
Predicate returning @c True if
this Species object\'s \'units\' attribute is set.

@return @c True if the \'units\' attribute of this Species is
set, @c False otherwise.
";


%feature("docstring") Species::isSetCharge "
Predicate returning @c True if this
Species object\'s \'charge\' attribute is set.

@return @c True if the \'charge\' attribute of this Species is
set, @c False otherwise.

@note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
attribute on Species is deprecated and in SBML Level&nbsp;3 it does
not exist at all.  Its use strongly discouraged.  Its presence is
considered a misfeature in earlier definitions of SBML because its
implications for the mathematics of a model were never defined, and in
any case, no known modeling system ever used it.  Instead, models take
account of charge values directly in their definitions of species by
(for example) having separate species identities for the charged and
uncharged versions of the same species.  This allows the condition to
affect model mathematics directly.  LibSBML retains this method for
easier compatibility with SBML Level&nbsp;1.
";


%feature("docstring") Species::isSetConversionFactor "
Predicate returning @c True if this
Species object\'s \'conversionFactor\' attribute is set.

@return @c True if the \'conversionFactor\' attribute of this Species is
set, @c False otherwise.

@note The \'conversionFactor\' attribute was introduced in SBML
Level&nbsp;3.  It does not exist on Species in SBML Levels&nbsp;1
and&nbsp;2.
";


%feature("docstring") Species::isSetBoundaryCondition "
Predicate returning @c True if this
Species object\'s \'boundaryCondition\' attribute is set.

@return @c True if the \'boundaryCondition\' attribute of this Species is
set, @c False otherwise.
";


%feature("docstring") Species::isSetHasOnlySubstanceUnits "
Predicate returning @c True if this
Species object\'s \'hasOnlySubstanceUnits\' attribute is set.

@return @c True if the \'hasOnlySubstanceUnits\' attribute of this Species is
set, @c False otherwise.

@note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
Level&nbsp;1.
";


%feature("docstring") Species::isSetConstant "
Predicate returning @c True if this
Species object\'s \'constant\' attribute is set.

@return @c True if the \'constant\' attribute of this Species is
set, @c False otherwise.

@note The attribute \'constant\' is only available in SBML Levels&nbsp;2
and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
";


%feature("docstring") Species::setId "
Sets the value of the \'id\' attribute of this Species.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") Species::setName "
Sets the value of the \'name\' attribute of this Species.

@par

The string in @p name is copied.

@param name the new name for the SBML object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Species::setSpeciesType "
Sets the \'speciesType\' attribute of this Species object.

@param sid the identifier of a SpeciesType object defined elsewhere
in this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@note The \'speciesType\' attribute is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.
";


%feature("docstring") Species::setCompartment "
Sets the \'compartment\' attribute of this Species object.

@param sid the identifier of a Compartment object defined elsewhere
in this Model.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Species::setInitialAmount "
Sets the \'initialAmount\' attribute of this Species and marks the field
as set.

This method also unsets the \'initialConcentration\' attribute.

@param value the value to which the \'initialAmount\' attribute should
be set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
";


%feature("docstring") Species::setInitialConcentration "
Sets the \'initialConcentration\' attribute of this Species and marks
the field as set.

This method also unsets the \'initialAmount\' attribute.

@param value the value to which the \'initialConcentration\' attribute
should be set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@note The attribute \'initialConcentration\' is only available in SBML
Level&nbsp;2 and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
";


%feature("docstring") Species::setSubstanceUnits "
Sets the \'substanceUnits\' attribute of this Species object.

@param sid the identifier of the unit to use.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Species::setSpatialSizeUnits "
(SBML Level&nbsp;2 Versions&nbsp;1&ndash;2) Sets the \'spatialSizeUnits\' attribute of this Species object.

@param sid the identifier of the unit to use.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@warning <span class=\'warning\'>In versions of SBML Level&nbsp;2 before
Version&nbsp;3, the class Species included an attribute called
\'spatialSizeUnits\', which allowed explicitly setting the units of size for
initial concentration.  This attribute was removed in SBML Level&nbsp;2
Version&nbsp;3.  LibSBML retains this attribute for compatibility with
older definitions of Level&nbsp;2, but its use is strongly discouraged
because it is incompatible with levels and versions of SBML beyond
Level&nbsp;2 Version&nbsp;2.</span>
";


%feature("docstring") Species::setUnits "
(SBML Level&nbsp;1 only) Sets the units of this Species object.

@param sname the identifier of the unit to use.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Species::setHasOnlySubstanceUnits "
Sets the \'hasOnlySubstanceUnits\' attribute of this Species object.

@param value boolean value for the \'hasOnlySubstanceUnits\' attribute.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
Level&nbsp;1.
";


%feature("docstring") Species::setBoundaryCondition "
Sets the \'boundaryCondition\' attribute of this Species object.

@param value boolean value for the \'boundaryCondition\' attribute.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
";


%feature("docstring") Species::setCharge "
Sets the \'charge\' attribute of this Species object.

@param value an integer to which to set the \'charge\' to.

@note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
attribute on Species is deprecated and in SBML Level&nbsp;3 it does
not exist at all.  Its use strongly discouraged.  Its presence is
considered a misfeature in earlier definitions of SBML because its
implications for the mathematics of a model were never defined, and in
any case, no known modeling system ever used it.  Instead, models take
account of charge values directly in their definitions of species by
(for example) having separate species identities for the charged and
uncharged versions of the same species.  This allows the condition to
affect model mathematics directly.  LibSBML retains this method for
easier compatibility with SBML Level&nbsp;1. 

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
";


%feature("docstring") Species::setConstant "
Sets the \'constant\' attribute of this Species object.

@param value a boolean value for the \'constant\' attribute.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@note The attribute \'constant\' is only available in SBML Levels&nbsp;2
and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
";


%feature("docstring") Species::setConversionFactor "
Sets the value of the \'conversionFactor\' attribute of this Species object.

The string in @p sid is copied.

@param sid the new conversionFactor for the Species.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink

@note The \'conversionFactor\' attribute was introduced in SBML
Level&nbsp;3.  It does not exist on Species in SBML Levels&nbsp;1
and&nbsp;2.
";


%feature("docstring") Species::unsetName "
Unsets the value of the \'name\' attribute of this Species object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") Species::unsetConstant "
Unsets the value of the \'constant\' attribute of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see isSetConstant()
@see setConstant()
@see getConstant()
";


%feature("docstring") Species::unsetSpeciesType "
Unsets the \'speciesType\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The attribute \'speciesType\' is only available in SBML
Level&nbsp;2 Versions&nbsp;2&ndash;4.
";


%feature("docstring") Species::unsetInitialAmount "
Unsets the \'initialAmount\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") Species::unsetInitialConcentration "
Unsets the \'initialConcentration\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The attribute \'initialConcentration\' is only available in SBML
Level&nbsp;2 and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
";


%feature("docstring") Species::unsetSubstanceUnits "
Unsets the \'substanceUnits\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") Species::unsetSpatialSizeUnits "
Unsets the \'spatialSizeUnits\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@warning <span class=\'warning\'>In versions of SBML Level&nbsp;2 before
Version&nbsp;3, the class Species included an attribute called
\'spatialSizeUnits\', which allowed explicitly setting the units of size for
initial concentration.  This attribute was removed in SBML Level&nbsp;2
Version&nbsp;3.  LibSBML retains this attribute for compatibility with
older definitions of Level&nbsp;2, but its use is strongly discouraged
because it is incompatible with levels and versions of SBML beyond
Level&nbsp;2 Version&nbsp;2.</span>
";


%feature("docstring") Species::unsetUnits "
Unsets the \'units\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") Species::unsetCharge "
Unsets the \'charge\' attribute
value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
attribute on Species is deprecated and in SBML Level&nbsp;3 it does
not exist at all.  Its use strongly discouraged.  Its presence is
considered a misfeature in earlier definitions of SBML because its
implications for the mathematics of a model were never defined, and in
any case, no known modeling system ever used it.  Instead, models take
account of charge values directly in their definitions of species by
(for example) having separate species identities for the charged and
uncharged versions of the same species.  This allows the condition to
affect model mathematics directly.  LibSBML retains this method for
easier compatibility with SBML Level&nbsp;1.
";


%feature("docstring") Species::unsetConversionFactor "
Unsets the \'conversionFactor\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@note The \'conversionFactor\' attribute was introduced in SBML
Level&nbsp;3.  It does not exist on Species in SBML Levels&nbsp;1
and&nbsp;2.
";


%feature("docstring") Species::unsetCompartment "
Unsets the \'compartment\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") Species::unsetBoundaryCondition "
Unsets the \'boundaryCondition\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") Species::unsetHasOnlySubstanceUnits "
Unsets the \'hasOnlySubstanceUnits\' attribute value of this Species object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") Species::getDerivedUnitDefinition "
Constructs and returns a UnitDefinition that corresponds to the units
of this Species\' amount or concentration.

Species in SBML have an attribute (\'substanceUnits\') for declaring the
units of measurement intended for the species\' amount or concentration
(depending on which one applies).  In the absence of a value given for
\'substanceUnits\', the units are taken from the enclosing Model\'s
definition of @c \'substance\' or @c \'substance\'/<em>(size of the
compartment)</em> in which the species is located, or finally, if
these are not redefined by the Model, the relevant SBML default units
for those quantities.  Following that procedure, the method
@if java Species.getDerivedUnitDefinition()@else getDerivedUnitDefinition()@endif@~
returns a UnitDefinition based on the
interpreted units of this species\'s amount or concentration.

Note that the functionality that facilitates unit analysis depends 
on the model as a whole.  Thus, in cases where the object has not 
been added to a model or the model itself is incomplete,
unit analysis is not possible and this method will return @c None.

Note also that unit declarations for Species are in terms of the @em
identifier of a unit, but this method returns a UnitDefinition object,
not a unit identifier.  It does this by constructing an appropriate
UnitDefinition.  Callers may find this particularly useful when used
in conjunction with the helper methods on UnitDefinition for comparing
different UnitDefinition objects.

In SBML Level&nbsp;2 specifications prior to Version&nbsp;3, Species
includes an additional attribute named \'spatialSizeUnits\', which
allows explicitly setting the units of size for initial concentration.
The @if java Species.getDerivedUnitDefinition()@else getDerivedUnitDefinition()@endif@~
takes this into account for models
expressed in SBML Level&nbsp;2 Versions&nbsp;1 and&nbsp;2.

@return a UnitDefinition that expresses the units of this 
Species, or @c None if one cannot be constructed.

@see getSubstanceUnits()
";


%feature("docstring") Species::getTypeCode "
Returns the libSBML type code for this SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_SPECIES SBML_SPECIES@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getElementName()
@see getPackageName()
";


%feature("docstring") Species::getElementName "
Returns the XML element name of this object, which for Species, is
always @c \'species\'.

@return the name of this element, i.e., @c \'species\'.
";


%feature("docstring") Species::writeElements "
@internal

@internal
";


%feature("docstring") Species::hasRequiredAttributes "
Predicate returning @c True if
all the required attributes for this Species object
have been set.

The required attributes for a Species object are:
@li \'id\' (or \'name\' in SBML Level&nbsp;1)
@li \'compartment\'
@li \'initialAmount\' (required in SBML Level&nbsp;1 only; optional otherwise)
@li \'hasOnlySubstanceUnits\' (required in SBML Level&nbsp;3; optional in SBML Level&nbsp;2)
@li \'boundaryCondition\' (required in SBML Level&nbsp;3; optional in Levels&nbsp;1 and&nbsp;2)
@li \'constant\' (required in SBML Level&nbsp;3; optional in SBML Level&nbsp;2)

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") Species::renameSIdRefs "
Replaces all uses of a given @c SIdRef type attribute value with another
value.

@par
In SBML, object identifiers are of a data type called <code>SId</code>.
In SBML Level&nbsp;3, an explicit data type called <code>SIdRef</code> was
introduced for attribute values that refer to <code>SId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to an identifier\', but the effective
data type was the same as <code>SIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>SIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all attributes and (if appropriate)
mathematical formulas in MathML content, comparing the referenced
identifiers to the value of @p oldid.  If any matches are found, the
matching values are replaced with @p newid.  The method does @em not
descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") Species::renameUnitSIdRefs "
Replaces all uses of a given @c UnitSIdRef type attribute value with
another value.

@par
In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
introduced for attribute values that refer to <code>UnitSId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to a unit identifier\', but the effective
data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all unit identifier attribute values
(including, if appropriate, inside mathematical formulas), comparing the
referenced unit identifiers to the value of @p oldid.  If any matches
are found, the matching values are replaced with @p newid.  The method
does @em not descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") Species::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") Species::readAttributes "
@internal

@internal
";


%feature("docstring") Species::readL1Attributes "
@internal

@internal
";


%feature("docstring") Species::readL2Attributes "
@internal

@internal
";


%feature("docstring") Species::readL3Attributes "
@internal

@internal
";


%feature("docstring") Species::writeAttributes "
@internal

@internal
";


%feature("docstring") ListOfSpecies::ListOfSpecies "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfSpecies(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfSpecies object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfSpecies object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfSpecies(long level, long version)</pre>

Creates a new ListOfSpecies object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfSpecies::clone "
Creates and returns a deep copy of this ListOfSpecies object.

@return the (deep) copy of this ListOfSpecies object.
";


%feature("docstring") ListOfSpecies::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., Species objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for objects contained in this list:
@link libsbml#SBML_SPECIES SBML_SPECIES@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfSpecies::getElementName "
Returns the XML element name of this object.

For ListOfSpeciess, the XML element name is @c \'listOfSpeciess\'.

@return the name of this element, i.e., @c \'listOfSpeciess\'.
";


%feature("docstring") ListOfSpecies::get "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(long n)</pre>

Get a Species from the ListOfSpecies.

@param n the index number of the Species to get.

@return the nth Species in this ListOfSpecies.
If the index @p n is invalid, @c None is returned.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(string sid)</pre>

Get a Species from the ListOfSpecies
based on its identifier.

@param sid a string representing the identifier 
of the Species to get.

@return Species in this ListOfSpecies
with the given @p sid or @c None if no such
Species exists.

@see get()
@see size()
";


%feature("docstring") ListOfSpecies::remove "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(long n)</pre>

Removes the nth item from this ListOfSpeciess items and returns a pointer to
it.

The caller owns the returned item and is responsible for deleting it.

@param n the index of the item to remove.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(string sid)</pre>

Removes item in this ListOfSpeciess items with the given identifier.

The caller owns the returned item and is responsible for deleting it.
If none of the items in this list have the identifier @p sid, then
@c None is returned.

@param sid the identifier of the item to remove.

@return the item removed.  As mentioned above, the caller owns the
returned item.
";


%feature("docstring") ListOfSpecies::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfSpecies::createObject "
@internal

@internal
";


%feature("docstring") Parameter "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html An SBML parameter: a named symbol with a value.

A Parameter is used in SBML to define a symbol associated with a value;
this symbol can then be used in mathematical formulas in a model.  By
default, parameters have constant value for the duration of a
simulation, and for this reason are called @em parameters instead of @em
variables in SBML, although it is crucial to understand that <em>SBML
parameters represent both concepts</em>.  Whether a given SBML
parameter is intended to be constant or variable is indicated by the
value of its \'constant\' attribute.

SBML\'s Parameter has a required attribute, \'id\', that gives the
parameter a unique identifier by which other parts of an SBML model
definition can refer to it.  A parameter can also have an optional
\'name\' attribute of type @c string.  Identifiers and names must be used
according to the guidelines described in the SBML specifications.

The optional attribute \'value\' determines the value (of type @c float)
assigned to the parameter.  A missing value for \'value\' implies that
the value either is unknown, or to be obtained from an external source,
or determined by an initial assignment.  The unit of measurement
associated with the value of the parameter can be specified using the
optional attribute \'units\'.  Here we only mention briefly some notable
points about the possible unit choices, but readers are urged to consult
the SBML specification documents for more information:
<ul>

<li> In SBML Level&nbsp;3, there are no constraints on the units that
can be assigned to parameters in a model; there are also no units to
inherit from the enclosing Model object (unlike the case for, e.g.,
Species and Compartment).

<li> In SBML Level&nbsp;2, the value assigned to the parameter\'s \'units\'
attribute must be chosen from one of the following possibilities: one of
the base unit identifiers defined in SBML; one of the built-in unit
identifiers @c \'substance\', @c \'time\', @c \'volume\', @c \'area\' or
@c \'length\'; or the identifier of a new unit defined in the list of unit
definitions in the enclosing Model structure.  There are no constraints
on the units that can be chosen from these sets.  There are no default
units for parameters.
</ul>

The Parameter structure has another boolean attribute named \'constant\'
that is used to indicate whether the parameter\'s value can vary during a
simulation.  (In SBML Level&nbsp;3, the attribute is mandatory and must
be given a value; in SBML Levels below Level&nbsp;3, the attribute is
optional.)  A value of @c True indicates the parameter\'s value cannot be
changed by any construct except InitialAssignment.  Conversely, if the
value of \'constant\' is @c False, other constructs in SBML, such as rules
and events, can change the value of the parameter.

SBML Level&nbsp;3 uses a separate object class, LocalParameter, for
parameters that are local to a Reaction\'s KineticLaw.  In Levels prior
to SBML Level&nbsp;3, the Parameter class is used both for definitions
of global parameters, as well as reaction-local parameters stored in a
list within KineticLaw objects.  Parameter objects that are local to a
reaction (that is, those defined within the KineticLaw structure of a
Reaction) cannot be changed by rules and therefore are <em>implicitly
always constant</em>; consequently, in SBML Level&nbsp;2, parameter
definitions within Reaction structures should @em not have their
\'constant\' attribute set to @c False.

What if a global parameter has its \'constant\' attribute set to @c False,
but the model does not contain any rules, events or other constructs
that ever change its value over time?  Although the model may be
suspect, this situation is not strictly an error.  A value of @c False
for \'constant\' only indicates that a parameter @em can change value, not
that it @em must.

As with all other major SBML components, Parameter is derived from
SBase, and the methods defined on SBase are available on Parameter.

@note The use of the term @em parameter in SBML sometimes leads to
confusion among readers who have a particular notion of what something
called \'parameter\' should be.  It has been the source of heated debate,
but despite this, no one has yet found an adequate replacement term that
does not have different connotations to different people and hence leads
to confusion among @em some subset of users.  Perhaps it would have been
better to have two constructs, one called @em constants and the other
called @em variables.  The current approach in SBML is simply more
parsimonious, using a single Parameter construct with the boolean flag
\'constant\' indicating which flavor it is.  In any case, readers are
implored to look past their particular definition of a @em parameter and
simply view SBML\'s Parameter as a single mechanism for defining both
constants and (additional) variables in a model.  (We write @em
additional because the species in a model are usually considered to be
the central variables.)  After all, software tools are not required to
expose to users the actual names of particular SBML constructs, and
thus tools can present to their users whatever terms their designers
feel best matches their target audience.

In SBML Level&nbsp;3 Version&nbsp;2, many restrictions were lifted 
requiring only Boolean values in Boolean contexts, and numeric
values in numeric contexts.  This means that a Parameter may now
be used as a Boolean, despite canonically having a numeric value.
To be consistent, one should always assign it a value of @c True
or @c False, and use it in Boolean contexts exclusively.  It would
be appropriate to give it an SBO value of 602 (\'Logical parameter\')
if one chooses to do this.

@see ListOfParameters
";


%feature("docstring") ListOfParameters "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of Parameter objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") Parameter::Parameter "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Parameter(SBMLNamespaces sbmlns)</pre>

Creates a new Parameter using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments. 

It is worth emphasizing that although this constructor does not take
an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
(identifier) attribute of a Parameter is required to have a value.
Thus, callers are cautioned to assign a value after calling this
constructor if no identifier is provided as an argument.  Setting the
identifier can be accomplished using the method
@if java setId(String id)@else setId()@endif.

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Parameter(long level, long version)</pre>

Creates a new Parameter using the given SBML @p level and @p version
values.

@param level a long integer, the SBML Level to assign to this Parameter.

@param version a long integer, the SBML Version to assign to this
Parameter.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>Parameter(Parameter orig)</pre>

Copy constructor; creates a copy of a Parameter.

@param orig the Parameter instance to copy.
";


%feature("docstring") Parameter::accept "
@internal

@internal
";


%feature("docstring") Parameter::clone "
Creates and returns a deep copy of this Parameter object.

@return the (deep) copy of this Parameter object.
";


%feature("docstring") Parameter::initDefaults "
Initializes the fields of this Parameter object to \'typical\' defaults
values.

The SBML Parameter component has slightly different aspects and
default attribute values in different SBML Levels and Versions.  Many
SBML object classes defined by libSBML have an initDefaults() method
to set the values to certain common defaults, based mostly on what
they are in SBML Level&nbsp;2.  In the case of Parameter, this method
only sets the value of the \'constant\' attribute to @c True.

@see getConstant()
@see isSetConstant()
@see setConstant()
";


%feature("docstring") Parameter::getId "
Returns the value of the \'id\' attribute of this Parameter.

@note Because of the inconsistent behavior of this function with 
respect to assignments and rules, it is now recommended to
use the getIdAttribute() function instead.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return the id of this Parameter.

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") Parameter::getName "
Returns the value of the \'name\' attribute of this Parameter object.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return the name of this SBML object, or the empty string if not set or unsettable.

@see getIdAttribute()
@see isSetName()
@see setName()
@see unsetName()
";


%feature("docstring") Parameter::getValue "
Returns the numerical value of this Parameter.

@return the value of the \'value\' attribute of this Parameter, as a
number of type @c float.

@note <b>It is crucial</b> that callers not blindly call
Parameter.getValue() without first using Parameter.isSetValue() to
determine whether a value has ever been set.  Otherwise, the value
return by Parameter.getValue() may not actually represent a value
assigned to the parameter.  The reason is simply that the data type
@c float in a program always has @em some value.  A separate test is
needed to determine whether the value is a true model value, or
uninitialized data in a computer\'s memory location.

@see isSetValue()
@see setValue()
@see getUnits()
";


%feature("docstring") Parameter::getUnits "
Returns the units defined for this Parameter.

The value of an SBML parameter\'s \'units\' attribute establishes the
unit of measurement associated with the parameter\'s value.

@return the value of the \'units\' attribute of this Parameter, as a
string.  An empty string indicates that no units have been assigned.

@note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

@see isSetUnits()
@see setUnits()
@see getValue()
";


%feature("docstring") Parameter::getConstant "
Returns the value of the \'constant\' attribute of this Parameter instance.

@return @c True if this Parameter is declared as being constant,
@c False otherwise.

@note Readers who view the documentation for LocalParameter may be
confused about the presence of this method.  LibSBML derives
LocalParameter from Parameter; however, this does not precisely match
the object hierarchy defined by SBML Level&nbsp;3, where
LocalParameter is derived directly from SBase and not Parameter.  We
believe this arrangement makes it easier for libSBML users to program
applications that work with both SBML Level&nbsp;2 and SBML
Level&nbsp;3, but programmers should also keep in mind this difference
exists.  A side-effect of libSBML\'s scheme is that certain methods on
LocalParameter that are inherited from Parameter do not actually have
relevance to LocalParameter objects.  An example of this is the
methods pertaining to Parameter\'s attribute \'constant\' (i.e.,
isSetConstant(), setConstant(), and getConstant()).

@see isSetConstant()
@see setConstant()
";


%feature("docstring") Parameter::isSetId "
Predicate returning @c True if this
Parameter\'s \'id\' attribute is set.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@return @c True if the \'id\' attribute of this SBML object is
set, @c False otherwise.

@note Because of the inconsistent behavior of this function with
respect to assignments and rules, it is recommended that callers
use isSetIdAttribute() instead.

@see getIdAttribute()
@see setIdAttribute()
@see unsetIdAttribute()
@see isSetIdAttribute()
";


%feature("docstring") Parameter::isSetName "
Predicate returning @c True if this
Parameter\'s \'name\' attribute is set.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return @c True if the \'name\' attribute of this SBML object is
set, @c False otherwise.

@see getName()
@see setName()
@see unsetName()
";


%feature("docstring") Parameter::isSetValue "
Predicate returning @c True if the
\'value\' attribute of this Parameter is set.

In SBML definitions after SBML Level&nbsp;1 Version&nbsp;1,
parameter values are optional and have no defaults.  If a model read
from a file does not contain a setting for the \'value\' attribute of a
parameter, its value is considered unset; it does not default to any
particular value.  Similarly, when a Parameter object is created in
libSBML, it has no value until given a value.  The
Parameter.isSetValue() method allows calling applications to
determine whether a given parameter\'s value has ever been set.

In SBML Level&nbsp;1 Version&nbsp;1, parameters are required to have
values and therefore, the value of a Parameter <b>should always be
set</b>.  In Level&nbsp;1 Version&nbsp;2 and beyond, the value is
optional and as such, the \'value\' attribute may or may not be set.

@return @c True if the value of this Parameter is set,
@c False otherwise.

@see getValue()
@see setValue()
";


%feature("docstring") Parameter::isSetUnits "
Predicate returning @c True if the
\'units\' attribute of this Parameter is set.

@return @c True if the \'units\' attribute of this Parameter is
set, @c False otherwise.

@note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.
";


%feature("docstring") Parameter::isSetConstant "
Predicate returning @c True if the
\'constant\' attribute of this Parameter is set.

@return @c True if the \'constant\' attribute of this Parameter is
set, @c False otherwise.

@note Readers who view the documentation for LocalParameter may be
confused about the presence of this method.  LibSBML derives
LocalParameter from Parameter; however, this does not precisely match
the object hierarchy defined by SBML Level&nbsp;3, where
LocalParameter is derived directly from SBase and not Parameter.  We
believe this arrangement makes it easier for libSBML users to program
applications that work with both SBML Level&nbsp;2 and SBML
Level&nbsp;3, but programmers should also keep in mind this difference
exists.  A side-effect of libSBML\'s scheme is that certain methods on
LocalParameter that are inherited from Parameter do not actually have
relevance to LocalParameter objects.  An example of this is the
methods pertaining to Parameter\'s attribute \'constant\' (i.e.,
isSetConstant(), setConstant(), and getConstant()).

@see getConstant()
@see setConstant()
";


%feature("docstring") Parameter::setId "
Sets the value of the \'id\' attribute of this Parameter.

@par
The string @p sid is copied.

@par
The identifier given by an object\'s \'id\' attribute value
is used to identify the object within the SBML model definition.
Other objects can refer to the component using this identifier.  The
data type of \'id\' is always <code>SId</code> or a type derived
from that, such as <code>UnitSId</code>, depending on the object in 
question.  All data types are defined as follows:
<pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
letter ::= \'a\'..\'z\',\'A\'..\'Z\'
digit  ::= \'0\'..\'9\'
idChar ::= letter | digit | \'_\'
SId    ::= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping,
the character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
is determined by an exact character sequence match; i.e., comparisons must
be performed in a case-sensitive manner.  This applies to all uses of
<code>SId</code>, <code>SIdRef</code>, and derived types.

Users need to be aware of some important API issues that are the result of
the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
of SBML objects.  To simplify the work of programmers, libSBML\'s API
provided get, set, check, and unset on the SBase object class itself
instead of on individual subobject classes. This made the
get/set/etc. methods uniformly available on all objects in the libSBML
API.  LibSBML simply returned empty strings or otherwise did not act when
the methods were applied to SBML objects that were not defined by the SBML
specification to have \'id\' or \'name\' attributes.  Additional complications
arose with the rule and assignment objects: InitialAssignment,
EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
the rule object hierarchy was different, and in addition, then as now,
they possess different attributes: \'variable\' (for the rules and event
assignments), \'symbol\' (for initial assignments), or neither (for
algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
would always return an empty string, and isSetId() would always return @c False for objects of these classes.

With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
Version&nbsp;2, it became necessary to introduce a new way to interact
with the attributes more consistently in libSBML to avoid breaking
backward compatibility in the behavior of the original \'id\' methods.  For
this reason, libSBML provides four functions (getIdAttribute(),
setIdAttribute(@if java String@endif), isSetIdAttribute(), and
unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
from SBase, regardless of the object\'s type.  <strong>These new methods
should be used instead of the older getId()/setId()/etc. methods</strong>
unless the old behavior is somehow necessary.  Regardless of the Level and
Version of the SBML, these functions allow client applications to use more
generalized code in some situations (for instance, when manipulating
objects that are all known to have identifiers).  If the object in
question does not posess an \'id\' attribute according to the SBML
specification for the Level and Version in use, libSBML will not allow the
identifier to be set, nor will it read or write \'id\' attributes for those
objects.

@param sid the string to use as the identifier of this object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@see getIdAttribute()
@see setIdAttribute()
@see isSetIdAttribute()
@see unsetIdAttribute()
";


%feature("docstring") Parameter::setName "
Sets the value of the \'name\' attribute of this Parameter.

The string in @p name is copied.

@param name the new name for the Parameter.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Parameter::setValue "
Sets the \'value\' attribute of this Parameter to the given @c float
value and marks the attribute as set.

@param value a @c float, the value to assign.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
";


%feature("docstring") Parameter::setUnits "
Sets the \'units\' attribute of this Parameter to a copy of the given
units identifier @p units.

@param units a string, the identifier of the units to assign to this
Parameter instance.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") Parameter::setConstant "
Sets the \'constant\' attribute of this Parameter to the given boolean
@p flag.

@param flag a boolean, the value for the \'constant\' attribute of this
Parameter instance.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink

@note Readers who view the documentation for LocalParameter may be
confused about the presence of this method.  LibSBML derives
LocalParameter from Parameter; however, this does not precisely match
the object hierarchy defined by SBML Level&nbsp;3, where
LocalParameter is derived directly from SBase and not Parameter.  We
believe this arrangement makes it easier for libSBML users to program
applications that work with both SBML Level&nbsp;2 and SBML
Level&nbsp;3, but programmers should also keep in mind this difference
exists.  A side-effect of libSBML\'s scheme is that certain methods on
LocalParameter that are inherited from Parameter do not actually have
relevance to LocalParameter objects.  An example of this is the
methods pertaining to Parameter\'s attribute \'constant\' (i.e.,
isSetConstant(), setConstant(), and getConstant()).

@see getConstant()
@see isSetConstant()
";


%feature("docstring") Parameter::unsetName "
Unsets the value of the \'name\' attribute of this Parameter.

@par
In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
moved to SBase directly, instead of being defined individually for many
(but not all) objects.  LibSBML has for a long time provided functions
defined on SBase itself to get, set, and unset those attributes, which 
would fail or otherwise return empty strings if executed on any object 
for which those attributes were not defined.  Now that all SBase objects 
define those attributes, those functions now succeed for any object with 
the appropriate level and version.

The \'name\' attribute is
optional and is not intended to be used for cross-referencing purposes
within a model.  Its purpose instead is to provide a human-readable
label for the component.  The data type of \'name\' is the type
<code>string</code> defined in XML Schema.  SBML imposes no
restrictions as to the content of \'name\' attributes beyond those
restrictions defined by the <code>string</code> type in XML Schema.

The recommended practice for handling \'name\' is as follows.  If a
software tool has the capability for displaying the content of \'name\'
attributes, it should display this content to the user as a
component\'s label instead of the component\'s \'id\'.  If the user
interface does not have this capability (e.g., because it cannot
display or use special characters in symbol names), or if the \'name\'
attribute is missing on a given component, then the user interface
should display the value of the \'id\' attribute instead.  (Script
language interpreters are especially likely to display \'id\' instead of
\'name\'.)

As a consequence of the above, authors of systems that automatically
generate the values of \'id\' attributes should be aware some systems
may display the \'id\'\'s to the user.  Authors therefore may wish to
take some care to have their software create \'id\' values that are: (a)
reasonably easy for humans to type and read; and (b) likely to be
meaningful, for example by making the \'id\' attribute be an abbreviated
form of the name attribute value.

An additional point worth mentioning is although there are
restrictions on the uniqueness of \'id\' values, there are no
restrictions on the uniqueness of \'name\' values in a model.  This
allows software applications leeway in assigning component identifiers.

Regardless of the level and version of the SBML, these functions allow
client applications to use more generalized code in some situations 
(for instance, when manipulating objects that are all known to have 
names).  If the object in question does not posess a \'name\' attribute 
according to the SBML specification for the Level and Version in use,
libSBML will not allow the name to be set, nor will it read or 
write \'name\' attributes for those objects.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see getName()
@see setName()
@see isSetName()
";


%feature("docstring") Parameter::unsetConstant "
Unsets the value of the \'constant\' attribute of this Parameter object.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink

@see isSetConstant()
@see setConstant()
@see getConstant()
";


%feature("docstring") Parameter::unsetValue "
Unsets the \'value\' attribute of this Parameter instance.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ This particular
function only does one thing irrespective of user input or 
object state, and thus will only return a single value:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink

In SBML Level&nbsp;1 Version&nbsp;1, parameters are required to have
values and therefore, the value of a Parameter <b>should always be
set</b>.  In SBML Level&nbsp;1 Version&nbsp;2 and beyond, the value
is optional and as such, the \'value\' attribute may or may not be set.
";


%feature("docstring") Parameter::unsetUnits "
Unsets the \'units\' attribute of this Parameter instance.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
";


%feature("docstring") Parameter::getDerivedUnitDefinition "
Constructs and returns a UnitDefinition that corresponds to the units
of this Parameter\'s value.

Parameters in SBML have an attribute (\'units\') for declaring the units
of measurement intended for the parameter\'s value.  <b>No defaults are
defined</b> by SBML in the absence of a definition for \'units\'.  This
method returns a UnitDefinition object based on the units declared for
this Parameter using its \'units\' attribute, or it returns @c None if
no units have been declared.

Note that unit declarations for Parameter objects are specified in
terms of the @em identifier of a unit (e.g., using setUnits()), but
@em this method returns a UnitDefinition object, not a unit
identifier.  It does this by constructing an appropriate
UnitDefinition.For SBML Level&nbsp;2 models, it will do this even when
the value of the \'units\' attribute is one of the special SBML
Level&nbsp;2 unit identifiers @c \'substance\', @c \'volume\', @c \'area\',
@c \'length\' or @c \'time\'.  Callers may find this useful in conjunction
with the helper methods provided by the UnitDefinition class for
comparing different UnitDefinition objects.

@return a UnitDefinition that expresses the units of this 
Parameter, or @c None if one cannot be constructed.

@note The libSBML system for unit analysis depends on the model as a
whole.  In cases where the Parameter object has not yet been added to
a model, or the model itself is incomplete, unit analysis is not
possible, and consequently this method will return @c None.

@see isSetUnits()
";


%feature("docstring") Parameter::getTypeCode "
Returns the libSBML type code for this SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_PARAMETER SBML_PARAMETER@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getElementName()
@see getPackageName()
";


%feature("docstring") Parameter::getElementName "
Returns the XML element name of this object, which for Parameter, is
always @c \'parameter\'.

@return the name of this element, i.e., @c \'parameter\'.
";


%feature("docstring") Parameter::writeElements "
@internal

@internal
";


%feature("docstring") Parameter::hasRequiredAttributes "
Predicate returning @c True if
all the required attributes for this Parameter object
have been set.

The required attributes for a Parameter object are:
@li \'id\' (or \'name\' in SBML Level&nbsp;1)
@li \'value\' (required in Level&nbsp;1, optional otherwise)

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") Parameter::renameUnitSIdRefs "
Renames all the @c UnitSIdRef attributes on this element.

@par
In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
introduced for attribute values that refer to <code>UnitSId</code> values; in
previous Levels of SBML, this data type did not exist and attributes were
simply described to as \'referring to a unit identifier\', but the effective
data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
Levels of SBML, even if the corresponding SBML specification did not
explicitly name the data type.

This method works by looking at all unit identifier attribute values
(including, if appropriate, inside mathematical formulas), comparing the
unit identifiers to the value of @p oldid.  If any matches are found,
the matching identifiers are replaced with @p newid.  The method does
@em not descend into child elements.

@param oldid the old identifier.
@param newid the new identifier.
";


%feature("docstring") Parameter::setCalculatingUnits "
@internal

@internal
";


%feature("docstring") Parameter::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") Parameter::readAttributes "
@internal

@internal
";


%feature("docstring") Parameter::readL1Attributes "
@internal

@internal
";


%feature("docstring") Parameter::readL2Attributes "
@internal

@internal
";


%feature("docstring") Parameter::readL3Attributes "
@internal

@internal
";


%feature("docstring") Parameter::writeAttributes "
@internal

@internal
";


%feature("docstring") Parameter::inferUnits "
@internal

@internal
";


%feature("docstring") Parameter::inferUnitsFromAssignments "
@internal

@internal
";


%feature("docstring") Parameter::inferUnitsFromRules "
@internal

@internal
";


%feature("docstring") Parameter::inferUnitsFromReactions "
@internal

@internal
";


%feature("docstring") Parameter::inferUnitsFromEvents "
@internal

@internal
";


%feature("docstring") Parameter::inferUnitsFromEvent "
@internal

@internal
";


%feature("docstring") Parameter::inferUnitsFromKineticLaw "
@internal

@internal
";


%feature("docstring") Parameter::getCalculatingUnits "
@internal

@internal
";


%feature("docstring") ListOfParameters::ListOfParameters "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfParameters(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfParameters object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfParameters object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfParameters(long level, long version)</pre>

Creates a new ListOfParameters object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfParameters::clone "
Creates and returns a deep copy of this ListOfParameters object.

@return the (deep) copy of this ListOfParameters object.
";


%feature("docstring") ListOfParameters::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., Parameter objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this objects contained in this list:
@link libsbml#SBML_PARAMETER SBML_PARAMETER@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfParameters::getElementName "
Returns the XML element name of this object.

For ListOfParameters, the XML element name is @c \'listOfParameters\'.

@return the name of this element, i.e., @c \'listOfParameters\'.
";


%feature("docstring") ListOfParameters::get "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(long n)</pre>

Returns the Parameter object located at position @p n within this
ListOfParameters instance.

@param n the index number of the Parameter to get.

@return the nth Parameter in this ListOfParameters.  If the index @p n
is out of bounds for the length of the list, then @c None is returned.

@see size()
@see get()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(string sid)</pre>

Returns the first Parameter object matching the given identifier.

@param sid a string representing the identifier of the Parameter to
get.

@return the Parameter object found.  The caller owns the returned
object and is responsible for deleting it.  If none of the items have
an identifier matching @p sid, then @c None is returned.

@see get()
@see size()
";


%feature("docstring") ListOfParameters::remove "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(long n)</pre>

Removes the nth item from this ListOfParameters, and returns a pointer
to it.

@param n the index of the item to remove.

@return the item removed.  The caller owns the returned object and is
responsible for deleting it.  If the index number @p n is out of
bounds for the length of the list, then @c None is returned.

@see size()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(string sid)</pre>

Removes the first Parameter object in this ListOfParameters
matching the given identifier, and returns a pointer to it.

@param sid the identifier of the item to remove.

@return the item removed.  The caller owns the returned object and is
responsible for deleting it.  If none of the items have an identifier
matching @p sid, then @c None is returned.
";


%feature("docstring") ListOfParameters::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfParameters::createObject "
@internal

@internal
";


%feature("docstring") LocalParameter "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A parameter inside an SBML <em>reaction</em> definition.

LocalParameter has been introduced in SBML Level&nbsp;3 to serve as the
object class for parameter definitions that are intended to be local to
a Reaction.  Objects of class LocalParameter never appear at the Model
level; they are always contained within ListOfLocalParameters lists
which are in turn contained within KineticLaw objects.

Like its global Parameter counterpart, the LocalParameter object class
is used to define a symbol associated with a value; this symbol can then
be used in a model\'s mathematical formulas (and specifically, for
LocalParameter, reaction rate formulas).  Unlike Parameter, the
LocalParameter class does not have a \'constant\' attribute: local
parameters within reactions are @em always constant.

LocalParameter has one required attribute, \'id\', to give the
parameter a unique identifier by which other parts of an SBML model
definition can refer to it.  A parameter can also have an optional
\'name\' attribute of type @c string.  Identifiers and names must be used
according to the guidelines described in the SBML specifications.

The optional attribute \'value\' determines the value (of type @c float)
assigned to the parameter.  A missing value for \'value\' implies that
the value either is unknown, or to be obtained from an external source,
or determined by an initial assignment.  The unit of measurement
associated with the value of the parameter can be specified using the
optional attribute \'units\'.  Here we only mention briefly some notable
points about the possible unit choices, but readers are urged to consult
the SBML specification documents for more information:
<ul>

<li> In SBML Level&nbsp;3, there are no constraints on the units that
can be assigned to parameters in a model; there are also no units to
inherit from the enclosing Model object.

<li> In SBML Level&nbsp;2, the value assigned to the parameter\'s \'units\'
attribute must be chosen from one of the following possibilities: one of
the base unit identifiers defined in SBML; one of the built-in unit
identifiers @c \'substance\', @c \'time\', @c \'volume\', @c \'area\' or
@c \'length\'; or the identifier of a new unit defined in the list of unit
definitions in the enclosing Model structure.  There are no constraints
on the units that can be chosen from these sets.  There are no default
units for local parameters.
</ul>

As with all other major SBML components, LocalParameter is derived from
SBase, and the methods defined on SBase are available on LocalParameter.

In SBML Level&nbsp;3 Version&nbsp;2, the scope of the LocalParameter 
was expanded slightly to officially encompass the entire Reaction
instead of just the KineticLaw in which it appears.  This has no
effect on models using only SBML Level&nbsp;3 Core constructs,
but has the potential to allow SBML Level&nbsp;3 Packages to
include elements in a Reaction that could reference a
LocalParameter from that Reaction\'s KineticLaw.  It also means that
no LocalParameter may have the same @c \'id\' as a referenced Species
in any SimpleSpeciesReference in that Reaction. 

@warning <span class=\'warning\'>LibSBML derives LocalParameter from
Parameter; however, this does not precisely match the object hierarchy
defined by SBML Level&nbsp;3, where LocalParameter is derived directly
from SBase and not Parameter.  We believe this arrangement makes it easier
for libSBML users to program applications that work with both SBML
Level&nbsp;2 and SBML Level&nbsp;3, but programmers should also keep in
mind this difference exists.  A side-effect of libSBML\'s scheme is that
certain methods on LocalParameter that are inherited from Parameter do not
actually have relevance to LocalParameter objects.  An example of this is
the methods pertaining to Parameter\'s attribute \'constant\' (i.e.,
isSetConstant(), setConstant(), and getConstant()).</span>

@see ListOfLocalParameters
@see KineticLaw
";


%feature("docstring") ListOfLocalParameters "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of LocalParameter objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") LocalParameter::LocalParameter "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>LocalParameter(SBMLNamespaces sbmlns)</pre>

Creates a new LocalParameter object with the given SBMLNamespaces
object @p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments. 

It is worth emphasizing that although this constructor does not take
an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
(identifier) attribute of a LocalParameter is required to have a value.
Thus, callers are cautioned to assign a value after calling this
constructor if no identifier is provided as an argument.  Setting the
identifier can be accomplished using the method
@if java setId(String id)@else setId()@endif.

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>LocalParameter(long level, long version)</pre>

Creates a new LocalParameter object with the given SBML @p level and
@p version values.

@param level a long integer, the SBML Level to assign to this
LocalParameter.

@param version a long integer, the SBML Version to assign to this
LocalParameter.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>LocalParameter(LocalParameter orig)</pre>

Copy constructor; creates a copy of a given LocalParameter object.

@param orig the LocalParameter instance to copy.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>LocalParameter(Parameter orig)</pre>

Copy constructor; creates a LocalParameter object by copying
the attributes of a given Parameter object.

@param orig the Parameter instance to copy.
";


%feature("docstring") LocalParameter::accept "
@internal

@internal
";


%feature("docstring") LocalParameter::clone "
Creates and returns a deep copy of this LocalParameter object.

@return the (deep) copy of this LocalParameter object.
";


%feature("docstring") LocalParameter::getDerivedUnitDefinition "
Constructs and returns a UnitDefinition that corresponds to the units
of this LocalParameter\'s value.

@par
LocalParameters in SBML have an attribute (\'units\') for declaring the
units of measurement intended for the parameter\'s value.  <b>No
defaults are defined</b> by SBML in the absence of a definition for
\'units\'.  This method returns a UnitDefinition object based on the
units declared for this LocalParameter using its \'units\' attribute, or
it returns @c None if no units have been declared.

Note that unit declarations for LocalParameter objects are specified
in terms of the @em identifier of a unit (e.g., using setUnits()), but
@em this method returns a UnitDefinition object, not a unit
identifier.  It does this by constructing an appropriate
UnitDefinition.  For SBML Level&nbsp;2 models, it will do this even
when the value of the \'units\' attribute is one of the predefined SBML
units @c \'substance\', @c \'volume\', @c \'area\', @c \'length\' or
@c \'time\'.  Callers may find this useful in conjunction with the helper
methods provided by the UnitDefinition class for comparing different
UnitDefinition objects.

@return a UnitDefinition that expresses the units of this 
LocalParameter, or @c None if one cannot be constructed.

@note The libSBML system for unit analysis depends on the model as a
whole.  In cases where the LocalParameter object has not yet been
added to a model, or the model itself is incomplete, unit analysis is
not possible, and consequently this method will return @c None.

@see isSetUnits()
";


%feature("docstring") LocalParameter::getTypeCode "
Returns the libSBML type code for this SBML object.

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for this object:
@link libsbml#SBML_LOCAL_PARAMETER SBML_LOCAL_PARAMETER@endlink (default).

@warning <span class=\'warning\'>The specific integer values of the possible
type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
packages,  To fully identify the correct code, <strong>it is necessary to
invoke both getPackageName() and getTypeCode()</strong> (or 
ListOf.getItemTypeCode()).</span>

@see getElementName()
@see getPackageName()
";


%feature("docstring") LocalParameter::getElementName "
Returns the XML element name of this object, which for LocalParameter,
is always @c \'localParameter\'.

@return the name of this element, i.e., @c \'localParameter\'.
";


%feature("docstring") LocalParameter::hasRequiredAttributes "
Predicate returning @c True if all the required attributes for this
LocalParameter object have been set.

The required attributes for a LocalParameter object are:
@li \'id\'
@li \'value\'

@return @c True if the required attributes have been set, @c False
otherwise.
";


%feature("docstring") LocalParameter::getConstant "
@internal

@internal
";


%feature("docstring") LocalParameter::isSetConstant "
@internal

@internal
";


%feature("docstring") LocalParameter::setConstant "
@internal

@internal
";


%feature("docstring") LocalParameter::unsetConstant "
@internal

@internal
";


%feature("docstring") LocalParameter::addExpectedAttributes "
@internal

@internal
";


%feature("docstring") LocalParameter::readAttributes "
@internal

@internal
";


%feature("docstring") LocalParameter::readL3Attributes "
@internal

@internal
";


%feature("docstring") LocalParameter::writeAttributes "
@internal

@internal
";


%feature("docstring") ListOfLocalParameters::ListOfLocalParameters "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfLocalParameters(SBMLNamespaces sbmlns)</pre>

Creates a new ListOfLocalParameters object.

The object is constructed such that it is valid for the SBML Level and
Version combination determined by the SBMLNamespaces object in @p
sbmlns.

@param sbmlns an SBMLNamespaces object that is used to determine the
characteristics of the ListOfLocalParameters object to be created.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>ListOfLocalParameters(long level, long version)</pre>

Creates a new ListOfLocalParameters object.

The object is constructed such that it is valid for the given SBML
Level and Version combination.

@param level the SBML Level.

@param version the Version within the SBML Level.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
";


%feature("docstring") ListOfLocalParameters::clone "
Creates and returns a deep copy of this ListOfLocalParameters object.

@return the (deep) copy of this ListOfLocalParameters object.
";


%feature("docstring") ListOfLocalParameters::getItemTypeCode "
Returns the libSBML type code for the objects contained in this ListOf
(i.e., LocalParameter objects, if the list is non-empty).

@par
LibSBML attaches an identifying code to every kind of SBML object.  These
are integer constants known as <em>SBML type codes</em>.  The names of all
the codes begin with the characters <code>SBML_</code>.
@if clike The set of possible type codes for core elements is defined in
the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
SBML Level&nbsp;3 packages define their own extra enumerations of type
codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
package).@endif@if java In the Java language interface for libSBML, the
type codes are defined as static integer constants in the interface class
{@link libsbmlConstants}.  @endif@if python In the Python language
interface for libSBML, the type codes are defined as static integer
constants in the interface class @link libsbml@endlink.@endif@if csharp In
the C# language interface for libSBML, the type codes are defined as
static integer constants in the interface class
@link libsbmlcs.libsbml@endlink.@endif@~  Note that different Level&nbsp;3
package plug-ins may use overlapping type codes; to identify the package
to which a given object belongs, call the 
<code>@if conly SBase_getPackageName()
@else SBase.getPackageName()
@endif</code>
method on the object.

The exception to this is lists:  all SBML-style list elements have the type 
@link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
are from.

@return the SBML type code for the objects contained in this ListOf:
@link libsbml#SBML_LOCAL_PARAMETER SBML_LOCAL_PARAMETER@endlink (default).

@see getElementName()
@see getPackageName()
";


%feature("docstring") ListOfLocalParameters::getElementName "
Returns the XML element name of this object.

For ListOfLocalParameters, the XML element name is @c \'listOfLocalParameters\'.

@return the name of this element, i.e., @c \'listOfLocalParameters\'.
";


%feature("docstring") ListOfLocalParameters::get "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(long n)</pre>

Returns the LocalParameter object located at position @p n within this
ListOfLocalParameters instance.

@param n the index number of the LocalParameter to get.

@return the item at position @p n.  The caller owns the returned
object and is responsible for deleting it.  If the index number @p n
is out of bounds for the length of the list, then @c None is returned.

@see size()
@see get()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>get(string sid)</pre>

Returns the first LocalParameter object matching the given identifier.

@param sid a string representing the identifier of the LocalParameter
to get.

@return the LocalParameter object found.  The caller owns the returned
object and is responsible for deleting it.  If none of the items have
an identifier matching @p sid, then @c None is returned.

@see get()
@see size()
";


%feature("docstring") ListOfLocalParameters::getElementBySId "
Returns the first child element found that has the given @p id in the
model-wide SId namespace, or @c None if no such object is found.

Note that LocalParameters, while they use the SId namespace, are not in
the model-wide SId namespace, so no LocalParameter object will be
returned from this function (and is the reason we override the base
ListOf::getElementBySId function here).

@param id string representing the id of the object to find.

@return pointer to the first element found with the given @p id.
";


%feature("docstring") ListOfLocalParameters::remove "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(long n)</pre>

Removes the nth item from this ListOfLocalParameters, and returns a
pointer to it.

@param n the index of the item to remove.

@return the item removed.  The caller owns the returned object and is
responsible for deleting it.  If the index number @p n is out of
bounds for the length of the list, then @c None is returned.

@see size()
@see remove()
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>remove(string sid)</pre>

Removes the first LocalParameter object in this ListOfLocalParameters
matching the given identifier, and returns a pointer to it.

@param sid the identifier of the item to remove.

@return the item removed.  The caller owns the returned object and is
responsible for deleting it.  If none of the items have an identifier
matching @p sid, then @c None is returned.
";


%feature("docstring") ListOfLocalParameters::getElementPosition "
@internal

@internal
";


%feature("docstring") ListOfLocalParameters::createObject "
@internal

@internal
";


%feature("docstring") InitialAssignment "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html An SBML <em>initial assignment</em>, evaluated once only.

SBML Level 2 Versions 2&ndash;4 and SBML Level&nbsp;3 provide two ways of assigning initial
values to entities in a model.  The simplest and most basic is to set
the values of the appropriate attributes in the relevant components; for
example, the initial value of a model parameter (whether it is a
constant or a variable) can be assigned by setting its \'value\' attribute
directly in the model definition.  However, this approach is not
suitable when the value must be calculated, because the initial value
attributes on different components such as species, compartments, and
parameters are single values and not mathematical expressions.  In those
situations, the InitialAssignment construct can be used; it permits the
calculation of the value of a constant or the initial value of a
variable from the values of @em other quantities in a model.

As explained below, the provision of InitialAssignment does not mean
that models necessarily must use this construct when defining initial
values of quantities in a model.  If a value can be set directly using
the relevant attribute of a component in a model, then that
approach may be more efficient and more portable to other software
tools.  InitialAssignment should be used when the other mechanism is
insufficient for the needs of a particular model.

The InitialAssignment construct has some similarities to AssignmentRule.
The main differences are: (a) an InitialAssignment can set the value of
a constant whereas an AssignmentRule cannot, and (b) unlike
AssignmentRule, an InitialAssignment definition only applies up to and
including the beginning of simulation time, i.e., <em>t <= 0</em>,
while an AssignmentRule applies at all times.

InitialAssignment has a required attribute, \'symbol\', whose value must
follow the guidelines for identifiers described in the %SBML
specification (e.g., Section 3.3 in the Level 2 Version 4
specification).  The value of this attribute in an InitialAssignment
object can be the identifier of a Compartment, Species, SpeciesReference 
(in SBML Level&nbsp;3),  global Parameter, or (as of SBML 
Level&nbsp;3 Version&nbsp;2) the identifier of a SBML Level&nbsp;3 
package element with mathematical meaning.  The InitialAssignment defines the
initial value of the constant or variable referred to by the \'symbol\'
attribute.  (The attribute\'s name is \'symbol\' rather than \'variable\'
because it may assign values to constants as well as variables in a
model.)  Note that an initial assignment cannot be made to reaction
identifiers, that is, the \'symbol\' attribute value of an
InitialAssignment cannot be an identifier that is the \'id\' attribute
value of a Reaction object in the model.  This is identical to a
restriction placed on rules.

InitialAssignment also has a \'math\' subelement that contains a
MathML expression used to calculate the value of the constant or the
initial value of the variable.  This subelement is required in SBML
Level&nbsp;2 and SBML Level&nbsp;3 Version&nbsp;1, but the requirement
was relaxed in SBML Level&nbsp;3 Version&nbsp;2, making it optional.
The units of the value computed by the formula in the \'math\' subelement 
should (in SBML Level&nbsp;2 Version&nbsp;4 and in SBML Level&nbsp;3) 
or must (in previous Versions) be identical to be the
units associated with the identifier given in the \'symbol\' attribute.
(That is, the units are the units of the species, compartment, or
parameter, as appropriate for the kind of object identified by the value
of \'symbol\'.)

InitialAssignment was introduced in SBML Level 2 Version 2.  It is not
available in SBML Level&nbsp;2 Version&nbsp;1 nor in any version of Level 1.

@section initassign-semantics Semantics of Initial Assignments

The value calculated by an InitialAssignment object overrides the value
assigned to the given symbol by the object defining that symbol.  For
example, if a compartment\'s \'size\' attribute is set in its definition,
and the model also contains an InitialAssignment having that
compartment\'s identifier as its \'symbol\' attribute value, then the
interpretation is that the \'size\' assigned in the Compartment object
should be ignored and the value assigned based on the computation
defined in the InitialAssignment.  Initial assignments can take place
for Compartment, Species, global Parameter, SpeciesReference (in 
Level&nbsp;3), and SBML Level&nbsp;3 package elements (in 
Level&nbsp;3 Version&nbsp;2), regardless of the value of their 
\'constant\' attribute.

The actions of all InitialAssignment objects are in general terms
the same, but differ in the precise details depending on the type
of variable being set:
<ul>
<li> <em>In the case of a species</em>, an InitialAssignment sets the
referenced species\' initial quantity (concentration or amount of
substance) to the value determined by the formula in the \'math\'
subelement.    The overall units of the formula should (in SBML
Level&nbsp;2 Version&nbsp;4 and in SBML Level&nbsp;3) or must (in previous Versions) be the same
as the units specified for the species.

<li> <em>In the case of a compartment</em>, an InitialAssignment sets
the referenced compartment\'s initial size to the size determined by the
formula in \'math\'.  The overall units of the formula should (in SBML
Level&nbsp;2 Version&nbsp;4 and in SBML Level&nbsp;3) or must (in previous Versions) be the same
as the units specified for the size of the compartment.

<li> <em>In the case of a parameter</em>, an InitialAssignment sets the
referenced parameter\'s initial value to that determined by the formula
in \'math\'.  The overall units of the formula should (in SBML
Level&nbsp;2 Version&nbsp;4 and SBML Level&nbsp;3) or must (in previous Versions) be the same
as the units defined for the parameter.  

<li> (For SBML Level&nbsp;3 only) <em>In the case of a species
reference</em>, an initial assignment sets the initial value of the 
stoichiometry of the referenced reactant or product to the value determined 
by the formula in \'math\'.  The unit associated with the value produced by 
the \'math\' formula should be consistent with the unit \'dimensionless\',
because reactant and product stoichiometries in reactions are dimensionless
quantities.

<li>(For SBML Level&nbsp;3 Version&nbsp;2 only) <em>In the case 
of an object from an SBML Level&nbsp;3 package</em>, an InitialAssignment 
sets the referenced object\'s initial value (however such values are 
defined by the package) to the value of the formula in math. The unit 
of measurement associated with the value produced by the formula 
should be the same as that object\'s units attribute value (if it has 
such an attribute), or be equal to the units of model components of 
that type (if objects of that class are defined by the package as 
having the same units).

</ul>

If the symbol attribute of an InitialAssignment object references 
an object in an SBML namespace that is not understood by the 
interpreter reading a given SBML document (that is, if the object 
is defined by an SBML Level&nbsp;3 package that the software does 
not support), the assignment must be ignored--the object\'s initial 
value will not need to be set, as the interpreter could not understand 
that package. If an interpreter cannot establish whether a referenced 
object is missing from the model or instead is defined in an SBML 
namespace not understood by the interpreter, it may produce a 
warning to the user. (The latter situation may only arise if an SBML 
package is present in the SBML document with a package:required 
attribute of \'true\'.)

In the context of a simulation, initial assignments establish values
that are in effect prior to and including the start of simulation time,
i.e., <em>t <= 0</em>.  Section 3.4.8 in the SBML Level&nbsp;2
Version&nbsp;4  and SBML Level&nbsp;3 specifications 
provides information about the interpretation of
assignments, rules, and entity values for simulation time up to and
including the start time <em>t = 0</em>; this is important for
establishing the initial conditions of a simulation if the model
involves expressions containing the <em>delay</em> \'csymbol\'.

There cannot be two initial assignments for the same symbol in a model;
that is, a model must not contain two or more InitialAssignment objects
that both have the same identifier as their \'symbol\' attribute value.  A
model must also not define initial assignments <em>and</em> assignment
rules for the same entity.  That is, there cannot be <em>both</em> an
InitialAssignment and an AssignmentRule for the same symbol in a model,
because both kinds of constructs apply prior to and at the start of
simulated time---allowing both to exist for a given symbol would
result in indeterminism).

The ordering of InitialAssignment objects is not significant.  The
combined set of InitialAssignment, AssignmentRule and KineticLaw
objects form a set of assignment statements that must be considered as a
whole.  The combined set of assignment statements should not contain
algebraic loops: a chain of dependency between these statements should
terminate.  (More formally, consider the directed graph of assignment
statements where nodes are a model\'s assignment statements and directed
arcs exist for each occurrence of a symbol in an assignment statement
\'math\' attribute.  The directed arcs in this graph start from the
statement assigning the symbol and end at the statement that contains
the symbol in their math elements.  Such a graph must be acyclic.)

Finally, it is worth being explicit about the expected behavior in the
following situation.  Suppose (1) a given symbol has a value <em>x</em>
assigned to it in its definition, and (2) there is an initial assignment
having the identifier as its \'symbol\' value and reassigning the value to
<em>y</em>, <em>and</em> (3) the identifier is also used in the
mathematical formula of a second initial assignment.  What value should
the second initial assignment use?  It is <em>y</em>, the value assigned
to the symbol by the first initial assignment, not whatever value was
given in the symbol\'s definition.  This follows directly from the
behavior described above: if an InitialAssignment object exists for a
given symbol, then the symbol\'s value is overridden by that initial
assignment.
";


%feature("docstring") ListOfInitialAssignments "
@sbmlpackage{core}

@htmlinclude pkg-marker-core.html A list of InitialAssignment objects.

@par
The various ListOf___ @if conly structures @else classes@endif@~ in SBML
are merely containers used for organizing the main components of an SBML
model.  In libSBML\'s implementation, ListOf___
@if conly data structures @else classes@endif@~ are derived from the
intermediate utility @if conly structure @else class@endif@~ ListOf, which
is not defined by the SBML specifications but serves as a useful
programmatic construct.  ListOf is itself is in turn derived from SBase,
which provides all of the various ListOf___
@if conly data structures @else classes@endif@~ with common features
defined by the SBML specification, such as \'metaid\' attributes and
annotations.

The relationship between the lists and the rest of an SBML model is
illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):

@htmlinclude listof-illustration.html

SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
Version&nbsp;2 allows
containers to contain zero or more of the relevant object, instead of 
requiring at least one.  As such, libsbml will write out an 
otherwise-empty ListOf___ element that has any optional attribute set 
(such as \'id\' or \'metaid\'), that has an optional child (such 
as a \'notes\' or \'annotation\'), or that has attributes or children set
from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
any other children.

Readers may wonder about the motivations for using the ListOf___
containers in SBML.  A simpler approach in XML might be to place the
components all directly at the top level of the model definition.  The
choice made in SBML is to group them within XML elements named after
%ListOf<em>Classname</em>, in part because it helps organize the
components.  More importantly, the fact that the container classes are
derived from SBase means that software tools can add information @em about
the lists themselves into each list container\'s \'annotation\'.

@see ListOfFunctionDefinitions
@see ListOfUnitDefinitions
@see ListOfCompartmentTypes
@see ListOfSpeciesTypes
@see ListOfCompartments
@see ListOfSpecies
@see ListOfParameters
@see ListOfInitialAssignments
@see ListOfRules
@see ListOfConstraints
@see ListOfReactions
@see ListOfEvents

@if conly
@note In the C API for libSBML, functions that in other language APIs
would be inherited by the various ListOf___ structures not shown in the
pages for the individual ListOf___\'s.  Instead, the functions are defined
on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
the many common functions available for manipulating ListOf___
structures</strong>.  The documentation for the individual ListOf___
structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
all of the functionality available. @endif@~
";


%feature("docstring") InitialAssignment::InitialAssignment "
This method has multiple variants; they differ in the arguments
 they accept.  Each variant is described separately below.

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>InitialAssignment(SBMLNamespaces sbmlns)</pre>

Creates a new InitialAssignment using the given SBMLNamespaces object
@p sbmlns.

@par
The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
information.  It is used to communicate the SBML Level, Version, and (in
Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
common approach to using libSBML\'s SBMLNamespaces facilities is to create an
SBMLNamespaces object somewhere in a program once, then hand that object
as needed to object constructors that accept SBMLNamespaces as arguments. 

@param sbmlns an SBMLNamespaces object.

@throws SBMLConstructorException
Thrown if the given @p sbmlns is inconsistent or incompatible
with this object.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>InitialAssignment(long level, long version)</pre>

Creates a new InitialAssignment using the given SBML @p level and @p version
values.

@param level a long integer, the SBML Level to assign to this InitialAssignment.

@param version a long integer, the SBML Version to assign to this
InitialAssignment.

@throws SBMLConstructorException
Thrown if the given @p level and @p version combination are invalid
or if this object is incompatible with the given level and version.

@note Attempting to add an object to an SBMLDocument having a different
combination of SBML Level, Version and XML namespaces than the object
itself will result in an error at the time a caller attempts to make the
addition.  A parent object must have compatible Level, Version and XML
namespaces.  (Strictly speaking, a parent may also have more XML
namespaces than a child, but the reverse is not permitted.)  The
restriction is necessary to ensure that an SBML model has a consistent
overall structure.  This requires callers to manage their objects
carefully, but the benefit is increased flexibility in how models can be
created by permitting callers to create objects bottom-up if desired.  In
situations where objects are not yet attached to parents (e.g.,
SBMLDocument), knowledge of the intented SBML Level and Version help
libSBML determine such things as whether it is valid to assign a
particular value to an attribute.
   

@par
<hr>
<span class='variant-sig-heading'>Method variant with the following signature</span>:
 <pre class='signature'>InitialAssignment(InitialAssignment orig)</pre>

Copy constructor; creates a copy of this InitialAssignment.

@param orig the object to copy.
";


%feature("docstring") InitialAssignment::accept "
@internal

@internal
";


%feature("docstring") InitialAssignment::clone "
Creates and returns a deep copy of this InitialAssignment object.

@return the (deep) copy of this InitialAssignment object.
";


%feature("docstring") InitialAssignment::getSymbol "
Get the value of the \'symbol\' attribute of this InitialAssignment.

@return the identifier string stored as the \'symbol\' attribute value
in this InitialAssignment.
";


%feature("docstring") InitialAssignment::getMath "
Get the mathematical formula of this InitialAssignment.

@return an ASTNode, the value of the \'math\' subelement of this
InitialAssignment, or @c None if the math is not set.
";


%feature("docstring") InitialAssignment::isSetSymbol "
Predicate returning @c True if this
InitialAssignment\'s \'symbol\' attribute is set.

@return @c True if the \'symbol\' attribute of this InitialAssignment
is set, @c False otherwise.
";


%feature("docstring") InitialAssignment::isSetMath "
Predicate returning @c True if this
InitialAssignment\'s \'math\' subelement contains a value.

@return @c True if the \'math\' for this InitialAssignment is set,
@c False otherwise.
";


%feature("docstring") InitialAssignment::setSymbol "
Sets the \'symbol\' attribute value of this InitialAssignment.

@param sid the identifier of an element defined in this model whose
value can be set.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") InitialAssignment::unsetSymbol "
Unsets the \'symbol\' attribute value of this InitialAssignment.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
";


%feature("docstring") InitialAssignment::setMath "
Sets the \'math\' subelement of this InitialAssignment.

The AST passed in @p math is copied.

@param math an AST containing the mathematical expression to
be used as the formula for this InitialAssignment.

@return integer value indicating success/failure of the
function.  @if clike The value is drawn from the
enumeration #OperationReturnValues_t. @endif@~ The possible values
returned by this function are:
@li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
@li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
";


%feature("docstring") InitialAssignment::getDerivedUnitDefinition "
Calculates and returns a UnitDefinition that expresses the units
of measurement assumed for the \'math\' expression of this
InitialAssignment.

@par
The units are calculated based on the mathematical expression in the
InitialAssignment and the model quantities referenced by
<code>&lt;ci&gt;</code> elements used within that expression.  The method
InitialAssignment.getDerivedUnitDefinition() returns the calculated
units, to the extent that libSBML can compute them.

@note The functionality that facilitates unit analysis depends on the
model as a whole.  Thus, in cases where the object has not been added to
a model or the model itself is incomplete, unit analysis is not possible
and this method will return @c None. 

@warning <span class=\'warning\'>Note that it is possible the \'math\'
expression in the InitialAssignment contains literal numbers or parameters
with undeclared units.  In those cases, it is not possible to calculate
the units of the overall expression without making assumptions.  LibSBML
does not make assumptions about the units, and
InitialAssignment.getDerivedUnitDefinition() only returns the units as
far as it is able to determine them.  For example, in an expression <em>X
+ Y</em>, if <em>X</em> has unambiguously-defined units and <em>Y</em>
does not, it will return the units of <em>X</em>.  When using this method,
<strong>it is critical that callers also invoke the method</strong>
Init