/*****************************************************************
 * gmerlin-avdecoder - a general purpose multimedia decoding library
 *
 * Copyright (c) 2001 - 2012 Members of the Gmerlin project
 * gmerlin-general@lists.sourceforge.net
 * http://gmerlin.sourceforge.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * *****************************************************************/

#include <string.h>
#include <stdlib.h>


#include <avdec_private.h>
#include <flac_header.h>


int bgav_flac_streaminfo_read(const uint8_t * ptr, bgav_flac_streaminfo_t * ret)
  {
  uint64_t tmp_1;

  ret->min_blocksize = GAVL_PTR_2_16BE(ptr); ptr += 2;
  ret->max_blocksize = GAVL_PTR_2_16BE(ptr); ptr += 2;
  ret->min_framesize = GAVL_PTR_2_24BE(ptr); ptr += 3;
  ret->max_framesize = GAVL_PTR_2_24BE(ptr); ptr += 3;

  tmp_1 = GAVL_PTR_2_64BE(ptr); ptr += 8;
  
  ret->samplerate      =   tmp_1 >> 44;
  ret->num_channels    = ((tmp_1 >> 41) & 0x7) + 1;
  ret->bits_per_sample = ((tmp_1 >> 36) & 0x1f) + 1;
  ret->total_samples   = (tmp_1 & 0xfffffffffLL);

  /* MD5 */
  memcpy(ret->md5, ptr, 16);
  return 1;
  }

void bgav_flac_streaminfo_dump(bgav_flac_streaminfo_t * s)
  {
  bgav_dprintf("FLAC Streaminfo\n");
  bgav_dprintf("  Blocksize [%d/%d]\n", s->min_blocksize,
               s->max_blocksize);
  bgav_dprintf("  Framesize [%d/%d]\n", s->min_framesize,
               s->max_framesize);
  bgav_dprintf("  Samplerate:      %d\n", s->samplerate);
  bgav_dprintf("  Num channels:    %d\n", s->num_channels);
  bgav_dprintf("  Bits per sample: %d\n", s->bits_per_sample);
  bgav_dprintf("  Total samples:   %" PRId64 "\n", s->total_samples);
  }

void bgav_flac_streaminfo_init_stream(bgav_flac_streaminfo_t * si, gavl_dictionary_t * s)
  {
  gavl_audio_format_t * fmt = gavl_stream_get_audio_format_nc(s);
  
  fmt->num_channels = si->num_channels;
  fmt->samplerate   = si->samplerate;

  gavl_stream_set_audio_bits(s, si->bits_per_sample);
  
  if(si->min_blocksize == si->max_blocksize)
    fmt->samples_per_frame = si->max_blocksize;
  
  switch(fmt->num_channels)
    {
    case 1:
      fmt->channel_locations[0] = GAVL_CHID_FRONT_CENTER;
      break;
    case 2:
      fmt->channel_locations[0] = GAVL_CHID_FRONT_LEFT;
      fmt->channel_locations[1] = GAVL_CHID_FRONT_RIGHT;
      break;
    case 3:
      fmt->channel_locations[0] = GAVL_CHID_FRONT_LEFT;
      fmt->channel_locations[1] = GAVL_CHID_FRONT_RIGHT;
      fmt->channel_locations[2] = GAVL_CHID_FRONT_CENTER;
      break;
    case 4:
      fmt->channel_locations[0] = GAVL_CHID_FRONT_LEFT;
      fmt->channel_locations[1] = GAVL_CHID_FRONT_RIGHT;
      fmt->channel_locations[2] = GAVL_CHID_REAR_LEFT;
      fmt->channel_locations[3] = GAVL_CHID_REAR_RIGHT;
      break;
    case 5:
      fmt->channel_locations[0] = GAVL_CHID_FRONT_LEFT;
      fmt->channel_locations[1] = GAVL_CHID_FRONT_RIGHT;
      fmt->channel_locations[2] = GAVL_CHID_FRONT_CENTER;
      fmt->channel_locations[3] = GAVL_CHID_REAR_LEFT;
      fmt->channel_locations[4] = GAVL_CHID_REAR_RIGHT;
      break;
    case 6:
      fmt->channel_locations[0] = GAVL_CHID_FRONT_LEFT;
      fmt->channel_locations[1] = GAVL_CHID_FRONT_RIGHT;
      fmt->channel_locations[2] = GAVL_CHID_FRONT_CENTER;
      fmt->channel_locations[3] = GAVL_CHID_LFE;
      fmt->channel_locations[4] = GAVL_CHID_REAR_LEFT;
      fmt->channel_locations[5] = GAVL_CHID_REAR_RIGHT;
      break;
    }
  }

#define ADVANCE(num) \
  ptr += num; \
  size -= num;

#define CHECK_LEN(num) \
  if(size < num) \
    return 0;

static int read_utf8(const uint8_t * ptr, int size, int64_t * ret)
  {
  uint8_t mask, val;
  int bytes, i;

  CHECK_LEN(1);
  
  /* 0xxxxxxx */
  if((ptr[0] & 0x80) == 0x00)
    {
    *ret = ptr[0];
    return 1;
    }
  
  mask = 0xff;
  val  = 0xfe;
  bytes = 6;
  
  /* 11111110 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx */
  /* 1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx */
  /* 111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx */
  /* 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx */
  /* 1110xxxx 10xxxxxx 10xxxxxx */
  /* 110xxxxx 10xxxxxx */
  
  while(bytes >= 1)
    {
    if((ptr[0] & mask) == val)
      {
      *ret = ptr[0] & ~mask;
      CHECK_LEN(bytes+1);
      for(i = 1; i <= bytes; i++)
        {
        if((ptr[i] & 0xc0) != 0x80)
          return 0;
        
        (*ret) <<= 6;
        *ret |= (ptr[i] & 0x3f);
        }
      return bytes + 1;
      }

    mask <<= 1;
    val <<= 1;
    bytes--;
    }
  return 0;
  }

/* CRC stuff taken from FLAC sourcecode */

static const uint8_t crc8_table[256] =
  {
    0x00, 0x07, 0x0E, 0x09, 0x1C, 0x1B, 0x12, 0x15,
    0x38, 0x3F, 0x36, 0x31, 0x24, 0x23, 0x2A, 0x2D,
    0x70, 0x77, 0x7E, 0x79, 0x6C, 0x6B, 0x62, 0x65,
    0x48, 0x4F, 0x46, 0x41, 0x54, 0x53, 0x5A, 0x5D,
    0xE0, 0xE7, 0xEE, 0xE9, 0xFC, 0xFB, 0xF2, 0xF5,
    0xD8, 0xDF, 0xD6, 0xD1, 0xC4, 0xC3, 0xCA, 0xCD,
    0x90, 0x97, 0x9E, 0x99, 0x8C, 0x8B, 0x82, 0x85,
    0xA8, 0xAF, 0xA6, 0xA1, 0xB4, 0xB3, 0xBA, 0xBD,
    0xC7, 0xC0, 0xC9, 0xCE, 0xDB, 0xDC, 0xD5, 0xD2,
    0xFF, 0xF8, 0xF1, 0xF6, 0xE3, 0xE4, 0xED, 0xEA,
    0xB7, 0xB0, 0xB9, 0xBE, 0xAB, 0xAC, 0xA5, 0xA2,
    0x8F, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9D, 0x9A,
    0x27, 0x20, 0x29, 0x2E, 0x3B, 0x3C, 0x35, 0x32,
    0x1F, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0D, 0x0A,
    0x57, 0x50, 0x59, 0x5E, 0x4B, 0x4C, 0x45, 0x42,
    0x6F, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7D, 0x7A,
    0x89, 0x8E, 0x87, 0x80, 0x95, 0x92, 0x9B, 0x9C,
    0xB1, 0xB6, 0xBF, 0xB8, 0xAD, 0xAA, 0xA3, 0xA4,
    0xF9, 0xFE, 0xF7, 0xF0, 0xE5, 0xE2, 0xEB, 0xEC,
    0xC1, 0xC6, 0xCF, 0xC8, 0xDD, 0xDA, 0xD3, 0xD4,
    0x69, 0x6E, 0x67, 0x60, 0x75, 0x72, 0x7B, 0x7C,
    0x51, 0x56, 0x5F, 0x58, 0x4D, 0x4A, 0x43, 0x44,
    0x19, 0x1E, 0x17, 0x10, 0x05, 0x02, 0x0B, 0x0C,
    0x21, 0x26, 0x2F, 0x28, 0x3D, 0x3A, 0x33, 0x34,
    0x4E, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5C, 0x5B,
    0x76, 0x71, 0x78, 0x7F, 0x6A, 0x6D, 0x64, 0x63,
    0x3E, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2C, 0x2B,
    0x06, 0x01, 0x08, 0x0F, 0x1A, 0x1D, 0x14, 0x13,
    0xAE, 0xA9, 0xA0, 0xA7, 0xB2, 0xB5, 0xBC, 0xBB,
    0x96, 0x91, 0x98, 0x9F, 0x8A, 0x8D, 0x84, 0x83,
    0xDE, 0xD9, 0xD0, 0xD7, 0xC2, 0xC5, 0xCC, 0xCB,
    0xE6, 0xE1, 0xE8, 0xEF, 0xFA, 0xFD, 0xF4, 0xF3
  };

static int crc8(const uint8_t * data, int len)
  {
  int i;
  uint8_t crc = 0;
  for(i = 0; i < len; i++)
    crc = crc8_table[crc ^ data[i]];
  return crc;
  }

/* CRC-16, poly = x^16 + x^15 + x^2 + x^0, init = 0 */

static unsigned int crc16_table[256] =
  {
    0x0000,  0x8005,  0x800f,  0x000a,  0x801b,  0x001e,  0x0014,  0x8011,
    0x8033,  0x0036,  0x003c,  0x8039,  0x0028,  0x802d,  0x8027,  0x0022,
    0x8063,  0x0066,  0x006c,  0x8069,  0x0078,  0x807d,  0x8077,  0x0072,
    0x0050,  0x8055,  0x805f,  0x005a,  0x804b,  0x004e,  0x0044,  0x8041,
    0x80c3,  0x00c6,  0x00cc,  0x80c9,  0x00d8,  0x80dd,  0x80d7,  0x00d2,
    0x00f0,  0x80f5,  0x80ff,  0x00fa,  0x80eb,  0x00ee,  0x00e4,  0x80e1,
    0x00a0,  0x80a5,  0x80af,  0x00aa,  0x80bb,  0x00be,  0x00b4,  0x80b1,
    0x8093,  0x0096,  0x009c,  0x8099,  0x0088,  0x808d,  0x8087,  0x0082,
    0x8183,  0x0186,  0x018c,  0x8189,  0x0198,  0x819d,  0x8197,  0x0192,
    0x01b0,  0x81b5,  0x81bf,  0x01ba,  0x81ab,  0x01ae,  0x01a4,  0x81a1,
    0x01e0,  0x81e5,  0x81ef,  0x01ea,  0x81fb,  0x01fe,  0x01f4,  0x81f1,
    0x81d3,  0x01d6,  0x01dc,  0x81d9,  0x01c8,  0x81cd,  0x81c7,  0x01c2,
    0x0140,  0x8145,  0x814f,  0x014a,  0x815b,  0x015e,  0x0154,  0x8151,
    0x8173,  0x0176,  0x017c,  0x8179,  0x0168,  0x816d,  0x8167,  0x0162,
    0x8123,  0x0126,  0x012c,  0x8129,  0x0138,  0x813d,  0x8137,  0x0132,
    0x0110,  0x8115,  0x811f,  0x011a,  0x810b,  0x010e,  0x0104,  0x8101,
    0x8303,  0x0306,  0x030c,  0x8309,  0x0318,  0x831d,  0x8317,  0x0312,
    0x0330,  0x8335,  0x833f,  0x033a,  0x832b,  0x032e,  0x0324,  0x8321,
    0x0360,  0x8365,  0x836f,  0x036a,  0x837b,  0x037e,  0x0374,  0x8371,
    0x8353,  0x0356,  0x035c,  0x8359,  0x0348,  0x834d,  0x8347,  0x0342,
    0x03c0,  0x83c5,  0x83cf,  0x03ca,  0x83db,  0x03de,  0x03d4,  0x83d1,
    0x83f3,  0x03f6,  0x03fc,  0x83f9,  0x03e8,  0x83ed,  0x83e7,  0x03e2,
    0x83a3,  0x03a6,  0x03ac,  0x83a9,  0x03b8,  0x83bd,  0x83b7,  0x03b2,
    0x0390,  0x8395,  0x839f,  0x039a,  0x838b,  0x038e,  0x0384,  0x8381,
    0x0280,  0x8285,  0x828f,  0x028a,  0x829b,  0x029e,  0x0294,  0x8291,
    0x82b3,  0x02b6,  0x02bc,  0x82b9,  0x02a8,  0x82ad,  0x82a7,  0x02a2,
    0x82e3,  0x02e6,  0x02ec,  0x82e9,  0x02f8,  0x82fd,  0x82f7,  0x02f2,
    0x02d0,  0x82d5,  0x82df,  0x02da,  0x82cb,  0x02ce,  0x02c4,  0x82c1,
    0x8243,  0x0246,  0x024c,  0x8249,  0x0258,  0x825d,  0x8257,  0x0252,
    0x0270,  0x8275,  0x827f,  0x027a,  0x826b,  0x026e,  0x0264,  0x8261,
    0x0220,  0x8225,  0x822f,  0x022a,  0x823b,  0x023e,  0x0234,  0x8231,
    0x8213,  0x0216,  0x021c,  0x8219,  0x0208,  0x820d,  0x8207,  0x0202
  };

static unsigned int crc16(const uint8_t *data, unsigned len)
  {
  unsigned int crc = 0;
  
  while(len--)
    crc = ((crc<<8) ^ crc16_table[(crc>>8) ^ *data++]) & 0xffff;
  
  return crc;
  }

int bgav_flac_check_crc(const uint8_t * ptr, int size)
  {
  unsigned int crc1, crc2;

  crc1 = crc16(ptr, size-2);
  ptr += size-2;
  crc2 = GAVL_PTR_2_16BE(ptr);
  return crc1 == crc2;
  }

int bgav_flac_frame_header_read(const uint8_t * ptr, int size,
                                bgav_flac_streaminfo_t * si,
                                bgav_flac_frame_header_t * ret)
  {
  int len;
  int tmp;
  int crc;
  
  const uint8_t * ptr_start = ptr;

  CHECK_LEN(2);
  tmp = GAVL_PTR_2_16BE(ptr);
  ADVANCE(2);

  /* Check sync code */
  if((tmp & 0xFFFC) != 0xFFF8)
    return 0;
  
  if(tmp & 0x0002) // Mandatory zero
    return 0;

  ret->blocking_strategy_code = tmp & 0x0001;

  /* Block size, sample_rate */
  CHECK_LEN(1);
  tmp = *ptr;
  ADVANCE(1);

  ret->block_size_code = (tmp >> 4) & 0x0f;
  ret->samplerate_code = tmp & 0x0f;

  /* Channel, sample size */
  
  CHECK_LEN(1);
  tmp = *ptr;
  ADVANCE(1);
  
  ret->channel_code     = (tmp >> 4) & 0x0f;
  ret->samplesize_code = (tmp >> 1) & 0x07;

  if(tmp & 0x01) // Mandatory zero
    return 0;

  /* Compute values */

  /* Sample number */
  if(ret->blocking_strategy_code == 0) // Fixed blocksize
    {
    if(si->min_blocksize != si->max_blocksize)
      return 0;
    
    len = read_utf8(ptr, size, &ret->sample_number);
    if(!len || (len > 6))
      return 0;
    ret->sample_number *= si->min_blocksize;
    ADVANCE(len);
    }
  else // Variable blocksize, encode sample number
    {
    len = read_utf8(ptr, size, &ret->sample_number);
    if(!len)
      return 0;
    ADVANCE(len);
    }

  /* Block size of this frame */

  if(ret->block_size_code == 0)
    return 0;
  else if(ret->block_size_code == 1)
    ret->blocksize = 192;
  else if(ret->block_size_code < 6)
    ret->blocksize = 576 * (1 << (ret->block_size_code - 2));
  else if(ret->block_size_code == 6)
    {
    CHECK_LEN(1);
    ret->blocksize = *ptr;
    ADVANCE(1);
    ret->blocksize++;
    }
  else if(ret->block_size_code == 7)
    {
    CHECK_LEN(2);
    ret->blocksize = GAVL_PTR_2_16BE(ptr);
    ADVANCE(2);
    ret->blocksize++;
    }
  else
    ret->blocksize = 256 * (1 << (ret->block_size_code - 8));
  
  /* Samplerate */
  switch(ret->samplerate_code)
    {
    case 0x0:
      ret->samplerate = si->samplerate;
      break;
    case 0x1:
      ret->samplerate = 88200;
      break;
    case 0x2:
      ret->samplerate = 176400;
      break;
    case 0x3:
      ret->samplerate = 192000;
      break;
    case 0x4:
      ret->samplerate = 8000;
      break;
    case 0x5:
      ret->samplerate = 16000;
      break;
    case 0x6:
      ret->samplerate = 22050;
      break;
    case 0x7:
      ret->samplerate = 24000;
      break;
    case 0x8:
      ret->samplerate = 32000;
      break;
    case 0x9:
      ret->samplerate = 44100;
      break;
    case 0xa:
      ret->samplerate = 48000;
      break;
    case 0xb:
      ret->samplerate = 96000;
      break;
    case 0xc:
      CHECK_LEN(1);
      ret->samplerate = *ptr;
      ADVANCE(1);
      ret->samplerate *= 1000;
      break;
    case 0xd:
      CHECK_LEN(2);
      ret->samplerate = GAVL_PTR_2_16BE(ptr);
      ADVANCE(2);
      break;
    case 0xe:
      CHECK_LEN(2);
      ret->samplerate = GAVL_PTR_2_16BE(ptr);
      ADVANCE(2);
      ret->samplerate *= 10;
      break;
    case 0xf:
      return 0;
      break;
    }

  /* Channels */
  if(ret->channel_code < 8)
    ret->num_channels = ret->channel_code + 1;
  else if(ret->channel_code < 11)
    ret->num_channels = 2;
  else
    ret->num_channels = 0; // Reserved

  /* Sample size */
  switch(ret->samplesize_code)
    {
    case 0:
      ret->samplesize = si->bits_per_sample;
      break;
    case 1:
      ret->samplesize = 8;
      break;
    case 2:
      ret->samplesize = 12;
      break;
    case 3:
      ret->samplesize = 0; // Reserved
      break;
    case 4:
      ret->samplesize = 16;
      break;
    case 5:
      ret->samplesize = 20;
      break;
    case 6:
      ret->samplesize = 24;
      break;
    case 7:
      ret->samplesize = 0; // Reserved
      break;
    }
  
  /* Get CRC */
  CHECK_LEN(1);
  crc = crc8(ptr_start, ptr - ptr_start);
  if(crc != *ptr)
    {
    //    fprintf(stderr, "CRC mismatch\n");
    return 0;
    }

  if((ret->samplesize != si->bits_per_sample) ||
     (ret->num_channels != si->num_channels) ||
     (ret->samplerate != si->samplerate))
    return 0;
  
  return 1;
  }

int bgav_flac_frame_header_equal(bgav_flac_frame_header_t * h1,
                                 bgav_flac_frame_header_t * h2)
  {
  if(h1->blocking_strategy_code != h2->blocking_strategy_code)
    return 0;
  if(h1->samplerate != h2->samplerate)
    return 0;
  if(h1->samplesize != h2->samplesize)
    return 0;
  //  if((h1->blocking_strategy_code == 0) &&
  //     (h1->blocksize != h2->blocksize))
  //    return 0;
  return 1;
  }


/* Seek table */

int bgav_flac_seektable_read(bgav_input_context_t * input,
                          bgav_flac_seektable_t * ret,
                          int size)
  {
  int i;
  ret->num_entries = size / 18;
  ret->entries = malloc(ret->num_entries * sizeof(*(ret->entries)));

  for(i = 0; i < ret->num_entries; i++)
    {
    if(!bgav_input_read_64_be(input, &ret->entries[i].sample_number) ||
       !bgav_input_read_64_be(input, &ret->entries[i].offset) ||
       !bgav_input_read_16_be(input, &ret->entries[i].num_samples))
      return 0;
    }
  return 1;
  }

void bgav_flac_seektable_dump(bgav_flac_seektable_t * t)
  {
  int i;
  bgav_dprintf("Seektable: %d entries\n", t->num_entries);
  for(i = 0; i < t->num_entries; i++)
    {
    bgav_dprintf("Sample: %" PRId64 ", Position: %" PRId64 ", Num samples: %d\n",
            t->entries[i].sample_number,
            t->entries[i].offset,
            t->entries[i].num_samples);
    }
  }

void bgav_flac_seektable_free(bgav_flac_seektable_t * t)
  {
  if(t->entries)
    free(t->entries);
  }

