! (C) Copyright 2005- ECMWF.
! 
! This software is licensed under the terms of the Apache Licence Version 2.0
! which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
! In applying this licence, ECMWF does not waive the privileges and immunities
! granted to it by virtue of its status as an intergovernmental organisation
! nor does it submit to any jurisdiction.
!

MODULE BYTES_IO_MOD

!**** Interface to BYTES_IO

!     Purpose.
!     --------
!     Fortran 90 Interface to calling byte IO 

!     Author.
!     -------
!        W.Deconinck and M.Hamrud     ECMWF

!     Modifications.
!     --------------
!        Original: 2016-01-28

!     ------------------------------------------------------------------

USE EC_PARKIND , ONLY : JPRD, JPIM, JPIB, JPRM
USE YOMHOOK  , ONLY : LHOOK, DR_HOOK, JPHOOK
USE MPL_MODULE, ONLY : MPL_ABORT

IMPLICIT NONE

PRIVATE :: JPRD, JPIM, JPIB, JPRM
PRIVATE :: LHOOK, DR_HOOK
PRIVATE :: MPL_ABORT

PRIVATE

INTERFACE BYTES_IO_READ
MODULE PROCEDURE BYTES_IO_READ_JPIM_INT
MODULE PROCEDURE BYTES_IO_READ_JPIM_INT_ARRAY
MODULE PROCEDURE BYTES_IO_READ_JPIM_REAL4
MODULE PROCEDURE BYTES_IO_READ_JPIM_REAL4_ARRAY
MODULE PROCEDURE BYTES_IO_READ_JPIM_REAL8
MODULE PROCEDURE BYTES_IO_READ_JPIM_REAL8_ARRAY
MODULE PROCEDURE BYTES_IO_READ_JPIM_CHAR_ARRAY

MODULE PROCEDURE BYTES_IO_READ_JPIB_INT
MODULE PROCEDURE BYTES_IO_READ_JPIB_INT_ARRAY
MODULE PROCEDURE BYTES_IO_READ_JPIB_REAL4
MODULE PROCEDURE BYTES_IO_READ_JPIB_REAL4_ARRAY
MODULE PROCEDURE BYTES_IO_READ_JPIB_REAL8
MODULE PROCEDURE BYTES_IO_READ_JPIB_REAL8_ARRAY
MODULE PROCEDURE BYTES_IO_READ_JPIB_CHAR_ARRAY
END INTERFACE

INTERFACE BYTES_IO_WRITE
MODULE PROCEDURE BYTES_IO_WRITE_JPIM_INT
MODULE PROCEDURE BYTES_IO_WRITE_JPIM_INT_ARRAY
MODULE PROCEDURE BYTES_IO_WRITE_JPIM_REAL4
MODULE PROCEDURE BYTES_IO_WRITE_JPIM_REAL4_ARRAY
MODULE PROCEDURE BYTES_IO_WRITE_JPIM_REAL8
MODULE PROCEDURE BYTES_IO_WRITE_JPIM_REAL8_ARRAY
MODULE PROCEDURE BYTES_IO_WRITE_JPIM_CHAR_ARRAY

MODULE PROCEDURE BYTES_IO_WRITE_JPIB_INT
MODULE PROCEDURE BYTES_IO_WRITE_JPIB_INT_ARRAY
MODULE PROCEDURE BYTES_IO_WRITE_JPIB_REAL4
MODULE PROCEDURE BYTES_IO_WRITE_JPIB_REAL4_ARRAY
MODULE PROCEDURE BYTES_IO_WRITE_JPIB_REAL8
MODULE PROCEDURE BYTES_IO_WRITE_JPIB_REAL8_ARRAY
MODULE PROCEDURE BYTES_IO_WRITE_JPIB_CHAR_ARRAY
END INTERFACE

PUBLIC :: BYTES_IO_OPEN
PUBLIC :: BYTES_IO_CLOSE
PUBLIC :: BYTES_IO_WRITE
PUBLIC :: BYTES_IO_READ
PUBLIC :: BYTES_IO_SEEK
PUBLIC :: BYTES_IO_TELL

INTEGER, PARAMETER, PUBLIC :: JPBYTES_IO_SUCCESS=0
INTEGER, PARAMETER :: JPBYTES_IO_COULD_NOT_OPEN_FILE=-1
INTEGER, PARAMETER :: JPBYTES_IO_COULD_NOT_CLOSE_FILE=-1
INTEGER, PARAMETER :: JPBYTES_IO_INVALID_FILE_NAME=-2
INTEGER, PARAMETER :: JPBYTES_IO_INVALID_OPEN_MODE=-3
INTEGER, PARAMETER :: JPBYTES_IO_END_OF_FILE=-1
INTEGER, PARAMETER :: JPBYTES_IO_SEEK_ERROR=-2
INTEGER, PARAMETER :: JPBYTES_IO_READ_JPIM_ERROR=-2
INTEGER, PARAMETER :: JPBYTES_IO_WRITE_JPIM_ERROR=-1

CONTAINS


SUBROUTINE BYTES_IO_OPEN(KFILE,CDPATH,CDMODE,KRET)
INTEGER(KIND=JPIM),INTENT(OUT)  :: KFILE
CHARACTER(LEN=*)  ,INTENT(IN)   :: CDPATH
CHARACTER(LEN=1)  ,INTENT(IN)   :: CDMODE
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_OPEN',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_OPEN(KFILE,CDPATH,CDMODE,IRET)
IF(PRESENT(KRET)) THEN
  KRET = IRET
ELSEIF (IRET < JPBYTES_IO_SUCCESS) THEN
  WRITE(0,*) 'BYTES_IO ',TRIM(CDPATH),' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_OPEN FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_OPEN',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_OPEN

SUBROUTINE BYTES_IO_CLOSE(KFILE,KRET)
INTEGER(KIND=JPIM),INTENT(IN)  :: KFILE
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_CLOSE',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_CLOSE(KFILE,IRET)
IF(PRESENT(KRET)) THEN
  KRET = IRET
ELSEIF (IRET < JPBYTES_IO_SUCCESS) THEN
  WRITE(0,*) 'BYTES_IO_CLOSE ',KFILE,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_CLOSE FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_CLOSE',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_CLOSE


SUBROUTINE BYTES_IO_SEEK(KFILE,KOFFSET,KWHENCE,KRET)
INTEGER(KIND=JPIM),INTENT(IN) :: KFILE
INTEGER(KIND=JPIM),INTENT(IN) :: KOFFSET 
INTEGER(KIND=JPIM),INTENT(IN) :: KWHENCE
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL :: KRET
INTEGER(KIND=JPIM) :: IRET
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_SEEK',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_SEEK(KFILE,KOFFSET,KWHENCE,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS) THEN
  WRITE(0,*) 'BYTES_IO_SEEK ',KFILE,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_SEEK FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_SEEK',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_SEEK


SUBROUTINE BYTES_IO_TELL(KFILE,KRET)
INTEGER(KIND=JPIM),INTENT(IN) :: KFILE
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL :: KRET
INTEGER(KIND=JPIM) :: IRET
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_SEEK',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_TELL(KFILE,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS) THEN
  WRITE(0,*) 'BYTES_IO_TELL ',KFILE,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_TELL FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_TELL',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_TELL

! ============================================
! BYTES_IO_READ_JPIM

SUBROUTINE BYTES_IO_READ_JPIM_INT(KFILE,KBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
INTEGER(KIND=JPIM),INTENT(OUT)   :: KBUF
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_INT',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,KBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIM_INT ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIM_INT FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_INT',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIM_INT

SUBROUTINE BYTES_IO_READ_JPIM_INT_ARRAY(KFILE,KBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
INTEGER(KIND=JPIM),INTENT(OUT)   :: KBUF(:)
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_INT_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,KBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIM_INT_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIM_INT_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_INT_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIM_INT_ARRAY

SUBROUTINE BYTES_IO_READ_JPIM_REAL4(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRM),INTENT(OUT)      :: PBUF
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_REAL4',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIM_REAL4 ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIM_REAL4 FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_REAL4',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIM_REAL4

SUBROUTINE BYTES_IO_READ_JPIM_REAL4_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRM),INTENT(OUT)      :: PBUF(:)
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_REAL4_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIM_REAL4_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIM_REAL4_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_REAL4_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIM_REAL4_ARRAY

SUBROUTINE BYTES_IO_READ_JPIM_REAL8(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRD),INTENT(OUT)      :: PBUF
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_REAL8',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIM_REAL8 ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIM_REAL8 FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_REAL8',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIM_REAL8

SUBROUTINE BYTES_IO_READ_JPIM_REAL8_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRD),INTENT(OUT)      :: PBUF(:)
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_REAL8_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIM_REAL8_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIM_REAL8_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_REAL8_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIM_REAL8_ARRAY

SUBROUTINE BYTES_IO_READ_JPIM_CHAR_ARRAY(KFILE,CDBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
CHARACTER(LEN=1)  ,INTENT(OUT)   :: CDBUF(:)
INTEGER(KIND=JPIM),INTENT(INOUT) :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_CHAR_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,CDBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIM_CHAR_ARRAY ',KFILE,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIM_CHAR_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIM_CHAR_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIM_CHAR_ARRAY


! ============================================
! BYTES_IO_WRITE_JPIM

SUBROUTINE BYTES_IO_WRITE_JPIM_INT(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
INTEGER(KIND=JPIM),INTENT(IN)    :: PBUF
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE

IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_INT',0,ZHOOK_HANDLE)

CALL C_BYTES_IO_WRITE(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIM_INT ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIM_INT FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_INT',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIM_INT

SUBROUTINE BYTES_IO_WRITE_JPIM_INT_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
INTEGER(KIND=JPIM),INTENT(IN)    :: PBUF(:)
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE

IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_INT',0,ZHOOK_HANDLE)

CALL C_BYTES_IO_WRITE(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIM_INT ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIM_INT FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_INT',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIM_INT_ARRAY

SUBROUTINE BYTES_IO_WRITE_JPIM_REAL4(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRM),INTENT(IN)    :: PBUF
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE

IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_REAL4',0,ZHOOK_HANDLE)

CALL C_BYTES_IO_WRITE(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIM_REAL4 ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIM_REAL4 FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_REAL4',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIM_REAL4

SUBROUTINE BYTES_IO_WRITE_JPIM_REAL4_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRM),INTENT(IN)       :: PBUF(:)
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_REAL4_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_WRITE(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIM_REAL4_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIM_REAL4_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_REAL4_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIM_REAL4_ARRAY


SUBROUTINE BYTES_IO_WRITE_JPIM_REAL8(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRD),INTENT(IN)    :: PBUF
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_REAL8',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_WRITE(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIM_REAL8 ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIM_REAL8 FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_REAL8',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIM_REAL8

SUBROUTINE BYTES_IO_WRITE_JPIM_REAL8_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRD),INTENT(IN)       :: PBUF(:)
INTEGER(KIND=JPIM),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_REAL8_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_WRITE(KFILE,PBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIM_REAL8_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIM_REAL8_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_REAL8_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIM_REAL8_ARRAY

SUBROUTINE BYTES_IO_WRITE_JPIM_CHAR_ARRAY(KFILE,CDBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
CHARACTER(LEN=1)  ,INTENT(IN)    :: CDBUF(:)
INTEGER(KIND=JPIM),INTENT(INOUT) :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL :: KRET
INTEGER(KIND=JPIM) :: IRET
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_CHAR_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_WRITE(KFILE,CDBUF,KBYTES,IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIM_CHAR_ARRAY ',KFILE,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIM_CHAR_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIM_CHAR_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIM_CHAR_ARRAY

! ============================================
! BYTES_IO_READ_JPIB

SUBROUTINE BYTES_IO_READ_JPIB_INT(KFILE,KBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
INTEGER(KIND=JPIM),INTENT(OUT)   :: KBUF
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_INT',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,KBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIB_INT ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIB_INT FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_INT',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIB_INT

SUBROUTINE BYTES_IO_READ_JPIB_INT_ARRAY(KFILE,KBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
INTEGER(KIND=JPIM),INTENT(OUT)   :: KBUF(:)
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_INT_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,KBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIB_INT_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIB_INT_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_INT_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIB_INT_ARRAY

SUBROUTINE BYTES_IO_READ_JPIB_REAL4(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRM),INTENT(OUT)      :: PBUF
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_REAL4',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIB_REAL4 ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIB_REAL4 FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_REAL4',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIB_REAL4

SUBROUTINE BYTES_IO_READ_JPIB_REAL4_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRM),INTENT(OUT)      :: PBUF(:)
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_REAL4_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIB_REAL4_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIB_REAL4_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_REAL4_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIB_REAL4_ARRAY

SUBROUTINE BYTES_IO_READ_JPIB_REAL8(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRD),INTENT(OUT)      :: PBUF
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_REAL8',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIB_REAL8 ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIB_REAL8 FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_REAL8',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIB_REAL8

SUBROUTINE BYTES_IO_READ_JPIB_REAL8_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRD),INTENT(OUT)      :: PBUF(:)
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_REAL8_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIB_REAL8_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIB_REAL8_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_REAL8_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIB_REAL8_ARRAY

SUBROUTINE BYTES_IO_READ_JPIB_CHAR_ARRAY(KFILE,CDBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
CHARACTER(LEN=1)  ,INTENT(OUT)   :: CDBUF(:)
INTEGER(KIND=JPIB),INTENT(IN) :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL   :: KRET
INTEGER(KIND=JPIM) :: IRET,ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_CHAR_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_READ(KFILE,CDBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS .AND. IRET < JPBYTES_IO_END_OF_FILE ) THEN
  WRITE(0,*) 'BYTES_IO_READ_JPIB_CHAR_ARRAY ',KFILE,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_READ_JPIB_CHAR_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_READ_JPIB_CHAR_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_READ_JPIB_CHAR_ARRAY


! ============================================
! BYTES_IO_WRITE_JPIB

SUBROUTINE BYTES_IO_WRITE_JPIB_INT(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
INTEGER(KIND=JPIM),INTENT(IN)    :: PBUF
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE

IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_INT',0,ZHOOK_HANDLE)

CALL C_BYTES_IO_WRITE(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIB_INT ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIB_INT FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_INT',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIB_INT

SUBROUTINE BYTES_IO_WRITE_JPIB_INT_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
INTEGER(KIND=JPIM),INTENT(IN)    :: PBUF(:)
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE

IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_INT',0,ZHOOK_HANDLE)

CALL C_BYTES_IO_WRITE(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIB_INT ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIB_INT FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_INT',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIB_INT_ARRAY

SUBROUTINE BYTES_IO_WRITE_JPIB_REAL4(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRM),INTENT(IN)    :: PBUF
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE

IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_REAL4',0,ZHOOK_HANDLE)

CALL C_BYTES_IO_WRITE(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIB_REAL4 ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIB_REAL4 FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_REAL4',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIB_REAL4

SUBROUTINE BYTES_IO_WRITE_JPIB_REAL4_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRM),INTENT(IN)       :: PBUF(:)
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_REAL4_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_WRITE(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIB_REAL4_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIB_REAL4_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_REAL4_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIB_REAL4_ARRAY


SUBROUTINE BYTES_IO_WRITE_JPIB_REAL8(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRD),INTENT(IN)    :: PBUF
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_REAL8',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_WRITE(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIB_REAL8 ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIB_REAL8 FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_REAL8',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIB_REAL8

SUBROUTINE BYTES_IO_WRITE_JPIB_REAL8_ARRAY(KFILE,PBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
REAL(KIND=JPRD),INTENT(IN)       :: PBUF(:)
INTEGER(KIND=JPIB),INTENT(IN)    :: KBYTES
INTEGER(KIND=JPIM),OPTIONAL,INTENT(OUT) :: KRET
INTEGER(KIND=JPIM) :: IRET
INTEGER(KIND=JPIM) :: ILEN
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_REAL8_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_WRITE(KFILE,PBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ENDIF
IF(IRET < JPBYTES_IO_SUCCESS ) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIB_REAL8_ARRAY ',KFILE,' ',KBYTES,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIB_REAL8_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_REAL8_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIB_REAL8_ARRAY

SUBROUTINE BYTES_IO_WRITE_JPIB_CHAR_ARRAY(KFILE,CDBUF,KBYTES,KRET)
INTEGER(KIND=JPIM),INTENT(IN)    :: KFILE
CHARACTER(LEN=1)  ,INTENT(IN)    :: CDBUF(:)
INTEGER(KIND=JPIB),INTENT(IN) :: KBYTES
INTEGER(KIND=JPIM),INTENT(OUT), OPTIONAL :: KRET
INTEGER(KIND=JPIM) :: IRET
REAL(KIND=JPHOOK) :: ZHOOK_HANDLE
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_CHAR_ARRAY',0,ZHOOK_HANDLE)
CALL C_BYTES_IO_WRITE(KFILE,CDBUF,INT(KBYTES,JPIM),IRET)
IF(PRESENT(KRET)) THEN
  KRET=IRET
ELSEIF(IRET < JPBYTES_IO_SUCCESS) THEN
  WRITE(0,*) 'BYTES_IO_WRITE_JPIB_CHAR_ARRAY ',KFILE,' FAILED',IRET
  CALL MPL_ABORT('BYTES_IO_WRITE_JPIB_CHAR_ARRAY FAILED')
ENDIF
IF (LHOOK) CALL DR_HOOK('BYTES_IO:BYTES_IO_WRITE_JPIB_CHAR_ARRAY',1,ZHOOK_HANDLE)
END SUBROUTINE BYTES_IO_WRITE_JPIB_CHAR_ARRAY

END MODULE BYTES_IO_MOD
