@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-association-get: Get the Association Descriptive File
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use Term::ReadKey;

my %opt = ( gateway => undef, template => 0, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  gateway=s
	  template
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $associationName = $ARGV[0];
my $targetFile      = $ARGV[1];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No association-name specified!\n" ) if not($associationName);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No target-file specified!\n" )      if not($targetFile);

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# If no Gateway is specified then use the default Gateway
$opt{gateway} = $controlChannel->getGatewayName()->result if ( not $opt{gateway} );

# Open the target file
open FILE, ">", $targetFile or die "Error creating file: " . $targetFile . "\n";
chmod 0600, $targetFile;

# Get the detail for the specified associationName
my $association =
  $controlChannel->getAssociation( $token, $opt{gateway}, $associationName, $opt{template} ? 'true' : 'false' )->valueof('//getAssociationResponse/return');

# Print the parameters
print FILE "##############################################################\n";
print FILE "# Main Parameters\n";
print FILE "##############################################################\n";
print FILE "\$name='" . ( $association->{name} ) . "';\n";
print FILE "\$active='" . ( $association->{active} eq 'true' ? 'yes' : 'no' ) . "';\n";
print FILE "\$comment='" . $association->{comment} . "';\n";
print FILE "\$grantedUserList='" . $association->{grantedUserList} . "';\n";
print FILE "\$directory='" . $association->{directory} . "';\n";
print FILE "\$hostName='" . $association->{hostName} . "';\n";
print FILE "\$login='" . $association->{login} . "';\n";
print FILE "\$protocol='" . $association->{protocol} . "';\n";
print FILE "\n";
print FILE "##############################################################\n";
print FILE "# Data\n";
print FILE "##############################################################\n";
print FILE "\$data='\n" . $association->{data} . "';\n";

# Close file
close FILE;

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-association-get - Get the Association Descriptive File

=head1 SYNOPSIS

B<ecaccess-association-get -version|-help|-manual>

B<ecaccess-association-get [-debug] [-gateway> I<name>B<] [-template]> I<association-name> I<target-file>

=head1 DESCRIPTION

Get the Descriptive File for the ECtrans Association specified by the I<association-name> parameter.
Once downloaded, the I<target-file> can be modified and processed through the B<ecaccess-association-put>
command.

When using the B<-template> option, if the Association already exists in the ECaccess Gateway Database,
then the command return with an error. In order to create a new Association similar to an existing one
the B<ecaccess-association-get> command can be used to retrieve the Descriptive File of the existing
Association. The file can then be updated and pushed to the ECaccess Gateway Database with the
B<ecaccess-association-put> command.

An ECtrans Association Descriptive File contains parameters for ECtrans of the general form $name='value'.
Each ECtrans Association has their own Descriptive File, and parameters in any given Descriptive File will be
passed to the ECtrans container whenever a transfer is requested to the corresponding Association.

Blank lines and leading spaces and tabs are ignored when they do not occur in single quoted strings. Lines
whose first non-space character is a pound-sign (#) are comments, and are ignored. Note that comments are
not allowed on the same line as parameters and don't persist when the Descriptive File is retrieved from
the Gateway (they might however be used when Descriptive Files are archived on your system).

The parameters are the following:

=over 8

=item I<active>

This is a boolean which indicate if the Association can be used by ECtrans or not (e.g. 'yes' or 'no'). You might
want to deactivate an Association but still keep it in the ECaccess Gateway Database for later.

=item I<comment>

This is a comment about your Association (e.g. 'Access to the archive system').

=item I<grantedUserList>

This is the list of ECMWF user identifiers which are allowed (other that you) to use this Association. Multiple users
should be separated by a column (e.g. 'abc,def,ghi').

=item I<directory>

This is the directory where to download/upload the files from/to (e.g. '/tmp/data').

=item I<hostName>

This is the name of the host to connect to (e.g. 'hostname.example.ms').

=item I<login>

This is the login to use to connect to the host specified in the I<hostName> parameter (e.g. 'anonymous').

=item I<protocol>

This is the protocol ECtrans will use to connect to the host specified in the I<hostName> parameter. The list of available
protocols for a Gateway can be displayed with the B<ecaccess-protocol-list> command (e.g. 'genericFtp').

=item I<data>

This parameter can be split over multiple lines and contains various options for the I<protocol> which have been selected
for the Association (these options are passed verbatim to the Module which implement the I<protocol>). In order to get the
list of available options for a protocol please use the command B<ecaccess-association-protocol> with the name of the B<protocol>.
The availabe options depends of the version of the ECaccess Gateway which is used to host the Association so it might be
that some options are available for some Gateways and not available for others. An unknown option will be silently ignored
by ECtrans. The format of an option is {protocol-shortname}.{option}={value} (e.g. ftp.port="21" would set the port option
of the genericFtp module to "21"). An example of this parameter is given in the EXAMPLES section below.

=back

=head1 ARGUMENTS

=over 8

=item I<association-name>

The name of the Association to retrieve the corresponding Descriptive File.

=item I<target-file>

The name of the file where to download the Descriptive File.

=back

=head1 OPTIONS

=over 8

=item B<-gateway> I<name>

This is the name of the ECaccess Gateway where the Association is installed.
It is by default the Gateway you are connected to. In order to get the name
of your current Gateway you can use the B<ecaccess-gateway-name> command. When
using the commands at ECMWF the default Gateway is always "ecaccess.ecmwf.int".

=item B<-template>

Allow creating a new Association Descriptive File for the B<ecaccess-association-put> command.
If the Association already exists then the command fails.

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-association-get -template> I<test> I<./test>

Create a new Association Descriptive File for an Association named I<test> and store it in your
current directory (file I<./test>). The file can then be edited and updated accordingly to your
requirements.

B<ecaccess-association-put -password> I<./test>

Create the Association with the previously updated I<./test> Association Descriptive File.

B<ecaccess-association-get -gateway> I<ecaccess.ecmwf.int> I<ginko> I<./ginko>

Get the Descriptive File for the existing I<ginko> Association on the I<ecaccess.ecmwf.int>
Gateway.

B<ecaccess-association-put -gateway> I<ecaccess.ecmwf.int> I<./ginko>

Push the Association back (once updated).

This is an example of an Association Descriptive File:

  ##############################################################
  # Main Parameters
  ##############################################################
  $active='yes'
  $comment='Access to the archive system'
  $grantedUserList='abc,def,jhi'
  $directory='/tmp/data'
  $hostName='hostname.example.ms'
  $login='anonymous'
  $protocol='genericFtp'

  ##############################################################
  # Data
  ##############################################################
  $data='
  ftp.mkdirs="yes"
  ftp.passive="no"
  ftp.port="21"
  ftp.suffix=".tmp"
  ftp.usetmp="yes"'

Please note the multiple lines in the I<data> parameter.

=head1 SEE ALSO

B<ecaccess-association-delete>, B<ecaccess-association-list>, B<ecaccess-association-protocol>,
B<ecaccess-association-put> and B<ecaccess>.

=cut

__END__
:endofperl
