## ------------------------------------------------------------------------
##
## SPDX-License-Identifier: LGPL-2.1-or-later
## Copyright (C) 2017 - 2022 by the deal.II authors
##
## This file is part of the deal.II library.
##
## Part of the source code is dual licensed under Apache-2.0 WITH
## LLVM-exception OR LGPL-2.1-or-later. Detailed license information
## governing the source code and code contributions can be found in
## LICENSE.md and CONTRIBUTING.md at the top level directory of deal.II.
##
## ------------------------------------------------------------------------

#
# Usage:
#   setup_unity_target("${_list_of_unity_source_files}" _includes_per_unity_file _output_src)
#
# This macro sets up some number of unity build targets given a list of files
# and the number of files to be included per unity file. It generates the unity
# build files and appends their names to _output_src.
#
macro(setup_unity_target _unity_include_src _n_includes_per_unity_file _output_src)
  #
  # The unity build files need access to the original source files:
  #
  include_directories(BEFORE ${CMAKE_CURRENT_SOURCE_DIR})

  set(_file_index 0)
  foreach(_unity_file ${_unity_include_src})
    math(EXPR _unity_prefix_index "${_file_index} / ${_n_includes_per_unity_file}")
    #
    # Get around the fact that CMake does not support nested lists by generating
    # a file name and then storing that file name in a list (which we will
    # uniquify later)
    #
    set(_unity_prefix "unity_${_unity_prefix_index}")
    list(APPEND _unity_prefixes ${_unity_prefix})
    list(APPEND ${_unity_prefix} ${_unity_file})

    math(EXPR _file_index "${_file_index} + 1")
  endforeach()

  list(REMOVE_DUPLICATES _unity_prefixes)

  foreach(_unity_prefix ${_unity_prefixes})
    set(_unity_file_name "${CMAKE_CURRENT_BINARY_DIR}/${_unity_prefix}.cc")
    set(_tmp_unity_file_name "${CMAKE_CURRENT_BINARY_DIR}/${_unity_prefix}_tmp.cc")

    #
    # Note the CMake weirdness: _output_src is actually a string containing the
    # variable name since we are in a macro...
    #
    set(${_output_src}
      ${${_output_src}}
      ${_unity_file_name}
      )
    file(WRITE "${_tmp_unity_file_name}"
"// This file was automatically generated by the deal.II CMake configuration as
// part of the unity build subsystem. Please do not edit this file directly;
// instead, make appropriate changes to the relevant CMakeLists.txt file.\n\n"
      )
    foreach(_unity_file ${${_unity_prefix}})
      file(APPEND "${_tmp_unity_file_name}" "#include \"${_unity_file}\"\n")
    endforeach()

    #
    # CMake regenerates all unity files as part of the configuration process,
    # even though they usually have not changed. Avoid some recompilation by
    # using the CONFIGURE_FILE command, which won't update the time stamp unless
    # the content has changed.
    #
    configure_file(${_tmp_unity_file_name} ${_unity_file_name} COPYONLY)
  endforeach()
endmacro()
