if (typeof Test === 'undefined') require('../tester');

Test('plus', function () {
    var n = 'null',
        N = 'NaN',
        I = 'Infinity';

    function t(addendA, addendB, expected) {
        Test.areEqual(String(expected), String(new BigNumber(addendA).plus(addendB)));
        //Test.areEqual(String(expected), String(new BigNumber(addendA).plus(new BigNumber(addendB))));
    }

    BigNumber.config({
        DECIMAL_PLACES: 20,
        ROUNDING_MODE: 4,
        RANGE: 1E9,
        EXPONENTIAL_AT: [-7, 21]
    });

    t(1, 0, 1);
    t(1, -0, 1);
    t(-1, 0, -1);
    t(-1, -0, -1);
    t(1, N, N);
    t(-1, N, N);
    t(1, I, I);
    t(1, -I, -I);
    t(-1, I, I);
    t(-1, -I, -I);
    t(0, 1, 1);
    t(0, -1, -1);
    t(-0, 1, 1);
    t(-0, -1, -1);

    //  IEEE 754 - 2008 section 6.3
    //  When the sum of two operands with opposite signs is exactly zero, the sign
    //  of that sum shall be +0 in all rounding-direction attributes except
    //  roundTowardNegative; under that attribute, the sign of an exact zero sum
    //  shall be −0.
    //  However, x + x = x −(−x) retains the same sign as x even when x is zero.
    BigNumber.config( {ROUNDING_MODE: 3} );
    Test.areEqual('-0', new BigNumber(0).plus(-0).valueOf());     //   0 + -0 = -0
    Test.areEqual('-0', new BigNumber(-0).plus(0).valueOf());     //  -0 +  0 = -0
    Test.areEqual('0', new BigNumber(0).plus(0).valueOf());       //   0 +  0 =  0
    Test.areEqual('-0', new BigNumber(-0).plus(-0).valueOf());    //  -0 + -0 = -0
    Test.areEqual('-0', new BigNumber(1).plus(-1).valueOf());     //   1 + -1 = -0
    Test.areEqual('-0', new BigNumber(-1).plus(1).valueOf());     //  -1 +  1 = -0

    BigNumber.config( {ROUNDING_MODE: 4} );
    Test.areEqual('0', new BigNumber(0).plus(-0).valueOf());      //   0 + -0 =  0
    Test.areEqual('0', new BigNumber(-0).plus(0).valueOf());      //  -0 +  0 =  0
    Test.areEqual('0', new BigNumber(0).plus(0).valueOf());       //   0 +  0 =  0
    Test.areEqual('-0', new BigNumber(-0).plus(-0).valueOf());    //  -0 + -0 = -0
    Test.areEqual('0', new BigNumber(1).plus(-1).valueOf());      //   1 + -1 =  0
    Test.areEqual('0', new BigNumber(-1).plus(1).valueOf());      //  -1 +  1 =  0

    t(0, N, N);
    t(-0, N, N);
    t(0, I, I);
    t(0, -I, -I);
    t(-0, I, I);
    t(-0, -I, -I);
    t(N, 1, N);
    t(N, -1, N);
    t(N, 0, N);
    t(N, -0, N);
    t(N, N, N);
    t(N, I, N);
    t(N, -I, N);
    t(I, 1, I);
    t(I, -1, I);
    t(-I, 1, -I);
    t(-I, -1, -I);
    t(I, 0, I);
    t(I, -0, I);
    t(-I, 0, -I);
    t(-I, -0, -I);
    t(I, N, N);
    t(-I, N, N);
    t(I, I, I);
    t(I, -I, N);
    t(-I, I, N);
    t(-I, -I, -I);

    t(1, '0', '1');
    t(1, '1', '2');
    t(1, '-45', '-44');
    t(1, '22', '23');
    t(1, 0144, '101');
    t(1, '0144', '145');
    t(1, '6.1915', '7.1915');
    t(1, '-1.02', '-0.02');
    t(1, '0.09', '1.09');
    t(1, '-0.0001', '0.9999');
    t(1, '8e5', '800001');
    t(1, '9E12', '9000000000001');
    t(1, '1e-14', '1.00000000000001');
    t(1, '3.345E-9', '1.000000003345');
    t(1, '-345.43e+4', '-3454299');
    t(1, '-94.12E+0', '-93.12');
    t(1, ' 4.001', '5.001');
    t(1, '4.001 ', '5.001');
    t(1, Number.POSITIVE_INFINITY, I);
    t(1, Number.NEGATIVE_INFINITY, -I);
    t('0', 0, '0');
    t(0, '+0', '0');
    t('0', '0', '0');
    t(3, -0, '3');
    t(9.654, 0, '9.654');
    t(0, '0.001', '0.001');
    t(0, '111.1111111110000', '111.111111111');
    t(N, '0', N);
    t(-1, 1, '0');
    t(-0.01, 0.01, '0');
    t(54, -54, '0');
    t(9.99, '-9.99', '0');
    t('0.0000023432495704937', '-0.0000023432495704937', '0');
    t(NaN, NaN, N);
    t(NaN, N, N);
    t(N, NaN, N);
    t(N, 4, N);
    t(N, '4534534.45435435', N);
    t(N, 99999.999, N);
    t(Infinity, '354.345341', I);
    t(3, -I, -I);
    t(-Infinity, -I, -I);
    t(-I, -Infinity, -I);
    t(I, '-999e999', I);
    t('1.21123e43', -I, -I);
    t('-999.0', I, I);
    t('657.342e-45', -I, -I);
    t(I, 123, I);
    t(100, 100, '200');
    t(-999.99, '0.01', '-999.98');
    t('10 ', 4, '14');
    t('03.333', -4, '-0.667');
    t(-1, -0.1, '-1.1');
    t(43534.5435, '0.054645', '43534.598145');
    t('99999', '1', '100000');
    t(' +3e0', 4, '7');

    t('-0.000000046', '0', '-4.6e-8');
    t('0', '-5.1', '-5.1');
    t('1.3', '2', '3.3');
    t('1.02', '1.2', '2.22');
    t('3.0', '0', '3');
    t('3', '31.9', '34.9');
    t('0', '-0.0000000000000712', '-7.12e-14');
    t('1.10', '5', '6.1');
    t('4.2', '-0.000000062', '4.199999938');
    t('1', '0', '1');
    t('-5.1', '1', '-4.1');
    t('0', '-1', '-1');
    t('699', '-4', '695');
    t('0', '-1', '-1');
    t('1.6', '-27.2', '-25.6');
    t('0', '-7', '-7');
    t('3.0', '-4', '-1');
    t('0', '-2.0', '-2');
    t('0', '-3', '-3');
    t('-2', '1', '-1');
    t('-9', '-1', '-10');
    t('2', '-1.1', '0.9');
    t('-5', '-3', '-8');
    t('7', '-37', '-30');
    t('-3', '-5.0', '-8');
    t('1.2', '-0.0000194', '1.1999806');
    t('0', '5', '5');
    t('0', '1', '1');
    t('-0.000000000000214', '0', '-2.14e-13');
    t('0', '0', '0');
    t('0', '-1', '-1');
    t('-3', '156', '153');
    t('231', '0.00000000408', '231.00000000408');
    t('0', '-1.7', '-1.7');
    t('-4.16', '0', '-4.16');
    t('0', '5.8', '5.8');
    t('1.5', '5', '6.5');
    t('4.0', '-6.19', '-2.19');
    t('-1.46', '-5.04', '-6.5');
    t('5.11', '6', '11.11');
    t('-2.11', '0', '-2.11');
    t('0.0067', '-5', '-4.9933');
    t('0', '2', '2');
    t('1.0', '-24.4', '-23.4');
    t('-0.000015', '-6', '-6.000015');
    t('1.5', '0', '1.5');
    t('4.1', '-3', '1.1');
    t('-2', '-1', '-3');
    t('3', '1.5', '4.5');
    t('-7.8', '-3', '-10.8');
    t('-32', '17.6', '-14.4');
    t('0', '0', '0');
    t('-47.4', '-1', '-48.4');
    t('15.4', '0.000014', '15.400014');
    t('7.2', '2.9', '10.1');
    t('-86.5', '-47.2', '-133.7');
    t('1.1', '-31.4', '-30.3');
    t('-121', '3', '-118');
    t('-4', '3', '-1');
    t('-3.98', '1.2', '-2.78');
    t('-4.90', '0', '-4.9');
    t('2.28', '0', '2.28');
    t('-0.0000000000051', '-0.0000000000000000236', '-5.1000236e-12');
    t('1', '-28', '-27');
    t('0', '-3.12', '-3.12');
    t('7', '24.9', '31.9');
    t('-7.8', '17', '9.2');
    t('1', '1', '2');
    t('0.00000000000000000016', '-2', '-1.99999999999999999984');
    t('6', '8.5', '14.5');
    t('1.10', '-1', '0.1');
    t('-1', '3.3', '2.3');
    t('4', '-1.3', '2.7');
    t('0', '2.09', '2.09');
    t('-1', '0', '-1');
    t('-1', '0', '-1');
    t('0', '8.1', '8.1');
    t('-3', '-4.96', '-7.96');
    t('9.73', '0', '9.73');
    t('1', '0', '1');
    t('-1', '-3', '-4');
    t('3', '-3.0', '0');
    t('-2.78', '-403', '-405.78');
    t('1', '-0.00063', '0.99937');
    t('2', '0', '2');
    t('3', '7', '10');
    t('-1', '0', '-1');
    t('-4.1', '-4', '-8.1');
    t('-5', '7', '2');
    t('-7', '-0.00000000000000000511', '-7.00000000000000000511');
    t('0', '0.000000000000000000233', '2.33e-19');
    t('1.2', '-8.5', '-7.3');
    t('2', '-2', '0');
    t('-24', '-5', '-29');
    t('-2.1', '0.0114', '-2.0886');
    t('8', '-5', '3');
    t('0.061', '12.1', '12.161');
    t('0', '2.7', '2.7');
    t('-0.00000871', '0', '-0.00000871');
    t('0', '0', '0');
    t('2', '-6.0', '-4');
    t('9', '-1.2', '7.8');
    t('7', '0', '7');
    t('0', '0.000000000000000213', '2.13e-16');
    t('2.5', '0', '2.5');
    t('0', '0.00211', '0.00211');
    t('6.4', '-15.7', '-9.3');
    t('1.5', '0', '1.5');
    t('-41', '0.113', '-40.887');
    t('-7.1', '2', '-5.1');
    t('6', '-1.6', '4.4');
    t('-1.2', '0', '-1.2');
    t('-3', '13.3', '10.3');
    t('0', '0', '0');
    t('0', '-105', '-105');
    t('-0.52', '-40.9', '-41.42');
    t('1', '0', '1');
    t('0', '0', '0');
    t('-5.1', '-0.00024', '-5.10024');
    t('-0.000000000000027', '6', '5.999999999999973');
    t('125', '-2', '123');
    t('2', '-365', '-363');
    t('6.2', '-55.1', '-48.9');
    t('4.9', '-6', '-1.1');
    t('0.0000000482', '0.0000000019', '5.01e-8');
    t('0', '1.7', '1.7');
    t('78.3', '2.2', '80.5');
    t('-53.9', '4.0', '-49.9');
    t('0', '2.1', '2.1');
    t('-1.0', '-143', '-144');
    t('-1', '2.2', '1.2');
    t('1', '84.9', '85.9');
    t('0', '26', '26');
    t('51', '0.000000000000000757', '51.000000000000000757');
    t('1.1', '-3.67', '-2.57');
    t('-1.2', '1.30', '0.1');
    t('-0.00000000000021', '0.0000000013', '1.29979e-9');
    t('-1.6', '-1', '-2.6');
    t('-2.0', '63', '61');
    t('-3', '7', '4');
    t('-221', '38', '-183');
    t('-1', '0', '-1');
    t('46.4', '2', '48.4');
    t('0', '0', '0');
    t('-1', '-0.0000000853', '-1.0000000853');
    t('79', '0.000190', '79.00019');
    t('0', '-8.59', '-8.59');
    t('1', '-1', '0');
    t('0.000000000000000000110', '-5.8', '-5.79999999999999999989');
    t('6', '3.86', '9.86');
    t('-9', '8', '-1');
    t('-1.0', '-45.9', '-46.9');
    t('-2', '1', '-1');
    t('17.3', '1', '18.3');
    t('0', '0.23', '0.23');
    t('1.14', '0', '1.14');
    t('-1.99', '-1', '-2.99');
    t('9', '0.0000000000000000000157', '9.0000000000000000000157');
    t('-11', '89', '78');
    t('0', '-13.9', '-13.9');
    t('0.00000000000015', '86', '86.00000000000015');
    t('278', '-2', '276');
    t('0', '-2.18', '-2.18');
    t('0', '-0.000000029', '-2.9e-8');
    t('-6', '-0.0000000045', '-6.0000000045');
    t('0', '-24.7', '-24.7');
    t('6.0', '124', '130');
    t('0.00089', '-0.117', '-0.11611');
    t('-0.94', '44', '43.06');
    t('52.1', '-4', '48.1');
    t('0', '-0.00000062', '-6.2e-7');
    t('2', '-0.000000000242', '1.999999999758');
    t('-6.2', '1', '-5.2');
    t('3.4', '1', '4.4');
    t('-1.5', '3.8', '2.3');
    t('3', '-1.27', '1.73');
    t('-1', '7', '6');
    t('-2.29', '-4.8', '-7.09');
    t('0', '0', '0');
    t('-5', '-0.0000000000000016', '-5.0000000000000016');
    t('2.0', '-1.5', '0.5');
    t('94.2', '-1.4', '92.8');
    t('37', '-0.000000000000000000028', '36.999999999999999999972');
    t('-0.00000000000000000750', '1', '0.9999999999999999925');
    t('1.5', '-1.7', '-0.2');
    t('-1', '20.0', '19');
    t('2.6', '0', '2.6');
    t('0', '-28.4', '-28.4');
    t('-12.1', '-14', '-26.1');
    t('1.7', '0.000000041', '1.700000041');
    t('9.5', '4', '13.5');
    t('2.8', '101', '103.8');
    t('0.000000022', '0', '2.2e-8');
    t('6', '28', '34');
    t('7', '-97', '-90');
    t('-1.7', '-3', '-4.7');
    t('107', '6.2', '113.2');
    t('-0.000000000000000118', '-2', '-2.000000000000000118');
    t('-0.000000000000000451', '-5.3', '-5.300000000000000451');
    t('0', '-1', '-1');
    t('0.0000055', '145', '145.0000055');
    t('0', '-8', '-8');
    t('0', '-2.7', '-2.7');
    t('-3', '0', '-3');
    t('-7', '7', '0');
    t('-1.1', '0', '-1.1');
    t('-92', '-1.4', '-93.4');
    t('-2.7', '-3.25', '-5.95');
    t('68.5', '509', '577.5');
    t('0', '0', '0');
    t('22.6', '-1', '21.6');
    t('373', '0', '373');
    t('0', '-5', '-5');
    t('32.2', '-7', '25.2');
    t('-1', '-1.7', '-2.7');
    t('-1.3', '0.0000000048', '-1.2999999952');
    t('5', '-5', '0');
    t('0', '11.9', '11.9');
    t('-0.82', '25', '24.18');
    t('0', '3.1', '3.1');
    t('0.000024', '6', '6.000024');
    t('10', '-0.000000116', '9.999999884');
    t('977', '0', '977');
    t('13', '-0.00000205', '12.99999795');
    t('-7', '-9.0', '-16');
    t('0', '1.05', '1.05');
    t('1', '0', '1');
    t('-10.1', '0', '-10.1');
    t('2.2', '-0.000000000000061', '2.199999999999939');
    t('0', '-0.0000085', '-0.0000085');
    t('3', '3.5', '6.5');
    t('1', '2.8', '3.8');
    t('-2', '-8.60', '-10.6');
    t('223', '9', '232');
    t('-20.4', '-213', '-233.4');
    t('0', '2', '2');
    t('-2.9', '-1.3', '-4.2');
    t('3.0', '0', '3');
    t('-5', '0.00000000000000000011', '-4.99999999999999999989');
    t('-0.000088', '70.4', '70.399912');
    t('-1', '-505', '-506');
    t('0', '-4', '-4');
    t('768', '1.1', '769.1');
    t('2', '0', '2');
    t('88', '1.4', '89.4');
    t('7.8', '0.0000000000000025', '7.8000000000000025');
    t('2.6', '-3.20', '-0.6');
    t('-24', '-2.6', '-26.6');
    t('0', '-1', '-1');
    t('-6', '0.00059', '-5.99941');
    t('14', '4.1', '18.1');
    t('-30.5', '1.48', '-29.02');
    t('-509', '5', '-504');
    t('-1', '3', '2');
    t('1.3', '0.000103', '1.300103');
    t('-2.8', '19.1', '16.3');
    t('10.07', '0.581', '10.651');
    t('3', '-2', '1');
    t('-29', '4', '-25');
    t('-3.80', '-48.2', '-52');
    t('6', '-21.3', '-15.3');
    t('3', '-1.7', '1.3');
    t('0', '0.00000000033', '3.3e-10');
    t('0.49', '0', '0.49');
    t('7', '1.1', '8.1');
    t('1', '-2.73', '-1.73');
    t('0', '-3.89', '-3.89');
    t('1.27', '9', '10.27');
    t('-0.00000000151', '-25', '-25.00000000151');
    t('-11.7', '0.000000000000014', '-11.699999999999986');

    t('3', '-7292337998569017257242651.028572395143563730317242', '-7.292337998569017257242648028572395143563730317242e+24');
    t('-340867371992331.2987436', '337.06898', '-340867371991994.2297636');
    t('10.55995846458487041359', '-11363016227498211122402159.8244', '-1.136301622749821112240214926444153541512958641e+25');
    t('-2493625481630720848504.2253423230119762824868856426', '-2.5572303947825744924097699541922801', '-2.4936254816307208485067825727177945507748966555967922801e+21');
    t('0.000000000000019478219401872667571256888553179333787767', '172.34675119', '172.346751190000019478219401872667571256888553179333787767');
    t('-34', '1679140391.9', '1679140357.9');
    t('-80049928601879665221091.2', '-422696.2614386', '-8.00499286018796656437874614386e+22');
    t('-2807730637205.01726166308462171615083', '0.0000000000000000000622355365575749096529347607937593112149767278', '-2807730637205.0172616630846217160885944634424250903470652392062406887850232722');
    t('2011299643403413077489630775210103349137877109', '-163066968.786271125332323161612905', '2.011299643403413077489630775210103348974810140213728874667676838387095e+45');
    t('-0.00000083671937317281562083962202394322135659', '7878332.295282735111804499965213784837870', '7878332.29528189839243132714959294521584605677864341');
    t('0.000000188701196610233915664001509614338431694523153', '-23707678356144422298078574459450831700.985', '-2.3707678356144422298078574459450831700984999811298803389766084335998490385661568305476847e+37');
    t('26203291462.747893543000897830970817', '2.39160879274042658647090', '26203291465.139502335741324417441717');
    t('-13.731026951', '0.000000000000519065974912984366352959010635376943677199693101323558', '-13.731026950999480934025087015633647040989364623056322800306898676442');
    t('42946999115270631061563709594990331564869354319610', '-689518233710514.17779877383431696209686368332723612515969802', '4.294699911527063106156370959499033087535112060909582220122616568303790313631667276387484030198e+49');
    t('-316537.13', '5.849231740', '-316531.28076826');
    t('-4483733651090758635203278665262327.55342', '0', '-4.48373365109075863520327866526232755342e+33');
    t('0', '-2599', '-2599');
    t('-5148.92870373', '-0.0000000000003511413413454972', '-5148.9287037300003511413413454972');
    t('-134406704543039418795782131496347812', '39838386849529610034769612280421774691.6785604641098', '3.97039801449865706159738301489254268796785604641098e+37');
    t('1.084309017285818872', '0.0000000000004353337515049604976685056082181', '1.0843090172862542057515049604976685056082181');
    t('81757261120729.3998391292', '0.00793', '81757261120729.4077691292');
    t('305717993676885329951937476257830866546431763156719336807', '-0.00000000000000000001482815967', '3.0571799367688532995193747625783086654643176315671933680699999999999999999998517184033e+56');
    t('203522961261530443.917820906776579027929', '-9185.139979750596690606', '203522961261521258.777841156179888421929');
    t('58302388.37179491476', '9936044410052508775.881727897883802', '9936044410110811164.253522812643802');
    t('0.00000000000000000005248972194638156020825779794502284697376', '-0.0388379024820383074988759583112', '-0.03883790248203830744638623636481843979174220205497715302624');
    t('0.000000001932861108418416669193839546006791915101090', '-0.000000000000103564871', '1.93275754354741666919383954600679191510109e-9');
    t('-97.5', '-3.1', '-100.6');
    t('-3644798362098266765793', '-11528555350.16536616701926798156538624434748578548375', '-3.64479836210979532114316536616701926798156538624434748578548375e+21');
    t('-371577606534.62392858950803524930468', '-0.00000000269386577418846418', '-371577606534.62392859220190102349314418');
    t('-691678464095856590.95634633025558439435402102', '0.000000030539948265', '-691678464095856590.95634629971563612935402102');
    t('0.000000003928034045864', '-39380632121496755483.22', '-39380632121496755483.219999996071965954136');
    t('-795.89085780158799717759425220514961346582715', '0.000706564963643740242942689304765914029848878612899171', '-795.890151236624353437351309515844847551797301121387100829');
    t('157678510.5484789671095156751921759005061', '0.0000003723795', '157678510.5484793394890156751921759005061');
    t('-1307.3', '2', '-1305.3');
    t('2952858872457.929554397223499', '13455320760120.8707665014', '16408179632578.800320898623499');
    t('-255588821522540889862114962636995141.122', '0.00000326851523835', '-2.5558882152254088986211496263699514112199673148476165e+35');
    t('0.00000000000014406509', '-560153923805366388086033', '-5.6015392380536638808603299999999999985593491e+23');
    t('-6.099826371', '28726540983564666452792216637044.9356006234', '2.87265409835646664527922166370388357742524e+31');
    t('8636541368105344305914147830.15649125124', '0.0000000000049754', '8.6365413681053443059141478301564912512449754e+27');
    t('-755310742571241039658903028788091708555.4969131794', '99330105954469.4', '-7.553107425712410396589029294579857540860969131794e+38');
    t('4020318299.7', '-670276060499479628473243989367746661643882288', '-6.702760604994796284732439893677466576235639883e+44');
    t('5095849414624671459392138104904515509922724138164588773.042', '149494908418030920.00255211471236475197791306529', '5.09584941462467145939213810490451551007221904658261969304455211471236475197791306529e+54');
    t('138761505665961028617702243300', '9857942288.8', '1.387615056659610286275601855888e+29');
    t('3804404287.52574', '16416902504', '20221306791.52574');
    t('-2055.8130', '0.00218842416232671530104849459457718', '-2055.81081157583767328469895150540542282');
    t('0.000000000000989697422064467113678', '0.697065625725549', '0.697065625726538697422064467113678');
    t('-30288347947284335685837645153586172422797922983.37915571667', '-1335.069192', '-3.028834794728433568583764515358617242279792431844834771667e+46');
    t('0.00054054406579017', '0.0020445688232423225788738738557412646944032455769768734602', '0.0025851128890324925788738738557412646944032455769768734602');
    t('-68994411860.328728', '48816504201613414531319144006465274996081421325116090708', '4.8816504201613414531319144006465274996081421256121678847671272e+55');
    t('-809830632609724847526268.50966586512675871946995', '45805247683771870541946686267298071051761.71368629398645', '4.580524768377186973211605365757322352549320402042885969128053005e+40');
    t('12.244685335891505325017912752605737279289', '-1953.157', '-1940.912314664108494674982087247394262720711');
    t('0.00000000000000012201816814103574356713495876418475275883288964', '-14514327327765802694345312637.28121049209345627407', '-1.451432732776580269434531263728121049209345615205183185896425643286504123581524724116711036e+28');
    t('-35760010680713914665457.416640208061422564137087770', '237677.40440', '-3.576001068071391442778001224020806142256413708777e+22');
    t('0.213764724863667178076578241157520057358951', '-195.381', '-195.167235275136332821923421758842479942641049');
    t('-4', '-20479253407195868023.7190887', '-20479253407195868027.7190887');
    t('-2636506939789983.9517900975', '7166515142782230578873295164.094395142727', '7.166515142779594071933505180142605045227e+27');
    t('0.00001177737517372381877477801315318', '90325802810983024805454390865176398367442693156.692576771', '9.032580281098302480545439086517639836744269315669258854837517372381877477801315318e+46');
    t('0.0000000000000006940972689786006700192372751360485539314987602213580801251', '-321024060655325518945438734389309904797132969999', '-3.210240606553255189454387343893099047971329699989999999999999993059027310213993299807627248639514460685012397786419198749e+47');
    t('-0.0000000000000000014803670473728307445', '-0.72148924607942442354320737482582513411299162884', '-0.72148924607942442502357442219865587861299162884');
    t('0.0000000000000001043810048169882311646196761962068118', '0.00000000000008989735712546153946422343611927620630428259597102874', '9.000173813027852769538805579547241311608259597102874e-14');
    t('0.0000000856354489413072352886007888280862870145666795091412287', '124165153666113222225049048396538922054986467641', '1.241651536661132222250490483965389220549864676410000000856354489413072352886007888280862870145666795091412287e+47');
    t('-0.0000000000000320757814372847561957016', '-81190157394157.37730', '-81190157394157.3773000000000320757814372847561957016');
    t('-4334587898155190922720532.94504694431', '85522.93', '-4.33458789815519092263501001504694431e+24');
    t('657.44690790930416714587', '-0.000000000000000000046335131044673181381027', '657.446907909304167145823664868955326818618973');
    t('29982214095954021.68565215319585', '-44628768214034.30768386111941549271096', '29937585327739987.37796829207643450728904');
    t('0.00000227708329764590268495570108111', '188682548499636418836270555181954622706559207', '1.8868254849963641883627055518195462270655920700000227708329764590268495570108111e+44');
    t('-105740086870.87902330897898941693939252887655925', '-0.00902789833804655794', '-105740086870.88805120731703597487939252887655925');
    t('5321976.194462', '0.00000000062660860859824908696344225671974056962030135135', '5321976.19446200062660860859824908696344225671974056962030135135');
    t('1400649337343672602231550299981618481761671319', '-109314621074190268.0231', '1.4006493373436726022315502998723038606874810509769e+45');
    t('2057998276732722646423663', '22081288', '2.057998276732722668504951e+24');
    t('0.0000000000000000017861940166096730915828677929399660926585274067', '19858.316504586', '19858.3165045860000000017861940166096730915828677929399660926585274067');
    t('-22618978889574488865872.1936326217809072732', '84743644438924408560024120708247557837387061303912', '8.47436444389244085600241206856285789478125724380398063673782190927268e+49');
    t('16899677', '0.0000000000000006200595313980106282035', '16899677.0000000000000006200595313980106282035');
    t('84223630138728231319265207385866523.7037949685268627985151', '98049376624950796086168448377.9', '8.42237281881048562700612935543149016037949685268627985151e+34');
    t('-23139798585733154920035503271.3293705839340975', '-203534508453530507372.39865759096937', '-2.31397987892676633735660106437280281749034675e+28');
    t('2132422616', '313.8131489937516147609', '2132422929.8131489937516147609');
    t('0.00000000031757080246638721492851445443', '3794846068194592901324442.377', '3.79484606819459290132444237700000031757080246638721492851445443e+24');
    t('-720523766692302305335100081586272082042593510557835022', '-48655616939922.2105', '-7.205237666923023053351000815862720820426421661747749442105e+53');
    t('-0.000000000000681090019432916245992942611570731948008', '-78075676178', '-78075676178.000000000000681090019432916245992942611570731948008');
    t('0.0000000000000000000326999427026228462203921452389557015799356287967578926', '-0.0000000000010075912482623982890', '-1.0075912155624555863771537796078547610442984200643712032421074e-12');
    t('-0.000001663549464265503879091759952912765557', '-2288485846519565.90882', '-2288485846519565.908821663549464265503879091759952912765557');
    t('15283021016799660238790294783012978152064246848.302809', '-255.33064438', '1.528302101679966023879029478301297815206424659297216462e+46');
    t('-2141275004855302313', '-14324404392255657678513289015273', '-1.4324404392257798953518144317586e+31');
    t('-0.000000000000000010204742781305685387176792401033809', '-1797802696763189176117325682047878480266832090717009269338', '-1.797802696763189176117325682047878480266832090717009269338000000000000000010204742781305685387176792401033809e+57');
    t('18393642772198629.04467960835946581820412498120401305795', '0.001275506381558716433091344352361195683041600198429562', '18393642772198629.045955114741024534637216325556374253633041600198429562');
    t('-1985677.0479797102021852689', '3421136183.732859', '3419150506.6848792897978147311');
    t('0.0000000000048772038513856900253878297299144253261148646117991377226', '-0.000000000000000000633993968849724597473885254797830639072', '4.8772032173917211756632322560291705282842255397991377226e-12');
    t('44812645985.34921549502557530796823798603', '-3581800787100727450085636788', '-3.58180078710072740527299080265078450497442469203176201397e+27');
    t('1079812992968985436999219069272969', '2', '1.079812992968985436999219069272971e+33');
    t('-1753778882883960239.7357211808838917234439093151132461', '-675.03089678041125557523363798883874525571164770741', '-1753778882883960914.76661796129514729867754730395199135571164770741');
    t('2.5', '79808580452454570.6021583877', '79808580452454573.1021583877');
    t('-2318172325.38510421', '1', '-2318172324.38510421');
    t('1374939.4348', '5.6626826003903229567653167590578', '1374945.0974826003903229567653167590578');
    t('-1433073.1818631797258961330292384136', '121881788197.262469654057', '121880355124.0806064743311038669707615864');
    t('2.494544723114530285809', '3576653', '3576655.494544723114530285809');
    t('7938514604943790100.9276107434535811520579868735844807580453', '20401419259674024936996903873847920688818394344', '2.04014192596740249369969038817864352937621844449276107434535811520579868735844807580453e+46');
    t('-0.00000000000000000659293074407431751262', '26538.5176499542676', '26538.51764995426759999340706925592568248738');
    t('171650791296', '207021136095822862734047.67963169458559659', '2.0702113609599451352534367963169458559659e+23');
    t('111202958445.9250724298025103213740842327570768605596136302', '1.537803126215', '111202958447.4628755560175103213740842327570768605596136302');
    t('-66398807.32298869', '-6090756873.58292331915049610499833418951253059357', '-6157155680.90591200915049610499833418951253059357');
    t('2353147.270338458726403618253', '0.021384377221532', '2353147.291722835947935618253');
    t('-148563993416893680237652940993302142753.35541587', '64239.868', '-1.4856399341689368023765294099330207851348741587e+38');
    t('0.00000000000000952989372890150229168153411', '0.000000001089981372067260502063827792831910332175826', '1.089990901960989403566119474366020332175826e-9');
    t('-0.000095942657103193473656868628869530806593380', '2056191383129368498489079272615326194', '2.05619138312936849848907927261532619399990405734289680652634313137113046919340662e+36');
    t('-322696702507520517171977833895787676173146086137211', '-2440854162625.37382074757014', '-3.2269670250752051717197783389578767617558694029983637382074757014e+50');
    t('-1211219183955140821584', '-5210569206996623901353622002846939989302.92283', '-5.21056920699662390256484118680208081088692283e+39');
    t('-0.0000000000000079666993349250232202', '489285196659741383.006297750838528045433123', '489285196659741383.0062977508385200787337880749767798');
    t('0.00000000077141109510594202471060992900492', '3208.3347', '3208.33470000077141109510594202471060992900492');
    t('-68024506176127817746586754376221734884', '0.15464521042533657299008608160', '-6.80245061761278177465867543762217348838453547895746634270099139184e+37');
    t('-145880984716.215', '2337129866657263570', '2337129720776278853.785');
    t('188974623061492483.142804194521468235379066778', '-0.00000038525080631123327317383933592', '188974623061492483.14280380927066192414579360416066408');
    t('-16356113742963595266465952932662', '-125355883826.39416781321896', '-1.635611374296359526659130881648839416781321896e+31');
    t('-6055715.8736074588411038198577757669624785013783338', '314649408270458779064780.10', '3.146494082704587730090642263925411588961801422242330375214986216662e+23');
    t('115404161731.4660458068936241750308873387559102', '-22544800348.350741696032151441813827609476762', '92859361383.1153041108614727332170597292791482');
    t('-15403035163985572960537283993932052281699167401054521716', '-4.51871358', '-1.540303516398557296053728399393205228169916740105452172051871358e+55');
    t('494.839191894412897545883335718', '-6838.818196825251112157347', '-6343.979004930838214611463664282');
    t('12.6', '0.00000023237282267118', '12.60000023237282267118');
    t('80998.93', '-0.000000000000000006657264362722375638995179208931019910', '80998.92999999999999999334273563727762436100482079106898009');
    t('24311907546797437.192938', '33293941690', '24311940840739127.192938');
    t('-2439553576035597703284083776782658338714854480.76670', '-552371860516159411553653432588968147375569940274.6', '-5.548114140921950092569375163657508057142847947553667e+47');
    t('-2', '267.41036926198544821143', '265.41036926198544821143');
    t('-221185414.39378617220971422653741297827696718276', '147716681512356798558369414201759187083768076717531.7', '1.4771668151235679855836941420175918708376785553211730621382779028577346258702172303281724e+50');
    t('27782844063780674567856973058456377099880', '0.42833855291455356', '2.778284406378067456785697305845637709988042833855291455356e+40');
    t('-6779946038658269907151725830212.3231241258972158956574549786', '-59152624443548.397860097634612838089', '-6.7799460386582699663043502737607209842235318287337464549786e+30');
    t('5583728', '-12088255617331322291590332122631397105695', '-1.2088255617331322291590332122631391521967e+40');
    t('-122.55488541428134', '-1849241463858352827933557649058195369965078195', '-1.84924146385835282793355764905819536996507831755488541428134e+45');
    t('-5.40620', '-7926553114777.34793', '-7926553114782.75413');
    t('-0.000253850655677758430078658974284641259587365835647', '9712716976.55714607561630095490035500071', '9712716976.556892224960623196470276341735715358740412634164353');
    t('168095465368167755.20132', '-0.000000000000100770026067662522650967001796393', '168095465368167755.201319999999899229973932337477349032998203607');
    t('-23084021.1912', '-0.0000000002552886320752142', '-23084021.1912000002552886320752142');
    t('-0.00000034692938443702777764811193148332', '-2291330567445.020091462', '-2291330567445.02009180892938443702777764811193148332');
    t('-0.000000000000000000500', '-3638839062523.16788353204231378745138113805005277262', '-3638839062523.16788353204231378795138113805005277262');
    t('-21291430264059347596', '-278661231050935040505252800716670488924003.0', '-2.78661231050935040505274092146934548271599e+41');
    t('-3937382504232.4631934324802032008', '-12670428.0658912205267960601919307115632', '-3937395174660.5290846530069992609919307115632');
    t('-146982.22450149250', '623504303083058604561989109312642337', '6.235043030830586045619891093124953547754985075e+35');
    t('253.01', '802625930614062.9705947195646834588541152620643', '802625930614315.9805947195646834588541152620643');
    t('115359268370848989850854005321447663172886737834785520883', '458.4807', '1.153592683708489898508540053214476631728867378347855213414807e+56');
    t('163.09176', '0.00000000000000001823481782581109744', '163.09176000000000001823481782581109744');
    t('-2213687948.07450556690935288', '21871988945935811925735.3643007186962826339', '2.18719889459335982377872897951517869297539e+22');
    t('-13600.9931855956842627546715', '137199858862267.530744593929324697303900300812048', '137199858848666.537558998245061942632400300812048');
    t('64302.84879', '-0.00000000000000000031884862211764', '64302.84878999999999999968115137788236');
    t('-21.0343150', '0.00003762594931905088097193777529518846', '-21.03427737405068094911902806222470481154');
    t('-43.2', '0.0000000000000000088209059726558824954701945990768141500402895240582621', '-43.1999999999999999911790940273441175045298054009231858499597104759417379');
    t('-21.716', '0.00010435033062418396646761217', '-21.71589564966937581603353238783');
    t('-14349223886609.8996911241', '591424', '-14349223295185.8996911241');
    t('387.146464770977292777877373605881', '0.000000000748129478828479148973975406341232062736061643261337037', '387.146464771725422256705852754854975406341232062736061643261337037');
    t('-5400652.348672', '-272186438747803380.1804043255', '-272186438753204032.5290763255');
    t('-21.3', '6.8', '-14.5');
    t('0.00000000103600815339389940326020721966409', '29645237.7502768697756870856787621204', '29645237.75027687081169523907266152366020721966409');
    t('-534005184647169042.131454403358674522', '0.005694323359477401172045805857601828035888769857500968445222', '-534005184647169042.125760079999197120827954194142398171964111230142499031554778');
    t('1862736418', '1', '1862736419');
    t('351490387188996259', '62.100460124428609287573', '351490387188996321.100460124428609287573');
    t('169499641692977052216453491793.9787615383724028133893', '0.00000000000000006361911', '1.6949964169297705221645349179397876153837240287700841e+29');
    t('-8', '3417322569.4831394174', '3417322561.4831394174');
    t('-838501752798.326614960580367718115', '-2022114720338558.047757485234', '-2022953222091356.374372445814367718115');
    t('-1549912.8026532077081092082547', '-45506131.43371799701886630176768733952185393859212', '-47056044.23637120472697551002238733952185393859212');
    t('-2183663034044253888858323830694952562946010.782', '1354851.73581347266611421085387946798291', '-2.18366303404425388885832383069495256159115904618652733388578914612053201709e+42');
    t('-0.0000000000046712261814706680606784284472438172230285748063131539', '918514605286328938996265974.307597316850270', '9.185146052863289389962659743075973168455987738185293319393215715527561827769714251936868461e+26');
    t('-0.0301991824416038881101099', '-4464346608', '-4464346608.0301991824416038881101099');
    t('3.98', '5218371.49516', '5218375.47516');
    t('4832441872354802689671', '-0.0000000000213837', '4.8324418723548026896709999999999786163e+21');
    t('0.000000000000000016298281', '-0.00000000348998960378566871408591706715068151875573445053079845095', '-3.48998958748738771408591706715068151875573445053079845095e-9');
    t('-2059458247822896700102858412401366415.403909858', '30110.7306736165788262027129954556', '-2.0594582478228967001028584124013363046732362414211737972870045444e+36');
    t('-106653', '-6554779180145425976', '-6554779180145532629');
    t('0', '-12771592550606134879262234468876731140678883443896', '-1.2771592550606134879262234468876731140678883443896e+49');
    t('-141742173848073082.100575461779634531', '-7', '-141742173848073089.100575461779634531');
    t('245.6735', '-3537763728425648898.536045229', '-3537763728425648652.862545229');
    t('23.71866884462009661132193', '48487130568434204612649872032673642473034', '4.848713056843420461264987203267364247305771866884462009661132193e+40');
    t('79401597909529525308.48420609229175', '2591403158790529529573511201.19', '2.59140323819212743910303650967420609229175e+27');
    t('284576176526552350', '-10', '284576176526552340');
    t('1.0682', '7861.283121183677379038987', '7862.351321183677379038987');
    t('-1.60444805', '0.0000028847098483848786064787560152209651320682639', '-1.6044451652901516151213935212439847790348679317361');
    t('-0.0000000018996053506896890911603732691263731955475477051763872', '-57581.875373867812973783', '-57581.8753738697125791336896890911603732691263731955475477051763872');
    t('0.00000000000000000012392510786805471753525331968033845561363410307741609902317', '0.000004349947715706492223712976943980034293671355123', '0.00000434994771570661614882084499869756954699103546145561363410307741609902317');
    t('128907.8518369701779', '-1', '128906.8518369701779');
    t('-21.35510004122675870004717674727620213', '0.00000000000000000001009606209896423416929193498290467', '-21.35510004122675870003708068517723789583070806501709533');
    t('-1036072.70', '30288483890840.7644829772783388118', '30288482854768.0644829772783388118');
    t('1', '0.00000000108663482163751960896', '1.00000000108663482163751960896');
    t('196690688540287471600138402631254097', '-1296924088245887989211255853', '1.96690687243363383354250413419998244e+35');
    t('1791156.834533749567213189130', '173011022354550416183638753635.63334944502450395340681', '1.7301102235455041618364054479246788319459171714253681e+29');
    t('1861430347575777575680356960079436184499.919613', '-612614849498276685996593027243', '1.861430346963162726182080274082843157256919613e+39');
    t('-7758655923622034030955.9291230128', '0.00594001420381222017096211329', '-7.75865592362203403095592318299859618777982903788671e+21');
    t('-5459.708512054323229825378150954057236451373947101', '0.00000000000000010467896268967026375601267236689054126519', '-5459.70851205432322972069918826438697269536127473410945873481');
    t('1.5357640654914', '26484040831501596185144229855201045526.8772', '2.64840408315015961851442298552010455284129640654914e+37');
    t('0.000000000000000000551032982567013358317393735391', '3951981877.70635', '3951981877.706350000000000000551032982567013358317393735391');
    t('19966337.40327464680217082908346850752429077', '0.0000000694117823464637414804852860124695677260107514259062814196', '19966337.4032747162139531755472099880095767824695677260107514259062814196');
    t('0.000000000000114387', '26.0780', '26.078000000000114387');
    t('-0.000000000000058566632687', '154445365803820719089161481686648591330608966505075815231', '1.54445365803820719089161481686648591330608966505075815230999999999999941433367313e+56');
    t('-81.9', '-290987195118120910.650828527355125621', '-290987195118120992.550828527355125621');
    t('0.0000000000129766844476802102046046856035673832204965536', '-15.50195731', '-15.5019573099870233155523197897953953143964326167795034464');
    t('5081846528', '22.8724', '5081846550.8724');
    t('-85904518857594634808347910122940606952070250543093.244309866', '87863849.691319671938088226', '-8.5904518857594634808347910122940606952070162679243552990194061911774e+49');
    t('-170442726618237824341261070261982330084077537445839818', '0.000000000000000780053862947758376946726067312229726087592094', '-1.70442726618237824341261070261982330084077537445839817999999999999999219946137052241623053273932687770273912407906e+53');
    t('413.7126477448411377495505703', '-316329.243682', '-315915.5310342551588622504494297');
    t('-0.00000000000000483378251152198797664837453061179141660023', '4083329356329023991816392216832315734019339465889233737667', '4.08332935632902399181639221683231573401933946588923373766699999999999999516621748847801202335162546938820858339977e+57');
    t('0.000000000004945130554715279352', '20393092845', '20393092845.000000000004945130554715279352');
    t('66.60837', '28592560366063.746528', '28592560366130.354898');
    t('-10824143490362152769146.8628434597262310327475178771', '-6.442', '-1.08241434903621527691533048434597262310327475178771e+22');
    t('-17.63', '-0.000000000000000012319', '-17.630000000000000012319');
    t('-134.1362825800766312455', '17351.358339090284637450403853', '17217.222056510208006204903853');
    t('0.000000000001327776', '-0.00005717732', '-0.000057177318672224');
    t('0.000081202465051128565', '9.25', '9.250081202465051128565');
    t('-49969297933006175912135', '201', '-4.9969297933006175911934e+22');
    t('-0.0000000000000000000351971832201257343216570588903268', '-614730534926067941867', '-614730534926067941867.0000000000000000000351971832201257343216570588903268');
    t('-4699376204284239198897468324308327.578823266', '0.01240127407873530917', '-4.69937620428423919889746832430832756642199192126469083e+33');
    t('-166859773331579667092627457017924681891', '1.177', '-1.66859773331579667092627457017924681889823e+38');
    t('297611546807476.7389967424', '517878390231.954956053920806357536', '298129425197708.693952796320806357536');
    t('2.26', '476121878051171262698479334452067496570446544422226', '4.7612187805117126269847933445206749657044654442222826e+50');
    t('0.000000000348934602334173547747920539533095838627739', '-0.000000000000827121544297495855005100518267691068913070523148', '3.48107480789876051892915439014828147558825929476852e-10');
    t('0.0000000016630321489412099873134278851', '33382216672582.0742647965724', '33382216672582.0742647982354321489412099873134278851');
    t('-19.715751399', '1455362393129824490954487356267083.8186754959749850983', '1.4553623931298244909544873562670641029240969749850983e+33');
    t('11152141361730026829285665961687.3139561070881149371766', '2.76963401441423333517597101345', '1.115214136173002682928566596169008359012150234827235257101345e+31');
    t('-44836120135503200078585015978983219367221886861731704', '-2', '-4.4836120135503200078585015978983219367221886861731706e+52');
    t('16.02069381131', '0.0000000000000042', '16.0206938113100042');
    t('-329678350725130.3958196996612366385400490474513156349559', '-3467', '-329678350728597.3958196996612366385400490474513156349559');
    t('-4182765105477793237555.273265434', '56254364076240107601891319154234794697144.298685574', '5.625436407624010759770855404875700145958902542014e+40');
    t('98796860384198464024305164555588491131671343895593228060535', '-1405573031776506147878053891673759594011202270592763', '9.8796858978625432247799016677534599457911749884390957467772e+58');
    t('1092363940258060395083098718642956733485643364', '-10598809886235236342944197021319583170547674110389928477485', '-1.0598809886234143979003938960924500071829031153656442834121e+58');
    t('37586361614.095', '-3148149077577', '-3110562715962.905');
    t('-254953108518285985', '-1.07910144271207423934923587934451351214', '-254953108518285986.07910144271207423934923587934451351214');
    t('2442.32751', '43535097.033480512037', '43537539.360990512037');
    t('91995040568885362527648040727723', '0.4786144932196936758637061866474810131052563763028', '9.19950405688853625276480407277234786144932196936758637061866474810131052563763028e+31');
    t('-23863678424756722926637928011000581.244001324799381062566619', '96688.916787457820799184195711813964401021274965053735523', '-2.3863678424756722926637928010903892327213866978581878370907186035598978725034946264477e+34');
    t('0', '-0.00000000000027852776765692', '-2.7852776765692e-13');
    t('-1623378710787743245737360669094938021186.3532080061729827', '0', '-1.6233787107877432457373606690949380211863532080061729827e+39');
    t('0', '0.000000049048339295559358355126332957758142200703099', '4.9048339295559358355126332957758142200703099e-8');
    t('6192378510550191959653347.846386529481410301978431652948347', '-0.38294548165530032173705676329607616564872132520398', '6.19237851055019195965334746344104782610998024137488965227083435127867479602e+24');
    t('-24283320565469963929558662982.9349694043', '2673821469225170314390', '-2.42833178916484947043883485929349694043e+28');
    t('218.110682218445321922075046444527', '0.01497687885540092217834', '218.125659097300722844253386444527');
    t('-150034638.4607772346143954532', '-20008571757063313138', '-20008571757213347776.4607772346143954532');
    t('-2971683662963108664767', '19264012460036395799353855810042830620021', '1.9264012460036395796382172147079721955254e+40');
    t('-5252175.40891737279869057', '-2027371894615285196604888264930073402643.5207', '-2.02737189461528519660488826493007865481892961737279869057e+39');
    t('-20871238425800671889182.829701547250', '-76082554384456.3158288776570471938928399650934647043692665519', '-2.08712385018832262736391455304249070471938928399650934647043692665519e+22');
    t('0', '-6932.702341315865', '-6932.702341315865');
    t('0.0000000000000000005738499200802402637492440447802302210350633678412977240457', '31318.68', '31318.6800000000000000005738499200802402637492440447802302210350633678412977240457');
    t('-934961659015198853393126550.25200823467372663246334', '74878058.044873358837373372754304', '-9.34961659015198853318248492207134875836353259709036e+26');
    t('150.22669', '3684561512708165421004.40592369696065395561261265524', '3.68456151270816542115463261369696065395561261265524e+21');
    t('-0.0000000000000015', '0.0001183766448622143702316173650526018028825820894748606870475', '0.0001183766448607143702316173650526018028825820894748606870475');
    t('-2.8', '-0.000000000001829913701194321808657770171290559944848203', '-2.800000000001829913701194321808657770171290559944848203');
    t('0.000018752442721586663741713313007598397434839252', '-396673155594371.6367526145400674503928659772089760230841322', '-396673155594371.636733862097345863729124263895968424686697360748');
    t('-13096091967075751663002949891851210', '-1477304896251731689853.4603171477120194670', '-1.3096091967077228967899201623541063460317147712019467e+34');
    t('-740512796205842509947742580189685788668.411', '-2824022128765.93600320598135905249475748957', '-7.4051279620584250994774258301370791743434700320598135905249475748957e+38');
    t('-0.00000000000000000007489577842583193917', '1.663748696735089', '1.66374869673508899992510422157416806083');
    t('-358935750965455.3592219004802490902523174719', '43371.1', '-358935750922084.2592219004802490902523174719');
    t('9667988395245566159878.653259', '0.000000000116385624231571218', '9.667988395245566159878653259000116385624231571218e+21');
    t('2758033402.5184130', '661106474725.93', '663864508128.448413');
    t('999070419142037555127.53', '-381460743151065544049291431867372559397611945291347286', '-3.8146074315106554404929143186737156032719280325379215847e+53');
    t('-92699966163766696.7490442006', '-1915615.8916014755799758113439947795642627', '-92699966165682312.6406456761799758113439947795642627');
    t('489937.3780742473458733440028228509907624', '-1', '489936.3780742473458733440028228509907624');
    t('-0.000000000000754489936', '-0.00000000000036618270314381889826166978763226', '-1.12067263914381889826166978763226e-12');
    t('390379828579195857322601563326484972906745.91988878145980541', '704108103.1382116', '3.9037982857919585732260156332648567701484905810038145980541e+41');
    t('-26.87', '2353822.691199775358305790461', '2353795.821199775358305790461');
    t('-1587155103407939753589555224978500385.2263024764001305020', '0.0000000000011377959206097569335266885356013552156589880648', '-1.5871551034079397535895552249785003852263024763989927060793902430664733114643986447843410119352e+36');
    t('-5708346691.2506', '-2970137689.07191265257072771', '-8678484380.32251265257072771');
    t('0.0000000000000002334849322956799148269551416529021258', '51972.5801195102957735093196127055803694295535', '51972.5801195102957737428045450012602842565086416529021258');
    t('-104080.3162441765370132760114', '0.000000390882311111306995576914162226185882353716', '-104080.316243785654702164704404423085837773814117646284');
    t('-1923634420932885580541323911945854', '-5442.2', '-1.9236344209328855805413239119512962e+33');
    t('269402746056235370532350773376.6611241372196686370390', '-2.398', '2.69402746056235370532350773374263124137219668637039e+29');
    t('4.6142', '-91954290028090809.5727705166649650', '-91954290028090804.958570516664965');
    t('0.0000000038792075674539308578456', '-0.00000000799999446190038', '-4.1207868944464491421544e-9');
    t('86889265743549292727829232652504603134424', '3966264277627964.8584259691377065', '8.68892657435492927278292366187688807623888584259691377065e+40');
    t('-0.01849718975477457', '42293783764.387302167416177115129958269578', '42293783764.368804977661402545129958269578');
    t('229183.9324129997835780196022124839', '-180.13280', '229003.7996129997835780196022124839');
    t('0.0000000000000004071', '-6632946124.9569670001116089493461386479567752', '-6632946124.9569670001116085422461386479567752');
    t('-79368388.12060961147325951196402', '-8818967839841756190843118.1854748170628718251708829084816359', '-8.8189678398417562702115063060844285361313371349029084816359e+24');
    t('-96.8977819169541179624268824', '-110441943210894571.8830072797816067381187179609278660643', '-110441943210894668.7807891967357247005456003609278660643');
    t('-36873417583534015.45598304691218697443', '1060166.917377718838271', '-36873417582473848.53860532807391597443');
    t('0.000000000000002700871444392', '-0.0605349026669693089396005912402318', '-0.0605349026669666080681561992402318');
    t('-0.000000000000053692523', '-0.00000137254604803', '-0.000001372546101722523');
    t('1074227134372692301132209354432413147.1963324', '0.000000000000003598619851546479146545077911695828898095102737795956459', '1.074227134372692301132209354432413147196332400000003598619851546479146545077911695828898095102737795956459e+36');
    t('-1017687657058637575723357458862923106911128795220693738771601', '4258563010.69702', '-1.01768765705863757572335745886292310691112879522068948020859030298e+60');
    t('15695318372.722371', '-7262689556596731591510081797054945.08235892482', '-7.26268955659673159151006610173657235998792482e+33');
    t('-0.184121579717', '-0.00000000000032197634799642162905185759509417879', '-0.18412157971732197634799642162905185759509417879');
    t('496711464563303349082978268361607314429719423836547893298', '0.00780349823846259456540557', '4.9671146456330334908297826836160731442971942383654789329800780349823846259456540557e+56');
    t('-303514.82542426177461216895784715957208157971', '-0.0000000000000010372854153009492220571724887918212517847890822413245347', '-303514.8254242617746132062432624605213036368824887918212517847890822413245347');
    t('-34126709757832', '-1034.8105337', '-34126709758866.8105337');
    t('-557553079116126732746.60346222066', '-194401.4', '-557553079116126927148.00346222066');
    t('5', '13969995', '13970000');
    t('0.000000000024420167617316425311147207326564663249988502924425073', '58388856055235574885008260342295638772198164.947520', '5.8388856055235574885008260342295638772198164947520000024420167617316425311147207326564663249988502924425073e+43');
    t('-1952712333.155776956436338241', '-0.000000000000001787', '-1952712333.155776956436340028');
    t('0.000000000000000000442399285756922519285614916177278059874646382070215708393687', '-0.0000000000007944556433362323923998778322490216082668198093259', '-7.94455200936946635477358546634105430988759934679517929784291606313e-13');
    t('-17.91', '1176950886190171466875.8668061191330515772843337822020031191', '1.1769508861901714668579568061191330515772843337822020031191e+21');
    t('157077985623370359640124976052501925', '28604427989766282.76371', '1.5707798562337035966872940404226820776371e+35');
    t('7453449.6233221933856360936994951958191632578932339027257', '7220.0', '7460669.6233221933856360936994951958191632578932339027257');
    t('19787312872.5997045141158923178398412505666161654557403133090', '-37306481586787392775866703659600367821024385.6498012', '-3.7306481586787392775866703659600348033711513050096685884107682160158749433383834544259686691e+43');
    t('0.0036377269513631890696573602180806431948860786178658756', '2028146793266265751425109925774562808347846641', '2.0281467932662657514251099257745628083478466410036377269513631890696573602180806431948860786178658756e+45');
    t('-69329355007729552.734184155872579911364124970', '49428086084838561002968128.908102465415944072794710779766855', '4.9428086015509205995238576173918309543364161430585809766855e+25');
    t('5.236129243946948546627393853048424524035753709977', '1458.28382091564243822112', '1463.519950159589386767747393853048424524035753709977');
    t('-125770.7986824578099574150670482067998529', '-54536494221899508990432759126.5854', '-5.45364942218995089904328848973840824578099574150670482067998529e+28');
    t('0.0000000000000000040282063397187671534478', '42.9342796393702', '42.9342796393702000040282063397187671534478');
    t('112255.6473762401', '625578631883141131859950104709.402446232383239610277', '6.25578631883141131859950216965049822472483239610277e+29');
    t('0.0000000000008204051137050998759261498906577146164344577915568026', '-914503544747584022218019464876', '-9.145035447475840222180194648759999999999991795948862949001240738501093422853835655422084431974e+29');
    t('0.000000000000032086696834814061554271827918462166063', '-7', '-6.999999999999967913303165185938445728172081537833937');
    t('-16102511193600.208625575', '0.0000000005921920429652733476192339', '-16102511193600.2086255744078079570347266523807661');
    t('0.05330917360748068', '-1387662245.82971965501508', '-1387662245.77641048140759932');
    t('332566083814742507.4381', '50225079040717.5', '332616308893783224.9381');
    t('0.0535616342876656884145', '40.302980205257854', '40.3565418395455196884145');
    t('2.095810329351761236733504648511', '1291032886241879180795380.39652395418615', '1.291032886241879180795382492334283537911236733504648511e+24');
    t('76.93666308278743385854202793941924424687995851136', '-0.0000031951856193404', '76.93665988760181451814202793941924424687995851136');
    t('-4089192.401545276102326047095158140681', '0.0000000000294587249643456047378078626482043840305033', '-4089192.4015452760728673221308125359431921373517956159694967');
    t('-0.0000000026917607386', '-0.000000001463738085548829246226730', '-4.15549882414882924622673e-9');
    t('5997.3860262975745', '-9.2570184551997909', '5988.1290078423747091');
    t('-0.0000001736810132395059213491577845055187737', '-16125521840386.2', '-16125521840386.2000001736810132395059213491577845055187737');
    t('-12.393', '-959175810182.97', '-959175810195.363');
    t('0.000000000000000012703677534683229539552879126254493256638930817', '-162613165341774647773791411159812774229455.506', '-1.62613165341774647773791411159812774229455505999999999999987296322465316770460447120873745506743361069183e+41');
    t('5765801580384047256456488623813074.84286854632242004465357', '165435022359303094.00576799365464765649', '5.76580158038404742189151098311616884863653997706770114357e+33');
    t('0.000024851933157201952923401889849591044056443558822429', '-19872544695921164383628718578914483460945374183201070203', '-1.9872544695921164383628718578914483460945374183201070202999975148066842798047076598110150408955943556441177571e+55');
    t('15967162444205725.5835876977581985895876178', '-22.09383479755539543658379', '15967162444205703.4897529002028031530038278');
    t('-1895.5870000978149', '158456457583275411.151255931089745677273050725', '158456457583273515.564255833274845677273050725');
    t('0.00000000000000006613200838614468668274572771438132', '0.000000000000009799052946232089973195569084478957663245622', '9.865184954618234659878314812193338983245622e-15');
    t('-0.00007217386', '-30380533.92750742177144456679338699', '-30380533.92757959563144456679338699');
    t('49321761839.8798408956884550', '104924407952098597601614114560902417392737539279.14066', '1.04924407952098597601614114560902417442059301119020500895688455e+47');
    t('-1691047173618367369.8797144744337845447480409090470639328', '-10750', '-1691047173618378119.8797144744337845447480409090470639328');
    t('-16791796.527002378128880210996658740114552744738867694', '115759936860192797305173026', '1.15759936860192797288381229472997621871119789003341259885447255261132306e+26');
    t('-1029316.894', '261548573279697971.910', '261548573278668655.016');
    t('0.0001296647057215427680944560864883468441013790931695', '-1054.824421', '-1054.8242913352942784572319055439135116531558986209068305');
    t('-46699431235320232532.67509849580', '0.000008979993955818623', '-46699431235320232532.675089515806044181377');
    t('-6298558991125571.36653922609195960880571', '-4113889298860378827260223247091017172.9329596', '-4.11388929886037882726652180608214274429949882609195960880571e+36');
    t('-2336086975140170284498.42499016188003935830', '1457086105105141790243380324424465.1603058805177211619', '1.4570861051028057032682401541399667353157186376818036e+33');
    t('13584368794590929.65056685978986415743426966', '1.22576526919', '13584368794590930.87633212897986415743426966');
    t('-1327351.46438658329883133911209577075273465014', '0.000000000005775966498832314718', '-1327351.46438658329305537261326345603473465014');
    t('-516531363720634698623139507801786560806', '-137346717602427886117612615306931.7600975293129786', '-5.165315010673523010510256254144018677377600975293129786e+38');
    t('-497191321049479298.6006597', '-1047994963280.836414', '-497192369044442579.4370737');
    t('-0.0000000000000002185586000243939351096925920847', '-0.0000000000009048521726041772916157', '-9.050707312042016855508096925920847e-13');
    t('6016595778895316729168438939349.8686687192', '5526181676412323789926982786789.74580518160', '1.15427774553076405190954217261396144739008e+31');
    t('3.0095', '-0.0000000000005426618174200484334', '3.0094999999994573381825799515666');
    t('-68807659807110289490038532736.1520217818672205564919342', '-523349129605691396252486008533.3780711514', '-5.921567894128016857425245412695300929332672205564919342e+29');
    t('-113364492557824.61981595', '-4469693.046218916056590', '-113364497027517.66603486605659');
    t('-0.00000000000164215561823273145124136513441258', '-4973023665107453759495929363162.8237739698', '-4.97302366510745375949592936316282377396980164215561823273145124136513441258e+30');
    t('35.9888417583081351968218538', '-14915.870931400193663433176830', '-14879.8820896418855282363549762');
    t('-2331.988493276549717942', '0.00000000000141823854809196715', '-2331.98849327654829970345190803285');
    t('-11371.3496728385913350997078680648168992940913566777190', '-2397016.60966829969942721806170794681095113', '-2408387.959341138290762317769576011627850424091356677719');
    t('3383157742756319045795578039331661679255299628783006818', '112.431', '3.383157742756319045795578039331661679255299628783006930431e+54');
    t('0.1199645670415880693938857655777249', '-44832078259011.6759', '-44832078259011.5559354329584119306061142344222751');
    t('-0.000074515', '0.000000000000822522086975415741166136452344243213662423269', '-0.000074514999177477913024584258833863547655756786337576731');
    t('-316363904412657873.82463253361', '6029091127919181004154328007134773005154138.82694336', '6.02909112791918100415432769077086859249626500231082639e+42');
    t('-2948135076910640.90049', '5761548892564240496281853871829076765874909400', '5.76154889256424049628185387182612863079799875909951e+45');
    t('58372178458636173167435580966904.9', '-0.0000000000000000069124', '5.83721784586361731674355809669048999999999999999930876e+31');
    t('-1030343159494169', '1.17687493992567335710622120899406991559327721710205732290350', '-1030343159494167.8231250600743266428937787910059300844067227828979426770965');
    t('-364369717.74', '1051.2619702659448865', '-364368666.4780297340551135');
    t('-14700708312683508124053613483934', '169203221928.16495122822999668712809968967', '-1.470070831268350812388441026200583504877177000331287190031033e+31');
    t('-7136587497.6210794583359262865362602', '0.000000000000000943850201474525427491095625665832833080048218', '-7136587497.621079458335925342686058725474572508904374334167166919951782');
    t('543052781691689788023396133911013281045', '-0.14820404005794765846900406675812481', '5.4305278169168978802339613391101328104485179595994205234153099593324187519e+38');
    t('110663366765368921.0866334279485908190131223654028', '36120532541712948438741340607422573739675370', '3.61205325417129484387413407180859405050442910866334279485908190131223654028e+43');
    t('380658.3', '-2065165618.375157186233', '-2064784960.075157186233');
    t('-17457782.9819883601', '942920318414698402627635914937488163385896', '9.429203184146984026276359149374881459281130180116399e+41');
    t('8629631914097.8362771910', '-4.034', '8629631914093.802277191');
    t('288071238497.1261824965452941330232', '1052', '288071239549.1261824965452941330232');
    t('78997931984403436704001039530.3024629578317209', '0.000343258751917970970676216110596811', '7.8997931984403436704001039530302806216583638870970676216110596811e+28');
    t('-2.25', '111337103342848345761099521859657477916973094722775.5951', '1.113371033428483457610995218596574779169730947227733451e+50');
    t('-1', '0.00000332919915648989163252787357017400732468280', '-0.9999966708008435101083674721264298259926753172');
    t('-16445332684883.22825', '-0.00000000000000012333460118914012271577551957339747141406', '-16445332684883.22825000000000012333460118914012271577551957339747141406');
    t('-31175222200344385644218736866483.2725089', '-3327088982173967964155665008077925357598', '-3.3270890133491901645000506522966622240812725089e+39');
    t('-1252051655804', '0.00000000264882354935793977626553543541414436396686075411715667541', '-1252051655803.99999999735117645064206022373446456458585563603313924588284332459');
    t('-0.0000000000000000000158865062136901543406179703019550687767953302019945978597', '1020942680079952748611824.185392743482507618487632660693', '1.0209426800799527486118241853927434825076184717461544793098456593820296980449312232046697980054021403e+24');
    t('-35898806424024840518', '1209273562014126837', '-34689532862010713681');
    t('0.00000000000000000063529152935584224393904', '0.000019230208356140680362397434569942324862475539', '0.000019230208356141315653926790412186263902475539');
    t('-33441632499.7607713725472193611665003418', '106413', '-33441526086.7607713725472193611665003418');
    t('-4.75634838524050727239455330', '0.00000222716225977301166476', '-4.75634615807824749938288854');
    t('-852521157106518977717317.959654055992630233016032351392', '-0.0000000000000000017823481878995487452645517241108404', '-8.525211571065189777173179596540559926302347983805392915487452645517241108404e+23');
    t('161134348969753.7818551261464', '14778587228410.68', '175912936198164.4618551261464');
    t('-369981.81', '434675717524696906710428616481.721', '4.34675717524696906710428246499911e+29');
    t('-2332861219206834364527.48368', '19149287947261990184026730628.5871', '1.914928561440077097719236610110342e+28');
    t('-0.00000000000000116369456154760102530923593100791059354951050808242146', '-81782712838440550651059561384159250420604069689985758.44650', '-8.178271283844055065105956138415925042060406968998575844650000000000116369456154760102530923593100791059354951050808242146e+52');
    t('-351.475892554479058', '-447032714556907039546625862961730678.7', '-4.47032714556907039546625862961731030175892554479058e+35');
    t('-0.0000000000005359206060244700158823650', '-0.0000000000000000003863162428565', '-5.35920992340712872382365e-13');
    t('2', '-0.00000000000000005866552459931', '1.99999999999999994133447540069');
    t('639324673786985183757123973071419.69', '0.000000000000000105695777609583429374882466829273', '6.39324673786985183757123973071419690000000000000105695777609583429374882466829273e+32');
    t('4294088930905823047177509660568176369997.858000657', '0.00000000190441051986057354054684652513790', '4.2940889309058230471775096605681763699978580006589044105198605735405468465251379e+39');
    t('81952.93187082317', '-3354935.93728410227615128341536014594398120471', '-3272983.00541327910615128341536014594398120471');
    t('22.4490', '-60024492865596734181918135790584892061552368', '-6.0024492865596734181918135790584892061552345551e+43');
    t('82042481481926776214220048262853946881327742847', '-8538522144679468521.62573001550037', '8.204248148192677621422004825431542473664827432537426998449963e+46');
    t('-4821483203793.49440644526137664', '-0.000004487693742178692907867', '-4821483203793.494410932955118818692907867');
    t('6.110882058707251378846629539352045716', '-418562303949284087382062254279932829.02', '-4.18562303949284087382062254279932822909117941292748621153370460647954284e+35');
    t('-0.00000000000000003499186801952540123648020382', '-65374665927301037435189057407673135706346962', '-6.537466592730103743518905740767313570634696200000000000000003499186801952540123648020382e+43');
    t('-117.491', '-35267005558911708.73667772625498979866', '-35267005558911826.22767772625498979866');
    t('-1305188291195132404665648036283152', '-48817.0270426507883050419161', '-1.3051882911951324046656480363319690270426507883050419161e+33');
    t('-332959.31909282242075209', '-0.031741578746012412923914', '-332959.350834401166764502923914');
    t('-17.1394344634', '-0.00000229726550687811210739718966133', '-17.13943676066550687811210739718966133');
    t('810780860238944784555214657769411196386947.54', '4.92693024082761080', '8.107808602389447845552146577694111963869524669302408276108e+41');
    t('-1048758998827591442543266558.0106480081485758097', '-94.076883827230674850527236', '-1.048758998827591442543266652087531835379250660227236e+27');
    t('704447412485396882279061367382703398302182548530873140734.6', '6.1441561936492973077606028', '7.044474124853968822790613673827033983021825485308731407407441561936492973077606028e+56');
    t('-9', '14240110.34382757891637692908119148', '14240101.34382757891637692908119148');
    t('-2.17399974', '-7.782135494523838665700444005467417478', '-9.956135234523838665700444005467417478');
    t('423939022366100.1303605421315647706922839530', '-148861179543023924.831222210', '-148437240520657824.700861667868435229307716047');
    t('-524884.99509383', '-8951228003529351713733925073.1782', '-8.95122800352935171373444995817329383e+27');
    t('-0.020796655578405', '368497345356.14', '368497345356.119203344421595');
    t('-1963433333827228036507.697737169766323166530902591', '-957846652507696110363', '-2.921279986334924146870697737169766323166530902591e+21');
    t('0.000001311', '-3369850772751642763158244569.8616096', '-3.369850772751642763158244569861608289e+27');
    t('4794.0', '-1192644326919702801393470130877439453668.01186106844617097', '-1.19264432691970280139347013087743944887401186106844617097e+39');
    t('-1021147107953300067786801337030572.294185965004276223', '-39540941781.309992029344807273', '-1.021147107953300067786840877972353604177994349083496e+33');
    t('-7044613941200081350320568560980315.797014086930', '-22.98', '-7.04461394120008135032056856098033877701408693e+33');
    t('0.0000090121286833758732990328348454967225169260742236026', '990483291162514723456237581.12', '9.904832911625147234562375811200090121286833758732990328348454967225169260742236026e+26');
    t('-26.736', '-50947076888389365.5543516223716887', '-50947076888389392.2903516223716887');
    t('0.00000000019379047403352963512763584433797131380295276579883848', '351648601080041724641347981598154974909945367236167357297', '3.5164860108004172464134798159815497490994536723616735729700000000019379047403352963512763584433797131380295276579883848e+56');
    t('-13975.741', '373024094875867055483269157586785.905835293458170355', '3.73024094875867055483269157572810164835293458170355e+32');
    t('-299.92', '-11054263.84681950544', '-11054563.76681950544');
    t('-1406576080.91', '-11.8662', '-1406576092.7762');
    t('0.00000000001874641346', '0.00000000000000486443153897052258296310659171509120488311396238084788', '1.875127789153897052258296310659171509120488311396238084788e-11');
    t('1205.34627565144026871663097', '1249.7837058417060142511952909501288363112688', '2455.1299814931462829678262609501288363112688');
    t('-4.39', '545534', '545529.61');
    t('-2916468.72915302357477562892725843748', '-78412.481653315', '-2994881.21080633857477562892725843748');
    t('1', '1157759807263227857254563934897', '1.157759807263227857254563934898e+30');
    t('-104251055525855379611.30896754', '-16664469.85', '-104251055525872044081.15896754');
    t('0.00000000000261749383970', '-11939719.526854341', '-11939719.5268543409973825061603');
    t('-0.0000000090677528576752114233457187000354303371555029332226226', '-0.0000054807823719971287602588898631186660', '-0.0000054898501248548039716822355818187014303371555029332226226');
    t('-116485999654579416404.1770995047267', '-1056453978415557512986444149851256428790784', '-1.0564539784155575129865606358509110082071881770995047267e+42');
    t('129605008229833515.738549411440577906', '-2966237394191984.299672767366997401377001384652892925560', '126638770835641531.43887664407358050462299861534710707444');
    t('38233691542980267658744125', '22629558718.76465481003636074', '3.823369154298029028830284376465481003636074e+25');
    t('-8533862254008034912764735.004170692', '0.00000000000002245605992555542408030328467190', '-8.5338622540080349127647350041706919999775439400744445759196967153281e+24');
    t('10553.65', '-0.00000049686576247164541003858453804', '10553.64999950313423752835458996141546196');
    t('-51954241699518441712828334574084047.068703826124873318564', '395669535095772787686098943620277851996703327535', '3.95669535095720833444399425178565023662129243487931296173875126681436e+47');
    t('-23122182991889866069296720914272472', '58837368013.02465555264983929119523508427711325303360206', '-2.312218299188986606929666207690445897534444735016070880476491572288674696639794e+34');
    t('0.00000000397559426', '-371081877267739.52921835969493214299461', '-371081877267739.52921835571933788299461');
    t('-1318006475307606890136618572999490756076602355.97', '632863313713664.4745327432714901307248136075', '-1.3180064753076068901366185729988578927628886914954672567285098692751863925e+45');
    t('-23439088778534600.8640688389613841', '16', '-23439088778534584.8640688389613841');
    t('21557692582940554231371458355850911425', '54.498980233719', '2.1557692582940554231371458355850911479498980233719e+37');
    t('-15841465782.23786068092567135289691949618', '-0.06849728241', '-15841465782.30635796333567135289691949618');
    t('0.0000000001081504196289698603171046129405333329174', '-1442109.88437873202181493001814004974080724589', '-1442109.8843787319136645103891701894237026329494666670826');
    t('-0.00000000001647', '282001122341733786209765075874760.2208698094645570', '2.82001122341733786209765075874760220869809448087e+32');
    t('99430939117933248217120.374139300891394898188889827982140', '0.00000000000000029379095872418716620806971498023154656656222336363941', '9.943093911793324821712037413930089139519197984855216930620806971498023154656656222336363941e+22');
    t('32.0', '-27722898.863296087173', '-27722866.863296087173');
    t('156106106814604967305174465335864706671583.70259565626257897', '-13917199525611', '1.5610610681460496730517446532194750714597270259565626257897e+41');
    t('0.000000000000012756394816575404961299', '7517.05', '7517.050000000000012756394816575404961299');
    t('16.12', '-420571324087664540438479.845', '-4.20571324087664540438463725e+23');
    t('2418460022267862452.4', '41.457986607816857109', '2418460022267862493.857986607816857109');
    t('148234016059793326337.640', '11.03882097673491443395332118707526905738860', '148234016059793326348.6788209767349144339533211870752690573886');
    t('590735628836667929357165331457381099840530274992323824', '-35.537', '5.90735628836667929357165331457381099840530274992323788463e+53');
    t('-91.305944234004155285775629491875981263029245580213970209', '-0.10312097106790790600287727466092359571108', '-91.409065205072063191778506766536904858740325580213970209');
    t('351474.4', '-409.9417808528084374954481354713200702998578', '351064.4582191471915625045518645286799297001422');
    t('0.00000000000000000268393845813954', '232389665408735', '232389665408735.00000000000000000268393845813954');
    t('0.000085437842245145501815518395009858127503498', '-228105253425581861', '-228105253425581860.999914562157754854498184481604990141872496502');
    t('113814704131627223061784854080682', '-0.00000000000000495917538613516689674', '1.1381470413162722306178485408068199999999999999504082461386483310326e+32');
    t('-104597143810254066499700', '-1320192161655050.18', '-1.0459714513044622815475018e+23');
    t('-808665613705806138579.9252448473426840', '-0.000000000463', '-808665613705806138579.925244847805684');
    t('-10652723155657612558630867.3337753966347961185835734448179360', '-1074473046919580926107363821097544239456092192199611', '-1.074473046919580926107363831750267395113704750830478333775396634796118583573444817936e+51');
    t('30.873', '-2877.8117541278', '-2846.9387541278');
    t('280550729.708', '15.0', '280550744.708');
    t('-165734300554398023710.52770', '44297520438', '-165734300510100503272.5277');
    t('0.00543767539426789750178885690131013907', '9444317260148.44999438456024566306', '9444317260148.45543205995451356056178885690131013907');
    t('3.906258287', '-101756814758789018743117855884.8664231869', '-1.017568147587890187431178558809601648999e+29');
    t('0.0000000000466441425', '-7925237.129420', '-7925237.1294199999533558575');
    t('-8793636.6549628644427603574935', '-168717.26735969139', '-8962353.9223225558327603574935');
    t('16229113210072', '-1.66', '16229113210070.34');
    t('-274', '5543767652349683046313915.6084123313153941360541851', '5.5437676523496830463136416084123313153941360541851e+24');
    t('697036466897177207083246447.2299025435243117175710148553', '0', '6.970364668971772070832464472299025435243117175710148553e+26');
    t('-46.3370059936', '10533716878.34472089', '10533716832.0077148964');
    t('10374324.55170082277424656551004841876', '27751581009979569031375460467333.939804424', '2.775158100997956903137547084165849150524677424656551004841876e+31');
    t('-26.207', '-0.0000000000000031037896000885366', '-26.2070000000000031037896000885366');
    t('0.00000008010', '18.21', '18.2100000801');
    t('4842596615336.5226989764856168408', '693189245669296426406216', '6.931892456741390230215525226989764856168408e+23');
    t('-0.0000000000000851077176424519554600753331960725209384331282631736', '-0.00000000032044460270651016683689341535385222681285333039475669967745', '-3.2052971042415261879235349068704829933379176352301987327745e-10');
    t('-45733275107439777530880615832524693352357490.41665', '-9002296385923358097661509924413196524814431890711992169', '-9.00229638596909137276894970194407714064695658406434965941665e+54');
    t('-0.00000000000000000156439224707874070263669523081315808081534434429692222172674', '0.00000000000000000004291737953641998', '-1.52147486754232072263669523081315808081534434429692222172674e-18');
    t('0.00000000198897553988595375857564654222391162510361117709636863903', '-10.007927501632868864725076236004253517352', '-10.00792749964389332483912247742860697512808837489638882290363136097');
    t('-0.0000000000000000823558627325', '0.000000118691410206725495', '1.186914101243696322675e-7');
    t('-6276754.7824301277909646649809', '-2300076804510676.99579229048', '-2300076810787431.7782224182709646649809');
    t('-1477245255838368818869339352082881086281616770202', '260.2', '-1.4772452558383688188693393520828810862816167699418e+48');
    t('-69931380572295540885943992404507908544953094482.1746111', '6493037948200.611359', '-6.99313805722955408859439924045079020519151462815632521e+46');
    t('-32658268317284858182413184281435089775970459834815.66170', '-0.000000000000342812261960547876364567574654890642067512409498', '-3.2658268317284858182413184281435089775970459834815661700000000342812261960547876364567574654890642067512409498e+49');
    t('-226963.2043', '-496091', '-723054.2043');
    t('-103912.138149562507', '-62.4', '-103974.538149562507');
    t('-2.2', '-192.817990', '-195.01799');
    t('222123997760197158313997248315192554300081415.469', '0.000000000012667203104882341577765952699206164', '2.22123997760197158313997248315192554300081415469000000012667203104882341577765952699206164e+44');
    t('131249246364141493511095170397038.4', '12005458301968.10861506047414540', '1.312492463641414935231006286990065086150604741454e+32');
    t('-6103920949696.969088220006245342108451032227019740900968', '-8610778484315116167388258277248987747817680', '-8.610778484315116167388258277255091668767376969088220006245342108451032227019740900968e+42');
    t('10233395574.3608814100201268800987627468958287183008442076', '2066089654793590701954', '2.0660896548038240975283608814100201268800987627468958287183008442076e+21');
    t('1054800822950425857387.3975342786210238442171979019733049494', '268210275857404541391650497878689.609927', '2.682102758584593422146009237360770074612786210238442171979019733049494e+32');
    t('0.0000000000000000005358606622145297463680878712323075380428171088961554207868', '21625227877279.763', '21625227877279.7630000000000000005358606622145297463680878712323075380428171088961554207868');
    t('0.0000000000000000038580163605093667407219262159717409196345947134', '26563841772035062352711123348805920.54972', '2.65638417720350623527111233488059205497200000000000038580163605093667407219262159717409196345947134e+34');
    t('0.0040413342647577519898276000725944344977704', '56147277645684.4193402', '56147277645684.4233815342647577519898276000725944344977704');
    t('4364458714.2682', '316321.1973', '4364775035.4655');
    t('-3442146624234450575848597310365404902.6280403978894069', '9600031560472464093658037177768209030975667.931701', '9.6000281183258398592074613291708986655707653036606021105931e+42');
    t('0', '-549407495633380422499474.55571333', '-5.4940749563338042249947455571333e+23');
    t('-37479660228.14714', '-1.75484', '-37479660229.90198');
    t('-26584742.26667643650870662855172761916351306', '-0.00000000000000187865453650020', '-26584742.26667643650870850720626411936351306');
    t('-58.001178998055584357671459355923764911439509141805', '-50935290675172160.97675237172778455', '-50935290675172218.977931369783368907671459355923764911439509141805');
    t('-0.00000000000000000034052335201240933961815921', '0.025619781818507958387831333698180', '0.02561978181850795804730798168577066038184079');
    t('-0.00000000000000008274050377833639593377852710426025817', '242065602.8095741290098057721936971088', '242065602.80957412900980568945319333046360406622147289573974183');
    t('-221524.27663105095567467682134', '-2258391802026441769618755584758918156197711875845897035768', '-2.25839180202644176961875558475891815619771187584589725729227663105095567467682134e+57');
    t('-0.0000000017893705878914010575881803996966689837370969008155370432', '-0.00000297259632962708146112815320511201052652', '-0.0000029743857002149728621857413855117071955037370969008155370432');
    t('-0.00000000000406710488085149344523', '-278522.037156181204952078706297015294979477127184963777538', '-278522.037156181209019183587148508740209477127184963777538');
    t('11973889508.580802', '1871682273132124693', '1871682285106014201.580802');
    t('0.0000000000000000033353113686655236115870191745605104608147650', '11625246.252195', '11625246.252195000000000003335311368665523611587019174560510460814765');
    t('-5318806059561785713953', '-0.0000000000000012146467404768921900862164865027461683067265853186116641', '-5.3188060595617857139530000000000000012146467404768921900862164865027461683067265853186116641e+21');
    t('-309.3', '-0.00000000000006218919975652358743864716', '-309.30000000000006218919975652358743864716');
    t('-1874575484506897571974850167460839993082172311.68616497703', '-0.0000000942559460072629437399413345982786975068449962297', '-1.8745754845068975719748501674608399930821723116861650712859460072629437399413345982786975068449962297e+45');
    t('-21', '-0.0000000000000397635708714407055658459', '-21.0000000000000397635708714407055658459');
    t('-1302141913979670987068496048', '-0.0000002335314231823837373064', '-1.3021419139796709870684960480000002335314231823837373064e+27');
    t('-8663116.663729764613058828364567651431', '609.8', '-8662506.863729764613058828364567651431');
    t('122805001870129.621993', '546.98190600859999', '122805001870676.60389900859999');
    t('132753213.74892150691420521615', '-2356523694.1', '-2223770480.35107849308579478385');
    t('-255163.150', '-22815991158761.86650580', '-22815991413925.0165058');
    t('0.00000000000000009834358', '27695886984688431669.7', '27695886984688431669.70000000000000009834358');
    t('0.000000000018235737891993707047558745084', '-0.0000027294384492006549909198104817456693956836003182', '-0.0000027294202134627629972127629230005853956836003182');
    t('-13126.0', '0.0000616967410806266771382642088892371968', '-13125.9999383032589193733228617357911107628032');
    t('720874433286547149.29011960', '-0.0000000000175150100264717579452770317006888697250136125181729', '720874433286547149.2901195999824849899735282420547229682993111302749863874818271');
    t('2905655849177158035847281414357098453.291834904651188', '-3', '2.905655849177158035847281414357098450291834904651188e+36');
    t('0', '-1475966921783333685644104279864374261258954134383883947232', '-1.475966921783333685644104279864374261258954134383883947232e+57');
    t('472816652216998927468244178551354910167729088.966726827004415', '177527765826935.24974826329344', '4.72816652216998927468244178551532437933556024216475090297855e+44');

    t('-274.5620579869', '78329265787241405566496082713913236369654243529457197134761863524633387852302663270044189362081880226597475301.025458805177854094773', '7.8329265787241405566496082713913236369654243529457197134761863524633387852302663270044189362081880226597475026463400818277854094773e+109');
    t('133883508068780764759485.5683529446087107456184862039990020823869300840561329290914475177233060368335', '-14.11123105188787211647630936581927502853383341149933197', '1.338835080687807647594714571218927208386291421768381797270538530966725568009590914475177233060368335e+23');
    t('-3199032706015694214818.7992938105687331661', '-7.87195', '-3.1990327060156942148266712438105687331661e+21');
    t('0.0000000000002295193152912890007348152875969581183350688584884783586480814376317641609774249', '11968142585488462774385062859935899134826448280897530108166161884029199683990896601020507477.320319', '1.19681425854884627743850628599358991348264482808975301081661618840291996839908966010205074773203190000002295193152912890007348152875969581183350688584884783586480814376317641609774249e+91');
    t('429850894793934827539600845168772.8751965913520453918272995739036873865390677578368381581396285338126446790691976041147571854', '-0.00000022801170604582813255353545523124229818428689101495876171494691321154484217536764378725027141584278379447842816929409', '4.2985089479393482753960084516877287519636334033934599916702036823215529676957354994714318086681886573146752435542874711339814972858415721620552157183070591e+32');
    t('0.12630479339914084342606977043625120835780195245007714525361728866248064543254204354136033894', '-57300622033341549450152055498951631223725420435.499102954248138500376888848777133938393663605560964812373961734244028047764768563', '-5.730062203334154945015205549895163122372542043537279816084899765695081907834088273003586165311088766712034444558154740233222651945863966106e+46');
    t('-0.000000000106720353331827', '4068913567129614779805806799030791537958519754660357531521550859320707940360600331986.6231458520200147185933974407694787756642025356767729344', '4.0689135671296147798058067990307915379585197546603575315215508593207079403606003319866231458519132943652615704407694787756642025356767729344e+84');
    t('1481.538992', '69979426854562140198395309', '6.9979426854562140198396790538992e+25');
    t('-481304712134904590633063526049619366756244902183946511856972789126658742068757286270722005910044731533367098.3473844315585478022917336214585042983934', '-9035651250061107335979601717303646.90745896455428', '-4.813047121349045906330635260496193667562449021839465118569727891266587420777929375207831132460243332506707452548433961128278022917336214585042983934e+107');
    t('5998474356563587104327566.3263164708068830115821012882772814947553985814910059469343679580428581460206', '-18057674949163421506117960386922177149977514555439908.159223871467357039682517588208442666597309817931705956866923', '-1.80576749491634215061179603809237027934139274511123418329074006604740281004162999311611718419112364407000099325550419571418539794e+52');
    t('641786663107582574187375964645432626362556087596248087198360354.064673626930039', '106200869851859350.797479350269434371025574271015473262363560762378677', '6.41786663107582574187375964645432626362556087702448957050219704862152977199473371025574271015473262363560762378677e+62');
    t('5280129522068699720461117125312358415979019162822.4201114673061632909516268898605984', '-205756343429440469562776948821275114561497830505910222682069.31376408871563202774509593696096457576461665289977493017361973671523262', '-2.0575634342416034004070824910081399743618547208993120351924689365262140946873679346904710036617576461665289977493017361973671523262e+59');
    t('-0.0000913314112194237567199921654806413644776347028968332389534560276307374133431440995704693847453686', '1.1', '1.0999086685887805762432800078345193586355223652971031667610465439723692625866568559004295306152546314');
    t('-2207121715490045.58', '-3295992738515109953967433108145888727360416859361202947062658620012196801551806622678192.686847483520554027598236734312161', '-3.295992738515109953967433108145888727360416859361202947062658620012196803758928338168238266847483520554027598236734312161e+87');
    t('61759391974251637508280834914990322126219035347147', '64943964591891233846314007885335.444346971691325618829', '6.1759391974251637573224799506881555972533043232482444346971691325618829e+49');
    t('-5540305476191589915887668399303961108947288288224548320', '0.0000294476901910227000564933914711035233697829465135452464583418791303186801762813', '-5.5403054761915899158876683993039611089472882882245483199999705523098089772999435066085288964766302170534864547535416581208696813198237187e+54');
    t('181474807531688120166683732972009536.64969213', '-3175669398983347716917468673419554.2177188765670101302869684383001638592478444069723506434104220110726821325272', '1.782991381327047724497662642985899824319732534329898697130315616998361407521555930276493565895779889273178674728e+35');
    t('-12493662650.119745851863596480945654415210619543132', '399561145206467750513891779469494975331120274361101613876792177943730662310953547064702574.65756653452860298452313916684555', '3.99561145206467750513891779469494975331120274361101613876792177943730662310953534571039924537820682665006503577484751634930456868e+89');
    t('-2166.664029173675712500257', '-0.00000000000067461152096758750665879707326349452924143528959808406172771844392251655077273026039982145730838447563291749346650348630096104418925967461686234281319', '-2166.66402917367638711177796758750665879707326349452924143528959808406172771844392251655077273026039982145730838447563291749346650348630096104418925967461686234281319');
    t('-55126691014038404667432234769742600667346582750812.76081290000540326323203936990298784699977785727177293701354598905047747033', '230955903911091180768255072886786497585594803905488997.065428', '2.3090077722007714236358764065201675498492745732273818430461509999459673676796063009701215300022214272822706298645401094952252967e+53');
    t('-12239899294934406744794741209740399921097535320596.080402725637531427417', '0.000000000000007975735707900012257888169944647551205385287889146629466680800976795', '-1.2239899294934406744794741209740399921097535320596080402725637523451681292099987742111830055352448794614712110853370533319199023205e+49');
    t('-0.00000000000000912462079748634949064954444536474698633137152955660092121181721134185040726847080723736415320180821995', '-28192491050430', '-28192491050430.00000000000000912462079748634949064954444536474698633137152955660092121181721134185040726847080723736415320180821995');
    t('5365712579646207337626428460702379954655063519', '-171091164429397358988968907423411173590072414448497446341010313844726796866187751610819558166794791576734732137347669225657216896755433103171081237', '-1.71091164429397358988968907423411173590072414448497446341010313844726796866187751610819558166794791571369019557701461888030788436053053148516017718e+146');
    t('-1981290034036713612.3587708893531360121978707494237', '438761731488571.030446306123818027751940714135233838', '-1980851272305225041.328324583229317984445930035288466162');
    t('-2755875412874743140386728151346', '2.8472810', '-2.755875412874743140386728151343152719e+30');
    t('-157.549092251345721782955172995326812019919432214305439543', '142439127771945282929407965570725127160359050662864293026220308331445962610.79480267325422', '1.42439127771945282929407965570725127160359050662864293026220308331445962453245710421908498217044827004673187980080567785694560457e+74');
    t('0.3491333927087766478511955867195720082891559879520214956234991486163801992970042826730252168795279033184196910108870648', '-756356333951812709085600894164769015473490895.271157080699', '-7.563563339518127090856008941647690154734908949220236879902233521488044132804279917108440120479785043765008513836198007029957173269747831204720966815803089891129352e+44');
    t('63261615358197872923073425992.67414929587646886049878412684052256635783684', '32232093735028677679866848653553607775797749686636316132824223211080228103818061093254438705652919', '3.223209373502867767986684865355360777579774968663631613282422321108029136543341929112736177907891167414929587646886049878412684052256635783684e+97');
    t('-0.0000000000000009295380014651692075387777504579033936499297749005685897989061480633040541931247624617087077690289771853068486993070710805656', '-74240982721590027061065744723039918113222511184444001668610337943', '-7.42409827215900270610657447230399181132225111844440016686103379430000000000000009295380014651692075387777504579033936499297749005685897989061480633040541931247624617087077690289771853068486993070710805656e+64');
    t('-38662088769617027684868436557208874287.531787', '-0.00000000000000000004404702485696046546186749454135837739919620970432790136280242735880480845505606335438999503830238610', '-3.86620887696170276848684365572088742875317870000000000000440470248569604654618674945413583773991962097043279013628024273588048084550560633543899950383023861e+37');
    t('0.0000005919093535654695188705799210533565699038', '-1050230870542550895325305037586388557344285623573695378599872476398998237594436121662.36921877939195132369', '-1.0502308705425508953253050375863885573442856235736953785998724763989982375944361216623692181874825977582204811294200789466434300962e+84');
    t('8531356548289.68', '0.0000000000010663385403102579270265567147867424805920', '8531356548289.680000000001066338540310257927026556714786742480592');
    t('-77352959873895499881.6761162233029992762254367454845614002262158269218399300505823906248006070788666306', '-0.000000763350249168968817759292626920653670602769749623937342230173000', '-77352959873895499881.6761169866532484451942545047771883208798864296915895539879246207978006070788666306');
    t('0.04585173770263682911835172946483907318715956718860423056098335799507649567731835', '-545336428117302.06514791235190', '-545336428117302.01929617464926317088164827053516092681284043281139576943901664200492350432268165');
    t('-2000777096546022469363680806530511943528746650101273997534.242598610793331542113919999385909', '-0.000000012960554065829921984561617848403788685435712387766345566655415445798299917451084000', '-2.000777096546022469363680806530511943528746650101273997534242598623753885607943841983947526848403788685435712387766345566655415445798299917451084e+57');
    t('-4637067617611402422338031964698807990446537844332708147280649406995000008716', '-3099749512371613407546572880261013353834801183114544503924317759.7679498142950006458622486041325542794064397790374869775038357487455568411693227469', '-4.6370676176145021718504035781063545633267988576865429484637639514989243264757679498142950006458622486041325542794064397790374869775038357487455568411693227469e+75');
    t('270678062855542005084307870824127678467656.762955855629442335926610514941014722825992247041', '282974792.523705817808', '2.70678062855542005084307870824127961442449286661673437442335926610514941014722825992247041e+41');
    t('-0.00000003225236579904856895487056015821433730938371824218643897684349658964835496773420825844721647906790768973764870027244455', '-0.004924810165331790880550482628308491946560589374047905819960040503118511618345', '-0.00492484241769758992911943749886865016089789875776614800639901734661510126669996773420825844721647906790768973764870027244455');
    t('-2873277.7031566090883', '369526947446567818357.08166567158618885089868282383289', '369526947446564945079.37850906249788885089868282383289');
    t('-2178986538025.6281171045', '-0.90007456268766834245637565755250114155754042861579405494550640641064108080486501982231697766610369506873393983', '-2178986538026.52819166718766834245637565755250114155754042861579405494550640641064108080486501982231697766610369506873393983');
    t('-12610827849005379185996946898193295000242048930.3057', '-50566291142615310315.413773094', '-1.2610827849005379185996946948759586142857359245719473094e+46');
    t('-0.72263391095435659584498561750405291423', '1.6716', '0.94896608904564340415501438249594708577');
    t('-0.00002308861674194925960442185653345749705673163043872636646918599512354359169533298098626366738286911364737665763586940744987791360736201819133314717054', '450814153774.5', '450814153774.49997691138325805074039557814346654250294326836956127363353081400487645640830466701901373633261713088635262334236413059255012208639263798180866685282946');
    t('-0.000000000000042613856152865288514141630996935837135169348614328862241783421937354219792185365968886244399960104759290619174907', '0.0000000000000000974480182403539706738405642564651574120855033994898433979613263953303312911354243605961011258564221775753219595811881303591336456508263682734', '-4.25164081346249345434677904326793719777572631109293723983854606109588894608942305445256482988342483371130438529474188118696408663543491736317266e-14');
    t('134411245152521998394744941365564060927709793292932748805716150.96086322753', '163717756163587942012853208387334527056988170322687172258242323330600618595724408258555875633809739823926575887736071927568012683', '1.6371775616358794201285320838733452705698817032268717225824232333073502984087693025695062057517530388485428568102900467637372883396086322753e+128');
    t('-1135858630959255507879446', '0.0000008656110212823553443864329542508631', '-1.1358586309592555078794459999991343889787176446556135670457491369e+24');
    t('-0.0000000003175675944478684925401856872596765776036876508094511873354931239253559389081387285155438349993170191128163568912691363523242026430941295514496034087951', '13806374012994737257255.1886211194405', '1.38063740129947372572551886211191229324055521315074598143127403234223963123491905488126645068760746440610918612714844561650006829808871836431087308636476757973569058704485503965912049e+22');
    t('-27708159800573272.0883231101900212685541939067996078907774725322218231069648770193298081146113593454964992974439864811', '1069672.96681867083', '-27708159799503599.1215044393600212685541939067996078907774725322218231069648770193298081146113593454964992974439864811');
    t('0.00000429880802966674008131137959417215280776628630203493369987886551523668068778978', '0.00000000077008967289490676642801409841947753617266544157841315779815188796831111777973779955219440658094362925853693153209608918067', '0.00000429957811933963498807780760827057228530245896747651211303666366712464899890755973779955219440658094362925853693153209608918067');
    t('0.0000000000000000000299565', '-73131153293465264.481', '-73131153293465264.4809999999999999999700435');
    t('133297778031366687408810593803463309652201134859997.4758048870790461770038771730220839594432975561216897245860371442353803167578892021729425784575887741', '2364940340353511166077790398873799899.851022395047720', '1.332977780313690523491509473146293874426000086598973268272821267661770038771730220839594432975561216897245860371442353803167578892021729425784575887741e+50');
    t('233341303466412748783092751385962706054310564979814063054720210979264.76320976947', '373546701810748884756.35246337986916104538633936243091901195', '2.3334130346641274878309275138596270605431056498018760975653095986402111567314933916104538633936243091901195e+68');
    t('-0.00000000000001298143460683533187542098532057471443650131347027927061505315402201377004104838055385241660827862931318282364149013913156553270415', '-3144.73816785889762413352612917538951639565049692192872179639873257494799763607259781490677831922245389370215618142714372657806598580728875889749', '-3144.73816785889763711496073601072139181663581749664315829771220285421861268922661982867681936760300774611876446005645690940170747594642032443019415');
    t('83823188981575692224309101153864758580078.5955259490170379255716122945113031579157318931038231', '-40028920687917.16848969', '8.38231889815756922243091011138358378921614270362590170379255716122945113031579157318931038231e+40');
    t('2808691776034467055726891904371173436595982589447978253042154698627671398488784', '9812791602520619.597754130838314809582665890845794489994575586', '2.808691776034467055726891904371173436595982589447978253042154708440463001009403597754130838314809582665890845794489994575586e+78');
    t('348504897735551663451493471155931032198603500912341395392663329685030763642073580083352277207617381115118735257611119505096273334530366023', '3012062864730334700837949872371998966828679.4047646544156150438268952921094746263', '3.485048977355516634514934711559310321986035009123413953926633296850307636420735800833522772076203931779834655923119574549686453334971947024047646544156150438268952921094746263e+137');
    t('93548859880851212916468686274573175909', '-75118428101074788562.506120603990328548341139932377168056', '9.3548859880851212841350258173498387346493879396009671451658860067622831944e+37');
    t('0.000000000000000000194012323991964077673853360201053367800566124970905963785098877971072180640313745271154543409391042526877924567117182739810', '0.0000000236189282059323667779215335', '2.361892820612637910191349757767385336020105336780056612497090596378509887797107218064031374527115454340939104252687792456711718273981e-8');
    t('-389202042550335503825154365205711290405537538112817615439726097659782032124231125784313789925', '0.00000003186474240443828135635273567868516420016402055460073651185', '-3.8920204255033550382515436520571129040553753811281761543972609765978203212423112578431378992499999996813525759556171864364726432131483579983597944539926348815e+92');
    t('175494756398169060188389869312520635210512554543656594330140', '-411530581933818049374887440657899352512027.09153247186642746768921170931268854307755443936717', '1.7549475639816905977685928737870258583562511388575724181811290846752813357253231078829068731145692244556063283e+59');
    t('4598950112574467183729719937465962791614617983103965921055089220439607671.599007540382070086491128969863554413197224811888258922835', '-26938781266346831074998405687771608325810543240305.7378923235384660560482541777264792878816806719888726231', '4.598950112574467183729692998684696444783542984698278149446763409896367365861115216843604030442874792137075125315544139899386299735e+72');
    t('57361809317883677060860671734348051270932088942529602386581.68900107141758948907', '-295802867132191509532524058', '5.736180931788367706086067173434775546806495675102006986252368900107141758948907e+58');
    t('-0.0000125601146369033513236980781240144461675622310937097773', '311729685945211093658295182', '3.117296859452110936582951819999874398853630966486763019218759855538324377689062902227e+26');
    t('-0.000000000000017031301308338958768641001044966021634165409838760713', '7354226011798172001332233249137409933327535976109274125337', '7.354226011798172001332233249137409933327535976109274125336999999999999982968698691661041231358998955033978365834590161239287e+57');
    t('-3.6', '-2815836861296137045153782405432811863217898809.59007', '-2.81583686129613704515378240543281186321789881319007e+45');
    t('1883757916919577797545040700765718683892962925168476980916591444220044498023865115.414486424', '-20632263651557034535417971371523039152817223695099485637652293509653426916474374628814088965383.41423390427269280942433875156452181764219', '-2.063226365155515077750105179372549411211645797641559267472712503267251032503015458431606510026799974748027269280942433875156452181764219e+94');
    t('0.0000000098475053710376389804550097928552217095921881011378913867422148039984790157547999437804576731605639912393200622917271102003', '-53035373614.717805924235005616321273', '-53035373614.7178059143875002452836340195449902071447782904078118988621086132577851960015209842452000562195423268394360087606799377082728897997');
    t('-0.0000811390350949663620350120726675225941477085450778', '-655.20852663603498591718010139782212', '-655.2086077750700808835421364098947875225941477085450778');
    t('2426894083677867432345228015404', '-101468490319724506034682885345783370059542917.575868248657413589', '-1.01468490319722079140599207478351024831527513575868248657413589e+44');
    t('0.30280', '0.000000000000000000254561077129670600967185643', '0.302800000000000000254561077129670600967185643');
    t('2460767093793617543980763007583218463976172951977824518682800860159902541914492786569.988063268009300275066475501733486973382818662', '33247165254498490053797.4453', '2.460767093793617543980763007583218463976172951977824518682800893407067796412982840367433363268009300275066475501733486973382818662e+84');
    t('0.00000000000297623885327264628872813944566975202437690710653591748231030925179641495339618475769115114527914606543143194147848430495597', '120677689870942459137299176289089761143754326820862009921425274349213001568104474801947554427316998849125038885919', '1.2067768987094245913729917628908976114375432682086200992142527434921300156810447480194755442731699884912503888591900000000000297623885327264628872813944566975202437690710653591748231030925179641495339618475769115114527914606543143194147848430495597e+113');
    t('0.00000000000000140253681305432813286110569633536186290552984308384603424304253311258844823992792716702555859199219343153509752', '0.00000000000680532732610522954462416342609639850515130341357440142880229', '6.80672986291828387275702453179273386701420894341748527483653304253311258844823992792716702555859199219343153509752e-12');
    t('-62741750762018440848873834757672851831867995632459744339895230073473.21603443732', '-874741125356670695571041484420725542382859690979070399875164371412.46', '-6.361649188737511154444487624209357737425085532343881473977039444488567603443732e+67');
    t('0.00000000000021176414395474510305287866523610550765676408675802', '256510633937671286139867882266916007823374156029717017148', '2.5651063393767128613986788226691600782337415602971701714800000000000021176414395474510305287866523610550765676408675802e+56');
    t('0.0000000000000000664215993897397955795237104020638588228731275806541568161715250236709545538098694857746988330740', '0.0000000000000072103606167686672117631228552986024686236649558161717746101379938704517818104841311722678845013668413146919184762655', '7.2767822161584070073426465657006663274465380833968259314263095188941227363642940006580425833344408413146919184762655e-15');
    t('-7562874182.83873823754546340019820202614555774800566970410202356651856', '-391433409037814397456500851811078785870089119160317598515862901999756095733578802714504574456.70518495722', '-3.9143340903781439745650085181107878587008911916031759851586290199975609573357880272206744863954392319476546340019820202614555774800566970410202356651856e+92');
    t('-0.0310379152847371895860555796583283692667844392505377548276647230042445', '0.00000082283513482603951054300917006814118588679056861856581955986306659552380016853864396371024709306483760781400794588331', '-0.03103709244960236354654503664915830112559855245996913626184516314117790447619983146135603628975290693516239218599205411669');
    t('87994958901421.56173641605454399949453740849330489135088167019977159972118641759021121261627327050792415022189885286236293442163107348031602461', '-100300365307636143430499359339922108619124158914798.44948', '-1.0030036530763614343049935933992210853112920001337688774358394545600050546259150669510864911832980022840027881358240978878738372672949207584977810114713763706557836892651968397539e+50');
    t('-1550007004.74515351306585502801', '57296100584443643484072467509091114834539973592176211806934477387549446567272546346798', '5.729610058444364348407246750909111483453997359217621180693447738754944656727099633979325484648693414497199e+85');
    t('19679349947742.994127711806874970698304897', '-8.6373642811', '19679349947734.356763430706874970698304897');
    t('54030292.9001', '0', '54030292.9001');
    t('651054592756506304745571531773834326.83644', '-13757776534587654.0116063063442543177687640729791412475434103606212206897637619312584893291977232262578', '6.510545927565063047318137552392466728248336936557456822312359270208587524565896393787793102362380687415106708022767737422e+35');
    t('-508728777596140418455412767451446817854358041268106849523.35989817998722029624762450551681595146802924988664', '-50262054005503525686132151227512844312491027044172912.40', '-5.0877903965014592198109889960267433069867053229515102243575989817998722029624762450551681595146802924988664e+56');
    t('-28915075134892302952316258686216304990690277355', '12.6179700286426790566', '-2.89150751348923029523162586862163049906902773423820299713573209434e+46');
    t('-0.000000000000000204506794915221132784856197218478429446796657919285394521228788556955524', '-86151293785140264277.3878186179195', '-86151293785140264277.387818617919500204506794915221132784856197218478429446796657919285394521228788556955524');
    t('42235147499429175944266679795088142906175521777163730652253755746712473163633468251357976', '-0.0023908655154', '4.22351474994291759442666797950881429061755217771637306522537557467124731636334682513579759976091344846e+88');
    t('591589.86', '0.00000000000000000001860329292788516991407643664465772674133322798829000748004481876019355253701810078529570613237182776549094598706836269877507206606048', '591589.86000000000000000001860329292788516991407643664465772674133322798829000748004481876019355253701810078529570613237182776549094598706836269877507206606048');
    t('-350651107084655269140529689112278274003508608230449884077441.95115833128262389', '0.000000000001117808991674427079320983053134086985164', '-3.50651107084655269140529689112278274003508608230449884077441951158331281506081008325572920679016946865913014836e+59');
    t('-39134659750772.877480206401712009933563545303653', '166958406518764640192354635927301907690317389127470622845773787156043063093969532575030322149816302478163328082727649181925765.252883933714', '1.66958406518764640192354635927301907690317389127470622845773787156043063093969532575030322149816302478163328082688514522174992375403727312287990066436454696347e+125');
    t('-172056481099057625655452551706739628179114255612273491148921284864736703885146667722.8388377144520839161622951529099672906561615777554908439149426', '-53188016746661398958851588477178244943851165029295827.73281273540583347372612339273431422', '-1.720564810990576256554525517067928161958609170112323427373984631096805550501759635505716504498579173898884185456442815106561615777554908439149426e+83');
    t('-101.2715534026922239603773241958969231368570306133332758128133966689708114732', '-0.0000000000069524253857033293796239251561117850631416003514681200491262005294123143215554850258908298728505228195487713881587714484422961615066116684637', '-101.2715534026991763857630275252765470620131423983964174131648647890199376737294123143215554850258908298728505228195487713881587714484422961615066116684637');
    t('-25333165941026', '-6.96905', '-25333165941032.96905');
    t('-23159930599967762034619306845640878338289007418537140350162347387358954515654697539296', '-0.0000000000025055077873630358869224326917384442359589577625318652368309033754969806200104209411667012418713252227441129227486821508788141961', '-2.31599305999677620346193068456408783382890074185371403501623473873589545156546975392960000000000025055077873630358869224326917384442359589577625318652368309033754969806200104209411667012418713252227441129227486821508788141961e+85');
    t('-272690054326355360717463915920945.79117473701250126513511144304624422023780335575219366', '-0.04646617289953036368927399160045218571026521306334789242996846430687659493840681112221758307480649419116688540343801556', '-2.7269005432635536071746391592094583764090991203162882438543464669640594806856881554155242996846430687659493840681112221758307480649419116688540343801556e+32');
    t('-0.000000000000000000824794592791708470343498439383184471216678775736929136783274734656085', '-15464590993770169004834442995125190019352837990780198498620045480008050531227509554782670484128441573926502719591921520634735942184', '-1.5464590993770169004834442995125190019352837990780198498620045480008050531227509554782670484128441573926502719591921520634735942184000000000000000000824794592791708470343498439383184471216678775736929136783274734656085e+130');
    t('-0.000025886438630912380429166846355627957951159888724105900108725398458013138566376225646226120608061663021', '-0.00000000000000000922559195021433452347561162265923135770528238859359860082308634808051034873284365047902141945318959511946199226', '-0.00002588643863092160602111706069015143356278254795546360539111399205661396165272430615657485345171214204241945318959511946199226');
    t('4208526064828419952094154623926319503361267122786727994718336255550699863529999970816217779807244768234375837054150463771544173571075249738067912741', '0.00000000000098970610', '4.2085260648284199520941546239263195033612671227867279947183362555506998635299999708162177798072447682343758370541504637715441735710752497380679127410000000000009897061e+147');
    t('782154833767130832317687050316770373560511399580384993381585732162357039470401244281856195.066130334377022511400443485932219487910616611626636101', '-0.00074331374743997058610568400177871943926460185889915970549597536944046651458673936138785020737671331878048877509692365190559231823', '7.8215483376713083231768705031677037356051139958038499338158573216235703947040124428185619506538702062958254081433780193044076847135200976773694129450402463055953348541326063861214979262328668121951122490307634809440768177e+89');
    t('42247383.064', '-0.0937875210571846395102908228397490109201012337594873628817471460764217967528951691863690241707606495782508382461321094202624928680498275', '42247382.9702124789428153604897091771602509890798987662405126371182528539235782032471048308136309758292393504217491617538678905797375071319501725');
    t('2576315957194062278091487692155019635858438293549959035082229215955048823715650588390178638212566542.511871428870771408965933726237972177952115', '14221275732102839762375835558162468599546489504738010046753467831312431.748866797310416772779810575075035266788827703', '2.576315957194062278091487692169240911590541133312334870640391684554595313220388598436932106043878974260738226181188181745744301313007444740942703e+99');
    t('-182341620.75963819', '0.000998366694277196028537550780159837727537078256289848938775652509142720967559421265698965121960614893238144882990784373782561165842054284561355', '-182341620.758639823305722803971462449219840162272462921743710151061224347490857279032440578734301034878039385106761855117009215626217438834157945715438645');
    t('842047793825512583448880280167368274851233502546714872003645854281826035260762377793954087704796997.25746974352168013713893221905548161407096643850', '-879630826.1', '8.420477938255125834488802801673682748512335025467148720036458542818260352607623777939540868251661711574697435216801371389322190554816140709664385e+98');
    t('-554969229244174864065407853658438.853', '0.000000000000000046413279985045478767808596461145151129534715272056', '-5.54969229244174864065407853658438852999999999999953586720014954521232191403538854848870465284727944e+32');
    t('0.0000004594903425141532404871962049647498137', '-932661375466478095759388985035813536638008334074283723919735.05355344249627897324401366029784201920498102254155227173893629037627164808010183', '-9.3266137546647809575938898503581353663800833407428372391973505355298300593645909077317310163705445516732254155227173893629037627164808010183e+59');
    t('55622408985624324409029884623142142230.3564337866286742532877692296727906360723088525106961633413366259718859274228008870960807098', '15964242335991562370631446343520536873365067961244.3048166767601763969347103883920143084337317316579770484121318738881953338102', '1.59642423360471847796170706679295667579882101034746612504633888506502224796180648049445060405841686732117534684998600812612330008870960807098e+49');
    t('4116.6343', '-13125881395295867999756142722598662945291700853448430819633768649050051199293316003551472339311180106761185979863040509.660957581', '-1.3125881395295867999756142722598662945291700853448430819633768649050051199293316003551472339311180106761185979863036393026657581e+118');
    t('-185.999886254076738335134043252415796249753374218519734071455', '-0.000000000000000001138037685382195961414405158218078543316186398616143756105152301597533', '-185.999886254076738336272080937797992211167779376737812614771186398616143756105152301597533');
    t('-198.9786279245', '1746794213.948368961266', '1746794014.969741036766');
    t('15874309893574389.8', '0', '15874309893574389.8');
    t('16588576987819.81660754169192038181188060714025162116858077319124964265625043196845919043822440466', '-89246600501318562304915309660228383388276620845488564802.496235049288379013778246594577495893634435949828063691098972563490365127', '-8.924660050131856230491530966022838338827660425691157698267962750759645863196636598743724427246585517663681404844272213152190593656177559534e+55');
    t('-127347284441930400932829141510983278557342554346089741297398296382897208587382099.5435745742690549551823615645727067', '-23318951044403467148495997370.7883388335', '-1.273472844419304009328291415109832785573425543460897646163493407863643570833794703319134077690549551823615645727067e+80');
    t('-5077543273110250868221405268190259167.97565436390533404678643216521675105958934642804771', '26014332970671415773600104084765496839411707080753222580030511682112031082495461045172183069324313539709881831959', '2.601433297067141577360010408476549683941170708075322258003051168211203108249038350189907281845609213444169157279102434563609466595321356783478324894041065357195229e+112');
    t('0.00000000725881132557751716179805811333894302315168261762649561806817469605831550855169620175635494174620728378066598111147687734973630755961025899', '434515885287.45022424035155504385780922882248468547935186640429869578572525153830676791908425738454517466', '434515885287.45022424761036636943532639062054279881829488955598131341222086960648146397739976593624137641635494174620728378066598111147687734973630755961025899');
    t('-0.000000000000005190715334165879366906953397614175481033368428717937549118481651827304967814441188123242137007460', '-25.81020207', '-25.81020207000000519071533416587936690695339761417548103336842871793754911848165182730496781444118812324213700746');
    t('-5250099.76439982', '1', '-5250098.76439982');
    t('1196038410000291625428567.6600927087933557619055721061645233907526288367', '274708477195241356599569', '1.4707468871955329820281366600927087933557619055721061645233907526288367e+24');
    t('1193283202443516969171199.42749207631310978334573154570736931304683327423627434514888492883630124033894042566872', '-2705842436304954369628829105664616526636750473333202873059187377027354726150010013782785345680025581966172299828118.857377088370514', '-2.70584243630495436962882910566461652663675047333320287305918737702735472615001001378278534448674237952265533065691942988501205740421665426845429263068695316672576372565485111507116369875966105957433128e+114');
    t('-100501387373.2156506022473571945511270816570649486799902805849591635129761139096179741555986743162675121226826359781143695356560', '-617089847302145796251638901404458614481659.476726016550865549', '-6.17089847302145796251638901404559115869032692376618798222743551127081657064948679990280584959163512976113909617974155598674316267512122682635978114369535656e+41');
    t('16333090935253715.586262', '-0.99663564938317815363068300949250859179309525545225', '16333090935253714.58962635061682184636931699050749140820690474454775');
    t('321070028467230447383995878020953.3888568546912021134358772343603579280658629722635803579108', '28832623126702540650471917064799132201501970153523973751436115142706664061743228360740052.89377744216939547659256989919800076278264836074', '2.88326231267025406504719170647991322015019701535239737517571851711738945091272242387610062826342968605975900284471335583586908485113330035803579108e+88');
    t('-0.00000000000000000003787025783187787162540090434491007131260214791563020212712224638154647089519712810087585510020487818551239718654495070616103779447174795619', '5152327951088416826370.49800740890621823749333954384917083961142478820631626896455924008645842950035037082504155566540807507', '5.15232795108841682637049800740890621823745546928601729296798602388386140619765195709217082822737322812444349508477021094696912414489979512181448760281345504929383896220552825204381e+21');
    t('-3781578965051244011802834529658700587745611157492889345003479370244990285997045726143735238169075073132478791.794030078', '-398406181131828029423189359168774589795627542.69216955808205240294903823537210247931306802', '-3.78157896505124401180283452965870058774561115749288934500347937064339646712887375556692459733784966292810633448619963608205240294903823537210247931306802e+108');
    t('-857281769723716632426451761545805487782866662040001135988125619524781917496404046.907015785954', '-0.00000000000073616976036884849929013414437985061753356336526778613936461244959197744911962137302874551', '-8.5728176972371663242645176154580548778286666204000113598812561952478191749640404690701578595473616976036884849929013414437985061753356336526778613936461244959197744911962137302874551e+80');
    t('3089627721247383947360751941786039174000627343590658682240341607417650', '-5343911762378383789737796312183683590895032635513074691383250730453', '3.084283809485005563571014145473855490409732310955145607548958356687197e+69');
    t('178191.66604428931197081347222738459872958', '-0.00000000457431668529', '178191.66604428473765412818222738459872958');
    t('-0.00000059402965420877250077049277056387386461138162956320009201234837822718242249019319520691853', '-85966889247898245.45722015', '-85966889247898245.45722074402965420877250077049277056387386461138162956320009201234837822718242249019319520691853');
    t('0.00000000000036703131', '-1690706009848928777.495021644332320103184605016856460586815056781662', '-1690706009848928777.495021644331953071874605016856460586815056781662');
    t('-1201941125816287616412820133589940378119107883736149474619667856427578308106640475701376366656637044297989392894431464368963531768836140765436577394', '30399.3500425', '-1.2019411258162876164128201335899403781191078837361494746196678564275783081066404757013763666566370442979893928944314643689635317688361407654365469946499575e+147');
    t('-5911232068256006917083493799668612278296458371177903401236223682450786815464114645404870555701', '-3666175952907291515352144888988964318204590.50095772343418568691895368', '-5.91123206825600691708349379966861227829645837117790706741217658974230216760900363436918876029150095772343418568691895368e+93');
    t('-2459545344690.7997076213355757525315040568953385869184764316593583226907', '4164315079449261632762812204636584837224819345915404704039302166.1662755649220', '4.1643150794492616327628122046365848372248193459154022444939574753665679435864242474684959431046614130815235683406416773093e+63');
    t('717000944256731362781243642636076865324406926670042', '-15881954495698535614971299471328513969444270448080565895751331092451780915657006942144214705830428876978218653.40439815564850025319446970516', '-1.588195449569853561497129947132851396944427044808056589575061409150752418429422569850157862896510447005154861140439815564850025319446970516e+109');
    t('2135955.98546847496701957917801628765161676501430902420565334363847226731093782482383515268139253106808269937151040195861', '-0.00000186143738178071752355591997816736943677781854280530890582737750128944', '2135955.98546661352963779846049273173163859764487224638711053832956643993343653538383515268139253106808269937151040195861');
    t('-2753001186629.1654476569387069958961438785379535103689665000111045009651593729849009490188549152460050154549811339425832561', '2994415432463123195.83557', '2994412679461936566.6701223430612930041038561214620464896310334999888954990348406270150990509811450847539949845450188660574167439');
    t('0.000000000000000018408379822942006989333252748149670155210581048845628953786929400969225780392593930060103', '-350080792300.4', '-350080792300.399999999999999981591620177057993010666747251850329844789418951154371046213070599030774219607406069939897');
    t('5854211899389381506945331507818660152894470704.9769422579022684177921767417413505638887249265068371213031063935537133330368277125538929754', '-6238089', '5.8542118993893815069453315078186601528882326159769422579022684177921767417413505638887249265068371213031063935537133330368277125538929754e+45');
    t('0.000000000000009383448945864756120696895884403358543889635556010530179055321923499853947812003801655721924', '-19052898127325499710380.9743286252434506021419112451035829611583', '-1.9052898127325499710380974328625243441218692965380347462264262415596641456110364443989469820944678076500146052187996198344278076e+22');
    t('-17256820969785465284434808952852690245011499274233791647620649273158761764638386391819039.862028829339196251615676315363239', '13.156452841246543538', '-1.7256820969785465284434808952852690245011499274233791647620649273158761764638386391819026705575988092652713615676315363239e+88');
    t('-665097561.949707', '-2162357984514826.9250074134451642962816183', '-2162358649612388.8747144134451642962816183');
    t('-1787583563218949877447505176411981054278266273447362590023794164049587059.8952802595404997239841103802373009222385203752749', '0.00000000000000000760974939459', '-1.7875835632189498774475051764119810542782662734473625900237941640495870598952802595404997163743609856473009222385203752749e+72');
    t('-97829644134932448583893137101111730259820.05765691989', '-0.00008062307539130036460992642410736502245711006627314391776591134965421657938611242879211258420030106690367460157448425', '-9.782964413493244858389313710111173025982005773754296539130036460992642410736502245711006627314391776591134965421657938611242879211258420030106690367460157448425e+40');
    t('176.49118337833630469590257828139881196863600409523', '35933.6589226268793460337176730393325086784341893690115486029833220604354746534', '36110.1501060052156507296202513207313206470701934642415486029833220604354746534');
    t('2317.37783', '-33432.9207570414', '-31115.5429270414');
    t('6637712385532453617053848832374521936910779.70621790725066804869722068097', '-2227675488368.765036410422350452356577021856408570307494950931297002601448393249899536', '6.637712385532453617053848832372294261422410941181496828317596340643659113591429692505049068702997398551606750100464e+42');
    t('-29796953630.4913026375328064186913399668764', '183881988687448174550.05812049319254873917759915275571715311496797038130265358357168647155176800096292869421', '183881988657651220919.56681785565974232048625918587931715311496797038130265358357168647155176800096292869421');
    t('13199507764.3046118900484236', '2889.225410430843647396962046960496506372239727670782057756607', '13199510653.530022320892070996962046960496506372239727670782057756607');
    t('21296975756.7155683578416319475809678174056385498809428133837225514886696657380', '-12089429986715170910650388697796567002415981764555993502477488839251139863938357832329451138126477861065923975862099475655105604012728', '-1.2089429986715170910650388697796567002415981764555993502477488839251139863938357832329451138126477861065923975862099475655084307036971284431642158368052419032182594361450119057186616277448511330334262e+133');
    t('-10171266580517816.0917067510409', '315013465687559910813490005858570828704.32710226291611151322994451850467532608107794030518681711210056375276848', '3.1501346568755991081347983459199031088823539551187521151322994451850467532608107794030518681711210056375276848e+38');
    t('-104378676461688732476982045085768915711531135396792471359807651838019425066099516.3887940021256850', '-0.0000000075463669108526190909746406521772574', '-1.043786764616887324769820450857689157115311353967924713598076518380194250660995163887940096720519108526190909746406521772574e+80');
    t('-554004567.5248864688113348892343007508509027941489605', '24036.8693849642440730', '-553980530.6555015045672618892343007508509027941489605');
    t('-0.13862781295596173373834834617362885552230456574647246122144469327684018389680003345722428926923436849384730389514581111500008577', '999130309696269960369411189504977138022060325178635804383777964396643757881930599444544232699066835337985580790738', '9.9913030969626996036941118950497713802206032517863580438377796439664375788193059944454423269906683533798558079073786137218704403826626165165382637114447769543425352753877855530672315981610319996654277571073076563150615269610485418888499991423e+113');
    t('111333740621290857055005859873614448428448428149963.63294760371993029666150267467043126547986051822019939851697686673961129595', '4999995459444511596846816663006330762507788928567760', '5.11132920006580245390182252287994521093623735671772363294760371993029666150267467043126547986051822019939851697686673961129595e+51');
    t('0.3638494649282355263819067061458269397737971067286076619347353611281169315189527491523244687688286', '-1790006094624306575957063565992185307582798107924013187.71', '-1.7900060946243065759570635659921853075827981079240131873461505350717644736180932938541730602262028932713923380652646388718830684810472508476755312311714e+54');
    t('-505517596445.1482', '13658590629245746', '13658085111649300.8518');
    t('-0.0000000000000000003526114927031853853794194624925423016782264', '-3648850034079477562581394480009152297642761812144625299013120030846235.2420343951205906', '-3.6488500340794775625813944800091522976427618121446252990131200308462352420343951205906003526114927031853853794194624925423016782264e+69');
    t('11151882673607076860108980480685547378102216722783817.5164597052664085858745970369334', '75.19', '1.11518826736070768601089804806855473781022167227838927064597052664085858745970369334e+52');
    t('721971138118087482240206073369515811598326847978080496736102280373630157207437819185711943223296522977798907649465741783364603042045658293259.583212980', '-140176921692721492746085636102764157797617196402727800746784082298282387.90860726314021850408568982802', '7.2197113811808748224020607336951581159832684797808049673610228037363001703051612646421919713766042021364111003226933905556385625796336001087167460571685978149591431017198e+140');
    t('-1.4681874863', '-3367963466800407715599.44244519159656022505642088971735092327417182287381', '-3.36796346680040771560091063267789656022505642088971735092327417182287381e+21');
    t('-0.0000000000000004372619977839161258710643393271233757307937114407266184997518559013565243060131612462006611450037791975727499469', '-0.000557380206825781899', '-0.0005573802068262191609977839161258710643393271233757307937114407266184997518559013565243060131612462006611450037791975727499469');
    t('889975261077.730985', '-444512813658612993665287376887226223666949336282284327.65111812283484181956398112803471368231154715336519231346350829789199178743774349985755167017', '-4.4451281365861299366528737688722622366694844630702324992013312283484181956398112803471368231154715336519231346350829789199178743774349985755167017e+53');
    t('4228.494108579658047102292', '0.000000000000000037431967516705914275', '4228.494108579658047139723967516705914275');
    t('5081895821751331032957687842323663780819055.0589539655583256', '3045235045891993287321120869322585059630654297351523276745951018307395.66088948484478706336724055686328883825124984361740259408994712721', '3.04523504589199328732112087440448088138198533030921111906961479912645071984345040311266336724055686328883825124984361740259408994712721e+69');
    t('0', '0.0014970897348316271399696665513926791886689361148588977237211047578120594034554421644608967215193', '0.0014970897348316271399696665513926791886689361148588977237211047578120594034554421644608967215193');
    t('-38086604857440351908282393909820732805333576481045075753031751734398766308', '-8466726560595694017777483540348440264157355649896385800844069368330876512659274241703013842599067504442486011483149390298', '-8.466726560595694017777483540348440264157355649934472405701509720239158906569094974508347419080112580195517763217548156606e+120');
    t('3.606', '-0.00000000226457426250954670156179129629125380257442683372667979', '3.60599999773542573749045329843820870370874619742557316627332021');
    t('0.00000000002432', '-2721049388922460690679999802237503439420773586591695831186196683218833781609933786136614218796838155186664711199485697050894450944907939921399737566841', '-2.72104938892246069067999980223750343942077358659169583118619668321883378160993378613661421879683815518666471119948569705089445094490793992139973756684099999999997568e+150');
    t('3419034647852504243.4', '0.00000000000000000001109196145046743', '3419034647852504243.40000000000000000001109196145046743');
    t('0.00000000000000001986294624152158258216394967520873041627413436833216557224760286108648966762394958388810402835384523225417563557', '-3796809284763021958367848219266230972066614510710559054.1146025612649983495607541466475364622687018577099048021869', '-3.79680928476302195836784821926623097206661451071055905411460256126499832969780790512595388010475218250117438591276563166783442775239713891351033237605041611189597164615476774582436443e+54');
    t('-204798018073795.6529529448002736801844846511974520623046560206000445277488879317921030460527861542787789702381941427349799245079342628083413', '-0.000079632857050636167683223005840115134570250229161360215975374259917315308577662215961670005725116845981841223176067219103818019091320131665778', '-204798018073795.653032577657324316352167874203292177439226270829205887964863306052020361361363816494740640243919259580961765731110330027445118019091320131665778');
    t('50550.28', '2121486761751437744953605262799815319.26905', '2.12148676175143774495360526279986586954905e+36');
    t('-889419638834002225522284149508929700476.172650890', '-424305840473677688810924633337769721763339720196709604554205748170445987058261155373223662505888873.82144770710851718819750629065194', '-4.2430584047367768881092463333776972176333972019670960455420663759008482106048667765737317143558934999409859710851718819750629065194e+98');
    t('-40760.965467', '-18942690834.2607446380197806', '-18942731595.2262116380197806');
    t('-21150396424144685285717528391770170039197157268878001492574230326386339596909894.1137843235394802222980337381818083735', '-6846760956505006842349358088765499381514.63658', '-2.11503964241446852857175283917701700392040040298345064994165796844751050962914087503643235394802222980337381818083735e+79');
    t('-5', '0.07853343679663140617550934559203677556312857282298714862538145367932278351423091814533627541733407490114923772813040559584747319434161156', '-4.92146656320336859382449065440796322443687142717701285137461854632067721648576908185466372458266592509885076227186959440415252680565838844');
    t('-1298989.19', '12967839159417800347248402834223327087001590.131521073718772815256761575699', '1.2967839159417800347248402834223327085702600941521073718772815256761575699e+43');
    t('-413624643703430579993792415.246', '-25047788866699.6065823802634', '-4.136246437034556277826591148525823802634e+26');
    t('-17002802435731054578579213050913648941015840438503895311063267366815.4760438', '0', '-1.70028024357310545785792130509136489410158404385038953110632673668154760438e+67');
    t('-0.0217487468807944116026287464540188864099826755097763413913523783549695801368408124880816141876938600', '-267928116215095659487396148857045537314243422826493604265786416560942224473274359760818137268229757952314852408659289722', '-2.6792811621509565948739614885704553731424342282649360426578641656094222447327435976081813726822975795231485240865928972202174874688079441160262874645401888640998267550977634139135237835496958013684081248808161418769386e+119');
    t('0.00000206058115026074270537347836316498830244149457757397506761076895276562936750930508221328', '-1156.25131151512762526101028', '-1156.25130945454647500026757462652163683501169755850542242602493238923104723437063249069491778672');
    t('8.52146', '8138996167927465829623641115119430224272617006293469478795384788419955586210311969.0911005211', '8.1389961679274658296236411151194302242726170062934694787953847884199555862103119776125605211e+81');
    t('-606792703514189222432342591095705024217062979215281726656222323623605437267298163559450093248410482334853425555387131742072494232274', '0.0000000000000075030205599538617170140424842633341870457951188905504524861678938064353730637261119815417488091644638014246916703266791', '-6.067927035141892224323425910957050242170629792152817266562223236236054372672981635594500932484104823348534255553871317420724942322739999999999999924969794400461382829859575157366658129542048811094495475138321061935646269362738880184582511908355361985753083296733209e+131');
    t('-1117646260533756972074615305111827894317502144279827222046925514965408293805196205607164567542139844488130299', '-8950710397293888186512785146937760727214387326213534247514635587853382873562774281822836130052011695722019446354140619086496577801', '-8.9507103972938881865139027931982944841864619415186460754089530899976627007848212073378015383458168919276266109216827589309847081e+129');
    t('17828132133.73128014639', '0.772844155917667013531128306734169323', '17828132134.504124302307667013531128306734169323');
    t('12409597293458966562697810991027046599347924039429686698803091382', '-127221396169370244097597696512688383579380113278994929674479641504219148260281326655899637.2600931768649107131099205359376530727886079199210691234204076', '-1.272213961693702440975976841030910901204135505811839386474330421562951088305946278528082552600931768649107131099205359376530727886079199210691234204076e+89');
    t('0.1848091145595494469555974732282088098544574667848', '-136107941934475938.0155896841', '-136107941934475937.8307805695404505530444025267717911901455425332152');
    t('-248947197724784384796094883777245432118187542578980970162991361.8625105461901333288766567084315189172715325691802', '1377349914239986.46343962270751134940329738658340912130', '-2.489471977247843847960948837772454321181875425776036202487513753990709234826219794733593218481097959715325691802e+62');
    t('-0.00000112660065275108830213180725478020363833427172529177370245297090375761920', '6361232368958886237.15488020607904', '6361232368958886237.1548790794783872489116978681927452197963616657282747082262975470290962423808');
    t('168065941825836172215.28538451779543155718349924675090', '-10020380537725454351590492009621817226308186635527799419512419.1501669170488428784662186956119820551', '-1.00203805377254543515904920096218172263080185695859735833402038647823992534113212827194488610820551e+61');
    t('25062938769541121.630478568522733066136124658086663843828053304849719245047', '2168109004490.945949015343314182595182591703278150478193635235835903', '25065106878545612.576427583866047248731307249789941994306246940085555148047');
    t('0.0000000127998547493889280163892364925927466380437055151407280212547688372695203587233106472126969776111613683412368835630499322381390265', '-21556437448948972049.5307444304476294707119579088623127759607498145900023581252', '-21556437448948972049.5307444176477747213230298924730762833680031765462968429844719787452311627304796412766893527873030223888386316587631164369500677618609735');
    t('-537626025690553743403625600240698190650679101924299218506047682820.61793992322367129916021469763743818692611951563337836906205744108', '0.000000000000078704422482734288110772342341176900316159707688584083247052769004920995218116598172244299647861085206345', '-5.37626025690553743403625600240698190650679101924299218506047682820617939923223592594737731963349327414583778338733062209354368856996752947230995079004781883401827755700352138914793655e+65');
    t('4936866312604172729288711401711374147250', '655874195881897624287118400860713782095501006074749065044610537625092.95069141905913994401259276838353408866', '6.5587419588189762428711840086565064840810517880403777644632191177234295069141905913994401259276838353408866e+68');
    t('-2382178805319965099104798246704598336951576480741895282006149578757854993414213685118', '-0.00000000000000002613467607235150454275071440605158933185803781754976383299500468377641084642463280487309618916557583346059257437473874331', '-2.38217880531996509910479824670459833695157648074189528200614957875785499341421368511800000000000000002613467607235150454275071440605158933185803781754976383299500468377641084642463280487309618916557583346059257437473874331e+84');
    t('-15946824.09950277073438320718451422330046117375794902861875105613916239700142602', '-0.00000000000010267855303483749668481765879881882622179514', '-15946824.09950277073448588573754906079714599141674784744497285127916239700142602');
    t('-581425628881991347289393948535594023431644044950894384975254331720033.7644030704523982311900267238328134677', '-999805005439068442805051046377911526225724977745109030159313147640897377464.83555602537429836574687154635825719', '-9.998055868646973247963983357718600618197484093891539810536981228952290974985999590958266965969368982701910706577e+74');
    t('60413295.642986556398', '-4226075045367317431795623702749235954814399556250656668352316.5728962846147542124931735298180367334281333643494998', '-4.2260750453673174317956237027492359548143995562506566079390209299097282167542124931735298180367334281333643494998e+60');
    t('-81794690209841974740.53672088428717597735688956390645130995150579644016165479898744379', '-7.88107898048618399872607957392671978662', '-81794690209841974748.41779986477335997608296913783317109657150579644016165479898744379');
    t('-1997717463163142285894367278389823014470638.3025530535651929616090187971946674805858924875948817390067051806853814519986698027983653385', '6', '-1.9977174631631422858943672783898230144706323025530535651929616090187971946674805858924875948817390067051806853814519986698027983653385e+42');
    t('-304875626379421812877053824267.89813338244318006612477499081284272413190571121891074444720931096507130284584180247091056', '-0.000000000000000010980103787919551156900136233767434931235794906561314315549096503926892808208116475271649840370149590820545219560834375927', '-3.04875626379421812877053824267898133382443180077104878778732393881032041944986345675683004217526385618394938306397803368208116475271649840370149590820545219560834375927e+29');
    t('-88815210861255646356014368338106338503849425126132.285406775673561905065', '0.000000000000000034992420410611731246495160913', '-8.8815210861255646356014368338106338503849425126132285406775673561870072579589388268753504839087e+49');
    t('-0.00000000000000237643259847411171950841188633341948147021355426920101030032158183296245118336593123026984501161674321953387313970740020', '-0.0000000874212145702142670184949308326411041422527293858413318840288404463319361607051392899032509388086790', '-8.74212169466468654926066503410529904756722108560548861532298507466535179936675904732691821690785240116167432195338731397074002e-8');
    t('-0.006555884028715480652', '6930118317156.576980190277', '6930118317156.570424306248284519348');
    t('39.384744166791242957726294455842839425402642808858550449900179430725398075422668018325305226767828926369540183994331444452642090521391283', '3260986081387.207629495556353019281390351729387452232726', '3260986081426.592373662347595977007684807572226877635368808858550449900179430725398075422668018325305226767828926369540183994331444452642090521391283');
    t('3908267232516297546694850213843945511916410859719238136818722842077.27108076292427902546404238314424734106746892397683459219351842029663495194', '-0.00000000000372286358808905369841368779659136965109619662993128616667816952276941560286526066255037987308726016479323693533782656017305522143883165700131792', '3.90826723251629754669485021384394551191641085971923813681872284207727108076292055616187595332944583365327087755432573839556358713412995678241723058439713473933744962012691273983520676306466217343982694477856116834299868208e+66');
    t('-0.00001556397239423792402602941437191887477933365239361893990546595775376870987107277583', '-1732082517354208969230613795180012699705122410188783671659220771290170536520910', '-1.73208251735420896923061379518001269970512241018878367165922077129017053652091000001556397239423792402602941437191887477933365239361893990546595775376870987107277583e+78');
    t('20451329027854657865397803520110269731953546381423082243753661273233534094096458728774066484280346143633534282606929883', '1.37062391834155371473359912213772362707849619840251030918103616690753987794703765279270817110400101918287849263050336091', '2.045132902785465786539780352011026973195354638142308224375366127323353409409645872877406648428034614363353428260692988437062391834155371473359912213772362707849619840251030918103616690753987794703765279270817110400101918287849263050336091e+118');
    t('-9873613763574093.9169949677113266651552362204071722363114406302496589482078244525416517', '-546631047066885391316821969502606459323760036081517906369260562129.5282969018129257869570273191088510708842914399537183835952083014376', '-5.466310470668853913168219695026064593237600360815277799830241362234452918695242524521122635395160233071957320702033773318030327539792517e+65');
    t('-720103743977684216550279573635737782.371525679181701893280139772', '0.00000000047196', '-7.20103743977684216550279573635737782371525678709741893280139772e+35');
    t('-124523382475.065647339231367286548664060598351481238512537358040678937053065205', '267195207211327546143.99593799833381494', '267195207086804163668.930290659102447653451335939401648518761487462641959321062946934795');
    t('-669248421565027510098621757579904855957549198616536004694615.801532771296796860509827700071860799103269705247335116543', '333905925959634844', '-6.69248421565027510098621757579904855957548864710610045059771801532771296796860509827700071860799103269705247335116543e+59');
    t('74741.757595631932381120479050874298796374442246866101939364', '10920811144639.430670345', '10920811219381.188265976932381120479050874298796374442246866101939364');
    t('1170639975951201522249787446299288957.534087576257621574901157627904404', '-0.00014785053034279838988159517', '1.170639975951201522249787446299288957533939725727278776511276032734404e+36');
    t('240641615318369852633021813201283453473338354696847465428407816540707799946328465329158700582983365200972956128970708198340862470221701303619', '-0.0000000000000000069337247648616552808723398415008620822832893391786175057036755718378355', '2.406416153183698526330218132012834534733383546968474654284078165407077999463284653291587005829833652009729561289707081983408624702217013036189999999999999999930662752351383447191276601584991379177167106608213824942963244281621645e+140');
    t('111714245279259151600970278454866645262571826000337873347902770371946919658476476.581635152136516883108103196', '-145713944701659169049317661586406655886350.64439112401759220725326887023191998402271373838960301529', '1.1171424527925915160097027845486664526242611205563621417885345271036051300259012593724402811892467585483432576808001597728626161039698471e+80');
    t('0.00000000000000000011659916431504181998777280021990151674872574004317563534059818105024140361220410041950905384756907730822832402571557634171797064450234198869785', '0.0000000000000034973095978104435865425237181247213055722049975492', '3.49742619697475862836251149092494120708895372328924317563534059818105024140361220410041950905384756907730822832402571557634171797064450234198869785e-15');
    t('5208299735981215934383073700988540115', '0.00000000008193999192179', '5.20829973598121593438307370098854011500000000008193999192179e+36');
    t('-0.000000000000287395113688235501898364131112180654248398999683488052724240112623987258282732482937', '-0.000000000005224156433448505010390917879000913540363966165400734673984060714544377832355795421', '-5.511551547136740512289282010113094194612365165084222726708300827168365090638527903937e-12');
    t('251290.029', '0.0000000408265742458950413927784669187015196483849964746', '251290.0290000408265742458950413927784669187015196483849964746');
    t('-0.002069526456699789023016398781994820749838585707371628', '-208989.20687136738198249707358292330050304648320312788', '-208989.208940893838682286096599322082497867233041713587371628');
    t('136830053104872291586007040221524635644507855229364036.990085839244669865877563203743313609845467275010746564518697656354953714652400', '12996360060717584879190082832087158897977012928919981.1724803374445157382323851607362941658942137', '1.498264131655898764651971230536117945424848681582840181625661766891856041099483644796077757396809750107465645186976563549537146524e+53');
    t('349923146008625248221940071805133632424745448085367309153304.41451', '1.279845995348543772724126048570984529358929917199046395358884958125978504501534072647803741667566840006664141118201898026492588582829', '3.49923146008625248221940071805133632424745448085367309153305694355995348543772724126048570984529358929917199046395358884958125978504501534072647803741667566840006664141118201898026492588582829e+59');
    t('-0.00002588190901774483141446130106109665568218409839877790718167670204180165388892638488618624373249215908731', '-0.000000000000000998380445484244673146797637364810042331013877503942848237747500650941845596181353662072946500712838803596129029770361640568986086478118', '-0.000025881909018743211859945545734243453319548908441108921059180644890039401389577326731782425086154232033810712838803596129029770361640568986086478118');
    t('0.000002397702115545082706325', '-0.86184468879219410762921439972937635313400330685547662030210928411728749240902108881486043419424702673819765398803611924680684916244185716500', '-0.861842291090078562546508074729376353134003306855476620302109284117287492409021088814860434194247026738197653988036119246806849162441857165');
    t('739247350528774318310976592372378823989.9340970329512293', '0.16461323830286423518914252503718472208992620706491311143767005406182583700046088073642382183228439964918345692170788', '7.3924735052877431831097659237237882399009871027125409353518914252503718472208992620706491311143767005406182583700046088073642382183228439964918345692170788e+38');
    t('20138770597416725450254724328732818742267620348802456988572150217738319484193183724229804103329780221759485486678', '-789269493613323078997155252278243776415270697547093524400899466707939123164245744251', '2.0138770597416725450254724327943549248654297269805301736293906441323048786646090199828904636621841098595239742427e+112');
    t('0.00000000000000000077718007226574178859573156113273718366023940333047610', '0.00000793842663906287702694882951591808277980436371652228248293328108479174722056639404425886138539535746075798843345', '0.00000793842663906365420702109525770667851136549645370594272233661156089174722056639404425886138539535746075798843345');
    t('-0.00073004266755307447297787100885736643137065171791847775170199', '533767943554264575564059388781105481229849071443170662645468384072829601890382711597239.43317227373401753588313752941835310264692801', '5.3376794355426457556405938878110548122984907144317066264546838407282960189038271159723943244223106646446141015965840949573621555735828208152224829801e+86');
    t('203795797050226528701909939354095.74227652473566285394054153851636316880904908195101127405638217584507290263629702856344914721596751573682519', '1048176497643', '2.0379579705022652870295811585173874227652473566285394054153851636316880904908195101127405638217584507290263629702856344914721596751573682519e+32');
    t('-88244004722847480146822008275080880695915873140049053843555316989873895145229459150435483330020885756477515260816717989568792030863738799712004', '0.0222020450217264733443', '-8.82440047228474801468220082750808806959158731400490538435553169898738951452294591504354833300208857564775152608167179895687920308637387997120039777979549782735266557e+142');
    t('-245.330813553080793', '-139455315144533860778775215600928891795189892914202233172560603256768538146961881693882554366322835347536572308601', '-1.39455315144533860778775215600928891795189892914202233172560603256768538146961881693882554366322835347536572308846330813553080793e+113');
    t('1312620952926614444601139109837919862847383911012476338291408744344284710456350.61901295', '0.000000005001841441858025051749066429', '1.312620952926614444601139109837919862847383911012476338291408744344284710456350619012955001841441858025051749066429e+78');
    t('-0.00000002568121501817024789950645', '0.00000152064801516710974357034365714164447372286990018294642975122815566995939556502510124016256086916899095831799319898986550479847', '0.00000149496680014893949567083720714164447372286990018294642975122815566995939556502510124016256086916899095831799319898986550479847');
    t('-50146741.71136917438523439037364041570643531039880309166796', '721.58', '-50146020.13136917438523439037364041570643531039880309166796');
    t('-0.000000000031373829058590280763397830914839643150936549070276134150319113360057425916282823335964570132205357792953166020744035018731306854352936', '1209934516502680876038974890161267538909081686211586564144614326182360480572106031540130067011132046106334621001880794855659284874', '1.209934516502680876038974890161267538909081686211586564144614326182360480572106031540130067011132046106334621001880794855659284873999999999968626170941409719236602169085160356849063450929723865849680886639942574083717176664035429867794642207046833979255964981268693145647064e+129');
    t('-13404342125936188871742118602022353067169949299128900358060758959144.618999657876884344030397439486785962997196322120897869261859638385586348924333', '-168915526978312.650128129957132282649873584573819887866036969239069391528581261172175429771', '-1.3404342125936188871742118602022353067169949299128900526976285937457269127787834016626680271024060605850863233291359967260790440899557761778695333e+67');
    t('1806842795835990618047801.084576277957835712505618234152320757352128501462413034062746347', '-50734288417028257924421286860848404602443885854272475391326979005189668017648768244414775061314743386961522762.585374718651305701111', '-5.0734288417028257924421286860848404602443885854272475391326979005189668017648768244412968218518907396343474961500798440693469988605381765847679242647871498537586965937253653e+109');
    t('-116120201222725247610527954599536747256108402740643449687724143269086567616449171607864628957429974416', '1148190510672125444192340349744778820699595561256574541295058422369656677509037208045122028374935585403541656043987495821745', '1.148190510672125444192224229543556095451985033301975004547802313966916034059349483901852941807319136231933791415030065847329e+123');
    t('-665059516301467503087062204795807129205981985820520012687625179.56870702735969931519884226663966750454', '293092076876324270859.1678050313417760917885816996452', '-6.6505951630146750308706220479580712920598169272844313636335432040090199601792322341026056699446750454e+62');
    t('-151941033837572720599600925337962250720110163460498264898016203736097072978877965550197312865391818', '275519398845322762752039373616165590950134853231913357677442.98089023579266433338582691582272222226713269975686273608935285642960', '-1.519410338375727205996009253379622507198346440616529421352641643624809073879278306969653995077143750191097642073356666141730841772777777328673002431372639106471435704e+98');
    t('774.27298241935872782935734853935110700980034942643', '-2335613505940656938918706780962782666653609805431040481805.54016', '-2.33561350594065693891870678096278266665360980543104048103126717758064127217064265146064889299019965057357e+57');
    t('-0.0000000000000000003606517118223376107369897317076138705797620611885910564859903965167670077845554058709312065326542523687696158170393314', '-1057235201.5625889131220869434', '-1057235201.5625889131220869437606517118223376107369897317076138705797620611885910564859903965167670077845554058709312065326542523687696158170393314');
    t('673588976883236537490.1143689352121199626156550908385672312863492899095087287923630764608919202964646650938598591666124915', '810684376454419924392927167848507180923498918725351613660889567923.47605095181300135828', '8.106843764544199243929271678485071809234989193989405905441261054135904198870251213208956550908385672312863492899095087287923630764608919202964646650938598591666124915e+65');
    t('-3430020671178955527172622925417363583260241197.62027095120829845', '-2697651899209144982439207207158758256041163754489319553122389855760271821650344473467567943237904607070691739749282365941241955206', '-2.69765189920914498243920720715875825604116375448931955312238985576027182165034447347099796390908356259786436267469972952450219640362027095120829845e+129');
    t('0.000000000000000000386684190', '-0.00000000000000593500383841354', '-5.93461715422354e-15');
    t('-229553467063496173115623569488071051955397518369782894211290517387417359092419.21316953918885760546743923347047910041156791448271162', '128761823394.73391412805150185026077404187271056727840511587820472574863102356882320762518910948767394645397112340047229286811934703310025622306', '-2.2955346706349617311562356948807105195539751836978289421129051738728859726902447925541113735575520666519159776853313316279860450689425136897643117679237481089051232605354602887659952770713188065296689974377694e+77');
    t('752803651337807242880562206786.554894468', '-0.000000000000000001071337', '7.52803651337807242880562206786554894467999999998928663e+29');
    t('1327499528941.259705731396165260452862352568124098842440934493864336643088192293192971501913307764464886787116200010928852119632015468364', '-0.0000013012400459295541997244614408586478387764368200', '1327499528941.259704430156119330898662628106683240194602158057044336643088192293192971501913307764464886787116200010928852119632015468364');
    t('-42732826522401556404680286738458727450161584.56982929732944313055951751542753245303762787700081314483', '-0.048575929228921161125311723255416867769014200391591416962565880220141310', '-4.273282652240155640468028673845872745016158461840522655836429168482923868294932080664207739240456179256588022014131e+43');
    t('-134317115975228516279277369440412329142525813492681218176.452257906478095711445159604268249481814005710', '878.87016124066018104272829846171161822902416', '-1.3431711597522851627927736944041232914252581349268121729758209666581791466871686114255663125278984571e+56');
    t('-67249264605177190041089081168585468792215793614566298512516668756953699746352828585714616446517294544062645676845293.91', '11785.189223441658736456882609485294922576241169', '-6.7249264605177190041089081168585468792215793614566298512516668756953699746352828585714616446517294544062645676833508720776558341263543117390514705077423758831e+115');
    t('473845447573029801116507496764870227913948278575637638312271.6929093', '220256599636715846396470132047447071430762023364054132738307.92410966839175642212437153019718871180', '6.941020472097456475129776288123172993447103019396917710505796170189683917564221243715301971887118e+59');
    t('61663987226727831177358451280.1273785663169955851813593', '7750.4529567880816153892714518857989477603684894511929835815714039337396479', '6.16639872267278311773584590305803353543986109744528111857989477603684894511929835815714039337396479e+28');
    t('1114249782085628259131.6300784647405324495788', '-75441254269515041497240125862191629.18821756865035216838014718712510162118417621758193335236088548825411631090241783', '-7.544125426951392724745804023393249755813910390981971880134718712510162118417621758193335236088548825411631090241783e+34');
    t('-47284766640683124441541777476988779051324489666892557826451642191773394572968326419750395845015488519009501143710.226604466706051616676759471', '-178533979356035013007.0486021615920523090681110580689173330395242698234732472413898657172920542549863', '-4.72847666406831244415417774769887790513244896668925578264516421917733945729683264197503958451940224983655361567172752066282981039257448705290689173330395242698234732472413898657172920542549863e+112');
    t('86971785918916738535599981140236508672686072922404606596498347463786815664561636214', '-38790942580106273917029.8616351336817911518100068189254182509141591442742642', '8.69717859189167385355999811402365086726860729224046065964983086728442355582877191841383648663182088481899931810745817490858408557257358e+82');
    t('229890914939905533517994656115939439222611214166185047720.6737', '-16450908510999308489324339587686508934693633813727800191296126619327316733314370883200772189175967292716357616129.25956476613892564270457543510303499', '-1.645090851099930848932433958768650893469363381372780019106623570438741119979637622708483274995335607855017256840858586476613892564270457543510303499e+112');
    t('8624299879022265728690568197694576473349674764163914006555775202613.80066', '-1951918312957524571038442996130836809607213863913860985460832910117284832599299718784244014.9462', '-1.95191831295752457103843437183095778734148517334566329088435956044252066868529316300904140114554e+90');
    t('0.000000000001791442922101442150265', '-176219478666778.39032', '-176219478666778.390319999998208557077898557849735');
    t('-0.0000000000000000000450909281658069443034906463413806497945746964537569542795485785087816', '20059612657232793812777346178916639350613903200274271932748642060630002343102688319925466687857659887751178879826484844955146.4794139791133416216349437', '2.00596126572327938127773461789166393506139032002742719327486420606300023431026883199254666878576598877511788798264848449551464794139791133416215898527718341930556965093536586193502054253035462430457204514214912184e+124');
    t('-6787468140564734605294950923721904660396026978942519331890016886817683748193913278435340067132052148355490986326961525834411933314297910293075891258', '299875808414865644293530981481903399783597394891051377970871372993113352087', '-6.787468140564734605294950923721904660396026978942519331890016886817683747894037470020474422838521166873587586543364130943360555343426537299962539171e+147');
    t('-2985845195375048154175460145131625566631330811.0698198381350592343482301558903611521655008133682589725943686764286694288782', '0.0000000000003440896409210193126584537498187185057931457434599306493995078695234961055215715664902', '-2.9858451953750481541754601451316255666313308110698198381347151447073091365777026984156820948624658268509087457792699210086765038944784284335098e+45');
    t('54523776480140693828539812402923.03280118600797412859863998481055730526509253892247361691821925554', '-3018017693744641785170272539537468727331662967020516461365187921.622252481531570142507295', '-3.01801769374464178517027253953741420355518282632668792155278499858945129552359601390865501518944269473490746107752638308178074446e+63');
    t('-148950590810862085091278929767052107224.8773813216218095795711623868958220961874', '1.75192', '-1.489505908108620850912789297670521072231254613216218095795711623868958220961874e+38');
    t('194487565128888901387639549948511413965636783508366826662059481869489007007469884500747433533356290730733602216171', '0.000000000000000000012232074647167060238846099337052578273155522711981842441546203086466766954706886327921', '1.94487565128888901387639549948511413965636783508366826662059481869489007007469884500747433533356290730733602216171000000000000000000012232074647167060238846099337052578273155522711981842441546203086466766954706886327921e+113');
    t('125779680121395715709804872384269463897503936971384019.82361436591484229964552907887172032', '-10705813706184352649413600364408885130948697893', '1.2577966941558200952545222297066909948861880602268612682361436591484229964552907887172032e+53');
    t('-2218129701029502833268536.3661385527823829459113120296928740', '25227558478464802399374718020711143953033396', '2.5227558478464802397156588319681641119764859633861447217617054088687970307126e+43');
    t('15922278874844168795624334393750706784541723182329429804868920436645833815588763261631529887040208280', '-26296.887413892620188356349250229', '1.5922278874844168795624334393750706784541723182329429804868920436645833815588763261631529887040181983112586107379811643650749771e+100');
    t('1869337163952851887099278621758788.4912775488811143022866151909022695202327782530199655924124637283188619937985055306138431', '25.12', '1.8693371639528518870992786217588136112775488811143022866151909022695202327782530199655924124637283188619937985055306138431e+33');
    t('-0.063341123812589258651794426063119957591863528264980035743077602252627967706982211651872386199270743459561294', '-0.000000000068714090541923937797208713394623709878340222046720399207', '-0.063341123881303349193718363860328670986487238143320257789798001459627967706982211651872386199270743459561294');
    t('-123065986062521895461889579855429054876.865438792794260927192006', '7084594017255215876752497968505767888079486468316419822128516563096248999466202123.38787540885020434', '7.084594017255215876752497968505767888079486345250433759606621101206669144037147246522436616055943412807994e+81');
    t('179397967161885408347722783468000236145781976956498009525966031884761882791548055245739198861467543753736467441189809830950974868855035870976.038552', '-6573750410552527157063328131505255371227573474471234861827269.61525370370996879661223', '1.7939796716188540834772278346800023614578197695649800952596603188476188279154804867198878830894038669040833593593443860337750039762017404370642329829629003120338777e+140');
    t('-139353860859081662533478815947566213452966893244463520929326236932221117460953128298317531935014131121470.4619756268309', '169812848393858427202383142897502241915472009842575593156979023589.9908950026432880351156036902370299251473942264824787465775921562579669735', '-1.393538608590816625334788159475662134527970803960696625021238537893236152190376562884749563418571520978804710806241876119648843963097629700748526057735175212534224078437420330265e+104');
    t('-0.00000000056135129922386681', '-595119376867293381235464924', '-5.9511937686729338123546492400000000056135129922386681e+26');
    t('9999400699648583439.910165407231452805204397263', '-239336712014958618084045176038.81761359335921757157899131356', '-2.3933671200495921738439659259890744818612776476637459405056e+29');
    t('0.000002623585411494924528522745861928371567645115623878', '-5636993000244072951029.8588310419570237', '-5.636993000244072951029858828418371612205075471477254138071628432354884376122e+21');
    t('-16469.08018532953044652581289843852712919053726120839851886233541371893823', '6420210.71382431387667941690927', '6403741.63363898434623289109637156147287080946273879160148113766458628106177');
    t('0.0000000001500611664251128666574118039477481282151196384889645376085403596652469864102626694482747', '263135384759861159877959298119330040825421757978037356875485701572152302094162243110', '2.631353847598611598779592981193300408254217579780373568754857015721523020941622431100000000001500611664251128666574118039477481282151196384889645376085403596652469864102626694482747e+83');
    t('-12317435659483.46', '-238178727623943.57770836536163790335018945716578328065826254', '-250496163283427.03770836536163790335018945716578328065826254');
    t('705138698371479924718654705441277066425', '-264458728957414529691672105440971342620756281273769616148413659.83139211549669420724473557409844984235683219270687462430060909545589521338825', '-2.6445872895741452969167140030227297114083156261906417487134723483139211549669420724473557409844984235683219270687462430060909545589521338825e+62');
    t('26845297854.21830300459071341439856406557876499767147640691523616826235445154220875683891282566002689437474534485000263914', '-787376261006598490249180340720309857956285707959849.0204685624653359640496048535548108297789532801386418120', '-7.8737626100659849024918034072030985795625886266199480216555787462254965104078797604583210747687322340564373764554845779124316108717433997310562525465514999736086e+50');
    t('55036049768.146079287001237352665049000674092', '-22143566056427860768945972894580679205849885116613091602005627.33897358443738987438843476838059361711849053157259882779343', '-2.214356605642786076894597289458067920584988511661303656595585919289429743615252172338576770650161711849053157259882779343e+61');
    t('-299420594788150782543317652372306174815941523345478009592090041773731019434882.753119389560259765940107169762142160679052728098688521999921', '-12453366463782072599491184208879912917463798673347144858330325.3505439835880291946926163036554982195666987139373567106718470730123422', '-2.994205947881507949966841161543787743071257322253909270558887151208758777652081036633731482889606327234734176403802457514420360452326717680730123422e+77');
    t('-256022815746044576351192361407997374095362780421849294332421070473692422546466754095230.656295542710250921847773406', '2439108134.33476347036328940422149059899605979832', '-2.5602281574604457635119236140799737409536278042184929433242107047369242254646431498709632153207234696151762628280700394020168e+86');
    t('-137304230559003045184479307675479065631585995407517838007075545485391216662724836', '-1173188488432045163948013790.3943256082040520817421175529478290364828174703361614075880892748696574370905742309087558', '-1.373042305590030451844793076754790656315859954075178391802640339174363806107386263943256082040520817421175529478290364828174703361614075880892748696574370905742309087558e+80');
    t('520707804236215170390228230926413098433541569396930548508375720908453157589569499547054273496534756616492929935502789', '275182891299865137595.3277925634463705149893539835817542834549123082412850143300233066529878447078650880805064450173373918062760558', '5.207078042362151703902282309264130984335415693969305485083757209084531575895694995470542734965350317993842298006403843277925634463705149893539835817542834549123082412850143300233066529878447078650880805064450173373918062760558e+116');
    t('2960407629256095014394492264783919104375530133643554797532898.80192159867955337779238068661889924', '276684036811079285530960359872097261579493580.0498287546202754687941154667048', '2.96040762925609529107852907586320463533589000574081637702647885175035329982884658649615332369924e+60');
    t('-5517355901446910.93833484348629903096547', '-642213064278646161225254.78548483470', '-6.4221306979600206267216572381967818629903096547e+23');
    t('152801618545615441308342159364435799274895358751942316139180186', '-44665364463201090323324.0002892895', '1.528016185456154413083421593644357992748506933874791150488568619997107105e+62');
    t('-42990335950341162574017016', '-155734841164152934379337800729168607491698641563818839018593201886.9108295077085945123926518226547', '-1.557348411641529343793378007291686074917416318997691801811672189029108295077085945123926518226547e+65');
    t('42132979000424460466648083857741221621925059', '230035200616600947476235404386815897149843926902113349799669418017872910933828', '2.30035200616600947476235404386815939282822927326573816447753275759094532858887e+77');
    t('-13721903019805428754977193786.436261994441376688889424250392449552098872713999965025277', '10194877.68498209368023', '-1.3721903019805428754966998908751279900761146688889424250392449552098872713999965025277e+28');
    t('0.0000000000000000000124951715848249957980344035116', '265848190673836152145387.561232311400411988900021332089088465444147046461212646850', '2.6584819067383615214538756123231140041198891251650367391346124218144997281264685e+23');
    t('-2478.622067383552773557059143603127982', '0.32513599534141365489000782897237000600555505333921918961022131484515789518698137637587', '-2478.29693138821135990216913577415561199399444494666078081038977868515484210481301862362413');
    t('1279542454.892945577837995361708396', '-135890698074837524994261181604433.067602982488449328126084049766962', '-1.35890698074837524994259902061978174657404650453966417688049766962e+32');
    t('0.00000011558150572741272723618033490943108053756815783338338515794075800230701762335698251644854707498749273387290233658497030939253228419', '10177968039270056306413680.184343176766129455291207155884626542169167703074423510915329626878514', '1.017796803927005630641368018434329234763518270393439206496145160024824064258134429871478481927200230701762335698251644854707498749273387290233658497030939253228419e+25');
    t('-1435068.618647657046101438708930395218818069312124176746881282174539906140539486722465500068653075714727202174518522871035823011751', '17.77', '-1435050.848647657046101438708930395218818069312124176746881282174539906140539486722465500068653075714727202174518522871035823011751');
    t('1133468285512298621716517199739886057755521547545846865283277167663070426262818957724352964984613242138379271360556', '-141537307411.00826429536495677705445', '1.13346828551229862171651719973988605775552154754584686528327716766307042626281895772435296498461324213823773405314499173570463504322294555e+114');
    t('-3578677705738240793482085438614614551372365641536961576081303950220125233969517175095644.5329094147681', '6370326210773977367247462308.43303220494101174275111504449143', '-3.57867770573824079348208543861461455137236564153696157608129757989391445999214992763333609987720982708825724888495550857e+87');
    t('4539369410251014415308963272777018911975282268127892682157655.2687008509', '0.000000000000000001809319965416452365878526603785327342157425587339349', '4.539369410251014415308963272777018911975282268127892682157655268700850900000001809319965416452365878526603785327342157425587339349e+60');
    t('-11612662163115798192491748988208694545917762161675688590981743152130989015346035128837953127650326301679404956035505966175985535696091516284466656479', '-129128951232580369482047739685813992608602054001765557698178402197589395906493483213995', '-1.1612662163115798192491748988208694545917762161675688590981743281259940247926404610885692813464318910281458957801063664354387733285487422777949870474e+148');
    t('0.0000720244123748728799957336555523697', '71791325249354506431814914086989', '7.17913252493545064318149140869890000720244123748728799957336555523697e+31');
    t('-133560.04', '0.0364781487240002253276365093066436254283459521449826381477605260349961957510696178085420533616152821352777161063017', '-133560.0035218512759997746723634906933563745716540478550173618522394739650038042489303821914579466383847178647222838936983');
    t('224439133349456441836858561541114452976410738191588.76028', '7143422387100306585012321998.95338744534500001350', '2.244391333494564418368657049635015532829957505135877136674453450000135e+50');
    t('0.000000000098363222378328329588864342385890675532349142951521770288606050467173967001464527333472294637844', '0.000005565523790105586010252525454', '0.000005565622153327964338582114318342385890675532349142951521770288606050467173967001464527333472294637844');
    t('-19302672950888869853468603175885975100580447783425209560751846890885724469162939072204904025589809680230949.15', '-5221907579579195185631992055573', '-1.930267295088886985346860317588597510058044778342520956075184689088572446916816097978448322077544167228652215e+106');
    t('446571007966420189521982224619283959966447939678911167542457919503539937579809547715586812124362669', '-0.000000001532941453246004602861225513798223406668763246640071116663063458244535088282298106516453270820501460262785607', '4.46571007966420189521982224619283959966447939678911167542457919503539937579809547715586812124362668999999998467058546753995397138774486201776593331236753359928883336936541755464911717701893483546729179498539737214393e+98');
    t('162256387512654913999487367376656275516', '-16092263592753754060348826948877883484399357352376898503109928037035273000095267537045095647020.9340480402341251915539575258', '-1.60922635927537540603488269488778834843993573523768985029476716495226180860957801696684393715049340480402341251915539575258e+94');
    t('-86.282711884059841689767705467465544410310216259432751729', '-92429065419108217418705866104650901.759952292192027196592545226813152124130603195434172370', '-9.2429065419108217418705866104650988042664176251868886360250694278696534440819454866924099e+34');
    t('-75868293.338801243263000230664880224491442311102621282662689408530004543', '2944972.7213068711096384504382609733072677', '-72923320.617494372153361780226619251184174611102621282662689408530004543');
    t('-238444972635059326925252856448297027255504596325494840989337749695326400426034906266359680165724.6162269656460992471659583387303201989744940449', '-63316025717469', '-2.384449726350593269252528564482970272555045963254948409893377496953264004260349063296757058831936162269656460992471659583387303201989744940449e+95');
    t('-0.00000000000000000002628376740857869992666514908906633928490609752241821541445544366572977726597123951321896262451176771066761534225489543173636033414059', '-0.00001057742323776455463981263889953456044073157406788745965110714274589491766184028326935', '-0.00001057742323776458092358004747823448710588066313422674455720466516411033211728394899912726597123951321896262451176771066761534225489543173636033414059');
    t('-0.00000000000003565805735839291765636742503', '0.000000000000000989774781039626', '-3.466828257735329165636742503e-14');
    t('0.00000000000000375655426038511007746679974331400659776884912884102085329769227537463698444', '546526458201961217312998279716410991.2240626147007447545948388', '5.4652645820196121731299827971641099122406261470074851114909918511007746679974331400659776884912884102085329769227537463698444e+35');
    t('-14278318423785251476284397565212.1237964463782332', '-37174052283987138.47044217178181655468224887601671623199409260448279688190513294184710815453374043139975669500607120979735080911231772347', '-1.427831842378528865033668155235059423861816004975468224887601671623199409260448279688190513294184710815453374043139975669500607120979735080911231772347e+31');
    t('-17330.438895896038328483940004', '8350320541724405487824535323446674.061386758095913411145749626626909634263800865346657', '8.350320541724405487824535323429343622490862057584927205745626626909634263800865346657e+33');
    t('-0.0000000000003788156589604414602956818964327013143296398421560231778437969245', '1634246393878053828481992729777312078472156.145364321882169640446334568077598604110397446683352082023376014305304686515469544910814539917743927216291855', '1.634246393878053828481992729777312078472156145364321881790824787374126617302922213964745369022442181219991127460889590969544910814539917743927216291855e+42');
    t('408110915553257916118522792717017858639986473034925463411181033631049938756007592', '-4438008.684761039161', '4.08110915553257916118522792717017858639986473034925463411181033631049938751569583315238960839e+80');
    t('-204903173380829791053251747', '-191455970489532344774803952466.94684645095468153608792233241808335603853244943133837112979971481605', '-1.9166087366291317456585720421394684645095468153608792233241808335603853244943133837112979971481605e+29');
    t('132721140690308763835711368030783660696277559510011717622014.3314196', '0.00554064704637147709273658476408607464726088564072789893070411751226529035714577466777850716116505710506906049773777298273003529699', '1.3272114069030876383571136803078366069627755951001171762201433696024704637147709273658476408607464726088564072789893070411751226529035714577466777850716116505710506906049773777298273003529699e+59');
    t('0.0000000000000000265193728860055071583880736', '-0.0000000000000000011892979550800567635892667658882072063', '2.53300749309254503947988068341117927937e-17');
    t('1924004.96231', '-7445082351600096.3980954490079933041653479391080501268670868688846572103759932357179141189615092741415678649052246409361594010562380231156977081147310', '-7445082349676091.435785449007993304165347939108050126867086868884657210375993235717914118961509274141567864905224640936159401056238023115697708114731');
    t('0.000000000000000000013903352680411740283535242027183387301940111406117407562975303647341986279', '0.000000193241459485855551124461468088220083733257451375352357494658457700846322832086915983440402233205', '1.93241459485869454477141879828503618975284634762654297606064575108409298135734257969719402233205e-7');
    t('-0.000000000003326499271235431532653228418183469394009927691863875127864766277232493921924691425777066908583580479571', '38538.5', '38538.499999999996673500728764568467346771581816530605990072308136124872135233722767506078075308574222933091416419520429');
    t('-20818471791668358551694729332150711.171307610', '-0.00708772819104515753090752104339593647529198186926535557761440185947368079580', '-2.08184717916683585516947293321507111783953381910451575309075210433959364752919818692653555776144018594736807958e+34');
    t('-752467347549701119616683978206438108', '0.00000000000000133182829836217784813241401889873', '-7.5246734754970111961668397820643810799999999999999866817170163782215186758598110127e+35');
    t('-71331.43876400219162800429885965449977637326182916850599237355086577534159328324037916425312333490817913501686724307', '-0.0000032139466257383829211495727047401282848726415697991943119851858819503701337097998682466465136331', '-71331.43876721613825374268178080407248111339011404114756217274517776052747523361051287405299158155469276811686724307');
    t('0.00000000003870179783985381063243324820910313', '-548412148702636016270138827097973238704618012095820424598631550144908156181914050173920.55528708694731', '-5.4841214870263601627013882709797323870461801209582042459863155014490815618191405017392055528708690860820216014618936756675179089687e+86');
    t('0.000000000000000159733765895', '2415466722403145928603501563265788955.65674020714185313333653211954046063741077995756304216513522037423497068589560605814929217258491053368146', '2.41546672240314592860350156326578895565674020714185329307029801454046063741077995756304216513522037423497068589560605814929217258491053368146e+36');
    t('227954506189379566132.3531049565054147523709893227464306251862462550700080825380345883105195752839361334678871895593932042077071912003898512892', '-0.000000000000000243005506005006402417515376832343523332630896909903184829530830475018969291441011676442441851780434411064620687143851249145363683686213768587121', '227954506189379566132.353104956505414509365483317740028207670869422726484749907137678407334745753105658448917898118381527765265339419955440224579312856148750854636316313786231412879');
    t('7973988940224355015214336919379953485029536', '-46903881814522001507986816.99248409698766603253677611753397276067931254075552578', '7.97398894022435496831045510485795197704271900751590301233396746322388246602723932068745924447422e+42');
    t('-28179553528461976563', '248852934964030510956645790197846644600695689011471340357218184219306881752129281687476727', '2.48852934964030510956645790197846644600695689011471340357218184219306853572575753225500164e+89');
    t('-0.00000038749442098332718476018064732023168357745188079514035581267126125927216662843067618614699072618739224559403486391045982462716534370262', '4925006594013147257537583640291677799280156031827105', '4.92500659401314725753758364029167779928015603182710499999961250557901667281523981935267976831642254811920485964418732873874072783337156932381385300927381260775440596513608954017537283465629738e+51');
    t('15886467257912495.75689320719759657353210920879083109402692609', '-1849440876713050806208487397730416627460.30935631710582208341731553877239172695260308652973787159595347848456382322348819127147788006459039327206090843', '-1.84944087671305080620847151126315871496455246310990822550988520632998156063292567699652973787159595347848456382322348819127147788006459039327206090843e+39');
    t('-5026233483715918355642138258166736347547068019235638147637409254751306580678549839', '0.00000000000019779052536990808660692', '-5.02623348371591835564213825816673634754706801923563814763740925475130658067854983899999999999980220947463009191339308e+81');
    t('0.000000000000000000062846132155934866382475838803414329167925640917238247929247187597144605858850', '-28375804842138366819355751075050288391048198685203076654182288294782980828652856275283802825108048295625084622829828083365524827680002246249410431', '-2.837580484213836681935575107505028839104819868520307665418228829478298082865285627528380282510804829562508462282982808336552482768000224624941043099999999999999999993715386784406513361752416119658567083207435908276175207075281240285539414115e+145');
    t('-48863470201397.143927278109695419367309919023804132', '-168361072783150231412254024286758558795677828805121174243492869928073465709719014.023363549742186056665143055249248475718140936364', '-1.68361072783150231412254024286758558795677828805121174243492869928122329179920411167290827851881476032452974273052607718140936364e+80');
    t('0.0000000000000018997661230877089423253849452578256073300966468605209561343912897439707821108359408910538145364970969403440848507153309', '-0.000385522197', '-0.0003855221969981002338769122910576746150547421743926699033531394790438656087102560292178891640591089461854635029030596559151492846691');
    t('865008.237475', '-141572245488815215559.668813721278969595937436750693180747505198056296244191362365981067', '-141572245488814350551.431338721278969595937436750693180747505198056296244191362365981067');
    t('4531430332.974369182979674290738630476176408845410151617472188112662759', '-0.000000000000002990088033905170812634080622222063618156981784899967819720909679583969232226858480299674669693921077115362022886', '4531430332.974369182979671300650596571005596211329529395408569955680974100032180279090320416030767773141519700325330306078922884637977114');
    t('1713604693153132011650052009647770010715862661891416010.39872031838645104', '-5209348059320160498592207950890250846.48', '1.71360469315313200644070395032760951212365471100116516391872031838645104e+54');
    t('298.64569377145669370010760887428305419049211767076463966663182890471105450765507454904166021', '0.0000000513282401911235340888997773718547011660029236548742535384576211684698905846807631716329563685023698', '298.6456938227849338912311429631828315623468188367675633215060824431686756761249651337224233816329563685023698');
    t('153158125588135275971624448998303827924375983937687494263.6', '522574.2172656390720935407437590923430209583762529523179633299440845173022407980283974345034782042', '1.531581255881352759716244489983038279243759839376880168378172656390720935407437590923430209583762529523179633299440845173022407980283974345034782042e+56');
    t('-37584369371625804653778917725.04710852052873625022089539878880649643091488064967', '387289160645786908822166966661304359256521.40416120883', '3.8728916064574932445279534085665058033879635705268830126374977910460121119350356908511935033e+41');
    t('333145493405211943110181488709393444800555889546483.25', '337039689069037487854189149366025543071617091671893617361194217844843723876472794060690078788349535879526770512127905359.772', '3.37039689069037487854189149366025543071617091671893617361194217844844057021966199272633188969838245272971571068017451843022e+119');
    t('-0.00000000000000000875947840124838794446714154793538138133101442657807977884554262860149896878671504', '6121126413234312706434650834319645665575176210124040301876334782352.2701746892166273345074659234578320327195185867786496894475416', '6.12112641323431270643465083431964566557517621012404030187633478235227017468921662732574798752220944408825237703884326830811652717342192022115445737139850103121328496e+66');
    t('-27101285266584198282389701705604910688467857152047550105271200946392221564235258269.0800011597617806021545806843947528587392169', '-93984277.78258808561', '-2.71012852665841982823897017056049106884678571520475501052712009463922215643292425468625892453717806021545806843947528587392169e+82');
    t('-19256051893600692288537586.59353458509306480025521794532051383696889407478585008231545758785', '-8963065351854.7878237484328880211387666531886258820788775118', '-1.925605189360965535388944138135833352595282139398459850913971904777158658585008231545758785e+25');
    t('11569497283022449921394189469855978299381812519350940701313946487504335257098102797076761168675477401606', '354878912159213729324887521064557506231700037558958924247470744332285142081086865543601683607743.711010177974', '1.1569497637901362080607918794743499363939318751050978260272870734975079589383244878163626712277161009349711010177974e+103');
    t('31410760334069399540603136630010363.370516', '-0.00000000000000000250028865888009060099213589080181239288708801576170886559433463009608557725740080566463813772802697909277043093845092263791985', '3.141076033406939954060313663001036337051599999999999749971134111990939900786410919818760711291198423829113440566536990391442274259919433536186227197302090722956906154907736208015e+34');
    t('48450334100677709.292858595098179054162833212046954235133498994154015488631434964418926099', '-5452278790911076.131563791561554078354880486603279', '42998055309766633.161294803536624975807952725443675235133498994154015488631434964418926099');
    t('-726362472599493167754185550127976323995131102302190474448.022897051088265272546787050801660', '-26697.1300027', '-7.2636247259949316775418555012797632399513110230219050114515289975108826527254678705080166e+56');
    t('-5169217768018238832429254316846393556631723341237749333387854823.66306235977379827807976904396539087620041043309411525365990764018230223245161', '21048346121126363354727928959697501228648890358940.20948083990111835371727450225140678862878629117909606507623912415702402888341685852', '-5.16921776801821778408313319048303882870276364373652068449749588345358151987267992436249454171398408757162414191501918858366851602527820356819314148e+63');
    t('0', '3880335331987075756173782777964278144918139052756795465605564193503574084038075', '3.880335331987075756173782777964278144918139052756795465605564193503574084038075e+78');
    t('1471118130122845147276641392737353012928182703107626659491246625014486338076282524169910132047730182612396587213027711271216.07461', '-9022200078219.61142095898083863316225395343911361206316438521617777', '1.47111813012284514727664139273735301292818270310762665949124662501448633807628252416991013204773018261239658720400551119299646318904101916136683774604656088638793683561478382223e+123');
    t('1961360640267', '-8237317626242593351740131071931066270126645544302536441310151862831895121602611.6685378785733350998', '-8.2373176262425933517401310719310662701266455443025364413101518628299337609623446685378785733350998e+78');
    t('250056193902937472244979.030455795040281239421914909457299497068011838527472729233340186411522', '-389637478200780623396903050063520125697228676163384342985662391907774888793755259018717015963204238731047741576614654499.6', '-3.89637478200780623396903050063520125697228676163384342985662391907774888793755259018717015963203988674853838639142409520569544204959718760578085090542700502931988161472527270766659813588478e+119');
    t('-41.0667685182669426103291702409386770119233174954666626675833072533459819908153', '-24683428468133023718789488350081878137388951325390325606955366999359.7544', '-2.46834284681330237187894883500818781373889513253903256069553669994008211685182669426103291702409386770119233174954666626675833072533459819908153e+67');
    t('-1465078045399636479827859198546113217231061854355552775', '286781605726130466373594446109778466785174261888370270594913242', '2.86781604261052420973957966281919268239061044657308416239360467e+62');
    t('-0.00000000000000004372082675620531277773860157001154109814365737979121603370264972266779787151695587487529163780797491077618453622889827822', '0.0000000000000000115010056472096691762701979992534492116764302820728655795640758846469523814776292792892505', '-3.221982110899564360146840357075809188646722709771835045413857383802084549003932659558604113780797491077618453622889827822e-17');
    t('-5915475824828715769525622713448197545310987686571625215824865265460902144119719.60667712865151405621927923268842958349404793882328736294067066', '8289599618691777653744294128182319598849442999.3869210944639310536363962387071546926', '-5.91547582482871576952562271344818925571136899479397147153073708314130329467672021975603418758300258288299398127489089404793882328736294067066e+78');
    t('-6781589283952862541042634770259255399053423179553784209358601916774267083862536888853055882080202198759767010533235152604760503522710009', '0.000000000162', '-6.781589283952862541042634770259255399053423179553784209358601916774267083862536888853055882080202198759767010533235152604760503522710008999999999838e+135');
    t('-7401754762477043478410731850924229267152.1385806493837710954211647499789972995005461407065862284241906466005572091609381993770', '6814772121230277085632920582844817329083526354461024137801607310774405037993936581203797318438915758471288490759159235041741129', '6.814772121230277085632920582844817329083526354461024137801607310774405037993936581203789916684153281427810080027308310812473976861419350616228904578835250021002700499453859293413771575809353399442790839061800623e+126');
    t('-8', '21559462366454603856476372397626468240404955163730871821082532003169939666641.3471758', '2.15594623664546038564763723976264682404049551637308718210825320031699396666333471758e+76');
    t('-10740658919737638642657973180452168052201470445', '0.0000000082551674495554316423778277737071293814860377807246652498052504231443776872267372379', '-1.07406589197376386426579731804521680522014704449999999917448325504445683576221722262928706185139622192753347501947495768556223127732627621e+46');
    t('95996905118671016719266871729.520398129746', '1424664524272137031863748414508.04407607261068488341354804735012808909858790022643537811522935701269430843913849', '1.52066142939080804858301528623756447420235668488341354804735012808909858790022643537811522935701269430843913849e+30');
    t('7545686522420.04', '-110343136.2377175', '7545576179283.8022825');
    t('-3778460127635156232285998080.7598066400530666299808051012', '-0.00056022982056213327166370574884842481363234', '-3.77846012763515623228599808076036686987362876325246880694884842481363234e+27');
    t('-205.787', '40023148.03084250896', '40022942.24384250896');
    t('-10469.58581784685565292886210019593298608566213', '0.00000000000000055769617173', '-10469.58581784685565237116592846593298608566213');
    t('-1697421104198498565372', '130856814871284869392577271881365150502943114887052960993846', '1.30856814871284869392577271881365150501245693782854462428474e+59');
    t('-0.000000414035245121759801970634606896989047228634707532', '-38870447038068433224878605004808090705226214661500574171036129739107587668370820311143094.957', '-3.8870447038068433224878605004808090705226214661500574171036129739107587668370820311143094957000414035245121759801970634606896989047228634707532e+88');
    t('-2028389003568384286508207197206771601505013066674015630980.40648010665804431311941439248759845544570040200920378245684409301143522362121397976187113415', '-0.00000000000000000001828002844616325557520485512882126384242054987542132167273332845067757', '-2.02838900356838428650820719720677160150501306667401563098040648010665804431313769442093376171102090525713802504629926464288685654529394730821254870415e+57');
    t('-0.00000188443084941099579692438787131849288418100707817192098239393848434584554053801905703996798261121316300897664557', '-3594409179355959594852497571834744071660256911.32474274552929134136288086348918314067885678172370772397', '-3.59440917935595959485249757183474407166025691132474462996014075235867778787705445917174096273078589589098239393848434584554053801905703996798261121316300897664557e+45');
    t('-0.85273', '-279017792692170227516916053414297917923905837082261296273831887211350843745548269287411294.07131470147929582735586577119', '-2.7901779269217022751691605341429791792390583708226129627383188721135084374554826928741129492404470147929582735586577119e+89');
    t('1465314670123686042148720875703178202039648.513007703872541159651633499734046400788', '777575543578333746326.8333028060', '1.465314670123686042149498451246756535785975346310509872541159651633499734046400788e+42');
    t('-26920347118432602525813368030981387.1405710723', '-9.7088124301334057558783763899332243970338509361338006057662744040169648799290478801660095182592839804430939342926', '-2.69203471184326025258133680309813968493835024334057558783763899332243970338509361338006057662744040169648799290478801660095182592839804430939342926e+34');
    t('1316027450661167908370945255095582684.489766006179613245381691072', '-27247880103214807605061248065261158542.238847334604', '-2.5931852652553639696690302810165575857749081328424386754618308928e+37');
    t('0.00000000000000060630789744883065327236848700992137069065181792024986218307439791524854229166347681155', '23583422975999413573947442313781193552150628103226534113902', '2.358342297599941357394744231378119355215062810322653411390200000000000000060630789744883065327236848700992137069065181792024986218307439791524854229166347681155e+58');
    t('-270309750628324546215936875977251097460648729.2740106846988640562244700895457240150700468834022435265506238503209610561582721812729822740951287073436', '6623510422802645083643613990747833105477640970454392196256901453016184756585217878929893750.897051821348362132957036238424470', '6.6235104228026450836436139907478331054776409701840824456285769068002478806079667814692450216230411366494980767325661488787459849299531165977564734493761496790389438417278187270177259048712926564e+90');
    t('56833947334627412648354391179890', '-0.00000000000000020974557897099293264846', '5.683394733462741264835439117988999999999999999979025442102900706735154e+31');
    t('-32374924556195581487094651860979067186648013640229524477572717027565077414917194641088084918165166494.4007630', '0.00000000000494296301119272380546991072751573073742664686905799391552910274849845882330521177213047667', '-3.237492455619558148709465186097906718664801364022952447757271702756507741491719464108808491816516649440076299999505703698880727619453008927248426926257335313094200608447089725150154117669478822786952333e+100');
    t('-1797058201757575606389544872285070938241126244.769314657904583268612759343503712876640776918509', '-123578693378839714952814254501905650483998928961025522981908260845901294079261880178955413386', '-1.23578693378839714952814254501905650483998928962822581183665836452290838951546951117196539630769314657904583268612759343503712876640776918509e+92');
    t('-225494018080889118537669713.9577313306531049801615408808777093224321509314', '-682519588810576318453197726941395968412861.368346014111618610229581846002026209517358035441594633629722053374906068945767849660172087', '-6.82519588810576543947215807830514506082575326077344764723590391122726879735531949508966841594633629722053374906068945767849660172087e+41');
    t('-146483103623737636574187312839618731854', '-164095488530068730324672.278737858685739790237796055945', '-1.46483103623737800669675842908349056526278737858685739790237796055945e+38');
    t('288065905331021752942013707735111579224784706910.9694544539500946696884975171333618909754943414924752306735638546017809645829364174230805972', '-1879997868031626.13530883243794860502747913408966454', '2.880659053310217529420137077351096992269166752848341456215121460646610183830436973509754943414924752306735638546017809645829364174230805972e+47');
    t('-30585397102026875650261239591045069897098845982837987425500176089534774480799.196611528585829637554', '-8.2', '-3.0585397102026875650261239591045069897098845982837987425500176089534774480807396611528585829637554e+76');
    t('0.0032143', '-0.00000327364505987922707888345463293127602039856777788995949286363555265326340970495', '0.00321102635494012077292111654536706872397960143222211004050713636444734673659029505');
    t('0.0000000000000122089', '-0.00000000009606197314249053393037900593870599194195536081517004235811304298818227827198075675001652859716008205203077623130159549042380478073889176477922188', '-9.604976424249053393037900593870599194195536081517004235811304298818227827198075675001652859716008205203077623130159549042380478073889176477922188e-11');
    t('-1519120.89697395603485201266905849077718172383773866152054', '-1252690569967.49008', '-1252692089088.38705395603485201266905849077718172383773866152054');
    t('135272242707147130668128741572727753877823517.5267897682843968097498958339590924822841849811607010672', '5323052067036940514642933493723371314528945383203996481350033868198902738290466758895359108447.5', '5.3230520670369405146429334937233713145289453832041317535927410153295708670320394866492369319650267897682843968097498958339590924822841849811607010672e+93');
    t('1373517886847509715943778293771477989783556837632514732917942', '-0.0002918853', '1.3735178868475097159437782937714779897835568376325147329179419997081147e+60');
    t('-7.74', '-0.677942863330392341680315597792114620456629945201714481742115371479319069952739078864753759584', '-8.417942863330392341680315597792114620456629945201714481742115371479319069952739078864753759584');
    t('0.000000000000020414435064555641261166870351580734289075091673294774203415476224497625497586758753668', '-295256622638190096478809549470566', '-2.95256622638190096478809549470565999999999999979585564935444358738833129648419265710924908326705225796584523775502374502413241246332e+32');
    t('0.00000000043262197364654223753368765660064439474065876064265664457730559389', '7732179149449785953403196073740747199538706014956074135203031279754730991083612056065.08514009015794758031660841', '7.73217914944978595340319607374074719953870601495607413520303127975473099108361205606508514009059056955396315064753368765660064439474065876064265664457730559389e+84');
    t('-31545120553641346250429.2041019063650382985', '-0.00000000000000322497856922376578447917222257195', '-3.154512055364134625042920410190636504152347856922376578447917222257195e+22');
    t('-0.00000001491640137712266436315173456229606458713951581451514404607758964925714049589638484422967721927256', '875664170041898483278650.4', '8.7566417004189848327865039999998508359862287733563684826543770393541286048418548485595392241035074285950410361515577032278072744e+23');
    t('0.000000014725515297168007', '-0.1721217425079357563094435985505347277188701680317422409696115500647420231', '-0.1721217277824204591414365985505347277188701680317422409696115500647420231');
    t('-0.0000000000000000074805926405562', '-32346407643566307755196.3614356592764197829600176806496', '-3.23464076435663077551963614356592764197904406103212058e+22');
    t('-313255500747185422178551717566317427599882739101058623548833613806257.69583741347200945021282349083047007656822121171291826489554745460530313085870235', '-0.0000000000000277730681342000948755480478956572325828011175067392454201615642985418337229601771222072988252633636962852384552843309099649046742455014449802537116792', '-3.132555007471854221785517175663174275998827391010586235488336138062576958374134720372232809576909253456246161168689455010660130541938507232924230008918337229601771222072988252633636962852384552843309099649046742455014449802537116792e+68');
    t('796875527749558859721941966594505581298971405919740366253609646289883816950441012', '0.000000000000000106051762207090679888452342439518714917029053485988097086863407642961157123252080954014447191265506482837715480', '7.9687552774955885972194196659450558129897140591974036625360964628988381695044101200000000000000010605176220709067988845234243951871491702905348598809708686340764296115712325208095401444719126550648283771548e+80');
    t('-2799308221255727632154376162818591439464534253442269002690075449594707354769913622', '-2653089999757580649494973458723447833943060689063627290.948', '-2.799308221255727632154376165471681439222114902937242461413523283537768043833540912948e+81');
    t('5230.6', '-1457159088427391036497085292606459576.080762789887425314064087931228782784478984069210492316', '-1.457159088427391036497085292606454345480762789887425314064087931228782784478984069210492316e+36');
    t('2265111540331847.44385', '0.0000000000000000026399730647357391221240416795810255729781114791438489591810343937780321771723301295667448186691762052166888791236524529182049438', '2265111540331847.4438500000000000026399730647357391221240416795810255729781114791438489591810343937780321771723301295667448186691762052166888791236524529182049438');
    t('-0.000000000000000000130351174611495679446350395997550522969873358970359485781', '6837379692457094204055811341371130548614625165355.865146', '6.837379692457094204055811341371130548614625165355865145999999999999869648825388504320553649604002449477030126641029640514219e+48');
    t('7777457612952579632757962073721379887227173983203369078658939826377158630604554480732826290762', '738745470.90386876186959774624', '7.77745761295257963275796207372137988722717398320336907865893982637715863060455448073356503623290386876186959774624e+93');
    t('2091587164056284398722371839269632038542800738624345517942031403189897878173896765128677273428424903855243972440613077101074439732042713569', '-4951053013270976201467725287896995979570086801989189606256031245422534338962187224417', '2.091587164056284398722371839269632038542800738624345512990978389918921676706171477231681293858338101866054366184581831678540100769855489152e+138');
    t('15596526663192771382682194043241293744360284406119405225927225287207668035938290350623.0136029816344460485336790911173723056262086400741246575531343349', '-0.00000929115562', '1.55965266631927713826821940432412937443602844061194052259272252872076680359382903506230135936904788260485336790911173723056262086400741246575531343349e+85');
    t('-0.143235913932920000943491996904423637', '105465669771635331021255530853326638414654835668447488251.44604956381929966', '1.05465669771635331021255530853326638414654835668447488251302813649886379659056508003095576363e+56');
    t('-131.66335836273650457236601594959663422039552341367012898208021864544080993', '0.000000000102821561249286522291590439126847620376937400555244959227688218488059751276284980726909295138051809044', '-131.663358362633683011116729427305043781268675793293191581524973686213121711511940248723715019273090704861948190956');
    t('-8211501623.467468766754729150340637983678978062348865062714034523840862885354594840', '4138.57064149204825240856224917824913', '-8211497484.89682727470647674177838880542984806234886506271403452384086288535459484');
    t('448392812.1018844354312957', '0.000001316332446359801420119858000840615513663646321599278426150890005786837699907597501844047946311701498712958992807897', '448392812.101885751763742059801420119858000840615513663646321599278426150890005786837699907597501844047946311701498712958992807897');
    t('-0.0000515081990743416226512970055332416111901', '9721676157492236482889538449332613888622823236564458642190.9068670871984057788931368828097465243944915994782715793742390978277016751898427038', '9.7216761574922364828895384493326138886228232365644586421909068155789993314372704855858042132827833014994782715793742390978277016751898427038e+57');
    t('0.00000016348993711366066763304174336589379906596596798355118', '27092941593289173149253532518.630793971858294366753', '2.709294159328917314925353251863079413534823148041366763304174336589379906596596798355118e+28');
    t('1750188741743624090823079726633157080090405425107855229635890.59733375', '0.00000000000000253442', '1.75018874174362409082307972663315708009040542510785522963589059733375000000253442e+60');
    t('-120.632964973', '5599333570130562863508293629418111213773068135437215884309301113811346990002242140', '5.599333570130562863508293629418111213773068135437215884309301113811346990002242019367035027e+81');
    t('-12956.8', '-28125616598.519875442790896131080979434706515929141364365696194890485115866976051969974226520962106271082896', '-28125629555.319875442790896131080979434706515929141364365696194890485115866976051969974226520962106271082896');
    t('0.000000291488949451719436740561621135104503110544381471384117075881655619830678129379740688277538395366270063296614253757547811832765458961027261', '742031645768480260456626682374347039419606586767893047218475327818790868744906.4174591102001653101442893099824528369555', '7.42031645768480260456626682374347039419606586767893047218475327818790868744906417459401689114761863726050544073972060003110544381471384117075881655619830678129379740688277538395366270063296614253757547811832765458961027261e+77');
    t('-87648513454369557979997099620411798538786483894671.07140', '-1', '-8.76485134543695579799970996204117985387864838946720714e+49');
    t('11680.216847571', '15545171141903075635187317512728638503.5120154558326692202036513424672740379112385241218565405732440132', '1.55451711419030756351873175127286501837288630268326692202036513424672740379112385241218565405732440132e+37');
    t('217933206105480443145202005696333545459278284908.49635092233713353554300078449954457018472363340', '851760835565720226699101062761084114408263270495330844773733382141014174507794413266810316.1227643422652856425756237116285709', '8.517608355657202266991010627610841144082634884285369502541765273430198708413398725450952246191152646024191781186244961281154701847236334e+89');
    t('131096.71', '-92311622283713248650111150643196971194314788501758170221195801273003488.29608754610346056452229667146067507425547000551040945788003726', '-9.231162228371324865011115064319697119431478850175817022119580127287239158608754610346056452229667146067507425547000551040945788003726e+70');
    t('-7.3', '-0.0000000091386734958004013759891573873629383131', '-7.3000000091386734958004013759891573873629383131');
    t('-83570037032769234234524544857963', '-16329401826664066011232.5825604925166033769207446067722340786326', '-8.35700370490986360611886108691955825604925166033769207446067722340786326e+31');
    t('207724189276.2354966467766137168630074255', '368.36464299671427627280350157241857056036300267791934105571800636754609729339278308276622409777940247912804080708853009235428453986', '207724189644.60013964349088998966650899791857056036300267791934105571800636754609729339278308276622409777940247912804080708853009235428453986');
    t('1036451349506118936704271313996579271029057412.5954933332585621063133644518747511868339417', '1919224338475644109', '1.0364513495061189367042713159158036095047015215954933332585621063133644518747511868339417e+45');
    t('4594184.672287496737761641402840568569137816292501654869263256716904015818', '9', '4594193.672287496737761641402840568569137816292501654869263256716904015818');
    t('50922584112695986965179537750579107265371590444561808302376506246479343493.0485900097493874734', '-105588622877722030471490473554330788644165258321036318195143132112931', '5.09224785240731092431490662601055529345829462793034872660583111033472305620485900097493874734e+73');
    t('8762733230291549920959.461299029519859816221596648161', '-0.0000000000000031719158', '8.762733230291549920959461299029519856644305796648161e+21');
    t('-0.000000000001088450644628700841462499123795267689859321764189574607336877911119745839851152785101218885203868', '225068092611409981730971194567763438117689044018909866535690426664845083547381487510884944286310222323340655938142842019712684', '2.25068092611409981730971194567763438117689044018909866535690426664845083547381487510884944286310222323340655938142842019712683999999999998911549355371299158537500876204732310140678235810425392663122088880254160148847214898781114796132e+125');
    t('-0.0000000739096483131943', '0.0077245546142820934802776687821282016447278885', '0.0077244807046337802859776687821282016447278885');
    t('13773756.65', '82341713701089392.952', '82341713714863149.602');
    t('0.210858539297226184666150871507845048599002989549496974738305', '893397778957253392646834827705840161788170516925929814290917857496573588954994819301103.84678150026', '8.93397778957253392646834827705840161788170516925929814290917857496573588954994819301104057640039557226184666150871507845048599002989549496974738305e+86');
    t('-311598106485014524446810253701.9', '-273597385676217579425078047906355.35858548639357', '-2.7390898378270259394952485816005725858548639357e+32');
    t('0.0000000000000010949761538307759705645612635463870582102620259', '-642669739487957339653187054422612017200108895155999822088234612866942591002244842.544328938390312037134789574351984286413492175445739315', '-6.426697394879573396531870544226120172001088951559998220882346128669425910022448425443289383903109421586357435760137218522286290586811047379741e+80');
    t('-1413496343761296278143773037357799924448219743939.91034833304743349454253527380297135637064998230', '-0.0967154985326414596487190403709602844792653613', '-1.4134963437612962781437730373577999244482197439400070638315800749541912543141739316408499153436e+48');
    t('88324430808176237552938.25850832460', '-1355506260987917851790849984926562224803792482721611489882549294623263720214603522264447050471402562893267497494827198102938336997669.605991', '-1.3555062609879178517908499849265622248037924827216114898825492946232637202146035222644470504714025628932674974065027672947620994447313474826754e+132');
    t('1913616274506284536238001245388057898244388034.32351451667730419851', '-0.000000000000007988642436925641812976903901681868680726375828237077834585228067490119284435811740642127567200765018765914463975898057', '1.913616274506284536238001245388057898244388034323514516677296209867563074358187023096098318131319273624171762922165414771932509880715564188259357872432799234981234085536024101943e+45');
    t('-0.00016620663104623655416647045', '-12686797041.8875795703189682', '-12686797041.88774577695001443655416647045');
    t('-955318682044891602024512908305401581438484362308395.3064608014', '2405264255243858252389318787930524.596646933647908', '-9.55318682044891599619248653061543329049165574377870709813867752092e+50');
    t('38251344747049710395611359429944', '0.000000000025044307069557002354176630069758745187110935106932295261690460', '3.825134474704971039561135942994400000000002504430706955700235417663006975874518711093510693229526169046e+31');
    t('6556082.32', '-0.000000000000001855587562856033456366238361224693109198376760772204394200539703419686910321614077786337510464951', '6556082.319999999999998144412437143966543633761638775306890801623239227795605799460296580313089678385922213662489535049');
    t('-601066282116.773646944861803892913655', '0.45869929801033616081412998252506077240747661632297824035019059504580212743644756970631731956591592148421555634', '-601066282116.31494764685146773209952501747493922759252338367702175964980940495419787256355243029368268043408407851578444366');
    t('-23.6504678041285333307941', '9322173289440937942221726663585040711544412727245532618749747902349893601059242945545713822289661475003325571215052971921099787751298.5', '9.3221732894409379422217266635850407115444127272455326187497479023498936010592429455457138222896614750033255712150529719210997877512748495321958714666692059e+132');
    t('13954.05335659426859124804949201546230433799858282911276550573060408965278409946479104625725605515184113476163', '54942938376898283461251972594178875731793459692.69288668376868', '5.494293837689828346125197259417887573179347364674624327803727124804949201546230433799858282911276550573060408965278409946479104625725605515184113476163e+46');
    t('38338476138349711098378.3943108022013227510', '184968761212298160660521125177275265078971984111374634082506646317710944619077959845438.025316842804835238303934013763002232693292615810218', '1.84968761212298160660521125177275265078971984111374634082506646356049420757427670943816419627645006157989303934013763002232693292615810218e+86');
    t('-535317800652152416.666011782506679594256929333292104206593745014', '206357700474622846234363174929527883904699924195458.1468414719961788534413928358700225430521698882933015922823192348467761048', '2.063577004746228462343631749295273485868992720430414808296894894992591844635025779183364584248742933015922823192348467761048e+50');
    t('2833257851650449534329564356827343715273182823625486392869199675024054992102672521.4119632200042590623653138732343284', '27874145222289935847607844819151291', '2.8332578516504495343295643568273437152731828236533605380914896108716628369218238124119632200042590623653138732343284e+81');
    t('0.00000000000000000002618552546853233266995429732175655051680150038461910732084518605757265', '-139.1176780483072', '-139.11767804830719999997381447453146766733004570267824344948319849961538089267915481394242735');
    t('-5231728462708550604235813229', '-91296185221416978065707502278539916045872.674264172531093353580151543119', '-9.1296185221422209794170210829144151859101674264172531093353580151543119e+40');
    t('0.000000106213737285740538348043261083184649640660300492624731818517303939576491624096820480026083363433', '-22201770785094963881965679358073543814802495287033739904412088335159142995847539301013.995757400073858039256583163281728323829172117', '-2.2201770785094963881965679358073543814802495287033739904412088335159142995847539301013995757293860120753516044815238467240644522476339699507375268181482696060423508375903179519973916636567e+85');
    t('-67911351036760640240904249364595686607959826796143043932178763791112125413384969479427858233781973657171072664684275848926309593074.2736408437759912', '-77343795081059041036255490247057579', '-6.79113510367606402409042493645956866079598267961430439321787637911121254133849694794278582337820510009661537237253121044165566506532736408437759912e+130');
    t('-365663287930535233624529826657412624347925682105954594671581052959938622', '6954200.1293731825636', '-3.656632879305352336245298266574126243479256821059545946715810529529844218706268174364e+71');
    t('-86904523910181630995087116839233134949554297554466494568778841810992441.93482682', '8131622.282657081038690', '-8.690452391018163099508711683923313494955429755446649456877884180286081965216973896131e+70');
    t('-1625898821893719962579186943572639572628075158134', '154905121706840975019322074123957122201752614177688556257506327062930427205660262365762413912612.3699041018353', '1.549051217068409750193220741239571222017526141760626574356126071003512402620876227931343387544783699041018353e+95');
    t('7155707429106159594597753.602', '-8825753984190829367467677923527023814575827126015236448.1769432522585118457684798455299849721603516850', '-8.825753984190829367467677923519868107146720966420638694574943252258511845768479845529984972160351685e+54');
    t('14702524453269326207746839725456981418982455947784576163899708846627563947982587259694248527512065356328701857996383.8405344660118236145367576964488171', '0.000000000000279801481414509128', '1.47025244532693262077468397254569814189824559477845761638997088466275639479825872596942485275120653563287018579963838405344660121034160181722055768171e+115');
    t('-225100394111865994732205121145289659406.81347348992302868556770481345355033428867007077210213924805668907831310175342503282790874586923576524594', '16536197888541991489345013693415581039371231147607496617201758269219830229742518663209266982586909041114114', '1.653619788854199148934501369341558103937123114760749661720175826921960512934840679721453477746576375145470718652651007697131443229518654644966571132992922789786075194331092168689824657496717209125413076423475406e+106');
    t('-147016978159508669835046590756636450424769212453306265233290290589028698490660334843817522777298255009234097047902836152977047399695261588489020507996', '-3385444909290343226731953993543122186484062389074939550327086795043944171', '-1.47016978159508669835046590756636450424769212453306265233290290589028698490663720288726813120524986963227640170089320215366122339245588675284064452167e+149');
    t('-199818369850480446138557902023384546574063876954063983588740826564571725789062130098498070856521310723010198313678089493931618667410', '-82297155933721780656979816142942540102465515796252492877737040', '-1.9981836985048044613855790202338454657406387695406398358874082656457180808621806382027872783633745366555030077919388574642449640445e+131');
    t('1224211946370320242988310.539593783414725138865000500768803244328120208411774168803312677659248392891734554652728756344327699042687781768182371130', '146158963047904046913411919077533655315762900865.56618168609343', '1.4615896304790404691341314328948002563600588917610577546950815513886500050076880324432812020841177416880331267765924839289173455465272875634432769904268778176818237113e+47');
    t('0.00000000000000000008559294799519157774769508766803', '-0.551609783500231605091545012587226546044881570184707190', '-0.55160978350023160500595206459203496829718648251667719');
    t('-0.00000000000000364248749061745076846426096689761350484416821596341761277974108029120654192812240270659121286544404842359764995', '0.00000000000000012915182634025629462517104694343718453070590970719003893358456085974950708638362321445540916204635318582523370886863628537009101151163541', '-3.51333566427719447383908991995417632031346230625622757384615651943145703484173877949213580370339769523777241624113136371462990898848836459e-15');
    t('-0.0000012159213941933105485030856059051020630869271620812934680783745913', '435177388107137395111544818911495302729591053552420773207670273108531988732273993785286.67035351430698075129626580', '4.351773881071373951115448189114953027295910535524207732076702731085319887322739937852866703522983855865579857172969143940948979369130728379187065319216254087e+86');
    t('-244127359165.119239441740', '-156775332029133323786340200106246008327217172695025168660963117705504796367803174489432334620737960393.807345425858853639626141', '-1.56775332029133323786340200106246008327217172695025168660963117705504796367803174489432334864865319558926584867598853639626141e+101');
    t('-1.3', '38822623294205506.9130785519203', '38822623294205505.6130785519203');
    t('-0.000000000000000200529323920298995078835589947427959287114325689776790905715952540392301944664052487332084825933878060366283983109838980691230645368608610691238118', '8.9235511422835689686003636129516379856485298', '8.923551142283568768071039692652642906812939852572040712885674310223209094284047459607698055335947512667915174066121939633716016890161019308769354631391389308761882');
    t('-2957229910068789430753016717681830773467703355331693598573435320975440031813155744826634542489276009415208500102073', '1387116781565930070372821993801771.117', '-2.957229910068789430753016717681830773467703355331693598573435320975440031813155743439517760923345939042386506300301883e+114');
    t('1700002853156086652196589662796447571836354710587680189245495702990262116528669877736.14002354840708066274724201698188364950277252310355226408545', '0.00000000000000001040670152065846069679024081862566530541721990959996195085462554562276036783860750708035', '1.70000285315608665219658966279644757183635471058768018924549570299026211652866987773614002354840708067315394353764034434629301334172921756950266990959996195085462554562276036783860750708035e+84');
    t('275376444375379941033818605.172279107271998841', '479542807197295799803972994083414159220618900334356109151914563415527998484615170546961833405749594109884901428553537510172052.3945910', '4.79542807197295799803972994083414159220618900334356109151914563415527998484615170546961833405749594385261345803933478543990657566870107271998841e+125');
    t('201072.84', '-18291007075346088677522680107496617786379481993162852.5235791884', '-1.82910070753460886775226801074966177863794819929617796835791884e+52');
    t('4.79698005010389753070269243940420243222736581061677999244886345403385181646604209917629845674090797104700400712679893585', '9488800450.896868361907362310326', '9488800455.69384841201125984102869243940420243222736581061677999244886345403385181646604209917629845674090797104700400712679893585');
    t('1079602565226551350070617572381861809880919380670631.1941110380393840344419101968428407597315561768248982401339351384570356479736868707938705396478', '-976453379501315909711332105625801476050587869483628203480762777882592729043296523674472960763347999942046454132907563626456686905612777665215490', '-9.764533795013159097113321056258014760505878694836282034807627778825927290432965236744729607622683973768199027828369460540748250957318582845448588058889619606159655580898031571592402684438231751017598660648615429643520263131292061294603522e+143');
    t('-0.00000000644271271718069271020345607284106825430330344982447309294205280695455856', '-4416.748357493646036829616437', '-4416.74835750008874954679712971020345607284106825430330344982447309294205280695455856');
    t('3445711155679699718853304199017871443351956851656285424813892934989061182900239514362410627083188779604842513142', '-0.0000000016502205307974697933105166599651672169912797927812530032512206464857476224692850052', '3.4457111556796997188533041990178714433519568516562854248138929349890611829002395143624106270831887796048425131419999999983497794692025302066894833400348327830087202072187469967487793535142523775307149948e+111');
    t('-0.000000000000000001721243717528274638265294074343297487403293013692904865481482446596237956150738926135318544866402636735967915186120920316527028', '0.000000000000000436433664131032429923892680858273396689173735933956', '4.34712420413504155285627386783930099201770442920263095134518517553403762043849261073864681455133597363264032084813879079683472972e-16');
    t('1204447398244110346590693828919979918701287883959208424098228210909783249592980543967052564154', '-1.202', '1.204447398244110346590693828919979918701287883959208424098228210909783249592980543967052564152798e+93');
    t('-0.00000000000000000003511', '-36043841896699925', '-36043841896699925.00000000000000000003511');
    t('-3025613431812032994475405709985.7993469679039433736384733579074386422934426781633847239705968789457410845', '-1002588615531.9649364944823485872292540973546318938467102', '-3.0256134318120329954779943255177642834623862919608677274552620705361401528781633847239705968789457410845e+30');
    t('-0.00000034223095611049552775891228859686202901218738308167914660966601362250385709703952480432247', '-972934111420524554955644957601045537.987527746161528768501935193004724606168511203759505291457031083199089815', '-9.7293411142052455495564495760104553798752808839248487899746295191701320303054021594688837313617769286510343750385709703952480432247e+35');
    t('-260.8020033716610908017940221805155296272264320025030641582384857795184385939216818072400908848372378517124042819164098083213511102035', '17001.46833548258145536106307267277476995298424211430854042389081593596901026449283938670366734353012622907027133923538851172235700848608201622', '16740.66633211092036455926905049225924032575781011180547626565233015645057167057115757946357645869288837735786705731897870340100589828258201622');
    t('4753081575672367265214388.9329438810676699994575479313858093392877635113982018254535256873963800876321781902066650227606038529159973094772009', '-7324419798632309384717050.254499146068562829441638', '-2.5713382229599421195026613215552650008928299840900686141906607122364886017981745464743126036199123678218097933349772393961470840026905227991e+24');
    t('261907813073953576038800700421412.5334378919400078993318520596126658082471977395259716516663364', '-0.0000000086951723661290447823645735627390797806202562817264370379556867453', '2.619078130739535760388007004214125334378832448355332028072772480922455081179589057153699398993620443132547e+32');
    t('-21057062244163808361926294417077492805710.769766499111898332671', '-540838928.18483069140526251205066101016721497478267885509539520800221', '-2.105706224416380836192629441707803364463895459719051716084472166101016721497478267885509539520800221e+40');
    t('-158255.62893406', '-31604958588557863785879282950522895919205970023779482993880870538808826806472714286351962262535559188830287123905862547286466', '-3.160495858855786378587928295052289591920597002377948299388087053880882680647271428635196226253555918883028712390586254744472162893406e+124');
    t('-0.00872805303919760722958348618910955647808183701772080233955945973054140753844133709507372345122366634238497203383151041725907235916579449430863', '-101853444464243101382759799835868417551993254203707661817114852', '-1.0185344446424310138275979983586841755199325420370766181711485200872805303919760722958348618910955647808183701772080233955945973054140753844133709507372345122366634238497203383151041725907235916579449430863e+62');
    t('-45631392014.68038550439368025039999458620767416686726', '-16083812762280564885084.27247572', '-1.608381276232619627709895286122439368025039999458620767416686726e+22');
    t('-1439626983041987850913974184983601119473089147100564746927844789096157891.4841085238151384299', '0.000000000000000000031951455735478942002429224112959031269033600382871577234131203139', '-1.439626983041987850913974184983601119473089147100564746927844789096157891484108523815138429868048544264521057997570775887040968730966399617128422765868796861e+72');
    t('-3773188261867111267717109.48643521972869867743466506227122436366440807932802199071587048', '-10773530952405638442737529189330991723231726109739.439911386600615356849445155509285190', '-1.077353095240563844273753296251925359034299382684892634660632931403428411021778050955366440807932802199071587048e+49');
    t('5276177017268.35565399021279131413495227030075', '33380791872.126', '5309557809140.48165399021279131413495227030075');
    t('-128.032971413512325119332251541126003042400653717001958956874845360597144465880718042889612181', '3394406625963422663090321402154499517163312515368646.5438894836165998162937', '3.394406625963422663090321402154499517163312515368518510918070104274696961448458873996957599346282998041043125154639402855534119281957110387819e+51');
    t('-0.00000000000000000319057978729978801053247482850854', '-5970871936587764739377055460041397843015391161297703354420885181193398466623676.491266129426', '-5.97087193658776473937705546004139784301539116129770335442088518119339846662367649126612942600000319057978729978801053247482850854e+78');
    t('-1059900849522366748033951746353058622121879424133545530750.9681536282', '0.0000643997702370519405494192784251815344112451050443536256021694283534059757585612317456776719019257679238083053', '-1.0599008495223667480339517463530586221218794241335455307509680892284297629480594505807215748184655887548949556463743978305716465940242414387682543223280980742320761916947e+57');
    t('912258587155780697120724221920358016075695895559841256853912713603948747888.0315678411874212150256690949682329867739341', '2.256661', '9.122585871557806971207242219203580160756958955598412568539127136039487478902882288411874212150256690949682329867739341e+74');
    t('-0.0000000000010627444491654438', '413858262712309329471602287282236771662.23037198528863973813156909657677576430461650127105324676789469187837272211762396717083479560627202188', '4.1385826271230932947160228728223677166223037198528757699368240365277677576430461650127105324676789469187837272211762396717083479560627202188e+38');
    t('10889284485250482936848754210451881764666355121565381521609141220124805252627779969564893095316988849809.140913232560678725', '265462397841083205283689782772736396048751613685021523052966574884200782708580692276859125750962879382252839837', '2.65462408730367690534172719621490606500633378351376644618348096493342002833385944904639095315855974699241689646140913232560678725e+110');
    t('-3940354048038348812232523779780977.34694963308484986947203317656721412409336941345032192250794544448871277436178166955124911913191273620514614922051333', '108461342576678368692525128056407844311709242137736115728865741966953.265427848893134917162575927126158297967830952', '1.0846134257667836869252512805640784037135519409938730349634196218597591847821580828504769054275055894417387446153854967807749205455551128722563821833044875088086808726379485385077948667e+68');
    t('26059607473911021896148097458633904533525723836841783.8770871138672438759194942708994459838051487010227902', '-455763990053647411202608788023.11837212224235256867341260022328839083293408', '2.60596074739110218961476416946438508861145212280537607587149916248913072460816706761575929722146210227902e+52');
    t('7485939927194657722950.3195815053046002170111816290687951', '3027.358', '7.4859399271946577259776775815053046002170111816290687951e+21');
    t('-0.00000000000005599703632158865779999224097099964286077696620548833919840351030138988682760507', '3.5', '3.49999999999994400296367841134220000775902900035713922303379451166080159648969861011317239493');
    t('-0.0000000000000000653970485009455424706348378806364512481181501902433188498531335390353104396782305272068893617841048746913384690805807406227445668817807', '-173224891091130915632980803868987591469417896552426790538575965839932016256122849190', '-1.732248910911309156329808038689875914694178965524267905385759658399320162561228491900000000000000000653970485009455424706348378806364512481181501902433188498531335390353104396782305272068893617841048746913384690805807406227445668817807e+83');


    t('1.06832442449657e+9', '5.31512565221766313e+1', '1068324477.6478265221766313');
    t('-8.9672012052076147472375857e-20', '6.391778063682e+4', '63917.780636819999999999910327987947923852527624143');
    t('0e+0', '3.803634150690184e+15', '3803634150690184');
    t('0e+0', '-2.250919143163409794163082811e+1', '-22.50919143163409794163082811');
    t('2.8352844e+1', '-9.8421821e+10', '-98421820971.647156');
    t('-3.96169309707604251e+17', '-3.28373582e+3', '-396169309707607534.73582');
    t('9.436349291685128603888e-8', '-2.264108528075248e-4', '-0.00022631648931460794871396112');
    t('0e+0', '7.2004864761158514004448e+17', '720048647611585140.04448');
    t('-4.8717347274e-19', '3.61737685608086525157404817e+26', '3.6173768560808652515740481699999999999999999951282652726e+26');
    t('-2.168389290841e+6', '5.223e+2', '-2167866.990841');
    t('-4.38725314493094e+12', '1.40054785697916026635598226844e+11', '-4247198359233.023973364401773156');
    t('7.1223378147664003e-7', '3e+0', '3.00000071223378147664003');
    t('-1.0656660973e+10', '-2.97209463056314635542851541186e+2', '-10656661270.209463056314635542851541186');
    t('1.2883277094e+1', '-5.54548048203261067457185e+22', '-5.5454804820326106745705616722906e+22');
    t('-1.504e-14', '2.71e+2', '270.99999999999998496');
    t('4.7287555806698e+13', '-3.5935329574452318e+14', '-312065739937825.18');
    t('5.4e-15', '-8.35468469007e+0', '-8.3546846900699946');
    t('-1.735565e+4', '-2e+0', '-17357.65');
    t('-6.89525e-18', '6.7229852843e+10', '67229852842.99999999999999999310475');
    t('-2.63508402e+2', '-2.478965e+4', '-25053.158402');
    t('6.4909990971742952e+9', '8.633411384049431980845536589e+27', '8.6334113840494319873365356861742952e+27');
    t('2.87817691607168594282e+13', '3.695e-1', '28781769160717.2289282');
    t('1.2558509833653622e-12', '9.46079802e+0', '9.4607980200012558509833653622');
    t('3.2795331525179582319e-17', '-9.431388519232451406922e+21', '-9.431388519232451406921999999999999999967204668474820417681e+21');
    t('-2.7943785679585251935425773745e+1', '1.177045839871050879e+18', '1177045839871050851.056214320414748064574226255');
    t('-6.73086857309438227946149e+24', '1.71905552920711054538516e-15', '-6.73086857309438227946148999999999999999828094447079288945461484e+24');
    t('1.9905292e+5', '3.0839223106e+3', '202136.8423106');
    t('8.31048846338017e+14', '2.83123673288364500197794e+22', '2.83123681598852963577964e+22');
    t('8.36725e+4', '0e+0', '83672.5');
    t('-2.110811501755e+8', '1.646035246034983555808013262e+16', '16460352249268685.38258013262');
    t('8e+0', '-8.45386934e+4', '-84530.6934');
    t('-1.5014313151e+11', '-1.3943193159404694868e+9', '-151537450825.9404694868');
    t('-6.19826535590383384678e+19', '1.45912696e+4', '-61982653559038323876.5304');
    t('-2.30128559243369e-20', '-1.74358e-1', '-0.1743580000000000000230128559243369');
    t('7e+0', '2.9175e+4', '29182');
    t('0e+0', '8.074398952e-13', '8.074398952e-13');
    t('-1.10169992553226e-6', '-1.489521419796568135223154e+3', '-1489.521420898268060755414');
    t('-1.2847222948669e-8', '6e+0', '5.999999987152777051331');
    t('-2.0748595e+6', '1.7737013267831817855508474e+17', '177370132676243319.05508474');
    t('-1.3e+1', '-5.537580936457497529591215129e+27', '-5.537580936457497529591215142e+27');
    t('1.53041098285404062449166636e-20', '-1.09698519e+5', '-109698.5189999999999999999846958901714595937550833364');
    t('-9.0549157165169081301152736658e+13', '1.71861116418514e+4', '-90549157147982.969659301336658');
    t('-6.84667133296448035e+4', '-1.2552e+3', '-69721.9133296448035');
    t('-3.38133320737533874e-3', '-3.6195193454909046017940982507e+23', '-3.6195193454909046017940982845133320737533874e+23');
    t('5.28063684109958995e+11', '3e+0', '528063684112.958995');
    t('5.6486677840434364930096077898e+6', '7.7345049656351983276666e+2', '5649441.2345400000128423744498');
    t('1.071698848e+5', '8.09702667e+6', '8204196.5548');
    t('-2.56438e+1', '-4.7752581981e-15', '-25.6438000000000047752581981');
    t('1.11255906527629e+3', '0e+0', '1112.55906527629');
    t('5.29214639434e+5', '0e+0', '529214.639434');
    t('1.907665907177150817777618e-16', '3.996484323e+1', '39.9648432300000001907665907177150817777618');
    t('-2.284817838850737484e+0', '1.70660375373e-10', '-2.284817838680077108627');
    t('-2e+0', '1.2057236765392598216859e+0', '-0.7942763234607401783141');
    t('9.5417e+3', '1.19709798011e-10', '9541.700000000119709798011');
    t('-2.1e-6', '1.73249589179219564051e-11', '-0.0000020999826750410820780435949');
    t('3.33966813434843e+8', '-4.33492101363039e+8', '-99525287.928196');
    t('-7.6863012322962e+0', '5.40256e+5', '540248.3136987677038');
    t('1.893278638203e+10', '-2.79127597942747e+9', '16141510402.60253');
    t('8.708419184498120859374e+8', '-2.125e+2', '870841705.9498120859374');
    t('-3.79886239868039785166793949e+26', '1.9e-15', '-3.798862398680397851667939489999999999999981e+26');
    t('1e+0', '-3.2066099723526e-8', '0.999999967933900276474');
    t('-2.101824983e-20', '3.3833950371681e+13', '33833950371680.99999999999999999997898175017');
    t('-6.1088406176222166146368e+22', '-2.00280388042170752227327721005e+18', '-6.109040898010258785389027327721005e+22');
    t('-2.0983e+0', '-1.1343932020174e+5', '-113441.41850174');
    t('-2e+0', '-1.19204063249021e+4', '-11922.4063249021');
    t('3e+0', '-2.989e+0', '0.011');
    t('-1.5828496509e+0', '-2.60683254419027318e-15', '-1.58284965090000260683254419027318');
    t('1.078440230361e-17', '-1.7739037620412430484912549e-12', '-1.7738929776389394384912549e-12');
    t('1.711170067502562238123015e+13', '2.259516522692505152e+18', '2259533634393180177.62238123015');
    t('-1.1997027e+5', '9e+0', '-119961.27');
    t('-6.7259257776601e+7', '5.960973996e-14', '-67259257.77660099999994039026004');
    t('2.85992183e+2', '-6.064117834028922826e+4', '-60355.18615728922826');
    t('7.934778e+6', '-9.4139862782877690738251378918e+28', '-9.413986278287769073824344414e+28');
    t('-6.263942705879383579638312222e+18', '3.2403266e+2', '-6263942705879383255.605652222');
    t('-2.329980632158e+5', '-1.08e-19', '-232998.063215800000000000108');
    t('-2.60095989544433e-14', '2.56e-5', '0.0000255999999739904010455567');
    t('-1.001038173511924331447439e+24', '-3e+0', '-1.001038173511924331447442e+24');
    t('3.68958378497693152e+6', '2.3811e+1', '3689607.59597693152');
    t('-7.20153952e-2', '-1.14726471710912029765e+1', '-11.5446625662912029765');
    t('4.66584406082341e+12', '-6.1465061983e+5', '4665843446172.79017');
    t('-7.088494e+5', '9.24436069228226918e+0', '-708840.15563930771773082');
    t('3.45085557305234282971e-8', '1.40482921012791804198897e+23', '1.404829210127918041988970000000345085557305234282971e+23');
    t('-2.8679709521880775607842e+21', '-5e+0', '-2.8679709521880775607892e+21');
    t('6.49984583317234152952e+17', '-2.9114532734232268763826e-20', '649984583317234152.951999999999999999970885467265767731236174');
    t('2.6e+0', '7.759622397664468658e+14', '775962239766449.4658');
    t('1.12464142221080461622674e-17', '4.39864275886353589274384e+23', '4.398642758863535892743840000000000000000112464142221080461622674e+23');
    t('-1.0485669677437140330013740955e+14', '5.37355740951565887959624082769e+5', '-104856696237015.662348571521590375917231');
    t('8.019083936288e+12', '2.1955080947893037e-5', '8019083936288.000021955080947893037');
    t('-6.2275998681220583433895e+8', '5.5575382458820654178885565e+23', '5.5575382458820591902886883779416566105e+23');
    t('-1e+0', '-1.8359922820027707246784e+6', '-1835993.2820027707246784');
    t('6.8845860200016e+9', '3.53338102476e+5', '6884939358.104076');
    t('9.89004923e-1', '9.5157252327953911020394e-15', '0.9890049230000095157252327953911020394');
    t('1.95366847613052662290905e-1', '4.61089e+1', '46.304266847613052662290905');
    t('1.52850891752130568e+1', '3.8790643093207712266859866e-16', '15.28508917521305718790643093207712266859866');
    t('1e+0', '-3.351589594322899987e-13', '0.9999999999996648410405677100013');
    t('4e+0', '-2.25635728971207153477e-17', '3.9999999999999999774364271028792846523');
    t('-7.1521e+0', '3e+0', '-4.1521');
    t('5.73130841108862822977457839205e-6', '-2.0226350480263253215e+10', '-20226350480.26324748369158891137177022542160795');
    t('2.38700916108487e+10', '4.6410211658583484277994722e+24', '4.6410211658583722978910830487e+24');
    t('-1.10178448051944844475771905272e-15', '-6.39395321758408376910624e+18', '-6393953217584083769.10624000000000110178448051944844475771905272');
    t('-3.013637e-14', '1.2228170261603e+3', '1222.81702616029996986363');
    t('5.701362229840513e+1', '-4.1266579254956732345e+1', '15.747043043448397655');
    t('-1.26549524e+4', '7.114747175586575423929900617e+4', '58492.51935586575423929900617');
    t('2e+0', '-6.19566611167700965419057181e-4', '1.999380433388832299034580942819');
    t('-1.59294729923996920972246e-17', '2.05841676458664250980624366171e+12', '2058416764586.6425098062436616940705270076003079027754');
    t('-3.611261022614801981543826e+5', '-5.4e+0', '-361131.5022614801981543826');
    t('-1.382436556533e+2', '-2.0005230510874070622654e+4', '-20143.474166527370622654');
    t('2.725e+1', '6.5907103031686003312192e+10', '65907103058.936003312192');
    t('1.7102043e+1', '1.630185891228e+5', '163035.6911658');
    t('1.06251e+1', '8.86362042342990151231977e+15', '8863620423429912.13741977');
    t('-2.0075903158e+3', '4.632965490155132e+15', '4632965490153124.4096842');
    t('1.57481186e+8', '1.9928e+2', '157481385.28');
    t('-4.4540881340084554082e+12', '-8.567827997230912244e-19', '-4454088134008.4554082000000000008567827997230912244');
    t('3.0149809456450947682607926e+26', '1.52e+4', '3.0149809456450947682609446e+26');
    t('3.55773e+1', '-1.296139601615599519887e-2', '35.56433860398384400480113');
    t('-2.88697527177e-7', '8.20473793e+9', '8204737929.999999711302472823');
    t('1.22209473e+8', '-2.071859795614941e+13', '-20718475746676.41');
    t('-4.29227481925874031992865289e+13', '-3.3e-4', '-42922748192587.4035292865289');
    t('-1.12e+0', '-3.656079060826e-5', '-1.12003656079060826');
    t('-5.6653957709162986e+0', '1.8713973e-1', '-5.4782560409162986');
    t('-7.598187175103e+12', '2.95663854017257699181848633626e+17', '295656255830082596.181848633626');
    t('-7.904686344760362698443617622e+19', '1.980883539639923746866474445e-3', '-79046863447603626984.434195336460360076253133525555');
    t('-1.887416922022595192028e+16', '-4.59e-20', '-18874169220225951.9202800000000000000459');
    t('-8.148255639724e+10', '2.891684342e+8', '-81193387963.04');
    t('6.23847007946890103e-19', '-8.4455421112902026239156794885e+9', '-8445542111.290202623915679487876152992053109897');
    t('-1.625684355377512651803808e+7', '-1.348895528074479183e+18', '-1348895528090736026.55377512651803808');
    t('-6.535179015515075206e+8', '-8.95738948608652714735e-10', '-653517901.551507521495738948608652714735');
    t('-8.153e-7', '1.65820611454448412e-13', '-8.15299834179388545551588e-7');
    t('5.051843689061792432745588e+1', '2.8243308408e+8', '282433134.59843689061792432745588');
    t('-1.8e+0', '9.44922368261817e+2', '943.122368261817');
    t('-4.6e+0', '1.636104301545649956e-6', '-4.599998363895698454350044');
    t('3.9366814707165145667343522e-12', '3.0389e+4', '30389.0000000000039366814707165145667343522');
    t('7.436644701432e+7', '2.6484854e+0', '74366449.6628054');
    t('6.072728e-6', '1.091e-13', '0.0000060727281091');
    t('-9.57628017436746e+6', '-5.39066706500255775e+13', '-53906680226305.75186746');
    t('4.09624190548576e-18', '-9.6749e+2', '-967.48999999999999999590375809451424');
    t('-8.6632e+3', '2.120751089392768905787515e+24', '2.1207510893927689057788518e+24');
    t('1.857536e+2', '6.75400474e+1', '253.2936474');
    t('-5.6e-18', '-2.0950506003186642599853e+3', '-2095.0506003186642599909');
    t('3.33788850991236125e+12', '-1.58862526437795e+14', '-155524637927882.63875');
    t('3.1483e+1', '2.28650378766881198016e+12', '2286503787700.29498016');
    t('1.234624954152162611e-19', '-7.4619712e+2', '-746.1971199999999999998765375045847837389');
    t('1.080098052386732090801e+17', '3.09363317193526857948304e-13', '108009805238673209.080100000000309363317193526857948304');
    t('-2.51e+0', '1.409033443601486029357e-19', '-2.5099999999999999998590966556398513970643');
    t('5.427853163e+2', '-1.29492978844337839542353e+23', '-1.294929788443378395418102146837e+23');
    t('-2.02e+2', '5.4765009943616783637932886e+25', '5.4765009943616783637932684e+25');
    t('6.0089899644e+8', '8.77139309e+7', '688612927.34');
    t('-2.52975e+2', '-8.063181606e-8', '-252.97500008063181606');
    t('4.5094208494899584309066e+0', '-1.103e+0', '3.4064208494899584309066');
    t('3.0782e+0', '8.6800615973178390115257e-7', '3.07820086800615973178390115257');
    t('1.242839052621980122206838e+18', '3.9218504671511828734123451441e+14', '1243231237668695240.49417923451441');
    t('3e+0', '9.0309539359826e+1', '93.309539359826');
    t('-7.79925520165e-12', '2.4e-16', '-7.79901520165e-12');
    t('-7.82454855580508259123083736e+4', '-1.4505891967290641474446448e+14', '-145058919751151.9003025153059123083736');
    t('-3.23795969916119488744e-19', '6.13973898934392e+6', '6139738.989343919999999999676204030083880511256');
    t('-3.2125638937067043056e+16', '6.26147244159010381015573613741e-5', '-32125638937067043.0559373852755840989618984426386259');
    t('-5.6388596462772628354639232512e+27', '-8.7402462294e+3', '-5.6388596462772628354639319914462294e+27');
    t('-5.18916166140020398584e+3', '-2.286072187548843381829615e+12', '-2286072192738.00504322981898584');
    t('4.07978e+3', '6.009589348894649131e+1', '4139.87589348894649131');
    t('1.24981916598699796711558874e+20', '-6.097e+0', '124981916598699796705.461874');
    t('3.40205966339e+9', '-5e+0', '3402059658.39');
    t('-2.011230243535947305604087574e-1', '8.09834868377e-6', '-0.2011149260049109605604087574');
    t('1.823304324438653941487094855e+6', '-4.755e-7', '1823304.324438178441487094855');
    t('1.984404125928134e-20', '-6.0867637168658727533522913e+25', '-6.086763716865872753352291299999999999999999998015595874071866e+25');
    t('2.15e-7', '1.40111e+5', '140111.000000215');
    t('1.8919e-20', '7.1771e+4', '71771.000000000000000000018919');
    t('9.95146e+3', '-3.9373688835252249931e+10', '-39373678883.792249931');
    t('-1.014255162652494738303244155e+7', '-2.67797715547e-5', '-10142551.62655172715458714155');
    t('-4.428e+0', '1.3437431538744606393264041272e+3', '1339.3151538744606393264041272');
    t('-2e+0', '-2.229409910360055e+4', '-22296.09910360055');
    t('4.0148389757764685838e+4', '3.07591618307944204e+15', '3075916183119590.429757764685838');
    t('1.5e+0', '3.204179e+2', '321.9179');
    t('1.7628168176292522230070264e-9', '-7.857162462055267e+5', '-785716.2462055249371831823707477769929736');
    t('-4.7068322333826667e+2', '2.6204796284915915378e+6', '2620008.94526825327113');
    t('-5.7849662e+6', '-1.67379927e+8', '-173164893.2');
    t('6.36341614538e+4', '-5.684708914970007106903e+11', '-568470827862.8392568903');
    t('-9.8170683973203026118362784916e-18', '-4.76099856150535148899823166528e+24', '-4.7609985615053514889982316652800000000000098170683973203026118362784916e+24');
    t('-1.450475441789747351396091722e-14', '9.5e-16', '-1.355475441789747351396091722e-14');
    t('-1.2401309543790470973e+12', '-3.283421840845701731666242363e-14', '-1240130954379.04709730000003283421840845701731666242363');
    t('3.8284741521518e+9', '1.040053292332476e+7', '3838874685.07512476');
    t('-3.629548186614033358094334214e+14', '3.88460431849184e-13', '-362954818661403.335809433421011539568150816');
    t('4.492353541896215e+4', '4.2278871477030354988345988e+12', '4227887192626.5709177967488');
    t('1.39125503887158081793020576e+12', '7.107400120480733e+4', '1391255109945.58202273753576');
    t('4.29162281457350835270745e-1', '-8.29e+0', '-7.860837718542649164729255');
    t('9.8480007978020274403298876e-14', '-9e+0', '-8.999999999999901519992021979725596701124');
    t('-6.065629590943247e+11', '2.1353939336e-5', '-606562959094.324678646060664');
    t('-1.90520093211683e-14', '2.6011329423597853e+3', '2601.1329423597852809479906788317');
    t('-5.258159e-8', '1.79707627e+3', '1797.07626994741841');
    t('4.39e-13', '3.909008772398642e-9', '3.909447772398642e-9');
    t('-5.5413711e+1', '1.72056051290943960725138e+22', '1.7205605129094396072458386289e+22');
    t('-2.653033e+6', '-2.3e+2', '-2653263');
    t('2.1e+0', '-1.03e-12', '2.09999999999897');
    t('4.5e+0', '5.57773e+5', '557777.5');
    t('-1.31252712310416354e-17', '-5.34133135e+6', '-5341331.3500000000000000131252712310416354');
    t('3.5452349200934969e+11', '8.46089262134099e+10', '439132418222.75959');
    t('-2.39151247297029e+0', '-3.4e+1', '-36.39151247297029');
    t('1.04933449400897990977690656e+16', '-2.0991e+0', '10493344940089796.9986690656');
    t('-2.3192197014588986e+10', '1.3582e-16', '-23192197014.58898599999999986418');
    t('7.142194053338425321066077e+5', '1.90025134243757962899e+1', '714238.4078472669079028976');
    t('-1.136854e-7', '9.38832e+5', '938831.9999998863146');
    t('3.32614494028772216103868273496e+27', '2.7733154489672324e+9', '3.3261449402877221638119981839272324e+27');
    t('8.53310299029659557937678946674e+29', '-2.171026229e-18', '8.53310299029659557937678946673999999999999999997828973771e+29');
    t('-8.7775827884702947387676403e+9', '-2.24368450892611e+10', '-31214427877.7313947387676403');
    t('-5.2404730569306871e-14', '-1.95066228095626569312714706e+26', '-1.95066228095626569312714706000000000000052404730569306871e+26');
    t('-1.3962539732462279142518e+9', '5.818337806672885323307992101e+8', '-814420192.5789393819210007899');
    t('-3e+0', '-3.758144520778459805048058852e+16', '-37581445207784601.05048058852');
    t('4.533812642503788216091413e-2', '0e+0', '0.04533812642503788216091413');
    t('2.2345667600317984449936934959e+28', '-2.312549676277765984005310857e+23', '2.23454363453503566733385344279143e+28');
    t('5.890444410932411361081252e+15', '3.886333434874616465e-9', '5890444410932411.361081255886333434874616465');
    t('7.0771471576147436232551549e+25', '3.31023407231546910086281e+14', '7.0771471576478459639783095910086281e+25');
    t('-4.579025795557e+9', '-9.412662879882994536113862e+19', '-94126628803408971156.69562');
    t('6.68976500113452229442881759e+16', '9.35885176e+2', '66897650011346158.8294641759');
    t('1.8481649705340526197387e+16', '-2.6037916468e+0', '18481649705340523.5935953532');
    t('4.57418412e+2', '1.2332172148052647e+6', '1233674.6332172647');
    t('1.11e+1', '-2.6e+0', '8.5');
    t('-2.1661e+3', '-3.7327283613e-10', '-2166.10000000037327283613');
    t('1.756224491703e-3', '4.5319398279654e+6', '4531939.829721624491703');
    t('-1.34077579411e+7', '8.24170643747189e+13', '82417050966960.9589');
    t('-2.0495814528228973658098e+10', '-5.20314966589367e-15', '-20495814528.22897365809800520314966589367');
    t('8.541190743467170007744156375e-13', '7.0401506297973216018710560963e+28', '7.04015062979732160187105609630000000000008541190743467170007744156375e+28');
    t('-9.61633628e+7', '-3.613868352359377835e+11', '-361482998598.7377835');
    t('-2.767001037506171092080557112e+14', '9.244954929589141444423604e-17', '-276700103750617.10920805571119990755045070410858555576396');
    t('1.89901678657288039300855105894e-13', '-1.58598406112309192413155018311e+25', '-1.5859840611230919241315501831099999999810098321342711960699144894106e+25');
    t('1.268441520208713701742930596e+15', '4.970632009043856853119754828e+18', '4971900450564065566.821497758596');
    t('-1.9274080087e-7', '-1.226582591e+3', '-1226.58259119274080087');
    t('7.6027378901641e+4', '5.6020622089981e+2', '76587.58512254081');
    t('-8.955651e+4', '-2.311647517004775402486236216e+27', '-2.31164751700477540248632577251e+27');
    t('-8.640119777677045398616e+19', '7.1041062169678654e-10', '-86401197776770453986.15999999928958937830321346');
    t('-4.9472e+1', '-9.5254025997e+4', '-95303.497997');
    t('4.9115e+4', '1.1091035133e+10', '11091084248');
    t('2.00178779e-3', '2.3e+0', '2.30200178779');
    t('-1.22635159837898393566542487e+7', '7.6174490056742349880855e-11', '-12263515.983789839280479758643257650119145');
    t('5e+0', '4.83e+0', '9.83');
    t('-5.18305e+1', '2.06741e+0', '-49.76309');
    t('4.732184851555942426024763e-20', '1.536847890671368e+15', '1536847890671368.00000000000000000004732184851555942426024763');
    t('1.88358776282573960911e+15', '-5.5178544e-9', '1883587762825739.6091099944821456');
    t('-7.250860069e+9', '-2.0722613389416e-6', '-7250860069.0000020722613389416');
    t('-9.37492384e-16', '-3.59438537290428941803686306e+26', '-3.59438537290428941803686306000000000000000937492384e+26');
    t('2.29612282112427409597796e+6', '-2.44261697753e+0', '2296120.37850729656597796');
    t('-4.410025312651288094581277836e+21', '-1.448656017e-16', '-4.4100253126512880945812778360000000001448656017e+21');
    t('-1.7181773728e+9', '-1.896321199e+7', '-1737140584.79');
    t('4.3098015497657959948133e+20', '1.274986092270260964754410066e+11', '430980155104078208708.3560964754410066');
    t('-2.1959570176e+10', '5.725576310292088e-18', '-21959570175.999999999999999994274423689707912');
    t('-1.44792248534417343390548623e+21', '1e+0', '-1.44792248534417343390448623e+21');
    t('1.040704879928354425713e+2', '4.86388848511334545e+2', '590.4593365041699875713');
    t('4.241468133452732043755761186e+16', '-2.73798399968495e+11', '42414407536127351.94255761186');
    t('-2e+0', '6.24224662705e+11', '624224662703');
    t('5.6641294006148610219533163e+0', '-2.12295174799e+11', '-212295174793.3358705993851389780466837');
    t('2.865e+0', '-4.095879034404159898232e-16', '2.8649999999999995904120965595840101768');
    t('1.897905e+1', '6.596474438004821e+15', '6596474438004839.97905');
    t('2.8032755954946389580459726817e+0', '-3.504845500434e+4', '-35045.6517287445053610419540273183');
    t('3.2100930836673e+7', '1.7556652634247164889788e+3', '32102686.5019364247164889788');
    t('3.94508726739917261971194668167e+20', '3.2678976e-12', '394508726739917261971.1946681670032678976');
    t('1.013478996e+6', '-1.35851240932e+11', '-135850227453.004');
    t('5.76471020553517e+10', '9.361663010412251704182487e+1', '57647102148.96833010412251704182487');
    t('8.039198206551479e-19', '-4.7304610153534734555e+11', '-473046101535.3473455499999999991960801793448521');
    t('1.08616611370746e+10', '1.63357530853972701257283441042e+29', '1.633575308539727012681451021790746e+29');
    t('8.985857919040188655e+18', '6.9122771945490484660606087488e+22', '6.9131757803409524849261087488e+22');
    t('8.55516146e+6', '-5.20868e-6', '8555161.45999479132');
    t('1.35007756418551e+5', '2.1871207843576e-10', '135007.75641855121871207843576');
    t('1.997393922577102e+15', '2.04297682524767e+14', '2201691605101869');
    t('-2.8299e+0', '-2.0533e+1', '-23.3629');
    t('9.801120776319968546478e-1', '-5.506831779512856926053754757e-6', '0.980106570800217341790873946245243');
    t('4.7906337242e+7', '-3.024012845303e+0', '47906334.217987154697');
    t('-1.27217277089e+6', '-2.5964435963219617e+8', '-260916532.40308617');
    t('-9.42191e-10', '2.844075292805153735859165701e+3', '2844.075292804211544859165701');
    t('-2.6967949047958223297482e+19', '9.67282581408114114e+13', '-26967852319700082486.0706');
    t('-1.1745879912966672172374876e+14', '-5.942752909258700258e-14', '-117458799129666.72172374876005942752909258700258');
    t('1.17492331131707e+14', '-1.85e+2', '117492331131522');
    t('3.8237936883565271658e-7', '-7.468507e+1', '-74.68506961762063116434728342');
    t('2.080597621127443899864e+7', '2.0363911e+2', '20806179.85038443899864');
    t('-1.317672639296246e+11', '-3.47791814748480321636668116e+26', '-3.477918147484804534039320456246e+26');
    t('4.02924099e+1', '5.28393737349807e+9', '5283937413.7904799');
    t('3.62917205826288627489788295e+18', '2.49970474597037787872e+12', '3629174557967632245.27576167');
    t('3.08272649390034127669268182e-2', '3e+0', '3.0308272649390034127669268182');
    t('5.85719579984856882552621456e+26', '-2e+0', '5.85719579984856882552621454e+26');
    t('-2.50225156633622568317671509e-9', '-2.2582829e+2', '-225.82829000250225156633622568317671509');
    t('-5.15193401834588205516587050994e-14', '1.4e+0', '1.3999999999999484806598165411794483412949006');
    t('1e+0', '8.8924e+0', '9.8924');
    t('-3.3e+0', '-7.8502049490351637397e-8', '-3.300000078502049490351637397');
    t('-7.33960473805391e+11', '7.439838102434947302e+0', '-733960473797.951161897565052698');
    t('-3.16e-5', '1.726737549992789265e+13', '17267375499927.8926184');
    t('4.270228010774903285941e+15', '3.46096007736828546415995945e+26', '3.46096007741098774426770848285941e+26');
    t('-2.10368325850606703058e+20', '-2.08133369e+0', '-210368325850606703060.08133369');
    t('1.7e+1', '-1.0122120905476e+13', '-10122120905459');
    t('-7.07684131786e+1', '2.63774583435255785325881638e+25', '2.63774583435255785325880930315868214e+25');
    t('9.1072366943654984307009963e+25', '1.439998771392437399072989e+2', '9.10723669436549843070101069998771392437399072989e+25');
    t('-2.291554e-12', '2.61075274155651389913495756e-20', '-2.2915539738924725844348610086504244e-12');
    t('1.7602843372492822519e+15', '-8.1404508102e+7', '1760284255844774.1499');
    t('-4e+0', '-1.944878631711881888860126e+0', '-5.944878631711881888860126');
    t('-2.187281084011878193e+17', '-7.474048902432652904641e-13', '-218728108401187819.3000000000007474048902432652904641');
    t('3.2e-19', '3.681598265268061504382107e+7', '36815982.65268061504382107032');
    t('-7.99963142249e+5', '-1.10759709047786981579e+17', '-110759709048586944.721249');
    t('8.6126004e+7', '-2.260854106319155e+12', '-2260767980315.155');
    t('-1.545857065710954713345527676e+7', '1.693211571e+2', '-15458401.33595244713345527676');
    t('1.7634085229179948402e-1', '-7.72257679334452942471286361e+10', '-77225767933.26895339483683661598');
    t('-4.09737779175073422506155472749e+17', '-9.371187455652828779751905571e-19', '-409737779175073422.5061554727490000009371187455652828779751905571');
    t('2.94436782641191186e+1', '-1.757436041e+0', '27.6862422231191186');
    t('1e+0', '6.393155e+4', '63932.55');
    t('6.94692049e-4', '1.1351884189267733e+13', '11351884189267.733694692049');
    t('-6.30461698646036622857426e-5', '-4.09097e+1', '-40.9097630461698646036622857426');
    t('-2.0402e-20', '1.67107872884e+7', '16710787.288399999999999999979598');
    t('5.532395277106440788e+9', '-7.24995e+5', '5531670282.106440788');
    t('6.000727e-18', '6.917267560158418e+11', '691726756015.841800000000000006000727');
    t('1.166055456823e+12', '-5.5020976743949e+12', '-4336042217571.9');
    t('-3.91173319151e-2', '4.41620271855e-17', '-0.0391173319150999558379728145');
    t('4.91856485935137e-1', '-1.08457423464500916876e+19', '-10845742346450091687.108143514064863');
    t('2.09824848244684254866384581e+26', '3.638246794566e+10', '2.0982484824468429124885252666e+26');
    t('-3.2e-8', '-7.140239683783e-17', '-3.200000007140239683783e-8');
    t('1.2946e+4', '6.4275853487855044216637711965e+27', '6.4275853487855044216637841425e+27');
    t('-2.31e+1', '-5.1e+0', '-28.2');
    t('-5.98100271250578949179e+8', '2.012527542898146781797867e+2', '-598100069.9978246593643218202133');
    t('-2.2616174364e+6', '1.808741032320373831865e+21', '1.8087410323203715702475636e+21');
    t('-1.59844056581e+11', '-3.677066761421338e-2', '-159844056581.03677066761421338');
    t('-3.070121910623175279e-4', '-8.191845955248936529857e+0', '-8.1921529674399988473849');
    t('3e+0', '-3.131384399e+0', '-0.131384399');
    t('5.493193351746495014e+9', '1.0058110782705526011700063e+10', '15551304134.452021025700063');
    t('3.7662480770326031342303292983e+26', '-1.532179017e+5', '3.766248077032603134228797119283e+26');
    t('2.2078732616074486433852e+19', '-8.0117272927216e+5', '22078732616073685261.12272784');
    t('-1.101469921604903937413697e+24', '-5.6493478665614358249763e+22', '-1.15796340027051829566346e+24');
    t('4.002068393e+2', '3.88674498621732e-19', '400.206839300000000000388674498621732');
    t('1.18924625762106479751351e+1', '2.12367151349326e-7', '11.8924627885777993244611');
    t('6.18899636825e+1', '-1.34517400616695669045e-3', '61.88861850849383304330955');
    t('-1.727166e+4', '-1.232546640708229439e-7', '-17271.6600001232546640708229439');
    t('-7.39e-16', '2.158174383154757e-7', '2.158174375764757e-7');
    t('5.95645368014790753037091178681e-18', '8.9194691702199e+13', '89194691702199.00000000000000000595645368014790753037091178681');
    t('2.4544990425537516895759e+6', '1.2777961e+5', '2582278.6525537516895759');
    t('-5.185388510885e+1', '-7.693787224e-8', '-51.85388518578787224');
    t('-2.7398396455549889037919557e+23', '-4.220983829023255e+0', '-2.73983964555498890379199790983829023255e+23');
    t('1.02153991719e+1', '9.3244708599174510285e-9', '10.2153991812244708599174510285');
    t('4.60472064169766494e+16', '-1.0328246092260039e+0', '46047206416976648.3671753907739961');
    t('-2.38454077437299e+14', '-4e+0', '-238454077437303');
    t('8.370371680400711505807e+17', '7.56e-7', '837037168040071150.580700756');
    t('2.10262406584177455194e+6', '-7.99673704574e+1', '2102544.09847131715194');
    t('1.35918051082554629399892854005e+7', '-9.05428949025616791486522594e+1', '13591714.5653605603783101367482406');
    t('4.19333658548894567e+0', '1e+0', '5.19333658548894567');
    t('-1.12684363429737223640594383707e+22', '8.25876483519322470373806e+21', '-3.0096715077804976603213783707e+21');
    t('0e+0', '6.04e+2', '604');
    t('1.8364537e+2', '1.0468696299566552286716e+22', '1.046869629956655228689964537e+22');
    t('-5.073984e+6', '6.9887615721504817196587e+8', '693802173.21504817196587');
    t('2.5457e+2', '1.341466050473471653464e+9', '1341466305.043471653464');
    t('5.448451501591168054051559931e+5', '6.906967417242e-6', '544845.1501660237728223979931');
    t('-1.194671553787672e+1', '-1.91896603190801227e+3', '-1930.91274744588899');
    t('1.279395975361696355442e+8', '-6.96569613221558540371e+7', '58282636.2140137815071');
    t('-5.97073932850348e-7', '-2.2291090891563813e+4', '-22291.090892160886932850348');
    t('1.464561173e+9', '1e+0', '1464561174');
    t('-9.4963284032193e+3', '6.16644216458418847e+10', '61664412149.5134814807');
    t('-5.4865356898608376e+9', '5e+0', '-5486535684.8608376');
    t('1.807335611418e+5', '0e+0', '180733.5611418');
    t('4.79648059622793e+11', '-5.79415144368e+3', '479648053828.64155632');
    t('-1.19816450616e+11', '-5.716149095e-7', '-119816450616.0000005716149095');
    t('-2.3561102073882235262820415e-17', '-1.608713271019924e-3', '-0.001608713271019947561102073882235262820415');
    t('4.34065400608679942e+8', '-6.6465252539670797297214992789e+4', '433998935.356140271202702785007211');
    t('3.0930179144e+10', '0e+0', '30930179144');
    t('5.366042e+2', '2.270835328764610647e+18', '2270835328764611183.6042');
    t('7.48097472370778786852679e-7', '9.4767671341211803315002015e+26', '9.47676713412118033150020150000000748097472370778786852679e+26');
    t('-8.23842e+4', '1.229458568566523723615668197597e+14', '122945856774268.1723615668197597');
    t('-8.7862458629211487549e+19', '-1.1019849690697e+0', '-87862458629211487550.1019849690697');
    t('0e+0', '1.089862696164931e+15', '1089862696164931');
    t('-1.4492938845e+8', '-1.42921263881025236036504955e+26', '-1.4292126388102523618143434345e+26');
    t('-1.6381521194899074907e+6', '5.6572651171333e-11', '-1638152.119489907434127348828667');
    t('-4.396054117495148392356145377e+21', '-1.829550398438512721241206e+17', '-4.3962370725349922436282694976e+21');
    t('3.742420087218951e+10', '-1.730936977e+2', '37424200699.0958123');
    t('-1.1285098282e+2', '3.0174920978669756069e-1', '-112.54923361021330243931');
    t('3.328692e-19', '-3.74463079e+10', '-37446307899.9999999999999999996671308');
    t('1.28232894392211733573999076556e+29', '1.149729331569322243219279e+20', '1.282328945071846667309313008779279e+29');
    t('1.07034e+2', '1.4997e+1', '122.031');
    t('-2.76584099e-19', '-3.3993789e-8', '-3.3993789000276584099e-8');
    t('-4.60271074562e+11', '-3.45574059291359666323689575e+15', '-3456200863988158.66323689575');
    t('1.73104292971e+10', '7.220485847821601e+10', '89515287775.31601');
    t('2.26079231553066e+14', '-3.429835324275129e+15', '-3203756092722063');
    t('4.909949450328607351257e+9', '1.693032494154686e+15', '1693037404104136.328607351257');
    t('2.804004587e+1', '3.054e-4', '28.04035127');
    t('-1.21373e+4', '-1.7051357521743463e+13', '-17051357533880.763');
    t('-2.010863577144665818021832751e+17', '-1.1990113e+7', '-201086357726456694.8021832751');
    t('-7.1235473e+6', '-3.145986004148e+9', '-3153109551.448');
    t('1.3665398036864664266039e+22', '4.089516e-2', '1.366539803686466426603904089516e+22');
    t('1.727799e+0', '-4.8149925e+4', '-48148.197201');
    t('6.6630744630002551658489e+11', '-5.8410410897522728193e+12', '-5174733643452.24730271511');
    t('2.0767364695045007623516126e+11', '6.25410192352696e+8', '208299057142.80277223516126');
    t('7.75e+0', '-1.569335603101854838490073823e-12', '7.749999999998430664396898145161509926177');
    t('5e+0', '-7.431618517651602908e+18', '-7431618517651602903');
    t('-1.5453e+3', '-3.9000399435568029e-6', '-1545.3000039000399435568029');
    t('-7.0748737648616353693e+0', '2.591961010757e+5', '259189.0262019351383646307');
    t('1.651e+4', '-2.8703125534636e+9', '-2870296043.4636');
    t('3.087702748515e-12', '8.3067008464351e+3', '8306.700846435103087702748515');
    t('7.7880835734433871606349974196e+9', '2.82154e+4', '7788111788.8433871606349974196');
    t('-1.9482797447534242447e+5', '4.172181002194178915e-13', '-194827.9744753424240527818997805821085');
    t('1.23e+0', '1.22e+0', '2.45');
    t('1.692421753e-9', '8.9292167159609532663074588e+4', '89292.167159611225084827588');
    t('-4.003138001375e+6', '-3.6225528729195e+13', '-36225532732333.001375');
    t('-1.043499514e+8', '1.85379394110541607112156303315e+27', '1.85379394110541607101721308175e+27');
    t('1.2e+0', '-2.2e+0', '-1');
    t('-4.59964147358e+0', '-1.245052236142679713264875449e+25', '-1.245052236142679713264875908964147358e+25');
    t('1.435745816367079985483e-17', '-2.2506199410005066316855e-13', '-2.2504763664188699236869517e-13');
    t('3.9212949955518e+12', '-6.253e+2', '3921294994926.5');
    t('3.272898632836139464918e+14', '-1.125321770879687081334515545e+6', '327289862158292.175612112918665484455');
    t('-7.0051742181e+0', '-3.558851591145309897e+6', '-3558858.596319527997');
    t('-1.17791e-5', '-2.8283187205369969776764952883e+28', '-2.82831872053699697767649528830000117791e+28');
    t('8.6191671e-12', '-8.66936689467088e+10', '-86693668946.7087999999913808329');
    t('3.67300220532035e-3', '-2.20592e+2', '-220.58832699779467965');
    t('5.307907765699952590606427e+24', '-8.116566770561e+12', '5.307907765691836023835866e+24');
    t('1.3693156074977e+1', '-1.86125e+3', '-1847.556843925023');
    t('-1.305776227522e+0', '3.1e+0', '1.794223772478');
    t('-5.8e+0', '-4.2864143878e-14', '-5.800000000000042864143878');
    t('7.3171907466448357479336251741e-7', '-3.6643373621770251e-1', '-0.36643300449862784551642520663748259');
    t('9.5507944480252871732e+8', '-1.473719301517056667482e+7', '940342251.78735815064518');
    t('-1.594802356375808285e+18', '2.2091549349442255368551009e+23', '2.2091389869206617787722509e+23');
    t('-1.38180012856133843768288e+23', '-1.4881225686551882e+12', '-1.381800128576219663369431882e+23');
    t('3.9944940533448e-11', '-1.42182826302e+9', '-1421828263.019999999960055059466552');
    t('2.1246e+0', '-1.190013e-18', '2.124599999999999998809987');
    t('-1.6725888942e+5', '5.22591577449581327e+5', '355332.688029581327');
    t('6.2523356350387526825216271e-1', '5.8015517e+6', '5801552.32523356350387526825216271');
    t('-3.5042195675619e+11', '-1.61923859937529243154197361e+19', '-16192386344174881071.6097361');
    t('5.326226512685426512311428421e+15', '-2.14826116e+2', '5326226512685211.686195428421');
    t('-2.033908204665857e-20', '5.0334987e-20', '2.999590495334143e-20');
    t('4.92122690634e+11', '-1e+0', '492122690633');
    t('-1.850639822486488995e+3', '2.2247255806523203254e+6', '2222874.940829833836405');
    t('-1.4652329507e-8', '-2.2e+0', '-2.200000014652329507');
    t('1.031983768655972e+3', '3.196384720063040238523e+7', '31964879.18439905835723');
    t('-3.95755151236389821e+0', '8.219752462201035868e+15', '8219752462201031.91044848763610179');
    t('4.960062e-7', '-4.48e-18', '4.9600619999552e-7');
    t('8.424039578e+8', '-1.3745220992603451351702e+8', '704951747.87396548648298');
    t('5.306671354863915213e-8', '9.108152565584071569e-2', '0.09108157872255426432915213');
    t('-1.469495615534e+1', '4.847e-3', '-14.69010915534');
    t('3.5853e+3', '1.7093298564986700250582085e+15', '1709329856502255.3250582085');
    t('5.482e+2', '-4.105575442089757e+15', '-4105575442089208.8');
    t('1.99e+2', '3.8145468189e-3', '199.0038145468189');
    t('-3.49258722592820006912e-2', '-1.949843151114522843231e+2', '-195.0192409837115663237912');
    t('-1.1936922e+2', '1.94667093809609e+10', '19466709261.59168');
    t('3.0612779617e+9', '5.5021749870968147533e-4', '3061277961.70055021749870968147533');
    t('1.6469577765954843637144527776e+25', '-3.53617804087992991e+1', '1.64695777659548436371444924142195912007009e+25');
    t('5.762086232617e+3', '-1e+0', '5761.086232617');
    t('-2.085492256313382505e+20', '2.73885062e-11', '-208549225631338250499.9999999999726114938');
    t('-1.0504694638041059389625e+14', '1.2117447209021816889456469472e+27', '1.21174472090207664199926653660610375e+27');
    t('3.266124155553289418147501e+25', '-1.196361921508326229864197918e+8', '3.26612415555328940618388178491673770135802082e+25');
    t('-3.681434944093545137202e-11', '-9.8172931644e-9', '-9.85410751384093545137202e-9');
    t('-5.5110319265e+9', '2.511e+2', '-5511031675.4');
    t('2.24384830388360906481037e+4', '-6.688e+4', '-44441.5169611639093518963');
    t('1.30685382594601602875155e-13', '-2.9e-7', '-2.89999869314617405398397124845e-7');
    t('1.256635121672088721271099e+24', '9.2e+0', '1.2566351216720887212711082e+24');
    t('4.25e+1', '2.097673647e+5', '209809.8647');
    t('-1.0834844778746335e+4', '5.724238385893731161972e-7', '-10834.8447781739111614106268838028');
    t('-1.3622153693079e+13', '-1.82203795563644950428091436626e+29', '-1.82203795563644964050245129705e+29');
    t('6.124567392189732e-10', '2.086068109112e-5', '0.0000208612935478592189732');
    t('-1.01e-15', '-1.6428548827105e+6', '-1642854.88271050000000101');
    t('-4.277650516254732251e-20', '5.58481757486e+1', '55.84817574859999999995722349483745267749');
    t('-3.9947160422255e+13', '1.230577e+0', '-39947160422253.769423');
    t('-6.97056460621311741e+17', '-1.19084697861e-14', '-697056460621311741.0000000000000119084697861');
    t('1.99e+0', '2.432670941489e+2', '245.2570941489');
    t('2.8e+0', '-6e-13', '2.7999999999994');
    t('1.48e+1', '-4.493331507903141119919e+20', '-449333150790314111977.1');
    t('1.5212827e+1', '2.37852664031269644800630723e+6', '2378541.85313969644800630723');
    t('-8.68972e+3', '-3.409629e+3', '-12099.349');
    t('8.7e+0', '1.053877931698399708032024498e+3', '1062.577931698399708032024498');
    t('-5.569516602e+2', '-7.674336919779554845819e+9', '-7674337476.731215045819');
    t('-2.33172e+4', '-1.1357215e+6', '-1159038.7');
    t('-7.728437483e+4', '-1.954891686102566032254774334e-1', '-77284.5703191686102566032254774334');
    t('-1.794108984118418586e+19', '4.29563121585293316012291166e+26', '4.29563103644203474828105306e+26');
    t('-2.5159616882775e+7', '0e+0', '-25159616.882775');
    t('-1.1497021250372501e+2', '8.3582955257533834133933719723e+22', '8.358295525753383413381874951049627499e+22');
    t('2.8314056900555843e+12', '4.23695829388274243636e-6', '2831405690055.58430423695829388274243636');
    t('-1.31709787273518408e+17', '8.014510794225900917778e+4', '-131709787273438262.89205774099082222');
    t('0e+0', '-1.629e+2', '-162.9');
    t('7.65253317420838491672142405e-2', '-7.99262139691085667872536301e+15', '-7992621396910856.6022000312679161508327857595');
    t('1.890960123000345821183126769e+14', '7.20788924150109e+9', '189103220189276.0832083126769');
    t('-1.3e+0', '8.319553612816369962e+7', '83195534.82816369962');
    t('-3.5558858e+7', '-8.10954e-10', '-35558858.000000000810954');
    t('-6.59635684e-11', '-1.12991511439649957747410439089e+29', '-1.129915114396499577474104390890000000000659635684e+29');
    t('2.84985498518035e+0', '-2.69695005783e+4', '-26966.65072331481965');
    t('-6.85881760429450007919e+20', '-4.2520891329373816896808041239e-11', '-685881760429450007919.000000000042520891329373816896808041239');
    t('-7.60611902357978e+5', '0e+0', '-760611.902357978');
    t('-2.8400853393896819e-12', '3.9346602572408205959634e+10', '39346602572.4082059596311599146606103181');
    t('-6.2918148769550413935353223e-6', '-1.46926127598445193e+7', '-14692612.7598508111148769550413935353223');
    t('-6.6e+0', '5.29e+2', '522.4');
    t('-5.32901866812117076e+16', '5.923499732512e+7', '-53290186621976710.27488');
    t('2.8826366994411923511e-4', '9.0827607747846942e+14', '908276077478469.42028826366994411923511');
    t('-1.16063858058205683e+17', '1.63197623001574371048e+1', '-116063858058205666.6802376998425628952');
    t('2.33448774537387e-8', '-2.344769e+1', '-23.4476899766551225462613');
    t('5.221324777342731268951204567e+27', '5.2831642468607302614700095e+25', '5.274156419811338571565904662e+27');
    t('3.176941232598488925334e+19', '-5.78631858983696974141e+20', '-546862446657712084887.66');
    t('-9.83148865e-8', '2.162567052649e-8', '-7.668921597351e-8');
    t('1.45780087267250652143848567879e+17', '2.31e+1', '145780087267250675.243848567879');
    t('1.938157e+6', '2.60157906e+1', '1938183.0157906');
    t('2.428e-6', '-5.322e+3', '-5321.999997572');
    t('8.2046484844e+3', '2.1549373877917e+14', '215493738787374.6484844');
    t('-8.834664968e+2', '-2.55430491293e+5', '-256313.9577898');
    t('2.3572045735847686172e+12', '-1.3818314700215998e+16', '-13815957495642413.2313828');
    t('-3.2657953413e-16', '4.1453273974121236811839e-18', '-3.224342067325878763188161e-16');
    t('3.9e+1', '9.93455000207993341247941942e-15', '39.00000000000000993455000207993341247941942');
    t('9.37900728e+0', '-7.836904346001529841e-17', '9.37900727999999992163095653998470159');
    t('-3.0891e+0', '-9.719267672741339831199e+21', '-9.7192676727413398312020891e+21');
    t('1.76267171902379081421e-4', '4.274711452739918e+15', '4274711452739918.000176267171902379081421');
    t('1.3371e-5', '-2.337338537516546e+15', '-2337338537516545.999986629');
    t('5.1552633575093e+3', '-4.78079e-3', '5155.2585767193');
    t('-5.5504722943333778362e+19', '8.8121340924893706238e+8', '-55504722942452564952.75106293762');
    t('-9.56289179492575435222173761e+9', '-1.5507589407486e+14', '-155085456966654.92575435222173761');
    t('1.813253984710071e+8', '-1.19293200963040761850166359388e+11', '-119111875564.569754750166359388');
    t('4.813328953637197259781466269e-12', '-4.47e+0', '-4.469999999995186671046362802740218533731');
});
