/*
 * Perform an ALPS INVENTORY and print out node attributes. This has the
 * disadvantage of not showing the allocation mode (interactive/batch/other).
 *
 * Nodes are printed in the currend ALPS_NIDORDER, which corresponds to
 * 'apstat -no' and 'apbasil -Q'.
 */
#include "basil_alps.h"

/* Print recursively to revert the LIFO order */
static void print_node(struct basil_node *node)
{
	struct basil_segment *seg;
	enum basil_proc_type arch = BPT_NONE;
	uint8_t total_cores = 0;
	uint32_t total_mem = 0;
	uint32_t cmhz;

	if (!node)
		return;
	print_node(node->next);

	/* FIXME: assuming uniform system for now */
	for (seg = node->seg_head; seg; seg = seg->next) {
		struct basil_node_processor *proc;
		struct basil_node_memory *mem;
		struct basil_label *label;
		uint8_t max_ordinal = 0;

		for (proc = seg->proc_head; proc; proc = proc->next) {
			if (proc->ordinal > max_ordinal)
				max_ordinal = proc->ordinal;
			arch = proc->arch;
			cmhz = proc->clock_mhz;
		}
		total_cores += max_ordinal + 1;

		for (mem = seg->mem_head; mem; mem = mem->next)
			total_mem += mem->page_size_kb * mem->page_count;

		for (label = seg->lbl_head; label; label = label->next)
			printf("%10s-%10s-%10s", label->name,
				nam_labeltype[label->type],
				nam_ldisp[label->disp]);
	}

	/*
	 * Note: In Basil 1.0 - 3.1, the Inventory response only lists
	 *       compute nodes, hence the test below (batch/interactive)
	 *       is only there to catch unexpected cases.
	 */
	printf("%6u %#8x %11s %8s %5d %8d %8d\n",
		node->node_id, node->node_id, node->name,
		(node->role == BNR_BATCH ||
		 node->role == BNR_INTER) ? "compute" : "UNKNOWN",
		total_cores, cmhz, total_mem >> 10);
}

int main(int ac, char **av)
{
	enum basil_version version  = get_basil_version();
	struct basil_inventory *inv = get_full_inventory(version);

	if (inv == NULL)
		err(1, "allocation of inventory data failed");
	else if (!inv->f->node_head)
		errx(0, "no node data");

	printf("   NID    (HEX)    NODENAME     TYPE CORES CLOCKMHZ AVAILMEM\n");
	print_node(inv->f->node_head);

	return EXIT_SUCCESS;
}
